/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.deadline.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.deadline.DeadlineAsyncClient;
import software.amazon.awssdk.services.deadline.model.DeadlineRequest;
import software.amazon.awssdk.services.deadline.model.GetFleetRequest;
import software.amazon.awssdk.services.deadline.model.GetFleetResponse;
import software.amazon.awssdk.services.deadline.model.GetJobRequest;
import software.amazon.awssdk.services.deadline.model.GetJobResponse;
import software.amazon.awssdk.services.deadline.model.GetLicenseEndpointRequest;
import software.amazon.awssdk.services.deadline.model.GetLicenseEndpointResponse;
import software.amazon.awssdk.services.deadline.model.GetQueueFleetAssociationRequest;
import software.amazon.awssdk.services.deadline.model.GetQueueFleetAssociationResponse;
import software.amazon.awssdk.services.deadline.model.GetQueueRequest;
import software.amazon.awssdk.services.deadline.model.GetQueueResponse;
import software.amazon.awssdk.services.deadline.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultDeadlineAsyncWaiter implements DeadlineAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final DeadlineAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<GetFleetResponse> fleetActiveWaiter;

    private final AsyncWaiter<GetJobResponse> jobCreateCompleteWaiter;

    private final AsyncWaiter<GetLicenseEndpointResponse> licenseEndpointDeletedWaiter;

    private final AsyncWaiter<GetLicenseEndpointResponse> licenseEndpointValidWaiter;

    private final AsyncWaiter<GetQueueFleetAssociationResponse> queueFleetAssociationStoppedWaiter;

    private final AsyncWaiter<GetQueueResponse> queueSchedulingWaiter;

    private final AsyncWaiter<GetQueueResponse> queueSchedulingBlockedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultDeadlineAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = DeadlineAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.fleetActiveWaiter = AsyncWaiter.builder(GetFleetResponse.class).acceptors(fleetActiveWaiterAcceptors())
                .overrideConfiguration(fleetActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.jobCreateCompleteWaiter = AsyncWaiter.builder(GetJobResponse.class).acceptors(jobCreateCompleteWaiterAcceptors())
                .overrideConfiguration(jobCreateCompleteWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.licenseEndpointDeletedWaiter = AsyncWaiter.builder(GetLicenseEndpointResponse.class)
                .acceptors(licenseEndpointDeletedWaiterAcceptors())
                .overrideConfiguration(licenseEndpointDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.licenseEndpointValidWaiter = AsyncWaiter.builder(GetLicenseEndpointResponse.class)
                .acceptors(licenseEndpointValidWaiterAcceptors())
                .overrideConfiguration(licenseEndpointValidWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.queueFleetAssociationStoppedWaiter = AsyncWaiter.builder(GetQueueFleetAssociationResponse.class)
                .acceptors(queueFleetAssociationStoppedWaiterAcceptors())
                .overrideConfiguration(queueFleetAssociationStoppedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.queueSchedulingWaiter = AsyncWaiter.builder(GetQueueResponse.class).acceptors(queueSchedulingWaiterAcceptors())
                .overrideConfiguration(queueSchedulingWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.queueSchedulingBlockedWaiter = AsyncWaiter.builder(GetQueueResponse.class)
                .acceptors(queueSchedulingBlockedWaiterAcceptors())
                .overrideConfiguration(queueSchedulingBlockedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<GetFleetResponse>> waitUntilFleetActive(GetFleetRequest getFleetRequest) {
        return fleetActiveWaiter.runAsync(() -> client.getFleet(applyWaitersUserAgent(getFleetRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetFleetResponse>> waitUntilFleetActive(GetFleetRequest getFleetRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return fleetActiveWaiter.runAsync(() -> client.getFleet(applyWaitersUserAgent(getFleetRequest)),
                fleetActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetJobResponse>> waitUntilJobCreateComplete(GetJobRequest getJobRequest) {
        return jobCreateCompleteWaiter.runAsync(() -> client.getJob(applyWaitersUserAgent(getJobRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetJobResponse>> waitUntilJobCreateComplete(GetJobRequest getJobRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return jobCreateCompleteWaiter.runAsync(() -> client.getJob(applyWaitersUserAgent(getJobRequest)),
                jobCreateCompleteWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetLicenseEndpointResponse>> waitUntilLicenseEndpointDeleted(
            GetLicenseEndpointRequest getLicenseEndpointRequest) {
        return licenseEndpointDeletedWaiter.runAsync(() -> client
                .getLicenseEndpoint(applyWaitersUserAgent(getLicenseEndpointRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetLicenseEndpointResponse>> waitUntilLicenseEndpointDeleted(
            GetLicenseEndpointRequest getLicenseEndpointRequest, WaiterOverrideConfiguration overrideConfig) {
        return licenseEndpointDeletedWaiter.runAsync(
                () -> client.getLicenseEndpoint(applyWaitersUserAgent(getLicenseEndpointRequest)),
                licenseEndpointDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetLicenseEndpointResponse>> waitUntilLicenseEndpointValid(
            GetLicenseEndpointRequest getLicenseEndpointRequest) {
        return licenseEndpointValidWaiter.runAsync(() -> client
                .getLicenseEndpoint(applyWaitersUserAgent(getLicenseEndpointRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetLicenseEndpointResponse>> waitUntilLicenseEndpointValid(
            GetLicenseEndpointRequest getLicenseEndpointRequest, WaiterOverrideConfiguration overrideConfig) {
        return licenseEndpointValidWaiter.runAsync(
                () -> client.getLicenseEndpoint(applyWaitersUserAgent(getLicenseEndpointRequest)),
                licenseEndpointValidWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetQueueFleetAssociationResponse>> waitUntilQueueFleetAssociationStopped(
            GetQueueFleetAssociationRequest getQueueFleetAssociationRequest) {
        return queueFleetAssociationStoppedWaiter.runAsync(() -> client
                .getQueueFleetAssociation(applyWaitersUserAgent(getQueueFleetAssociationRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetQueueFleetAssociationResponse>> waitUntilQueueFleetAssociationStopped(
            GetQueueFleetAssociationRequest getQueueFleetAssociationRequest, WaiterOverrideConfiguration overrideConfig) {
        return queueFleetAssociationStoppedWaiter.runAsync(
                () -> client.getQueueFleetAssociation(applyWaitersUserAgent(getQueueFleetAssociationRequest)),
                queueFleetAssociationStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetQueueResponse>> waitUntilQueueScheduling(GetQueueRequest getQueueRequest) {
        return queueSchedulingWaiter.runAsync(() -> client.getQueue(applyWaitersUserAgent(getQueueRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetQueueResponse>> waitUntilQueueScheduling(GetQueueRequest getQueueRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return queueSchedulingWaiter.runAsync(() -> client.getQueue(applyWaitersUserAgent(getQueueRequest)),
                queueSchedulingWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetQueueResponse>> waitUntilQueueSchedulingBlocked(GetQueueRequest getQueueRequest) {
        return queueSchedulingBlockedWaiter.runAsync(() -> client.getQueue(applyWaitersUserAgent(getQueueRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetQueueResponse>> waitUntilQueueSchedulingBlocked(GetQueueRequest getQueueRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return queueSchedulingBlockedWaiter.runAsync(() -> client.getQueue(applyWaitersUserAgent(getQueueRequest)),
                queueSchedulingBlockedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetFleetResponse>> fleetActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetFleetResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "CREATE_FAILED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "UPDATE_FAILED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetJobResponse>> jobCreateCompleteWaiterAcceptors() {
        List<WaiterAcceptor<? super GetJobResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("lifecycleStatus").value(), "CREATE_COMPLETE");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("lifecycleStatus").value(), "UPDATE_IN_PROGRESS");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("lifecycleStatus").value(), "UPDATE_FAILED");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("lifecycleStatus").value(), "UPDATE_SUCCEEDED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("lifecycleStatus").value(), "UPLOAD_FAILED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("lifecycleStatus").value(), "CREATE_FAILED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetLicenseEndpointResponse>> licenseEndpointDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetLicenseEndpointResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "NOT_READY");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetLicenseEndpointResponse>> licenseEndpointValidWaiterAcceptors() {
        List<WaiterAcceptor<? super GetLicenseEndpointResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "READY");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "NOT_READY");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetQueueFleetAssociationResponse>> queueFleetAssociationStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetQueueFleetAssociationResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "STOPPED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetQueueResponse>> queueSchedulingWaiterAcceptors() {
        List<WaiterAcceptor<? super GetQueueResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "SCHEDULING");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetQueueResponse>> queueSchedulingBlockedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetQueueResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "SCHEDULING_BLOCKED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration fleetActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(180);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration jobCreateCompleteWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration licenseEndpointDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(234);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration licenseEndpointValidWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(114);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration queueFleetAssociationStoppedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration queueSchedulingWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(70);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration queueSchedulingBlockedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(30);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static DeadlineAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends DeadlineRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements DeadlineAsyncWaiter.Builder {
        private DeadlineAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public DeadlineAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public DeadlineAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public DeadlineAsyncWaiter.Builder client(DeadlineAsyncClient client) {
            this.client = client;
            return this;
        }

        public DeadlineAsyncWaiter build() {
            return new DefaultDeadlineAsyncWaiter(this);
        }
    }
}
