/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.deadline.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.deadline.DeadlineClient;
import software.amazon.awssdk.services.deadline.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.deadline.model.DeadlineRequest;
import software.amazon.awssdk.services.deadline.model.GetFleetRequest;
import software.amazon.awssdk.services.deadline.model.GetFleetResponse;
import software.amazon.awssdk.services.deadline.model.GetJobRequest;
import software.amazon.awssdk.services.deadline.model.GetJobResponse;
import software.amazon.awssdk.services.deadline.model.GetLicenseEndpointRequest;
import software.amazon.awssdk.services.deadline.model.GetLicenseEndpointResponse;
import software.amazon.awssdk.services.deadline.model.GetQueueFleetAssociationRequest;
import software.amazon.awssdk.services.deadline.model.GetQueueFleetAssociationResponse;
import software.amazon.awssdk.services.deadline.model.GetQueueRequest;
import software.amazon.awssdk.services.deadline.model.GetQueueResponse;
import software.amazon.awssdk.services.deadline.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultDeadlineWaiter implements DeadlineWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final DeadlineClient client;

    private final AttributeMap managedResources;

    private final Waiter<GetFleetResponse> fleetActiveWaiter;

    private final Waiter<GetJobResponse> jobCreateCompleteWaiter;

    private final Waiter<GetLicenseEndpointResponse> licenseEndpointDeletedWaiter;

    private final Waiter<GetLicenseEndpointResponse> licenseEndpointValidWaiter;

    private final Waiter<GetQueueFleetAssociationResponse> queueFleetAssociationStoppedWaiter;

    private final Waiter<GetQueueResponse> queueSchedulingWaiter;

    private final Waiter<GetQueueResponse> queueSchedulingBlockedWaiter;

    private DefaultDeadlineWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = DeadlineClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.fleetActiveWaiter = Waiter.builder(GetFleetResponse.class).acceptors(fleetActiveWaiterAcceptors())
                .overrideConfiguration(fleetActiveWaiterConfig(builder.overrideConfiguration)).build();
        this.jobCreateCompleteWaiter = Waiter.builder(GetJobResponse.class).acceptors(jobCreateCompleteWaiterAcceptors())
                .overrideConfiguration(jobCreateCompleteWaiterConfig(builder.overrideConfiguration)).build();
        this.licenseEndpointDeletedWaiter = Waiter.builder(GetLicenseEndpointResponse.class)
                .acceptors(licenseEndpointDeletedWaiterAcceptors())
                .overrideConfiguration(licenseEndpointDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.licenseEndpointValidWaiter = Waiter.builder(GetLicenseEndpointResponse.class)
                .acceptors(licenseEndpointValidWaiterAcceptors())
                .overrideConfiguration(licenseEndpointValidWaiterConfig(builder.overrideConfiguration)).build();
        this.queueFleetAssociationStoppedWaiter = Waiter.builder(GetQueueFleetAssociationResponse.class)
                .acceptors(queueFleetAssociationStoppedWaiterAcceptors())
                .overrideConfiguration(queueFleetAssociationStoppedWaiterConfig(builder.overrideConfiguration)).build();
        this.queueSchedulingWaiter = Waiter.builder(GetQueueResponse.class).acceptors(queueSchedulingWaiterAcceptors())
                .overrideConfiguration(queueSchedulingWaiterConfig(builder.overrideConfiguration)).build();
        this.queueSchedulingBlockedWaiter = Waiter.builder(GetQueueResponse.class)
                .acceptors(queueSchedulingBlockedWaiterAcceptors())
                .overrideConfiguration(queueSchedulingBlockedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<GetFleetResponse> waitUntilFleetActive(GetFleetRequest getFleetRequest) {
        return fleetActiveWaiter.run(() -> client.getFleet(applyWaitersUserAgent(getFleetRequest)));
    }

    @Override
    public WaiterResponse<GetFleetResponse> waitUntilFleetActive(GetFleetRequest getFleetRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return fleetActiveWaiter.run(() -> client.getFleet(applyWaitersUserAgent(getFleetRequest)),
                fleetActiveWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetJobResponse> waitUntilJobCreateComplete(GetJobRequest getJobRequest) {
        return jobCreateCompleteWaiter.run(() -> client.getJob(applyWaitersUserAgent(getJobRequest)));
    }

    @Override
    public WaiterResponse<GetJobResponse> waitUntilJobCreateComplete(GetJobRequest getJobRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return jobCreateCompleteWaiter.run(() -> client.getJob(applyWaitersUserAgent(getJobRequest)),
                jobCreateCompleteWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetLicenseEndpointResponse> waitUntilLicenseEndpointDeleted(
            GetLicenseEndpointRequest getLicenseEndpointRequest) {
        return licenseEndpointDeletedWaiter
                .run(() -> client.getLicenseEndpoint(applyWaitersUserAgent(getLicenseEndpointRequest)));
    }

    @Override
    public WaiterResponse<GetLicenseEndpointResponse> waitUntilLicenseEndpointDeleted(
            GetLicenseEndpointRequest getLicenseEndpointRequest, WaiterOverrideConfiguration overrideConfig) {
        return licenseEndpointDeletedWaiter.run(
                () -> client.getLicenseEndpoint(applyWaitersUserAgent(getLicenseEndpointRequest)),
                licenseEndpointDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetLicenseEndpointResponse> waitUntilLicenseEndpointValid(
            GetLicenseEndpointRequest getLicenseEndpointRequest) {
        return licenseEndpointValidWaiter.run(() -> client.getLicenseEndpoint(applyWaitersUserAgent(getLicenseEndpointRequest)));
    }

    @Override
    public WaiterResponse<GetLicenseEndpointResponse> waitUntilLicenseEndpointValid(
            GetLicenseEndpointRequest getLicenseEndpointRequest, WaiterOverrideConfiguration overrideConfig) {
        return licenseEndpointValidWaiter.run(() -> client.getLicenseEndpoint(applyWaitersUserAgent(getLicenseEndpointRequest)),
                licenseEndpointValidWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetQueueFleetAssociationResponse> waitUntilQueueFleetAssociationStopped(
            GetQueueFleetAssociationRequest getQueueFleetAssociationRequest) {
        return queueFleetAssociationStoppedWaiter.run(() -> client
                .getQueueFleetAssociation(applyWaitersUserAgent(getQueueFleetAssociationRequest)));
    }

    @Override
    public WaiterResponse<GetQueueFleetAssociationResponse> waitUntilQueueFleetAssociationStopped(
            GetQueueFleetAssociationRequest getQueueFleetAssociationRequest, WaiterOverrideConfiguration overrideConfig) {
        return queueFleetAssociationStoppedWaiter.run(
                () -> client.getQueueFleetAssociation(applyWaitersUserAgent(getQueueFleetAssociationRequest)),
                queueFleetAssociationStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetQueueResponse> waitUntilQueueScheduling(GetQueueRequest getQueueRequest) {
        return queueSchedulingWaiter.run(() -> client.getQueue(applyWaitersUserAgent(getQueueRequest)));
    }

    @Override
    public WaiterResponse<GetQueueResponse> waitUntilQueueScheduling(GetQueueRequest getQueueRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return queueSchedulingWaiter.run(() -> client.getQueue(applyWaitersUserAgent(getQueueRequest)),
                queueSchedulingWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetQueueResponse> waitUntilQueueSchedulingBlocked(GetQueueRequest getQueueRequest) {
        return queueSchedulingBlockedWaiter.run(() -> client.getQueue(applyWaitersUserAgent(getQueueRequest)));
    }

    @Override
    public WaiterResponse<GetQueueResponse> waitUntilQueueSchedulingBlocked(GetQueueRequest getQueueRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return queueSchedulingBlockedWaiter.run(() -> client.getQueue(applyWaitersUserAgent(getQueueRequest)),
                queueSchedulingBlockedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetFleetResponse>> fleetActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetFleetResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "UPDATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=UPDATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetJobResponse>> jobCreateCompleteWaiterAcceptors() {
        List<WaiterAcceptor<? super GetJobResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("lifecycleStatus").value(), "CREATE_COMPLETE");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("lifecycleStatus").value(), "UPDATE_IN_PROGRESS");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("lifecycleStatus").value(), "UPDATE_FAILED");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("lifecycleStatus").value(), "UPDATE_SUCCEEDED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("lifecycleStatus").value(), "UPLOAD_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (lifecycleStatus=UPLOAD_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("lifecycleStatus").value(), "CREATE_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (lifecycleStatus=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetLicenseEndpointResponse>> licenseEndpointDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetLicenseEndpointResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "NOT_READY");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=NOT_READY) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetLicenseEndpointResponse>> licenseEndpointValidWaiterAcceptors() {
        List<WaiterAcceptor<? super GetLicenseEndpointResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "READY");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("status").value(), "NOT_READY");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (status=NOT_READY) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetQueueFleetAssociationResponse>> queueFleetAssociationStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetQueueFleetAssociationResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "STOPPED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetQueueResponse>> queueSchedulingWaiterAcceptors() {
        List<WaiterAcceptor<? super GetQueueResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "SCHEDULING");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetQueueResponse>> queueSchedulingBlockedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetQueueResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "SCHEDULING_BLOCKED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration fleetActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(180);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration jobCreateCompleteWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration licenseEndpointDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(234);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration licenseEndpointValidWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(114);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration queueFleetAssociationStoppedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration queueSchedulingWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(70);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration queueSchedulingBlockedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(30);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(10)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static DeadlineWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends DeadlineRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements DeadlineWaiter.Builder {
        private DeadlineClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public DeadlineWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public DeadlineWaiter.Builder client(DeadlineClient client) {
            this.client = client;
            return this;
        }

        public DeadlineWaiter build() {
            return new DefaultDeadlineWaiter(this);
        }
    }
}
