/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.deadline.endpoints.internal;

import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletionException;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsExecutionAttribute;
import software.amazon.awssdk.awscore.endpoints.AwsEndpointAttribute;
import software.amazon.awssdk.awscore.endpoints.authscheme.EndpointAuthScheme;
import software.amazon.awssdk.awscore.endpoints.authscheme.SigV4AuthScheme;
import software.amazon.awssdk.awscore.endpoints.authscheme.SigV4aAuthScheme;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.SelectedAuthScheme;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.interceptor.Context;
import software.amazon.awssdk.core.interceptor.ExecutionAttributes;
import software.amazon.awssdk.core.interceptor.ExecutionInterceptor;
import software.amazon.awssdk.core.interceptor.SdkExecutionAttribute;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.http.SdkHttpRequest;
import software.amazon.awssdk.http.auth.aws.signer.AwsV4HttpSigner;
import software.amazon.awssdk.http.auth.aws.signer.AwsV4aHttpSigner;
import software.amazon.awssdk.http.auth.aws.signer.RegionSet;
import software.amazon.awssdk.http.auth.spi.scheme.AuthSchemeOption;
import software.amazon.awssdk.identity.spi.Identity;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.services.deadline.endpoints.DeadlineEndpointParams;
import software.amazon.awssdk.services.deadline.endpoints.DeadlineEndpointProvider;
import software.amazon.awssdk.utils.CollectionUtils;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class DeadlineResolveEndpointInterceptor implements ExecutionInterceptor {
    @Override
    public SdkRequest modifyRequest(Context.ModifyRequest context, ExecutionAttributes executionAttributes) {
        SdkRequest result = context.request();
        if (AwsEndpointProviderUtils.endpointIsDiscovered(executionAttributes)) {
            return result;
        }
        DeadlineEndpointProvider provider = (DeadlineEndpointProvider) executionAttributes
                .getAttribute(SdkInternalExecutionAttribute.ENDPOINT_PROVIDER);
        try {
            long resolveEndpointStart = System.nanoTime();
            DeadlineEndpointParams endpointParams = ruleParams(result, executionAttributes);
            Endpoint endpoint = provider.resolveEndpoint(endpointParams).join();
            Duration resolveEndpointDuration = Duration.ofNanos(System.nanoTime() - resolveEndpointStart);
            Optional<MetricCollector> metricCollector = executionAttributes
                    .getOptionalAttribute(SdkExecutionAttribute.API_CALL_METRIC_COLLECTOR);
            metricCollector.ifPresent(mc -> mc.reportMetric(CoreMetric.ENDPOINT_RESOLVE_DURATION, resolveEndpointDuration));
            if (!AwsEndpointProviderUtils.disableHostPrefixInjection(executionAttributes)) {
                Optional<String> hostPrefix = hostPrefix(executionAttributes.getAttribute(SdkExecutionAttribute.OPERATION_NAME),
                        result);
                if (hostPrefix.isPresent()) {
                    endpoint = AwsEndpointProviderUtils.addHostPrefix(endpoint, hostPrefix.get());
                }
            }
            List<EndpointAuthScheme> endpointAuthSchemes = endpoint.attribute(AwsEndpointAttribute.AUTH_SCHEMES);
            SelectedAuthScheme<?> selectedAuthScheme = executionAttributes
                    .getAttribute(SdkInternalExecutionAttribute.SELECTED_AUTH_SCHEME);
            if (endpointAuthSchemes != null && selectedAuthScheme != null) {
                selectedAuthScheme = authSchemeWithEndpointSignerProperties(endpointAuthSchemes, selectedAuthScheme);
                executionAttributes.putAttribute(SdkInternalExecutionAttribute.SELECTED_AUTH_SCHEME, selectedAuthScheme);
            }
            executionAttributes.putAttribute(SdkInternalExecutionAttribute.RESOLVED_ENDPOINT, endpoint);
            setMetricValues(endpoint, executionAttributes);
            return result;
        } catch (CompletionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof SdkClientException) {
                throw (SdkClientException) cause;
            } else {
                throw SdkClientException.create("Endpoint resolution failed", cause);
            }
        }
    }

    @Override
    public SdkHttpRequest modifyHttpRequest(Context.ModifyHttpRequest context, ExecutionAttributes executionAttributes) {
        Endpoint resolvedEndpoint = executionAttributes.getAttribute(SdkInternalExecutionAttribute.RESOLVED_ENDPOINT);
        if (resolvedEndpoint.headers().isEmpty()) {
            return context.httpRequest();
        }
        SdkHttpRequest.Builder httpRequestBuilder = context.httpRequest().toBuilder();
        resolvedEndpoint.headers().forEach((name, values) -> {
            values.forEach(v -> httpRequestBuilder.appendHeader(name, v));
        });
        return httpRequestBuilder.build();
    }

    public static DeadlineEndpointParams ruleParams(SdkRequest request, ExecutionAttributes executionAttributes) {
        DeadlineEndpointParams.Builder builder = DeadlineEndpointParams.builder();
        builder.region(AwsEndpointProviderUtils.regionBuiltIn(executionAttributes));
        builder.useDualStack(AwsEndpointProviderUtils.dualStackEnabledBuiltIn(executionAttributes));
        builder.useFips(AwsEndpointProviderUtils.fipsEnabledBuiltIn(executionAttributes));
        builder.endpoint(AwsEndpointProviderUtils.endpointBuiltIn(executionAttributes));
        setContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME), request);
        setStaticContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME));
        setOperationContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME), request);
        return builder.build();
    }

    private static void setContextParams(DeadlineEndpointParams.Builder params, String operationName, SdkRequest request) {
    }

    private static void setStaticContextParams(DeadlineEndpointParams.Builder params, String operationName) {
    }

    private <T extends Identity> SelectedAuthScheme<T> authSchemeWithEndpointSignerProperties(
            List<EndpointAuthScheme> endpointAuthSchemes, SelectedAuthScheme<T> selectedAuthScheme) {
        for (EndpointAuthScheme endpointAuthScheme : endpointAuthSchemes) {
            if (!endpointAuthScheme.schemeId().equals(selectedAuthScheme.authSchemeOption().schemeId())) {
                continue;
            }
            AuthSchemeOption.Builder option = selectedAuthScheme.authSchemeOption().toBuilder();
            if (endpointAuthScheme instanceof SigV4AuthScheme) {
                SigV4AuthScheme v4AuthScheme = (SigV4AuthScheme) endpointAuthScheme;
                if (v4AuthScheme.isDisableDoubleEncodingSet()) {
                    option.putSignerProperty(AwsV4HttpSigner.DOUBLE_URL_ENCODE, !v4AuthScheme.disableDoubleEncoding());
                }
                if (v4AuthScheme.signingRegion() != null) {
                    option.putSignerProperty(AwsV4HttpSigner.REGION_NAME, v4AuthScheme.signingRegion());
                }
                if (v4AuthScheme.signingName() != null) {
                    option.putSignerProperty(AwsV4HttpSigner.SERVICE_SIGNING_NAME, v4AuthScheme.signingName());
                }
                return new SelectedAuthScheme<>(selectedAuthScheme.identity(), selectedAuthScheme.signer(), option.build());
            }
            if (endpointAuthScheme instanceof SigV4aAuthScheme) {
                SigV4aAuthScheme v4aAuthScheme = (SigV4aAuthScheme) endpointAuthScheme;
                if (v4aAuthScheme.isDisableDoubleEncodingSet()) {
                    option.putSignerProperty(AwsV4aHttpSigner.DOUBLE_URL_ENCODE, !v4aAuthScheme.disableDoubleEncoding());
                }
                if (!CollectionUtils.isNullOrEmpty(v4aAuthScheme.signingRegionSet())) {
                    RegionSet regionSet = RegionSet.create(v4aAuthScheme.signingRegionSet());
                    option.putSignerProperty(AwsV4aHttpSigner.REGION_SET, regionSet);
                }
                if (v4aAuthScheme.signingName() != null) {
                    option.putSignerProperty(AwsV4aHttpSigner.SERVICE_SIGNING_NAME, v4aAuthScheme.signingName());
                }
                return new SelectedAuthScheme<>(selectedAuthScheme.identity(), selectedAuthScheme.signer(), option.build());
            }
            throw new IllegalArgumentException("Endpoint auth scheme '" + endpointAuthScheme.name()
                    + "' cannot be mapped to the SDK auth scheme. Was it declared in the service's model?");
        }
        return selectedAuthScheme;
    }

    private static void setOperationContextParams(DeadlineEndpointParams.Builder params, String operationName, SdkRequest request) {
    }

    private static Optional<String> hostPrefix(String operationName, SdkRequest request) {
        switch (operationName) {
        case "AssociateMemberToFarm": {
            return Optional.of("management.");
        }
        case "AssociateMemberToFleet": {
            return Optional.of("management.");
        }
        case "AssociateMemberToJob": {
            return Optional.of("management.");
        }
        case "AssociateMemberToQueue": {
            return Optional.of("management.");
        }
        case "AssumeFleetRoleForRead": {
            return Optional.of("management.");
        }
        case "AssumeFleetRoleForWorker": {
            return Optional.of("scheduling.");
        }
        case "AssumeQueueRoleForRead": {
            return Optional.of("management.");
        }
        case "AssumeQueueRoleForUser": {
            return Optional.of("management.");
        }
        case "AssumeQueueRoleForWorker": {
            return Optional.of("scheduling.");
        }
        case "BatchGetJobEntity": {
            return Optional.of("scheduling.");
        }
        case "CopyJobTemplate": {
            return Optional.of("management.");
        }
        case "CreateBudget": {
            return Optional.of("management.");
        }
        case "CreateFarm": {
            return Optional.of("management.");
        }
        case "CreateFleet": {
            return Optional.of("management.");
        }
        case "CreateJob": {
            return Optional.of("management.");
        }
        case "CreateLicenseEndpoint": {
            return Optional.of("management.");
        }
        case "CreateLimit": {
            return Optional.of("management.");
        }
        case "CreateMonitor": {
            return Optional.of("management.");
        }
        case "CreateQueue": {
            return Optional.of("management.");
        }
        case "CreateQueueEnvironment": {
            return Optional.of("management.");
        }
        case "CreateQueueFleetAssociation": {
            return Optional.of("management.");
        }
        case "CreateQueueLimitAssociation": {
            return Optional.of("management.");
        }
        case "CreateStorageProfile": {
            return Optional.of("management.");
        }
        case "CreateWorker": {
            return Optional.of("scheduling.");
        }
        case "DeleteBudget": {
            return Optional.of("management.");
        }
        case "DeleteFarm": {
            return Optional.of("management.");
        }
        case "DeleteFleet": {
            return Optional.of("management.");
        }
        case "DeleteLicenseEndpoint": {
            return Optional.of("management.");
        }
        case "DeleteLimit": {
            return Optional.of("management.");
        }
        case "DeleteMeteredProduct": {
            return Optional.of("management.");
        }
        case "DeleteMonitor": {
            return Optional.of("management.");
        }
        case "DeleteQueue": {
            return Optional.of("management.");
        }
        case "DeleteQueueEnvironment": {
            return Optional.of("management.");
        }
        case "DeleteQueueFleetAssociation": {
            return Optional.of("management.");
        }
        case "DeleteQueueLimitAssociation": {
            return Optional.of("management.");
        }
        case "DeleteStorageProfile": {
            return Optional.of("management.");
        }
        case "DeleteWorker": {
            return Optional.of("management.");
        }
        case "DisassociateMemberFromFarm": {
            return Optional.of("management.");
        }
        case "DisassociateMemberFromFleet": {
            return Optional.of("management.");
        }
        case "DisassociateMemberFromJob": {
            return Optional.of("management.");
        }
        case "DisassociateMemberFromQueue": {
            return Optional.of("management.");
        }
        case "GetBudget": {
            return Optional.of("management.");
        }
        case "GetFarm": {
            return Optional.of("management.");
        }
        case "GetFleet": {
            return Optional.of("management.");
        }
        case "GetJob": {
            return Optional.of("management.");
        }
        case "GetLicenseEndpoint": {
            return Optional.of("management.");
        }
        case "GetLimit": {
            return Optional.of("management.");
        }
        case "GetMonitor": {
            return Optional.of("management.");
        }
        case "GetQueue": {
            return Optional.of("management.");
        }
        case "GetQueueEnvironment": {
            return Optional.of("management.");
        }
        case "GetQueueFleetAssociation": {
            return Optional.of("management.");
        }
        case "GetQueueLimitAssociation": {
            return Optional.of("management.");
        }
        case "GetSession": {
            return Optional.of("management.");
        }
        case "GetSessionAction": {
            return Optional.of("management.");
        }
        case "GetSessionsStatisticsAggregation": {
            return Optional.of("management.");
        }
        case "GetStep": {
            return Optional.of("management.");
        }
        case "GetStorageProfile": {
            return Optional.of("management.");
        }
        case "GetStorageProfileForQueue": {
            return Optional.of("management.");
        }
        case "GetTask": {
            return Optional.of("management.");
        }
        case "GetWorker": {
            return Optional.of("management.");
        }
        case "ListAvailableMeteredProducts": {
            return Optional.of("management.");
        }
        case "ListBudgets": {
            return Optional.of("management.");
        }
        case "ListFarmMembers": {
            return Optional.of("management.");
        }
        case "ListFarms": {
            return Optional.of("management.");
        }
        case "ListFleetMembers": {
            return Optional.of("management.");
        }
        case "ListFleets": {
            return Optional.of("management.");
        }
        case "ListJobMembers": {
            return Optional.of("management.");
        }
        case "ListJobParameterDefinitions": {
            return Optional.of("management.");
        }
        case "ListJobs": {
            return Optional.of("management.");
        }
        case "ListLicenseEndpoints": {
            return Optional.of("management.");
        }
        case "ListLimits": {
            return Optional.of("management.");
        }
        case "ListMeteredProducts": {
            return Optional.of("management.");
        }
        case "ListMonitors": {
            return Optional.of("management.");
        }
        case "ListQueueEnvironments": {
            return Optional.of("management.");
        }
        case "ListQueueFleetAssociations": {
            return Optional.of("management.");
        }
        case "ListQueueLimitAssociations": {
            return Optional.of("management.");
        }
        case "ListQueueMembers": {
            return Optional.of("management.");
        }
        case "ListQueues": {
            return Optional.of("management.");
        }
        case "ListSessionActions": {
            return Optional.of("management.");
        }
        case "ListSessions": {
            return Optional.of("management.");
        }
        case "ListSessionsForWorker": {
            return Optional.of("management.");
        }
        case "ListStepConsumers": {
            return Optional.of("management.");
        }
        case "ListStepDependencies": {
            return Optional.of("management.");
        }
        case "ListSteps": {
            return Optional.of("management.");
        }
        case "ListStorageProfiles": {
            return Optional.of("management.");
        }
        case "ListStorageProfilesForQueue": {
            return Optional.of("management.");
        }
        case "ListTagsForResource": {
            return Optional.of("management.");
        }
        case "ListTasks": {
            return Optional.of("management.");
        }
        case "ListWorkers": {
            return Optional.of("management.");
        }
        case "PutMeteredProduct": {
            return Optional.of("management.");
        }
        case "SearchJobs": {
            return Optional.of("management.");
        }
        case "SearchSteps": {
            return Optional.of("management.");
        }
        case "SearchTasks": {
            return Optional.of("management.");
        }
        case "SearchWorkers": {
            return Optional.of("management.");
        }
        case "StartSessionsStatisticsAggregation": {
            return Optional.of("management.");
        }
        case "TagResource": {
            return Optional.of("management.");
        }
        case "UntagResource": {
            return Optional.of("management.");
        }
        case "UpdateBudget": {
            return Optional.of("management.");
        }
        case "UpdateFarm": {
            return Optional.of("management.");
        }
        case "UpdateFleet": {
            return Optional.of("management.");
        }
        case "UpdateJob": {
            return Optional.of("management.");
        }
        case "UpdateLimit": {
            return Optional.of("management.");
        }
        case "UpdateMonitor": {
            return Optional.of("management.");
        }
        case "UpdateQueue": {
            return Optional.of("management.");
        }
        case "UpdateQueueEnvironment": {
            return Optional.of("management.");
        }
        case "UpdateQueueFleetAssociation": {
            return Optional.of("management.");
        }
        case "UpdateQueueLimitAssociation": {
            return Optional.of("management.");
        }
        case "UpdateSession": {
            return Optional.of("management.");
        }
        case "UpdateStep": {
            return Optional.of("management.");
        }
        case "UpdateStorageProfile": {
            return Optional.of("management.");
        }
        case "UpdateTask": {
            return Optional.of("management.");
        }
        case "UpdateWorker": {
            return Optional.of("scheduling.");
        }
        case "UpdateWorkerSchedule": {
            return Optional.of("scheduling.");
        }
        default:
            return Optional.empty();
        }
    }

    private void setMetricValues(Endpoint endpoint, ExecutionAttributes executionAttributes) {
        if (endpoint.attribute(AwsEndpointAttribute.METRIC_VALUES) != null) {
            executionAttributes.getOptionalAttribute(SdkInternalExecutionAttribute.BUSINESS_METRICS).ifPresent(
                    metrics -> endpoint.attribute(AwsEndpointAttribute.METRIC_VALUES).forEach(v -> metrics.addMetric(v)));
        }
    }
}
