/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.detective;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.detective.model.AcceptInvitationRequest;
import software.amazon.awssdk.services.detective.model.AcceptInvitationResponse;
import software.amazon.awssdk.services.detective.model.ConflictException;
import software.amazon.awssdk.services.detective.model.CreateGraphRequest;
import software.amazon.awssdk.services.detective.model.CreateGraphResponse;
import software.amazon.awssdk.services.detective.model.CreateMembersRequest;
import software.amazon.awssdk.services.detective.model.CreateMembersResponse;
import software.amazon.awssdk.services.detective.model.DeleteGraphRequest;
import software.amazon.awssdk.services.detective.model.DeleteGraphResponse;
import software.amazon.awssdk.services.detective.model.DeleteMembersRequest;
import software.amazon.awssdk.services.detective.model.DeleteMembersResponse;
import software.amazon.awssdk.services.detective.model.DetectiveException;
import software.amazon.awssdk.services.detective.model.DetectiveRequest;
import software.amazon.awssdk.services.detective.model.DisassociateMembershipRequest;
import software.amazon.awssdk.services.detective.model.DisassociateMembershipResponse;
import software.amazon.awssdk.services.detective.model.GetMembersRequest;
import software.amazon.awssdk.services.detective.model.GetMembersResponse;
import software.amazon.awssdk.services.detective.model.InternalServerException;
import software.amazon.awssdk.services.detective.model.ListGraphsRequest;
import software.amazon.awssdk.services.detective.model.ListGraphsResponse;
import software.amazon.awssdk.services.detective.model.ListInvitationsRequest;
import software.amazon.awssdk.services.detective.model.ListInvitationsResponse;
import software.amazon.awssdk.services.detective.model.ListMembersRequest;
import software.amazon.awssdk.services.detective.model.ListMembersResponse;
import software.amazon.awssdk.services.detective.model.RejectInvitationRequest;
import software.amazon.awssdk.services.detective.model.RejectInvitationResponse;
import software.amazon.awssdk.services.detective.model.ResourceNotFoundException;
import software.amazon.awssdk.services.detective.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.detective.model.ValidationException;
import software.amazon.awssdk.services.detective.paginators.ListGraphsIterable;
import software.amazon.awssdk.services.detective.paginators.ListInvitationsIterable;
import software.amazon.awssdk.services.detective.paginators.ListMembersIterable;
import software.amazon.awssdk.services.detective.transform.AcceptInvitationRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.CreateGraphRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.CreateMembersRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.DeleteGraphRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.DeleteMembersRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.DisassociateMembershipRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.GetMembersRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.ListGraphsRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.ListInvitationsRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.ListMembersRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.RejectInvitationRequestMarshaller;

/**
 * Internal implementation of {@link DetectiveClient}.
 *
 * @see DetectiveClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDetectiveClient implements DetectiveClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultDetectiveClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Amazon Detective is currently in preview.
     * </p>
     * <p>
     * Accepts an invitation for the member account to contribute data to a behavior graph. This operation can only be
     * called by an invited member account.
     * </p>
     * <p>
     * The request provides the ARN of behavior graph.
     * </p>
     * <p>
     * The member account status in the graph must be <code>INVITED</code>.
     * </p>
     *
     * @param acceptInvitationRequest
     * @return Result of the AcceptInvitation operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.AcceptInvitation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/AcceptInvitation" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AcceptInvitationResponse acceptInvitation(AcceptInvitationRequest acceptInvitationRequest) throws ConflictException,
            InternalServerException, ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            DetectiveException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AcceptInvitationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AcceptInvitationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AcceptInvitationRequest, AcceptInvitationResponse>()
                .withOperationName("AcceptInvitation").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(acceptInvitationRequest)
                .withMarshaller(new AcceptInvitationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Amazon Detective is currently in preview.
     * </p>
     * <p>
     * Creates a new behavior graph for the calling account, and sets that account as the master account. This operation
     * is called by the account that is enabling Detective.
     * </p>
     * <p>
     * The operation also enables Detective for the calling account in the currently selected Region. It returns the ARN
     * of the new behavior graph.
     * </p>
     * <p>
     * <code>CreateGraph</code> triggers a process to create the corresponding data tables for the new behavior graph.
     * </p>
     * <p>
     * An account can only be the master account for one behavior graph within a Region. If the same account calls
     * <code>CreateGraph</code> with the same master account, it always returns the same behavior graph ARN. It does not
     * create a new behavior graph.
     * </p>
     *
     * @param createGraphRequest
     * @return Result of the CreateGraph operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.CreateGraph
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/CreateGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateGraphResponse createGraph(CreateGraphRequest createGraphRequest) throws ConflictException,
            InternalServerException, AwsServiceException, SdkClientException, DetectiveException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateGraphResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateGraphRequest, CreateGraphResponse>()
                .withOperationName("CreateGraph").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createGraphRequest)
                .withMarshaller(new CreateGraphRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Amazon Detective is currently in preview.
     * </p>
     * <p>
     * Sends a request to invite the specified AWS accounts to be member accounts in the behavior graph. This operation
     * can only be called by the master account for a behavior graph.
     * </p>
     * <p>
     * <code>CreateMembers</code> verifies the accounts and then sends invitations to the verified accounts.
     * </p>
     * <p>
     * The request provides the behavior graph ARN and the list of accounts to invite.
     * </p>
     * <p>
     * The response separates the requested accounts into two lists:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The accounts that <code>CreateMembers</code> was able to start the verification for. This list includes member
     * accounts that are being verified, that have passed verification and are being sent an invitation, and that have
     * failed verification.
     * </p>
     * </li>
     * <li>
     * <p>
     * The accounts that <code>CreateMembers</code> was unable to process. This list includes accounts that were already
     * invited to be member accounts in the behavior graph.
     * </p>
     * </li>
     * </ul>
     *
     * @param createMembersRequest
     * @return Result of the CreateMembers operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws ServiceQuotaExceededException
     *         This request would cause the number of member accounts in the behavior graph to exceed the maximum
     *         allowed. A behavior graph cannot have more than 1000 member accounts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.CreateMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/CreateMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateMembersResponse createMembers(CreateMembersRequest createMembersRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, DetectiveException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateMembersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateMembersRequest, CreateMembersResponse>()
                .withOperationName("CreateMembers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createMembersRequest)
                .withMarshaller(new CreateMembersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Amazon Detective is currently in preview.
     * </p>
     * <p>
     * Disables the specified behavior graph and queues it to be deleted. This operation removes the graph from each
     * member account's list of behavior graphs.
     * </p>
     * <p>
     * <code>DeleteGraph</code> can only be called by the master account for a behavior graph.
     * </p>
     *
     * @param deleteGraphRequest
     * @return Result of the DeleteGraph operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.DeleteGraph
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DeleteGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteGraphResponse deleteGraph(DeleteGraphRequest deleteGraphRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteGraphResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteGraphRequest, DeleteGraphResponse>()
                .withOperationName("DeleteGraph").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteGraphRequest)
                .withMarshaller(new DeleteGraphRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Amazon Detective is currently in preview.
     * </p>
     * <p>
     * Deletes one or more member accounts from the master account behavior graph. This operation can only be called by
     * a Detective master account. That account cannot use <code>DeleteMembers</code> to delete their own account from
     * the behavior graph. To disable a behavior graph, the master account uses the <code>DeleteGraph</code> API method.
     * </p>
     *
     * @param deleteMembersRequest
     * @return Result of the DeleteMembers operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.DeleteMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DeleteMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteMembersResponse deleteMembers(DeleteMembersRequest deleteMembersRequest) throws ConflictException,
            InternalServerException, ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            DetectiveException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteMembersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteMembersRequest, DeleteMembersResponse>()
                .withOperationName("DeleteMembers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteMembersRequest)
                .withMarshaller(new DeleteMembersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Amazon Detective is currently in preview.
     * </p>
     * <p>
     * Removes the member account from the specified behavior graph. This operation can only be called by a member
     * account that has the <code>ENABLED</code> status.
     * </p>
     *
     * @param disassociateMembershipRequest
     * @return Result of the DisassociateMembership operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.DisassociateMembership
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DisassociateMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateMembershipResponse disassociateMembership(DisassociateMembershipRequest disassociateMembershipRequest)
            throws ConflictException, InternalServerException, ResourceNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, DetectiveException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateMembershipResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DisassociateMembershipRequest, DisassociateMembershipResponse>()
                .withOperationName("DisassociateMembership").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(disassociateMembershipRequest)
                .withMarshaller(new DisassociateMembershipRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Amazon Detective is currently in preview.
     * </p>
     * <p>
     * Returns the membership details for specified member accounts for a behavior graph.
     * </p>
     *
     * @param getMembersRequest
     * @return Result of the GetMembers operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.GetMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/GetMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetMembersResponse getMembers(GetMembersRequest getMembersRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMembersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetMembersRequest, GetMembersResponse>()
                .withOperationName("GetMembers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getMembersRequest)
                .withMarshaller(new GetMembersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Amazon Detective is currently in preview.
     * </p>
     * <p>
     * Returns the list of behavior graphs that the calling account is a master of. This operation can only be called by
     * a master account.
     * </p>
     * <p>
     * Because an account can currently only be the master of one behavior graph within a Region, the results always
     * contain a single graph.
     * </p>
     *
     * @param listGraphsRequest
     * @return Result of the ListGraphs operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListGraphs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListGraphs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGraphsResponse listGraphs(ListGraphsRequest listGraphsRequest) throws InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGraphsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListGraphsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListGraphsRequest, ListGraphsResponse>()
                .withOperationName("ListGraphs").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listGraphsRequest)
                .withMarshaller(new ListGraphsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Amazon Detective is currently in preview.
     * </p>
     * <p>
     * Returns the list of behavior graphs that the calling account is a master of. This operation can only be called by
     * a master account.
     * </p>
     * <p>
     * Because an account can currently only be the master of one behavior graph within a Region, the results always
     * contain a single graph.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listGraphs(software.amazon.awssdk.services.detective.model.ListGraphsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListGraphsIterable responses = client.listGraphsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.detective.paginators.ListGraphsIterable responses = client.listGraphsPaginator(request);
     *     for (software.amazon.awssdk.services.detective.model.ListGraphsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListGraphsIterable responses = client.listGraphsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGraphs(software.amazon.awssdk.services.detective.model.ListGraphsRequest)} operation.</b>
     * </p>
     *
     * @param listGraphsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListGraphs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListGraphs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGraphsIterable listGraphsPaginator(ListGraphsRequest listGraphsRequest) throws InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        return new ListGraphsIterable(this, applyPaginatorUserAgent(listGraphsRequest));
    }

    /**
     * <p>
     * Amazon Detective is currently in preview.
     * </p>
     * <p>
     * Retrieves the list of open and accepted behavior graph invitations for the member account. This operation can
     * only be called by a member account.
     * </p>
     * <p>
     * Open invitations are invitations that the member account has not responded to.
     * </p>
     * <p>
     * The results do not include behavior graphs for which the member account declined the invitation. The results also
     * do not include behavior graphs that the member account resigned from or was removed from.
     * </p>
     *
     * @param listInvitationsRequest
     * @return Result of the ListInvitations operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListInvitations" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListInvitationsResponse listInvitations(ListInvitationsRequest listInvitationsRequest) throws InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListInvitationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListInvitationsRequest, ListInvitationsResponse>()
                .withOperationName("ListInvitations").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listInvitationsRequest)
                .withMarshaller(new ListInvitationsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Amazon Detective is currently in preview.
     * </p>
     * <p>
     * Retrieves the list of open and accepted behavior graph invitations for the member account. This operation can
     * only be called by a member account.
     * </p>
     * <p>
     * Open invitations are invitations that the member account has not responded to.
     * </p>
     * <p>
     * The results do not include behavior graphs for which the member account declined the invitation. The results also
     * do not include behavior graphs that the member account resigned from or was removed from.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listInvitations(software.amazon.awssdk.services.detective.model.ListInvitationsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListInvitationsIterable responses = client.listInvitationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.detective.paginators.ListInvitationsIterable responses = client
     *             .listInvitationsPaginator(request);
     *     for (software.amazon.awssdk.services.detective.model.ListInvitationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListInvitationsIterable responses = client.listInvitationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInvitations(software.amazon.awssdk.services.detective.model.ListInvitationsRequest)} operation.</b>
     * </p>
     *
     * @param listInvitationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListInvitations" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListInvitationsIterable listInvitationsPaginator(ListInvitationsRequest listInvitationsRequest)
            throws InternalServerException, ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        return new ListInvitationsIterable(this, applyPaginatorUserAgent(listInvitationsRequest));
    }

    /**
     * <p>
     * Amazon Detective is currently in preview.
     * </p>
     * <p>
     * Retrieves the list of member accounts for a behavior graph. Does not return member accounts that were removed
     * from the behavior graph.
     * </p>
     *
     * @param listMembersRequest
     * @return Result of the ListMembers operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListMembersResponse listMembers(ListMembersRequest listMembersRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListMembersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListMembersRequest, ListMembersResponse>()
                .withOperationName("ListMembers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listMembersRequest)
                .withMarshaller(new ListMembersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Amazon Detective is currently in preview.
     * </p>
     * <p>
     * Retrieves the list of member accounts for a behavior graph. Does not return member accounts that were removed
     * from the behavior graph.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listMembers(software.amazon.awssdk.services.detective.model.ListMembersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListMembersIterable responses = client.listMembersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.detective.paginators.ListMembersIterable responses = client.listMembersPaginator(request);
     *     for (software.amazon.awssdk.services.detective.model.ListMembersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListMembersIterable responses = client.listMembersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMembers(software.amazon.awssdk.services.detective.model.ListMembersRequest)} operation.</b>
     * </p>
     *
     * @param listMembersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListMembersIterable listMembersPaginator(ListMembersRequest listMembersRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        return new ListMembersIterable(this, applyPaginatorUserAgent(listMembersRequest));
    }

    /**
     * <p>
     * Amazon Detective is currently in preview.
     * </p>
     * <p>
     * Rejects an invitation to contribute the account data to a behavior graph. This operation must be called by a
     * member account that has the <code>INVITED</code> status.
     * </p>
     *
     * @param rejectInvitationRequest
     * @return Result of the RejectInvitation operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.RejectInvitation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/RejectInvitation" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RejectInvitationResponse rejectInvitation(RejectInvitationRequest rejectInvitationRequest) throws ConflictException,
            InternalServerException, ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            DetectiveException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RejectInvitationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RejectInvitationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RejectInvitationRequest, RejectInvitationResponse>()
                .withOperationName("RejectInvitation").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(rejectInvitationRequest)
                .withMarshaller(new RejectInvitationRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DetectiveException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends DetectiveRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
