/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.detective;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.detective.model.AcceptInvitationRequest;
import software.amazon.awssdk.services.detective.model.AcceptInvitationResponse;
import software.amazon.awssdk.services.detective.model.ConflictException;
import software.amazon.awssdk.services.detective.model.CreateGraphRequest;
import software.amazon.awssdk.services.detective.model.CreateGraphResponse;
import software.amazon.awssdk.services.detective.model.CreateMembersRequest;
import software.amazon.awssdk.services.detective.model.CreateMembersResponse;
import software.amazon.awssdk.services.detective.model.DeleteGraphRequest;
import software.amazon.awssdk.services.detective.model.DeleteGraphResponse;
import software.amazon.awssdk.services.detective.model.DeleteMembersRequest;
import software.amazon.awssdk.services.detective.model.DeleteMembersResponse;
import software.amazon.awssdk.services.detective.model.DetectiveException;
import software.amazon.awssdk.services.detective.model.DisassociateMembershipRequest;
import software.amazon.awssdk.services.detective.model.DisassociateMembershipResponse;
import software.amazon.awssdk.services.detective.model.GetMembersRequest;
import software.amazon.awssdk.services.detective.model.GetMembersResponse;
import software.amazon.awssdk.services.detective.model.InternalServerException;
import software.amazon.awssdk.services.detective.model.ListGraphsRequest;
import software.amazon.awssdk.services.detective.model.ListGraphsResponse;
import software.amazon.awssdk.services.detective.model.ListInvitationsRequest;
import software.amazon.awssdk.services.detective.model.ListInvitationsResponse;
import software.amazon.awssdk.services.detective.model.ListMembersRequest;
import software.amazon.awssdk.services.detective.model.ListMembersResponse;
import software.amazon.awssdk.services.detective.model.RejectInvitationRequest;
import software.amazon.awssdk.services.detective.model.RejectInvitationResponse;
import software.amazon.awssdk.services.detective.model.ResourceNotFoundException;
import software.amazon.awssdk.services.detective.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.detective.model.StartMonitoringMemberRequest;
import software.amazon.awssdk.services.detective.model.StartMonitoringMemberResponse;
import software.amazon.awssdk.services.detective.model.ValidationException;
import software.amazon.awssdk.services.detective.paginators.ListGraphsIterable;
import software.amazon.awssdk.services.detective.paginators.ListInvitationsIterable;
import software.amazon.awssdk.services.detective.paginators.ListMembersIterable;

/**
 * Service client for accessing Amazon Detective. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * Detective uses machine learning and purpose-built visualizations to help you analyze and investigate security issues
 * across your Amazon Web Services (AWS) workloads. Detective automatically extracts time-based events such as login
 * attempts, API calls, and network traffic from AWS CloudTrail and Amazon Virtual Private Cloud (Amazon VPC) flow logs.
 * It also extracts findings detected by Amazon GuardDuty.
 * </p>
 * <p>
 * The Detective API primarily supports the creation and management of behavior graphs. A behavior graph contains the
 * extracted data from a set of member accounts, and is created and managed by a master account.
 * </p>
 * <p>
 * Every behavior graph is specific to a Region. You can only use the API to manage graphs that belong to the Region
 * that is associated with the currently selected endpoint.
 * </p>
 * <p>
 * A Detective master account can use the Detective API to do the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Enable and disable Detective. Enabling Detective creates a new behavior graph.
 * </p>
 * </li>
 * <li>
 * <p>
 * View the list of member accounts in a behavior graph.
 * </p>
 * </li>
 * <li>
 * <p>
 * Add member accounts to a behavior graph.
 * </p>
 * </li>
 * <li>
 * <p>
 * Remove member accounts from a behavior graph.
 * </p>
 * </li>
 * </ul>
 * <p>
 * A member account can use the Detective API to do the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * View the list of behavior graphs that they are invited to.
 * </p>
 * </li>
 * <li>
 * <p>
 * Accept an invitation to contribute to a behavior graph.
 * </p>
 * </li>
 * <li>
 * <p>
 * Decline an invitation to contribute to a behavior graph.
 * </p>
 * </li>
 * <li>
 * <p>
 * Remove their account from a behavior graph.
 * </p>
 * </li>
 * </ul>
 * <p>
 * All API actions are logged as CloudTrail events. See <a
 * href="https://docs.aws.amazon.com/detective/latest/adminguide/logging-using-cloudtrail.html">Logging Detective API
 * Calls with CloudTrail</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public interface DetectiveClient extends SdkClient {
    String SERVICE_NAME = "detective";

    /**
     * Create a {@link DetectiveClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static DetectiveClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link DetectiveClient}.
     */
    static DetectiveClientBuilder builder() {
        return new DefaultDetectiveClientBuilder();
    }

    /**
     * <p>
     * Accepts an invitation for the member account to contribute data to a behavior graph. This operation can only be
     * called by an invited member account.
     * </p>
     * <p>
     * The request provides the ARN of behavior graph.
     * </p>
     * <p>
     * The member account status in the graph must be <code>INVITED</code>.
     * </p>
     *
     * @param acceptInvitationRequest
     * @return Result of the AcceptInvitation operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.AcceptInvitation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/AcceptInvitation" target="_top">AWS API
     *      Documentation</a>
     */
    default AcceptInvitationResponse acceptInvitation(AcceptInvitationRequest acceptInvitationRequest) throws ConflictException,
            InternalServerException, ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            DetectiveException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Accepts an invitation for the member account to contribute data to a behavior graph. This operation can only be
     * called by an invited member account.
     * </p>
     * <p>
     * The request provides the ARN of behavior graph.
     * </p>
     * <p>
     * The member account status in the graph must be <code>INVITED</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link AcceptInvitationRequest.Builder} avoiding the need
     * to create one manually via {@link AcceptInvitationRequest#builder()}
     * </p>
     *
     * @param acceptInvitationRequest
     *        A {@link Consumer} that will call methods on {@link AcceptInvitationRequest.Builder} to create a request.
     * @return Result of the AcceptInvitation operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.AcceptInvitation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/AcceptInvitation" target="_top">AWS API
     *      Documentation</a>
     */
    default AcceptInvitationResponse acceptInvitation(Consumer<AcceptInvitationRequest.Builder> acceptInvitationRequest)
            throws ConflictException, InternalServerException, ResourceNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, DetectiveException {
        return acceptInvitation(AcceptInvitationRequest.builder().applyMutation(acceptInvitationRequest).build());
    }

    /**
     * <p>
     * Creates a new behavior graph for the calling account, and sets that account as the master account. This operation
     * is called by the account that is enabling Detective.
     * </p>
     * <p>
     * Before you try to enable Detective, make sure that your account has been enrolled in Amazon GuardDuty for at
     * least 48 hours. If you do not meet this requirement, you cannot enable Detective. If you do meet the GuardDuty
     * prerequisite, then when you make the request to enable Detective, it checks whether your data volume is within
     * the Detective quota. If it exceeds the quota, then you cannot enable Detective.
     * </p>
     * <p>
     * The operation also enables Detective for the calling account in the currently selected Region. It returns the ARN
     * of the new behavior graph.
     * </p>
     * <p>
     * <code>CreateGraph</code> triggers a process to create the corresponding data tables for the new behavior graph.
     * </p>
     * <p>
     * An account can only be the master account for one behavior graph within a Region. If the same account calls
     * <code>CreateGraph</code> with the same master account, it always returns the same behavior graph ARN. It does not
     * create a new behavior graph.
     * </p>
     *
     * @param createGraphRequest
     * @return Result of the CreateGraph operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ServiceQuotaExceededException
     *         This request cannot be completed for one of the following reasons.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The request would cause the number of member accounts in the behavior graph to exceed the maximum
     *         allowed. A behavior graph cannot have more than 1000 member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The request would cause the data rate for the behavior graph to exceed the maximum allowed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Detective is unable to verify the data rate for the member account. This is usually because the member
     *         account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.CreateGraph
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/CreateGraph" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateGraphResponse createGraph(CreateGraphRequest createGraphRequest) throws ConflictException,
            InternalServerException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, DetectiveException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a new behavior graph for the calling account, and sets that account as the master account. This operation
     * is called by the account that is enabling Detective.
     * </p>
     * <p>
     * Before you try to enable Detective, make sure that your account has been enrolled in Amazon GuardDuty for at
     * least 48 hours. If you do not meet this requirement, you cannot enable Detective. If you do meet the GuardDuty
     * prerequisite, then when you make the request to enable Detective, it checks whether your data volume is within
     * the Detective quota. If it exceeds the quota, then you cannot enable Detective.
     * </p>
     * <p>
     * The operation also enables Detective for the calling account in the currently selected Region. It returns the ARN
     * of the new behavior graph.
     * </p>
     * <p>
     * <code>CreateGraph</code> triggers a process to create the corresponding data tables for the new behavior graph.
     * </p>
     * <p>
     * An account can only be the master account for one behavior graph within a Region. If the same account calls
     * <code>CreateGraph</code> with the same master account, it always returns the same behavior graph ARN. It does not
     * create a new behavior graph.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateGraphRequest.Builder} avoiding the need to
     * create one manually via {@link CreateGraphRequest#builder()}
     * </p>
     *
     * @param createGraphRequest
     *        A {@link Consumer} that will call methods on {@link CreateGraphRequest.Builder} to create a request.
     * @return Result of the CreateGraph operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ServiceQuotaExceededException
     *         This request cannot be completed for one of the following reasons.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The request would cause the number of member accounts in the behavior graph to exceed the maximum
     *         allowed. A behavior graph cannot have more than 1000 member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The request would cause the data rate for the behavior graph to exceed the maximum allowed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Detective is unable to verify the data rate for the member account. This is usually because the member
     *         account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.CreateGraph
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/CreateGraph" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateGraphResponse createGraph(Consumer<CreateGraphRequest.Builder> createGraphRequest) throws ConflictException,
            InternalServerException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, DetectiveException {
        return createGraph(CreateGraphRequest.builder().applyMutation(createGraphRequest).build());
    }

    /**
     * <p>
     * Sends a request to invite the specified AWS accounts to be member accounts in the behavior graph. This operation
     * can only be called by the master account for a behavior graph.
     * </p>
     * <p>
     * <code>CreateMembers</code> verifies the accounts and then sends invitations to the verified accounts.
     * </p>
     * <p>
     * The request provides the behavior graph ARN and the list of accounts to invite.
     * </p>
     * <p>
     * The response separates the requested accounts into two lists:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The accounts that <code>CreateMembers</code> was able to start the verification for. This list includes member
     * accounts that are being verified, that have passed verification and are being sent an invitation, and that have
     * failed verification.
     * </p>
     * </li>
     * <li>
     * <p>
     * The accounts that <code>CreateMembers</code> was unable to process. This list includes accounts that were already
     * invited to be member accounts in the behavior graph.
     * </p>
     * </li>
     * </ul>
     *
     * @param createMembersRequest
     * @return Result of the CreateMembers operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws ServiceQuotaExceededException
     *         This request cannot be completed for one of the following reasons.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The request would cause the number of member accounts in the behavior graph to exceed the maximum
     *         allowed. A behavior graph cannot have more than 1000 member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The request would cause the data rate for the behavior graph to exceed the maximum allowed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Detective is unable to verify the data rate for the member account. This is usually because the member
     *         account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.CreateMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/CreateMembers" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateMembersResponse createMembers(CreateMembersRequest createMembersRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, DetectiveException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Sends a request to invite the specified AWS accounts to be member accounts in the behavior graph. This operation
     * can only be called by the master account for a behavior graph.
     * </p>
     * <p>
     * <code>CreateMembers</code> verifies the accounts and then sends invitations to the verified accounts.
     * </p>
     * <p>
     * The request provides the behavior graph ARN and the list of accounts to invite.
     * </p>
     * <p>
     * The response separates the requested accounts into two lists:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The accounts that <code>CreateMembers</code> was able to start the verification for. This list includes member
     * accounts that are being verified, that have passed verification and are being sent an invitation, and that have
     * failed verification.
     * </p>
     * </li>
     * <li>
     * <p>
     * The accounts that <code>CreateMembers</code> was unable to process. This list includes accounts that were already
     * invited to be member accounts in the behavior graph.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateMembersRequest.Builder} avoiding the need to
     * create one manually via {@link CreateMembersRequest#builder()}
     * </p>
     *
     * @param createMembersRequest
     *        A {@link Consumer} that will call methods on {@link CreateMembersRequest.Builder} to create a request.
     * @return Result of the CreateMembers operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws ServiceQuotaExceededException
     *         This request cannot be completed for one of the following reasons.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The request would cause the number of member accounts in the behavior graph to exceed the maximum
     *         allowed. A behavior graph cannot have more than 1000 member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The request would cause the data rate for the behavior graph to exceed the maximum allowed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Detective is unable to verify the data rate for the member account. This is usually because the member
     *         account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.CreateMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/CreateMembers" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateMembersResponse createMembers(Consumer<CreateMembersRequest.Builder> createMembersRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, ServiceQuotaExceededException,
            AwsServiceException, SdkClientException, DetectiveException {
        return createMembers(CreateMembersRequest.builder().applyMutation(createMembersRequest).build());
    }

    /**
     * <p>
     * Disables the specified behavior graph and queues it to be deleted. This operation removes the graph from each
     * member account's list of behavior graphs.
     * </p>
     * <p>
     * <code>DeleteGraph</code> can only be called by the master account for a behavior graph.
     * </p>
     *
     * @param deleteGraphRequest
     * @return Result of the DeleteGraph operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.DeleteGraph
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DeleteGraph" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteGraphResponse deleteGraph(DeleteGraphRequest deleteGraphRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Disables the specified behavior graph and queues it to be deleted. This operation removes the graph from each
     * member account's list of behavior graphs.
     * </p>
     * <p>
     * <code>DeleteGraph</code> can only be called by the master account for a behavior graph.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteGraphRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteGraphRequest#builder()}
     * </p>
     *
     * @param deleteGraphRequest
     *        A {@link Consumer} that will call methods on {@link DeleteGraphRequest.Builder} to create a request.
     * @return Result of the DeleteGraph operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.DeleteGraph
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DeleteGraph" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteGraphResponse deleteGraph(Consumer<DeleteGraphRequest.Builder> deleteGraphRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, DetectiveException {
        return deleteGraph(DeleteGraphRequest.builder().applyMutation(deleteGraphRequest).build());
    }

    /**
     * <p>
     * Deletes one or more member accounts from the master account behavior graph. This operation can only be called by
     * a Detective master account. That account cannot use <code>DeleteMembers</code> to delete their own account from
     * the behavior graph. To disable a behavior graph, the master account uses the <code>DeleteGraph</code> API method.
     * </p>
     *
     * @param deleteMembersRequest
     * @return Result of the DeleteMembers operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.DeleteMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DeleteMembers" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteMembersResponse deleteMembers(DeleteMembersRequest deleteMembersRequest) throws ConflictException,
            InternalServerException, ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            DetectiveException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes one or more member accounts from the master account behavior graph. This operation can only be called by
     * a Detective master account. That account cannot use <code>DeleteMembers</code> to delete their own account from
     * the behavior graph. To disable a behavior graph, the master account uses the <code>DeleteGraph</code> API method.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteMembersRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteMembersRequest#builder()}
     * </p>
     *
     * @param deleteMembersRequest
     *        A {@link Consumer} that will call methods on {@link DeleteMembersRequest.Builder} to create a request.
     * @return Result of the DeleteMembers operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.DeleteMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DeleteMembers" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteMembersResponse deleteMembers(Consumer<DeleteMembersRequest.Builder> deleteMembersRequest)
            throws ConflictException, InternalServerException, ResourceNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, DetectiveException {
        return deleteMembers(DeleteMembersRequest.builder().applyMutation(deleteMembersRequest).build());
    }

    /**
     * <p>
     * Removes the member account from the specified behavior graph. This operation can only be called by a member
     * account that has the <code>ENABLED</code> status.
     * </p>
     *
     * @param disassociateMembershipRequest
     * @return Result of the DisassociateMembership operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.DisassociateMembership
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DisassociateMembership"
     *      target="_top">AWS API Documentation</a>
     */
    default DisassociateMembershipResponse disassociateMembership(DisassociateMembershipRequest disassociateMembershipRequest)
            throws ConflictException, InternalServerException, ResourceNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, DetectiveException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes the member account from the specified behavior graph. This operation can only be called by a member
     * account that has the <code>ENABLED</code> status.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DisassociateMembershipRequest.Builder} avoiding the
     * need to create one manually via {@link DisassociateMembershipRequest#builder()}
     * </p>
     *
     * @param disassociateMembershipRequest
     *        A {@link Consumer} that will call methods on {@link DisassociateMembershipRequest.Builder} to create a
     *        request.
     * @return Result of the DisassociateMembership operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.DisassociateMembership
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DisassociateMembership"
     *      target="_top">AWS API Documentation</a>
     */
    default DisassociateMembershipResponse disassociateMembership(
            Consumer<DisassociateMembershipRequest.Builder> disassociateMembershipRequest) throws ConflictException,
            InternalServerException, ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            DetectiveException {
        return disassociateMembership(DisassociateMembershipRequest.builder().applyMutation(disassociateMembershipRequest)
                .build());
    }

    /**
     * <p>
     * Returns the membership details for specified member accounts for a behavior graph.
     * </p>
     *
     * @param getMembersRequest
     * @return Result of the GetMembers operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.GetMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/GetMembers" target="_top">AWS API
     *      Documentation</a>
     */
    default GetMembersResponse getMembers(GetMembersRequest getMembersRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the membership details for specified member accounts for a behavior graph.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetMembersRequest.Builder} avoiding the need to
     * create one manually via {@link GetMembersRequest#builder()}
     * </p>
     *
     * @param getMembersRequest
     *        A {@link Consumer} that will call methods on {@link GetMembersRequest.Builder} to create a request.
     * @return Result of the GetMembers operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.GetMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/GetMembers" target="_top">AWS API
     *      Documentation</a>
     */
    default GetMembersResponse getMembers(Consumer<GetMembersRequest.Builder> getMembersRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        return getMembers(GetMembersRequest.builder().applyMutation(getMembersRequest).build());
    }

    /**
     * <p>
     * Returns the list of behavior graphs that the calling account is a master of. This operation can only be called by
     * a master account.
     * </p>
     * <p>
     * Because an account can currently only be the master of one behavior graph within a Region, the results always
     * contain a single graph.
     * </p>
     *
     * @param listGraphsRequest
     * @return Result of the ListGraphs operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListGraphs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListGraphs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListGraphsResponse listGraphs(ListGraphsRequest listGraphsRequest) throws InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the list of behavior graphs that the calling account is a master of. This operation can only be called by
     * a master account.
     * </p>
     * <p>
     * Because an account can currently only be the master of one behavior graph within a Region, the results always
     * contain a single graph.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListGraphsRequest.Builder} avoiding the need to
     * create one manually via {@link ListGraphsRequest#builder()}
     * </p>
     *
     * @param listGraphsRequest
     *        A {@link Consumer} that will call methods on {@link ListGraphsRequest.Builder} to create a request.
     * @return Result of the ListGraphs operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListGraphs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListGraphs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListGraphsResponse listGraphs(Consumer<ListGraphsRequest.Builder> listGraphsRequest) throws InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        return listGraphs(ListGraphsRequest.builder().applyMutation(listGraphsRequest).build());
    }

    /**
     * <p>
     * Returns the list of behavior graphs that the calling account is a master of. This operation can only be called by
     * a master account.
     * </p>
     * <p>
     * Because an account can currently only be the master of one behavior graph within a Region, the results always
     * contain a single graph.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listGraphs(software.amazon.awssdk.services.detective.model.ListGraphsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListGraphsIterable responses = client.listGraphsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.detective.paginators.ListGraphsIterable responses = client.listGraphsPaginator(request);
     *     for (software.amazon.awssdk.services.detective.model.ListGraphsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListGraphsIterable responses = client.listGraphsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGraphs(software.amazon.awssdk.services.detective.model.ListGraphsRequest)} operation.</b>
     * </p>
     *
     * @param listGraphsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListGraphs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListGraphs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListGraphsIterable listGraphsPaginator(ListGraphsRequest listGraphsRequest) throws InternalServerException,
            ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the list of behavior graphs that the calling account is a master of. This operation can only be called by
     * a master account.
     * </p>
     * <p>
     * Because an account can currently only be the master of one behavior graph within a Region, the results always
     * contain a single graph.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listGraphs(software.amazon.awssdk.services.detective.model.ListGraphsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListGraphsIterable responses = client.listGraphsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.detective.paginators.ListGraphsIterable responses = client.listGraphsPaginator(request);
     *     for (software.amazon.awssdk.services.detective.model.ListGraphsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListGraphsIterable responses = client.listGraphsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGraphs(software.amazon.awssdk.services.detective.model.ListGraphsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListGraphsRequest.Builder} avoiding the need to
     * create one manually via {@link ListGraphsRequest#builder()}
     * </p>
     *
     * @param listGraphsRequest
     *        A {@link Consumer} that will call methods on {@link ListGraphsRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListGraphs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListGraphs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListGraphsIterable listGraphsPaginator(Consumer<ListGraphsRequest.Builder> listGraphsRequest)
            throws InternalServerException, ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        return listGraphsPaginator(ListGraphsRequest.builder().applyMutation(listGraphsRequest).build());
    }

    /**
     * <p>
     * Retrieves the list of open and accepted behavior graph invitations for the member account. This operation can
     * only be called by a member account.
     * </p>
     * <p>
     * Open invitations are invitations that the member account has not responded to.
     * </p>
     * <p>
     * The results do not include behavior graphs for which the member account declined the invitation. The results also
     * do not include behavior graphs that the member account resigned from or was removed from.
     * </p>
     *
     * @param listInvitationsRequest
     * @return Result of the ListInvitations operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListInvitations" target="_top">AWS API
     *      Documentation</a>
     */
    default ListInvitationsResponse listInvitations(ListInvitationsRequest listInvitationsRequest)
            throws InternalServerException, ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the list of open and accepted behavior graph invitations for the member account. This operation can
     * only be called by a member account.
     * </p>
     * <p>
     * Open invitations are invitations that the member account has not responded to.
     * </p>
     * <p>
     * The results do not include behavior graphs for which the member account declined the invitation. The results also
     * do not include behavior graphs that the member account resigned from or was removed from.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListInvitationsRequest.Builder} avoiding the need
     * to create one manually via {@link ListInvitationsRequest#builder()}
     * </p>
     *
     * @param listInvitationsRequest
     *        A {@link Consumer} that will call methods on {@link ListInvitationsRequest.Builder} to create a request.
     * @return Result of the ListInvitations operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListInvitations" target="_top">AWS API
     *      Documentation</a>
     */
    default ListInvitationsResponse listInvitations(Consumer<ListInvitationsRequest.Builder> listInvitationsRequest)
            throws InternalServerException, ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        return listInvitations(ListInvitationsRequest.builder().applyMutation(listInvitationsRequest).build());
    }

    /**
     * <p>
     * Retrieves the list of open and accepted behavior graph invitations for the member account. This operation can
     * only be called by a member account.
     * </p>
     * <p>
     * Open invitations are invitations that the member account has not responded to.
     * </p>
     * <p>
     * The results do not include behavior graphs for which the member account declined the invitation. The results also
     * do not include behavior graphs that the member account resigned from or was removed from.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listInvitations(software.amazon.awssdk.services.detective.model.ListInvitationsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListInvitationsIterable responses = client.listInvitationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.detective.paginators.ListInvitationsIterable responses = client
     *             .listInvitationsPaginator(request);
     *     for (software.amazon.awssdk.services.detective.model.ListInvitationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListInvitationsIterable responses = client.listInvitationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInvitations(software.amazon.awssdk.services.detective.model.ListInvitationsRequest)} operation.</b>
     * </p>
     *
     * @param listInvitationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListInvitations" target="_top">AWS API
     *      Documentation</a>
     */
    default ListInvitationsIterable listInvitationsPaginator(ListInvitationsRequest listInvitationsRequest)
            throws InternalServerException, ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the list of open and accepted behavior graph invitations for the member account. This operation can
     * only be called by a member account.
     * </p>
     * <p>
     * Open invitations are invitations that the member account has not responded to.
     * </p>
     * <p>
     * The results do not include behavior graphs for which the member account declined the invitation. The results also
     * do not include behavior graphs that the member account resigned from or was removed from.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listInvitations(software.amazon.awssdk.services.detective.model.ListInvitationsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListInvitationsIterable responses = client.listInvitationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.detective.paginators.ListInvitationsIterable responses = client
     *             .listInvitationsPaginator(request);
     *     for (software.amazon.awssdk.services.detective.model.ListInvitationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListInvitationsIterable responses = client.listInvitationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInvitations(software.amazon.awssdk.services.detective.model.ListInvitationsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListInvitationsRequest.Builder} avoiding the need
     * to create one manually via {@link ListInvitationsRequest#builder()}
     * </p>
     *
     * @param listInvitationsRequest
     *        A {@link Consumer} that will call methods on {@link ListInvitationsRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListInvitations" target="_top">AWS API
     *      Documentation</a>
     */
    default ListInvitationsIterable listInvitationsPaginator(Consumer<ListInvitationsRequest.Builder> listInvitationsRequest)
            throws InternalServerException, ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        return listInvitationsPaginator(ListInvitationsRequest.builder().applyMutation(listInvitationsRequest).build());
    }

    /**
     * <p>
     * Retrieves the list of member accounts for a behavior graph. Does not return member accounts that were removed
     * from the behavior graph.
     * </p>
     *
     * @param listMembersRequest
     * @return Result of the ListMembers operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListMembers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListMembersResponse listMembers(ListMembersRequest listMembersRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the list of member accounts for a behavior graph. Does not return member accounts that were removed
     * from the behavior graph.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListMembersRequest.Builder} avoiding the need to
     * create one manually via {@link ListMembersRequest#builder()}
     * </p>
     *
     * @param listMembersRequest
     *        A {@link Consumer} that will call methods on {@link ListMembersRequest.Builder} to create a request.
     * @return Result of the ListMembers operation returned by the service.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListMembers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListMembersResponse listMembers(Consumer<ListMembersRequest.Builder> listMembersRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, DetectiveException {
        return listMembers(ListMembersRequest.builder().applyMutation(listMembersRequest).build());
    }

    /**
     * <p>
     * Retrieves the list of member accounts for a behavior graph. Does not return member accounts that were removed
     * from the behavior graph.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listMembers(software.amazon.awssdk.services.detective.model.ListMembersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListMembersIterable responses = client.listMembersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.detective.paginators.ListMembersIterable responses = client.listMembersPaginator(request);
     *     for (software.amazon.awssdk.services.detective.model.ListMembersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListMembersIterable responses = client.listMembersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMembers(software.amazon.awssdk.services.detective.model.ListMembersRequest)} operation.</b>
     * </p>
     *
     * @param listMembersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListMembers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListMembersIterable listMembersPaginator(ListMembersRequest listMembersRequest) throws InternalServerException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the list of member accounts for a behavior graph. Does not return member accounts that were removed
     * from the behavior graph.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listMembers(software.amazon.awssdk.services.detective.model.ListMembersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListMembersIterable responses = client.listMembersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.detective.paginators.ListMembersIterable responses = client.listMembersPaginator(request);
     *     for (software.amazon.awssdk.services.detective.model.ListMembersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListMembersIterable responses = client.listMembersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMembers(software.amazon.awssdk.services.detective.model.ListMembersRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListMembersRequest.Builder} avoiding the need to
     * create one manually via {@link ListMembersRequest#builder()}
     * </p>
     *
     * @param listMembersRequest
     *        A {@link Consumer} that will call methods on {@link ListMembersRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.ListMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListMembers" target="_top">AWS API
     *      Documentation</a>
     */
    default ListMembersIterable listMembersPaginator(Consumer<ListMembersRequest.Builder> listMembersRequest)
            throws InternalServerException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, DetectiveException {
        return listMembersPaginator(ListMembersRequest.builder().applyMutation(listMembersRequest).build());
    }

    /**
     * <p>
     * Rejects an invitation to contribute the account data to a behavior graph. This operation must be called by a
     * member account that has the <code>INVITED</code> status.
     * </p>
     *
     * @param rejectInvitationRequest
     * @return Result of the RejectInvitation operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.RejectInvitation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/RejectInvitation" target="_top">AWS API
     *      Documentation</a>
     */
    default RejectInvitationResponse rejectInvitation(RejectInvitationRequest rejectInvitationRequest) throws ConflictException,
            InternalServerException, ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            DetectiveException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Rejects an invitation to contribute the account data to a behavior graph. This operation must be called by a
     * member account that has the <code>INVITED</code> status.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link RejectInvitationRequest.Builder} avoiding the need
     * to create one manually via {@link RejectInvitationRequest#builder()}
     * </p>
     *
     * @param rejectInvitationRequest
     *        A {@link Consumer} that will call methods on {@link RejectInvitationRequest.Builder} to create a request.
     * @return Result of the RejectInvitation operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.RejectInvitation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/RejectInvitation" target="_top">AWS API
     *      Documentation</a>
     */
    default RejectInvitationResponse rejectInvitation(Consumer<RejectInvitationRequest.Builder> rejectInvitationRequest)
            throws ConflictException, InternalServerException, ResourceNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, DetectiveException {
        return rejectInvitation(RejectInvitationRequest.builder().applyMutation(rejectInvitationRequest).build());
    }

    /**
     * <p>
     * Sends a request to enable data ingest for a member account that has a status of
     * <code>ACCEPTED_BUT_DISABLED</code>.
     * </p>
     * <p>
     * For valid member accounts, the status is updated as follows.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If Detective enabled the member account, then the new status is <code>ENABLED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If Detective cannot enable the member account, the status remains <code>ACCEPTED_BUT_DISABLED</code>.
     * </p>
     * </li>
     * </ul>
     *
     * @param startMonitoringMemberRequest
     * @return Result of the StartMonitoringMember operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ServiceQuotaExceededException
     *         This request cannot be completed for one of the following reasons.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The request would cause the number of member accounts in the behavior graph to exceed the maximum
     *         allowed. A behavior graph cannot have more than 1000 member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The request would cause the data rate for the behavior graph to exceed the maximum allowed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Detective is unable to verify the data rate for the member account. This is usually because the member
     *         account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li>
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.StartMonitoringMember
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/StartMonitoringMember"
     *      target="_top">AWS API Documentation</a>
     */
    default StartMonitoringMemberResponse startMonitoringMember(StartMonitoringMemberRequest startMonitoringMemberRequest)
            throws ConflictException, InternalServerException, ResourceNotFoundException, ServiceQuotaExceededException,
            ValidationException, AwsServiceException, SdkClientException, DetectiveException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Sends a request to enable data ingest for a member account that has a status of
     * <code>ACCEPTED_BUT_DISABLED</code>.
     * </p>
     * <p>
     * For valid member accounts, the status is updated as follows.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If Detective enabled the member account, then the new status is <code>ENABLED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If Detective cannot enable the member account, the status remains <code>ACCEPTED_BUT_DISABLED</code>.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartMonitoringMemberRequest.Builder} avoiding the
     * need to create one manually via {@link StartMonitoringMemberRequest#builder()}
     * </p>
     *
     * @param startMonitoringMemberRequest
     *        A {@link Consumer} that will call methods on {@link StartMonitoringMemberRequest.Builder} to create a
     *        request.
     * @return Result of the StartMonitoringMember operation returned by the service.
     * @throws ConflictException
     *         The request attempted an invalid action.
     * @throws InternalServerException
     *         The request was valid but failed because of a problem with the service.
     * @throws ResourceNotFoundException
     *         The request refers to a nonexistent resource.
     * @throws ServiceQuotaExceededException
     *         This request cannot be completed for one of the following reasons.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The request would cause the number of member accounts in the behavior graph to exceed the maximum
     *         allowed. A behavior graph cannot have more than 1000 member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The request would cause the data rate for the behavior graph to exceed the maximum allowed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Detective is unable to verify the data rate for the member account. This is usually because the member
     *         account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li>
     * @throws ValidationException
     *         The request parameters are invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DetectiveException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DetectiveClient.StartMonitoringMember
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/StartMonitoringMember"
     *      target="_top">AWS API Documentation</a>
     */
    default StartMonitoringMemberResponse startMonitoringMember(
            Consumer<StartMonitoringMemberRequest.Builder> startMonitoringMemberRequest) throws ConflictException,
            InternalServerException, ResourceNotFoundException, ServiceQuotaExceededException, ValidationException,
            AwsServiceException, SdkClientException, DetectiveException {
        return startMonitoringMember(StartMonitoringMemberRequest.builder().applyMutation(startMonitoringMemberRequest).build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of("api.detective");
    }
}
