/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.detective;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.detective.model.AcceptInvitationRequest;
import software.amazon.awssdk.services.detective.model.AcceptInvitationResponse;
import software.amazon.awssdk.services.detective.model.ConflictException;
import software.amazon.awssdk.services.detective.model.CreateGraphRequest;
import software.amazon.awssdk.services.detective.model.CreateGraphResponse;
import software.amazon.awssdk.services.detective.model.CreateMembersRequest;
import software.amazon.awssdk.services.detective.model.CreateMembersResponse;
import software.amazon.awssdk.services.detective.model.DeleteGraphRequest;
import software.amazon.awssdk.services.detective.model.DeleteGraphResponse;
import software.amazon.awssdk.services.detective.model.DeleteMembersRequest;
import software.amazon.awssdk.services.detective.model.DeleteMembersResponse;
import software.amazon.awssdk.services.detective.model.DetectiveException;
import software.amazon.awssdk.services.detective.model.DetectiveRequest;
import software.amazon.awssdk.services.detective.model.DisassociateMembershipRequest;
import software.amazon.awssdk.services.detective.model.DisassociateMembershipResponse;
import software.amazon.awssdk.services.detective.model.GetMembersRequest;
import software.amazon.awssdk.services.detective.model.GetMembersResponse;
import software.amazon.awssdk.services.detective.model.InternalServerException;
import software.amazon.awssdk.services.detective.model.ListGraphsRequest;
import software.amazon.awssdk.services.detective.model.ListGraphsResponse;
import software.amazon.awssdk.services.detective.model.ListInvitationsRequest;
import software.amazon.awssdk.services.detective.model.ListInvitationsResponse;
import software.amazon.awssdk.services.detective.model.ListMembersRequest;
import software.amazon.awssdk.services.detective.model.ListMembersResponse;
import software.amazon.awssdk.services.detective.model.RejectInvitationRequest;
import software.amazon.awssdk.services.detective.model.RejectInvitationResponse;
import software.amazon.awssdk.services.detective.model.ResourceNotFoundException;
import software.amazon.awssdk.services.detective.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.detective.model.StartMonitoringMemberRequest;
import software.amazon.awssdk.services.detective.model.StartMonitoringMemberResponse;
import software.amazon.awssdk.services.detective.model.ValidationException;
import software.amazon.awssdk.services.detective.paginators.ListGraphsPublisher;
import software.amazon.awssdk.services.detective.paginators.ListInvitationsPublisher;
import software.amazon.awssdk.services.detective.paginators.ListMembersPublisher;
import software.amazon.awssdk.services.detective.transform.AcceptInvitationRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.CreateGraphRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.CreateMembersRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.DeleteGraphRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.DeleteMembersRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.DisassociateMembershipRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.GetMembersRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.ListGraphsRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.ListInvitationsRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.ListMembersRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.RejectInvitationRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.StartMonitoringMemberRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link DetectiveAsyncClient}.
 *
 * @see DetectiveAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDetectiveAsyncClient implements DetectiveAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultDetectiveAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultDetectiveAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Accepts an invitation for the member account to contribute data to a behavior graph. This operation can only be
     * called by an invited member account.
     * </p>
     * <p>
     * The request provides the ARN of behavior graph.
     * </p>
     * <p>
     * The member account status in the graph must be <code>INVITED</code>.
     * </p>
     *
     * @param acceptInvitationRequest
     * @return A Java Future containing the result of the AcceptInvitation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The request attempted an invalid action.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.AcceptInvitation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/AcceptInvitation" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptInvitationResponse> acceptInvitation(AcceptInvitationRequest acceptInvitationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptInvitationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptInvitation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcceptInvitationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptInvitationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptInvitationRequest, AcceptInvitationResponse>()
                            .withOperationName("AcceptInvitation")
                            .withMarshaller(new AcceptInvitationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(acceptInvitationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = acceptInvitationRequest.overrideConfiguration().orElse(null);
            CompletableFuture<AcceptInvitationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new behavior graph for the calling account, and sets that account as the administrator account. This
     * operation is called by the account that is enabling Detective.
     * </p>
     * <p>
     * Before you try to enable Detective, make sure that your account has been enrolled in Amazon GuardDuty for at
     * least 48 hours. If you do not meet this requirement, you cannot enable Detective. If you do meet the GuardDuty
     * prerequisite, then when you make the request to enable Detective, it checks whether your data volume is within
     * the Detective quota. If it exceeds the quota, then you cannot enable Detective.
     * </p>
     * <p>
     * The operation also enables Detective for the calling account in the currently selected Region. It returns the ARN
     * of the new behavior graph.
     * </p>
     * <p>
     * <code>CreateGraph</code> triggers a process to create the corresponding data tables for the new behavior graph.
     * </p>
     * <p>
     * An account can only be the administrator account for one behavior graph within a Region. If the same account
     * calls <code>CreateGraph</code> with the same administrator account, it always returns the same behavior graph
     * ARN. It does not create a new behavior graph.
     * </p>
     *
     * @param createGraphRequest
     * @return A Java Future containing the result of the CreateGraph operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The request attempted an invalid action.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ServiceQuotaExceededException This request cannot be completed for one of the following reasons.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The request would cause the number of member accounts in the behavior graph to exceed the maximum
     *         allowed. A behavior graph cannot have more than 1000 member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The request would cause the data rate for the behavior graph to exceed the maximum allowed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Detective is unable to verify the data rate for the member account. This is usually because the member
     *         account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.CreateGraph
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/CreateGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGraphResponse> createGraph(CreateGraphRequest createGraphRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGraph");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateGraphResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGraphResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGraphRequest, CreateGraphResponse>()
                            .withOperationName("CreateGraph").withMarshaller(new CreateGraphRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createGraphRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createGraphRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateGraphResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends a request to invite the specified AWS accounts to be member accounts in the behavior graph. This operation
     * can only be called by the administrator account for a behavior graph.
     * </p>
     * <p>
     * <code>CreateMembers</code> verifies the accounts and then invites the verified accounts. The administrator can
     * optionally specify to not send invitation emails to the member accounts. This would be used when the
     * administrator manages their member accounts centrally.
     * </p>
     * <p>
     * The request provides the behavior graph ARN and the list of accounts to invite.
     * </p>
     * <p>
     * The response separates the requested accounts into two lists:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The accounts that <code>CreateMembers</code> was able to start the verification for. This list includes member
     * accounts that are being verified, that have passed verification and are to be invited, and that have failed
     * verification.
     * </p>
     * </li>
     * <li>
     * <p>
     * The accounts that <code>CreateMembers</code> was unable to process. This list includes accounts that were already
     * invited to be member accounts in the behavior graph.
     * </p>
     * </li>
     * </ul>
     *
     * @param createMembersRequest
     * @return A Java Future containing the result of the CreateMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>ServiceQuotaExceededException This request cannot be completed for one of the following reasons.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The request would cause the number of member accounts in the behavior graph to exceed the maximum
     *         allowed. A behavior graph cannot have more than 1000 member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The request would cause the data rate for the behavior graph to exceed the maximum allowed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Detective is unable to verify the data rate for the member account. This is usually because the member
     *         account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.CreateMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/CreateMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMembersResponse> createMembers(CreateMembersRequest createMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMembersRequest, CreateMembersResponse>()
                            .withOperationName("CreateMembers")
                            .withMarshaller(new CreateMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createMembersRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createMembersRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables the specified behavior graph and queues it to be deleted. This operation removes the graph from each
     * member account's list of behavior graphs.
     * </p>
     * <p>
     * <code>DeleteGraph</code> can only be called by the administrator account for a behavior graph.
     * </p>
     *
     * @param deleteGraphRequest
     * @return A Java Future containing the result of the DeleteGraph operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.DeleteGraph
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DeleteGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGraphResponse> deleteGraph(DeleteGraphRequest deleteGraphRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGraph");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteGraphResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGraphResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGraphRequest, DeleteGraphResponse>()
                            .withOperationName("DeleteGraph").withMarshaller(new DeleteGraphRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteGraphRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteGraphRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteGraphResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes one or more member accounts from the administrator account's behavior graph. This operation can only be
     * called by a Detective administrator account. That account cannot use <code>DeleteMembers</code> to delete their
     * own account from the behavior graph. To disable a behavior graph, the administrator account uses the
     * <code>DeleteGraph</code> API method.
     * </p>
     *
     * @param deleteMembersRequest
     * @return A Java Future containing the result of the DeleteMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The request attempted an invalid action.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.DeleteMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DeleteMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMembersResponse> deleteMembers(DeleteMembersRequest deleteMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMembersRequest, DeleteMembersResponse>()
                            .withOperationName("DeleteMembers")
                            .withMarshaller(new DeleteMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteMembersRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteMembersRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the member account from the specified behavior graph. This operation can only be called by a member
     * account that has the <code>ENABLED</code> status.
     * </p>
     *
     * @param disassociateMembershipRequest
     * @return A Java Future containing the result of the DisassociateMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The request attempted an invalid action.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.DisassociateMembership
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DisassociateMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateMembershipResponse> disassociateMembership(
            DisassociateMembershipRequest disassociateMembershipRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateMembershipRequest, DisassociateMembershipResponse>()
                            .withOperationName("DisassociateMembership")
                            .withMarshaller(new DisassociateMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateMembershipRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = disassociateMembershipRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DisassociateMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the membership details for specified member accounts for a behavior graph.
     * </p>
     *
     * @param getMembersRequest
     * @return A Java Future containing the result of the GetMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.GetMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/GetMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetMembersResponse> getMembers(GetMembersRequest getMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMembersRequest, GetMembersResponse>().withOperationName("GetMembers")
                            .withMarshaller(new GetMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getMembersRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getMembersRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of behavior graphs that the calling account is an administrator account of. This operation can
     * only be called by an administrator account.
     * </p>
     * <p>
     * Because an account can currently only be the administrator of one behavior graph within a Region, the results
     * always contain a single behavior graph.
     * </p>
     *
     * @param listGraphsRequest
     * @return A Java Future containing the result of the ListGraphs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.ListGraphs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListGraphs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListGraphsResponse> listGraphs(ListGraphsRequest listGraphsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGraphsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGraphs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGraphsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListGraphsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGraphsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGraphsRequest, ListGraphsResponse>().withOperationName("ListGraphs")
                            .withMarshaller(new ListGraphsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listGraphsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listGraphsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListGraphsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of behavior graphs that the calling account is an administrator account of. This operation can
     * only be called by an administrator account.
     * </p>
     * <p>
     * Because an account can currently only be the administrator of one behavior graph within a Region, the results
     * always contain a single behavior graph.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listGraphs(software.amazon.awssdk.services.detective.model.ListGraphsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListGraphsPublisher publisher = client.listGraphsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListGraphsPublisher publisher = client.listGraphsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.detective.model.ListGraphsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.detective.model.ListGraphsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGraphs(software.amazon.awssdk.services.detective.model.ListGraphsRequest)} operation.</b>
     * </p>
     *
     * @param listGraphsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.ListGraphs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListGraphs" target="_top">AWS API
     *      Documentation</a>
     */
    public ListGraphsPublisher listGraphsPaginator(ListGraphsRequest listGraphsRequest) {
        return new ListGraphsPublisher(this, applyPaginatorUserAgent(listGraphsRequest));
    }

    /**
     * <p>
     * Retrieves the list of open and accepted behavior graph invitations for the member account. This operation can
     * only be called by a member account.
     * </p>
     * <p>
     * Open invitations are invitations that the member account has not responded to.
     * </p>
     * <p>
     * The results do not include behavior graphs for which the member account declined the invitation. The results also
     * do not include behavior graphs that the member account resigned from or was removed from.
     * </p>
     *
     * @param listInvitationsRequest
     * @return A Java Future containing the result of the ListInvitations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.ListInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListInvitations" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListInvitationsResponse> listInvitations(ListInvitationsRequest listInvitationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInvitationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInvitations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInvitationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInvitationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInvitationsRequest, ListInvitationsResponse>()
                            .withOperationName("ListInvitations")
                            .withMarshaller(new ListInvitationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listInvitationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listInvitationsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListInvitationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the list of open and accepted behavior graph invitations for the member account. This operation can
     * only be called by a member account.
     * </p>
     * <p>
     * Open invitations are invitations that the member account has not responded to.
     * </p>
     * <p>
     * The results do not include behavior graphs for which the member account declined the invitation. The results also
     * do not include behavior graphs that the member account resigned from or was removed from.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listInvitations(software.amazon.awssdk.services.detective.model.ListInvitationsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListInvitationsPublisher publisher = client.listInvitationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListInvitationsPublisher publisher = client.listInvitationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.detective.model.ListInvitationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.detective.model.ListInvitationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInvitations(software.amazon.awssdk.services.detective.model.ListInvitationsRequest)} operation.</b>
     * </p>
     *
     * @param listInvitationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.ListInvitations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListInvitations" target="_top">AWS API
     *      Documentation</a>
     */
    public ListInvitationsPublisher listInvitationsPaginator(ListInvitationsRequest listInvitationsRequest) {
        return new ListInvitationsPublisher(this, applyPaginatorUserAgent(listInvitationsRequest));
    }

    /**
     * <p>
     * Retrieves the list of member accounts for a behavior graph. Does not return member accounts that were removed
     * from the behavior graph.
     * </p>
     *
     * @param listMembersRequest
     * @return A Java Future containing the result of the ListMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.ListMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListMembersResponse> listMembers(ListMembersRequest listMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMembersRequest, ListMembersResponse>()
                            .withOperationName("ListMembers").withMarshaller(new ListMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listMembersRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listMembersRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the list of member accounts for a behavior graph. Does not return member accounts that were removed
     * from the behavior graph.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listMembers(software.amazon.awssdk.services.detective.model.ListMembersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListMembersPublisher publisher = client.listMembersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.detective.paginators.ListMembersPublisher publisher = client.listMembersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.detective.model.ListMembersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.detective.model.ListMembersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMembers(software.amazon.awssdk.services.detective.model.ListMembersRequest)} operation.</b>
     * </p>
     *
     * @param listMembersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.ListMembers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListMembers" target="_top">AWS API
     *      Documentation</a>
     */
    public ListMembersPublisher listMembersPaginator(ListMembersRequest listMembersRequest) {
        return new ListMembersPublisher(this, applyPaginatorUserAgent(listMembersRequest));
    }

    /**
     * <p>
     * Rejects an invitation to contribute the account data to a behavior graph. This operation must be called by a
     * member account that has the <code>INVITED</code> status.
     * </p>
     *
     * @param rejectInvitationRequest
     * @return A Java Future containing the result of the RejectInvitation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The request attempted an invalid action.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.RejectInvitation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/RejectInvitation" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RejectInvitationResponse> rejectInvitation(RejectInvitationRequest rejectInvitationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rejectInvitationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectInvitation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RejectInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RejectInvitationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RejectInvitationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RejectInvitationRequest, RejectInvitationResponse>()
                            .withOperationName("RejectInvitation")
                            .withMarshaller(new RejectInvitationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(rejectInvitationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = rejectInvitationRequest.overrideConfiguration().orElse(null);
            CompletableFuture<RejectInvitationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends a request to enable data ingest for a member account that has a status of
     * <code>ACCEPTED_BUT_DISABLED</code>.
     * </p>
     * <p>
     * For valid member accounts, the status is updated as follows.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If Detective enabled the member account, then the new status is <code>ENABLED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If Detective cannot enable the member account, the status remains <code>ACCEPTED_BUT_DISABLED</code>.
     * </p>
     * </li>
     * </ul>
     *
     * @param startMonitoringMemberRequest
     * @return A Java Future containing the result of the StartMonitoringMember operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The request attempted an invalid action.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ServiceQuotaExceededException This request cannot be completed for one of the following reasons.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The request would cause the number of member accounts in the behavior graph to exceed the maximum
     *         allowed. A behavior graph cannot have more than 1000 member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The request would cause the data rate for the behavior graph to exceed the maximum allowed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Detective is unable to verify the data rate for the member account. This is usually because the member
     *         account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li></li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.StartMonitoringMember
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/StartMonitoringMember"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartMonitoringMemberResponse> startMonitoringMember(
            StartMonitoringMemberRequest startMonitoringMemberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMonitoringMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMonitoringMember");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartMonitoringMemberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartMonitoringMemberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartMonitoringMemberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartMonitoringMemberRequest, StartMonitoringMemberResponse>()
                            .withOperationName("StartMonitoringMember")
                            .withMarshaller(new StartMonitoringMemberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startMonitoringMemberRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startMonitoringMemberRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<StartMonitoringMemberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DetectiveException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends DetectiveRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
