/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.detective.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details about a member account that was invited to contribute to a behavior graph.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MemberDetail implements SdkPojo, Serializable, ToCopyableBuilder<MemberDetail.Builder, MemberDetail> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountId").getter(getter(MemberDetail::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountId").build()).build();

    private static final SdkField<String> EMAIL_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EmailAddress").getter(getter(MemberDetail::emailAddress)).setter(setter(Builder::emailAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EmailAddress").build()).build();

    private static final SdkField<String> GRAPH_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GraphArn").getter(getter(MemberDetail::graphArn)).setter(setter(Builder::graphArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GraphArn").build()).build();

    private static final SdkField<String> MASTER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MasterId").getter(getter(MemberDetail::masterId)).setter(setter(Builder::masterId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MasterId").build()).build();

    private static final SdkField<String> ADMINISTRATOR_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AdministratorId").getter(getter(MemberDetail::administratorId)).setter(setter(Builder::administratorId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdministratorId").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(MemberDetail::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> DISABLED_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DisabledReason").getter(getter(MemberDetail::disabledReasonAsString))
            .setter(setter(Builder::disabledReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisabledReason").build()).build();

    private static final SdkField<Instant> INVITED_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("InvitedTime")
            .getter(getter(MemberDetail::invitedTime))
            .setter(setter(Builder::invitedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InvitedTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> UPDATED_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdatedTime")
            .getter(getter(MemberDetail::updatedTime))
            .setter(setter(Builder::updatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdatedTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Double> PERCENT_OF_GRAPH_UTILIZATION_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("PercentOfGraphUtilization").getter(getter(MemberDetail::percentOfGraphUtilization))
            .setter(setter(Builder::percentOfGraphUtilization))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PercentOfGraphUtilization").build())
            .build();

    private static final SdkField<Instant> PERCENT_OF_GRAPH_UTILIZATION_UPDATED_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("PercentOfGraphUtilizationUpdatedTime")
            .getter(getter(MemberDetail::percentOfGraphUtilizationUpdatedTime))
            .setter(setter(Builder::percentOfGraphUtilizationUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("PercentOfGraphUtilizationUpdatedTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            EMAIL_ADDRESS_FIELD, GRAPH_ARN_FIELD, MASTER_ID_FIELD, ADMINISTRATOR_ID_FIELD, STATUS_FIELD, DISABLED_REASON_FIELD,
            INVITED_TIME_FIELD, UPDATED_TIME_FIELD, PERCENT_OF_GRAPH_UTILIZATION_FIELD,
            PERCENT_OF_GRAPH_UTILIZATION_UPDATED_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String accountId;

    private final String emailAddress;

    private final String graphArn;

    private final String masterId;

    private final String administratorId;

    private final String status;

    private final String disabledReason;

    private final Instant invitedTime;

    private final Instant updatedTime;

    private final Double percentOfGraphUtilization;

    private final Instant percentOfGraphUtilizationUpdatedTime;

    private MemberDetail(BuilderImpl builder) {
        this.accountId = builder.accountId;
        this.emailAddress = builder.emailAddress;
        this.graphArn = builder.graphArn;
        this.masterId = builder.masterId;
        this.administratorId = builder.administratorId;
        this.status = builder.status;
        this.disabledReason = builder.disabledReason;
        this.invitedTime = builder.invitedTime;
        this.updatedTime = builder.updatedTime;
        this.percentOfGraphUtilization = builder.percentOfGraphUtilization;
        this.percentOfGraphUtilizationUpdatedTime = builder.percentOfGraphUtilizationUpdatedTime;
    }

    /**
     * <p>
     * The AWS account identifier for the member account.
     * </p>
     * 
     * @return The AWS account identifier for the member account.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The AWS account root user email address for the member account.
     * </p>
     * 
     * @return The AWS account root user email address for the member account.
     */
    public final String emailAddress() {
        return emailAddress;
    }

    /**
     * <p>
     * The ARN of the behavior graph that the member account was invited to.
     * </p>
     * 
     * @return The ARN of the behavior graph that the member account was invited to.
     */
    public final String graphArn() {
        return graphArn;
    }

    /**
     * <p>
     * Deprecated. Instead of <code>MasterId</code>, use <code>AdministratorId</code>.
     * </p>
     * <p>
     * The AWS account identifier of the administrator account for the behavior graph.
     * </p>
     * 
     * @return Deprecated. Instead of <code>MasterId</code>, use <code>AdministratorId</code>.</p>
     *         <p>
     *         The AWS account identifier of the administrator account for the behavior graph.
     */
    public final String masterId() {
        return masterId;
    }

    /**
     * <p>
     * The AWS account identifier of the administrator account for the behavior graph.
     * </p>
     * 
     * @return The AWS account identifier of the administrator account for the behavior graph.
     */
    public final String administratorId() {
        return administratorId;
    }

    /**
     * <p>
     * The current membership status of the member account. The status can have one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INVITED</code> - Indicates that the member was sent an invitation but has not yet responded.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VERIFICATION_IN_PROGRESS</code> - Indicates that Detective is verifying that the account identifier and
     * email address provided for the member account match. If they do match, then Detective sends the invitation. If
     * the email address and account identifier don't match, then the member cannot be added to the behavior graph.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VERIFICATION_FAILED</code> - Indicates that the account and email address provided for the member account
     * do not match, and Detective did not send an invitation to the account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ENABLED</code> - Indicates that the member account accepted the invitation to contribute to the behavior
     * graph.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACCEPTED_BUT_DISABLED</code> - Indicates that the member account accepted the invitation but is prevented
     * from contributing data to the behavior graph. <code>DisabledReason</code> provides the reason why the member
     * account is not enabled.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Member accounts that declined an invitation or that were removed from the behavior graph are not included.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MemberStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current membership status of the member account. The status can have one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INVITED</code> - Indicates that the member was sent an invitation but has not yet responded.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VERIFICATION_IN_PROGRESS</code> - Indicates that Detective is verifying that the account identifier
     *         and email address provided for the member account match. If they do match, then Detective sends the
     *         invitation. If the email address and account identifier don't match, then the member cannot be added to
     *         the behavior graph.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VERIFICATION_FAILED</code> - Indicates that the account and email address provided for the member
     *         account do not match, and Detective did not send an invitation to the account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ENABLED</code> - Indicates that the member account accepted the invitation to contribute to the
     *         behavior graph.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACCEPTED_BUT_DISABLED</code> - Indicates that the member account accepted the invitation but is
     *         prevented from contributing data to the behavior graph. <code>DisabledReason</code> provides the reason
     *         why the member account is not enabled.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Member accounts that declined an invitation or that were removed from the behavior graph are not
     *         included.
     * @see MemberStatus
     */
    public final MemberStatus status() {
        return MemberStatus.fromValue(status);
    }

    /**
     * <p>
     * The current membership status of the member account. The status can have one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INVITED</code> - Indicates that the member was sent an invitation but has not yet responded.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VERIFICATION_IN_PROGRESS</code> - Indicates that Detective is verifying that the account identifier and
     * email address provided for the member account match. If they do match, then Detective sends the invitation. If
     * the email address and account identifier don't match, then the member cannot be added to the behavior graph.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VERIFICATION_FAILED</code> - Indicates that the account and email address provided for the member account
     * do not match, and Detective did not send an invitation to the account.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ENABLED</code> - Indicates that the member account accepted the invitation to contribute to the behavior
     * graph.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACCEPTED_BUT_DISABLED</code> - Indicates that the member account accepted the invitation but is prevented
     * from contributing data to the behavior graph. <code>DisabledReason</code> provides the reason why the member
     * account is not enabled.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Member accounts that declined an invitation or that were removed from the behavior graph are not included.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MemberStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current membership status of the member account. The status can have one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INVITED</code> - Indicates that the member was sent an invitation but has not yet responded.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VERIFICATION_IN_PROGRESS</code> - Indicates that Detective is verifying that the account identifier
     *         and email address provided for the member account match. If they do match, then Detective sends the
     *         invitation. If the email address and account identifier don't match, then the member cannot be added to
     *         the behavior graph.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VERIFICATION_FAILED</code> - Indicates that the account and email address provided for the member
     *         account do not match, and Detective did not send an invitation to the account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ENABLED</code> - Indicates that the member account accepted the invitation to contribute to the
     *         behavior graph.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACCEPTED_BUT_DISABLED</code> - Indicates that the member account accepted the invitation but is
     *         prevented from contributing data to the behavior graph. <code>DisabledReason</code> provides the reason
     *         why the member account is not enabled.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Member accounts that declined an invitation or that were removed from the behavior graph are not
     *         included.
     * @see MemberStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * For member accounts with a status of <code>ACCEPTED_BUT_DISABLED</code>, the reason that the member account is
     * not enabled.
     * </p>
     * <p>
     * The reason can have one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>VOLUME_TOO_HIGH</code> - Indicates that adding the member account would cause the data volume for the
     * behavior graph to be too high.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VOLUME_UNKNOWN</code> - Indicates that Detective is unable to verify the data volume for the member
     * account. This is usually because the member account is not enrolled in Amazon GuardDuty.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #disabledReason}
     * will return {@link MemberDisabledReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #disabledReasonAsString}.
     * </p>
     * 
     * @return For member accounts with a status of <code>ACCEPTED_BUT_DISABLED</code>, the reason that the member
     *         account is not enabled.</p>
     *         <p>
     *         The reason can have one of the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>VOLUME_TOO_HIGH</code> - Indicates that adding the member account would cause the data volume for
     *         the behavior graph to be too high.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VOLUME_UNKNOWN</code> - Indicates that Detective is unable to verify the data volume for the member
     *         account. This is usually because the member account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li>
     * @see MemberDisabledReason
     */
    public final MemberDisabledReason disabledReason() {
        return MemberDisabledReason.fromValue(disabledReason);
    }

    /**
     * <p>
     * For member accounts with a status of <code>ACCEPTED_BUT_DISABLED</code>, the reason that the member account is
     * not enabled.
     * </p>
     * <p>
     * The reason can have one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>VOLUME_TOO_HIGH</code> - Indicates that adding the member account would cause the data volume for the
     * behavior graph to be too high.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VOLUME_UNKNOWN</code> - Indicates that Detective is unable to verify the data volume for the member
     * account. This is usually because the member account is not enrolled in Amazon GuardDuty.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #disabledReason}
     * will return {@link MemberDisabledReason#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #disabledReasonAsString}.
     * </p>
     * 
     * @return For member accounts with a status of <code>ACCEPTED_BUT_DISABLED</code>, the reason that the member
     *         account is not enabled.</p>
     *         <p>
     *         The reason can have one of the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>VOLUME_TOO_HIGH</code> - Indicates that adding the member account would cause the data volume for
     *         the behavior graph to be too high.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VOLUME_UNKNOWN</code> - Indicates that Detective is unable to verify the data volume for the member
     *         account. This is usually because the member account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li>
     * @see MemberDisabledReason
     */
    public final String disabledReasonAsString() {
        return disabledReason;
    }

    /**
     * <p>
     * The date and time that Detective sent the invitation to the member account. The value is in milliseconds since
     * the epoch.
     * </p>
     * 
     * @return The date and time that Detective sent the invitation to the member account. The value is in milliseconds
     *         since the epoch.
     */
    public final Instant invitedTime() {
        return invitedTime;
    }

    /**
     * <p>
     * The date and time that the member account was last updated. The value is in milliseconds since the epoch.
     * </p>
     * 
     * @return The date and time that the member account was last updated. The value is in milliseconds since the epoch.
     */
    public final Instant updatedTime() {
        return updatedTime;
    }

    /**
     * <p>
     * The member account data volume as a percentage of the maximum allowed data volume. 0 indicates 0 percent, and 100
     * indicates 100 percent.
     * </p>
     * <p>
     * Note that this is not the percentage of the behavior graph data volume.
     * </p>
     * <p>
     * For example, the data volume for the behavior graph is 80 GB per day. The maximum data volume is 160 GB per day.
     * If the data volume for the member account is 40 GB per day, then <code>PercentOfGraphUtilization</code> is 25. It
     * represents 25% of the maximum allowed data volume.
     * </p>
     * 
     * @return The member account data volume as a percentage of the maximum allowed data volume. 0 indicates 0 percent,
     *         and 100 indicates 100 percent.</p>
     *         <p>
     *         Note that this is not the percentage of the behavior graph data volume.
     *         </p>
     *         <p>
     *         For example, the data volume for the behavior graph is 80 GB per day. The maximum data volume is 160 GB
     *         per day. If the data volume for the member account is 40 GB per day, then
     *         <code>PercentOfGraphUtilization</code> is 25. It represents 25% of the maximum allowed data volume.
     */
    public final Double percentOfGraphUtilization() {
        return percentOfGraphUtilization;
    }

    /**
     * <p>
     * The date and time when the graph utilization percentage was last updated.
     * </p>
     * 
     * @return The date and time when the graph utilization percentage was last updated.
     */
    public final Instant percentOfGraphUtilizationUpdatedTime() {
        return percentOfGraphUtilizationUpdatedTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(emailAddress());
        hashCode = 31 * hashCode + Objects.hashCode(graphArn());
        hashCode = 31 * hashCode + Objects.hashCode(masterId());
        hashCode = 31 * hashCode + Objects.hashCode(administratorId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(disabledReasonAsString());
        hashCode = 31 * hashCode + Objects.hashCode(invitedTime());
        hashCode = 31 * hashCode + Objects.hashCode(updatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(percentOfGraphUtilization());
        hashCode = 31 * hashCode + Objects.hashCode(percentOfGraphUtilizationUpdatedTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MemberDetail)) {
            return false;
        }
        MemberDetail other = (MemberDetail) obj;
        return Objects.equals(accountId(), other.accountId()) && Objects.equals(emailAddress(), other.emailAddress())
                && Objects.equals(graphArn(), other.graphArn()) && Objects.equals(masterId(), other.masterId())
                && Objects.equals(administratorId(), other.administratorId())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(disabledReasonAsString(), other.disabledReasonAsString())
                && Objects.equals(invitedTime(), other.invitedTime()) && Objects.equals(updatedTime(), other.updatedTime())
                && Objects.equals(percentOfGraphUtilization(), other.percentOfGraphUtilization())
                && Objects.equals(percentOfGraphUtilizationUpdatedTime(), other.percentOfGraphUtilizationUpdatedTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MemberDetail").add("AccountId", accountId()).add("EmailAddress", emailAddress())
                .add("GraphArn", graphArn()).add("MasterId", masterId()).add("AdministratorId", administratorId())
                .add("Status", statusAsString()).add("DisabledReason", disabledReasonAsString())
                .add("InvitedTime", invitedTime()).add("UpdatedTime", updatedTime())
                .add("PercentOfGraphUtilization", percentOfGraphUtilization())
                .add("PercentOfGraphUtilizationUpdatedTime", percentOfGraphUtilizationUpdatedTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "EmailAddress":
            return Optional.ofNullable(clazz.cast(emailAddress()));
        case "GraphArn":
            return Optional.ofNullable(clazz.cast(graphArn()));
        case "MasterId":
            return Optional.ofNullable(clazz.cast(masterId()));
        case "AdministratorId":
            return Optional.ofNullable(clazz.cast(administratorId()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "DisabledReason":
            return Optional.ofNullable(clazz.cast(disabledReasonAsString()));
        case "InvitedTime":
            return Optional.ofNullable(clazz.cast(invitedTime()));
        case "UpdatedTime":
            return Optional.ofNullable(clazz.cast(updatedTime()));
        case "PercentOfGraphUtilization":
            return Optional.ofNullable(clazz.cast(percentOfGraphUtilization()));
        case "PercentOfGraphUtilizationUpdatedTime":
            return Optional.ofNullable(clazz.cast(percentOfGraphUtilizationUpdatedTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MemberDetail, T> g) {
        return obj -> g.apply((MemberDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MemberDetail> {
        /**
         * <p>
         * The AWS account identifier for the member account.
         * </p>
         * 
         * @param accountId
         *        The AWS account identifier for the member account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The AWS account root user email address for the member account.
         * </p>
         * 
         * @param emailAddress
         *        The AWS account root user email address for the member account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emailAddress(String emailAddress);

        /**
         * <p>
         * The ARN of the behavior graph that the member account was invited to.
         * </p>
         * 
         * @param graphArn
         *        The ARN of the behavior graph that the member account was invited to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder graphArn(String graphArn);

        /**
         * <p>
         * Deprecated. Instead of <code>MasterId</code>, use <code>AdministratorId</code>.
         * </p>
         * <p>
         * The AWS account identifier of the administrator account for the behavior graph.
         * </p>
         * 
         * @param masterId
         *        Deprecated. Instead of <code>MasterId</code>, use <code>AdministratorId</code>.</p>
         *        <p>
         *        The AWS account identifier of the administrator account for the behavior graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterId(String masterId);

        /**
         * <p>
         * The AWS account identifier of the administrator account for the behavior graph.
         * </p>
         * 
         * @param administratorId
         *        The AWS account identifier of the administrator account for the behavior graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder administratorId(String administratorId);

        /**
         * <p>
         * The current membership status of the member account. The status can have one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INVITED</code> - Indicates that the member was sent an invitation but has not yet responded.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VERIFICATION_IN_PROGRESS</code> - Indicates that Detective is verifying that the account identifier and
         * email address provided for the member account match. If they do match, then Detective sends the invitation.
         * If the email address and account identifier don't match, then the member cannot be added to the behavior
         * graph.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VERIFICATION_FAILED</code> - Indicates that the account and email address provided for the member
         * account do not match, and Detective did not send an invitation to the account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ENABLED</code> - Indicates that the member account accepted the invitation to contribute to the
         * behavior graph.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACCEPTED_BUT_DISABLED</code> - Indicates that the member account accepted the invitation but is
         * prevented from contributing data to the behavior graph. <code>DisabledReason</code> provides the reason why
         * the member account is not enabled.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Member accounts that declined an invitation or that were removed from the behavior graph are not included.
         * </p>
         * 
         * @param status
         *        The current membership status of the member account. The status can have one of the following
         *        values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INVITED</code> - Indicates that the member was sent an invitation but has not yet responded.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VERIFICATION_IN_PROGRESS</code> - Indicates that Detective is verifying that the account
         *        identifier and email address provided for the member account match. If they do match, then Detective
         *        sends the invitation. If the email address and account identifier don't match, then the member cannot
         *        be added to the behavior graph.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VERIFICATION_FAILED</code> - Indicates that the account and email address provided for the
         *        member account do not match, and Detective did not send an invitation to the account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ENABLED</code> - Indicates that the member account accepted the invitation to contribute to the
         *        behavior graph.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACCEPTED_BUT_DISABLED</code> - Indicates that the member account accepted the invitation but is
         *        prevented from contributing data to the behavior graph. <code>DisabledReason</code> provides the
         *        reason why the member account is not enabled.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Member accounts that declined an invitation or that were removed from the behavior graph are not
         *        included.
         * @see MemberStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MemberStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current membership status of the member account. The status can have one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INVITED</code> - Indicates that the member was sent an invitation but has not yet responded.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VERIFICATION_IN_PROGRESS</code> - Indicates that Detective is verifying that the account identifier and
         * email address provided for the member account match. If they do match, then Detective sends the invitation.
         * If the email address and account identifier don't match, then the member cannot be added to the behavior
         * graph.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VERIFICATION_FAILED</code> - Indicates that the account and email address provided for the member
         * account do not match, and Detective did not send an invitation to the account.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ENABLED</code> - Indicates that the member account accepted the invitation to contribute to the
         * behavior graph.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACCEPTED_BUT_DISABLED</code> - Indicates that the member account accepted the invitation but is
         * prevented from contributing data to the behavior graph. <code>DisabledReason</code> provides the reason why
         * the member account is not enabled.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Member accounts that declined an invitation or that were removed from the behavior graph are not included.
         * </p>
         * 
         * @param status
         *        The current membership status of the member account. The status can have one of the following
         *        values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INVITED</code> - Indicates that the member was sent an invitation but has not yet responded.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VERIFICATION_IN_PROGRESS</code> - Indicates that Detective is verifying that the account
         *        identifier and email address provided for the member account match. If they do match, then Detective
         *        sends the invitation. If the email address and account identifier don't match, then the member cannot
         *        be added to the behavior graph.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VERIFICATION_FAILED</code> - Indicates that the account and email address provided for the
         *        member account do not match, and Detective did not send an invitation to the account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ENABLED</code> - Indicates that the member account accepted the invitation to contribute to the
         *        behavior graph.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACCEPTED_BUT_DISABLED</code> - Indicates that the member account accepted the invitation but is
         *        prevented from contributing data to the behavior graph. <code>DisabledReason</code> provides the
         *        reason why the member account is not enabled.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Member accounts that declined an invitation or that were removed from the behavior graph are not
         *        included.
         * @see MemberStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MemberStatus
         */
        Builder status(MemberStatus status);

        /**
         * <p>
         * For member accounts with a status of <code>ACCEPTED_BUT_DISABLED</code>, the reason that the member account
         * is not enabled.
         * </p>
         * <p>
         * The reason can have one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>VOLUME_TOO_HIGH</code> - Indicates that adding the member account would cause the data volume for the
         * behavior graph to be too high.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VOLUME_UNKNOWN</code> - Indicates that Detective is unable to verify the data volume for the member
         * account. This is usually because the member account is not enrolled in Amazon GuardDuty.
         * </p>
         * </li>
         * </ul>
         * 
         * @param disabledReason
         *        For member accounts with a status of <code>ACCEPTED_BUT_DISABLED</code>, the reason that the member
         *        account is not enabled.</p>
         *        <p>
         *        The reason can have one of the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>VOLUME_TOO_HIGH</code> - Indicates that adding the member account would cause the data volume
         *        for the behavior graph to be too high.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VOLUME_UNKNOWN</code> - Indicates that Detective is unable to verify the data volume for the
         *        member account. This is usually because the member account is not enrolled in Amazon GuardDuty.
         *        </p>
         *        </li>
         * @see MemberDisabledReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MemberDisabledReason
         */
        Builder disabledReason(String disabledReason);

        /**
         * <p>
         * For member accounts with a status of <code>ACCEPTED_BUT_DISABLED</code>, the reason that the member account
         * is not enabled.
         * </p>
         * <p>
         * The reason can have one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>VOLUME_TOO_HIGH</code> - Indicates that adding the member account would cause the data volume for the
         * behavior graph to be too high.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VOLUME_UNKNOWN</code> - Indicates that Detective is unable to verify the data volume for the member
         * account. This is usually because the member account is not enrolled in Amazon GuardDuty.
         * </p>
         * </li>
         * </ul>
         * 
         * @param disabledReason
         *        For member accounts with a status of <code>ACCEPTED_BUT_DISABLED</code>, the reason that the member
         *        account is not enabled.</p>
         *        <p>
         *        The reason can have one of the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>VOLUME_TOO_HIGH</code> - Indicates that adding the member account would cause the data volume
         *        for the behavior graph to be too high.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VOLUME_UNKNOWN</code> - Indicates that Detective is unable to verify the data volume for the
         *        member account. This is usually because the member account is not enrolled in Amazon GuardDuty.
         *        </p>
         *        </li>
         * @see MemberDisabledReason
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MemberDisabledReason
         */
        Builder disabledReason(MemberDisabledReason disabledReason);

        /**
         * <p>
         * The date and time that Detective sent the invitation to the member account. The value is in milliseconds
         * since the epoch.
         * </p>
         * 
         * @param invitedTime
         *        The date and time that Detective sent the invitation to the member account. The value is in
         *        milliseconds since the epoch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder invitedTime(Instant invitedTime);

        /**
         * <p>
         * The date and time that the member account was last updated. The value is in milliseconds since the epoch.
         * </p>
         * 
         * @param updatedTime
         *        The date and time that the member account was last updated. The value is in milliseconds since the
         *        epoch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedTime(Instant updatedTime);

        /**
         * <p>
         * The member account data volume as a percentage of the maximum allowed data volume. 0 indicates 0 percent, and
         * 100 indicates 100 percent.
         * </p>
         * <p>
         * Note that this is not the percentage of the behavior graph data volume.
         * </p>
         * <p>
         * For example, the data volume for the behavior graph is 80 GB per day. The maximum data volume is 160 GB per
         * day. If the data volume for the member account is 40 GB per day, then <code>PercentOfGraphUtilization</code>
         * is 25. It represents 25% of the maximum allowed data volume.
         * </p>
         * 
         * @param percentOfGraphUtilization
         *        The member account data volume as a percentage of the maximum allowed data volume. 0 indicates 0
         *        percent, and 100 indicates 100 percent.</p>
         *        <p>
         *        Note that this is not the percentage of the behavior graph data volume.
         *        </p>
         *        <p>
         *        For example, the data volume for the behavior graph is 80 GB per day. The maximum data volume is 160
         *        GB per day. If the data volume for the member account is 40 GB per day, then
         *        <code>PercentOfGraphUtilization</code> is 25. It represents 25% of the maximum allowed data volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentOfGraphUtilization(Double percentOfGraphUtilization);

        /**
         * <p>
         * The date and time when the graph utilization percentage was last updated.
         * </p>
         * 
         * @param percentOfGraphUtilizationUpdatedTime
         *        The date and time when the graph utilization percentage was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentOfGraphUtilizationUpdatedTime(Instant percentOfGraphUtilizationUpdatedTime);
    }

    static final class BuilderImpl implements Builder {
        private String accountId;

        private String emailAddress;

        private String graphArn;

        private String masterId;

        private String administratorId;

        private String status;

        private String disabledReason;

        private Instant invitedTime;

        private Instant updatedTime;

        private Double percentOfGraphUtilization;

        private Instant percentOfGraphUtilizationUpdatedTime;

        private BuilderImpl() {
        }

        private BuilderImpl(MemberDetail model) {
            accountId(model.accountId);
            emailAddress(model.emailAddress);
            graphArn(model.graphArn);
            masterId(model.masterId);
            administratorId(model.administratorId);
            status(model.status);
            disabledReason(model.disabledReason);
            invitedTime(model.invitedTime);
            updatedTime(model.updatedTime);
            percentOfGraphUtilization(model.percentOfGraphUtilization);
            percentOfGraphUtilizationUpdatedTime(model.percentOfGraphUtilizationUpdatedTime);
        }

        public final String getAccountId() {
            return accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        public final String getEmailAddress() {
            return emailAddress;
        }

        @Override
        public final Builder emailAddress(String emailAddress) {
            this.emailAddress = emailAddress;
            return this;
        }

        public final void setEmailAddress(String emailAddress) {
            this.emailAddress = emailAddress;
        }

        public final String getGraphArn() {
            return graphArn;
        }

        @Override
        public final Builder graphArn(String graphArn) {
            this.graphArn = graphArn;
            return this;
        }

        public final void setGraphArn(String graphArn) {
            this.graphArn = graphArn;
        }

        public final String getMasterId() {
            return masterId;
        }

        @Override
        public final Builder masterId(String masterId) {
            this.masterId = masterId;
            return this;
        }

        public final void setMasterId(String masterId) {
            this.masterId = masterId;
        }

        public final String getAdministratorId() {
            return administratorId;
        }

        @Override
        public final Builder administratorId(String administratorId) {
            this.administratorId = administratorId;
            return this;
        }

        public final void setAdministratorId(String administratorId) {
            this.administratorId = administratorId;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(MemberStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getDisabledReason() {
            return disabledReason;
        }

        @Override
        public final Builder disabledReason(String disabledReason) {
            this.disabledReason = disabledReason;
            return this;
        }

        @Override
        public final Builder disabledReason(MemberDisabledReason disabledReason) {
            this.disabledReason(disabledReason == null ? null : disabledReason.toString());
            return this;
        }

        public final void setDisabledReason(String disabledReason) {
            this.disabledReason = disabledReason;
        }

        public final Instant getInvitedTime() {
            return invitedTime;
        }

        @Override
        public final Builder invitedTime(Instant invitedTime) {
            this.invitedTime = invitedTime;
            return this;
        }

        public final void setInvitedTime(Instant invitedTime) {
            this.invitedTime = invitedTime;
        }

        public final Instant getUpdatedTime() {
            return updatedTime;
        }

        @Override
        public final Builder updatedTime(Instant updatedTime) {
            this.updatedTime = updatedTime;
            return this;
        }

        public final void setUpdatedTime(Instant updatedTime) {
            this.updatedTime = updatedTime;
        }

        public final Double getPercentOfGraphUtilization() {
            return percentOfGraphUtilization;
        }

        @Override
        public final Builder percentOfGraphUtilization(Double percentOfGraphUtilization) {
            this.percentOfGraphUtilization = percentOfGraphUtilization;
            return this;
        }

        public final void setPercentOfGraphUtilization(Double percentOfGraphUtilization) {
            this.percentOfGraphUtilization = percentOfGraphUtilization;
        }

        public final Instant getPercentOfGraphUtilizationUpdatedTime() {
            return percentOfGraphUtilizationUpdatedTime;
        }

        @Override
        public final Builder percentOfGraphUtilizationUpdatedTime(Instant percentOfGraphUtilizationUpdatedTime) {
            this.percentOfGraphUtilizationUpdatedTime = percentOfGraphUtilizationUpdatedTime;
            return this;
        }

        public final void setPercentOfGraphUtilizationUpdatedTime(Instant percentOfGraphUtilizationUpdatedTime) {
            this.percentOfGraphUtilizationUpdatedTime = percentOfGraphUtilizationUpdatedTime;
        }

        @Override
        public MemberDetail build() {
            return new MemberDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
