/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.detective;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.detective.internal.DetectiveServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.detective.model.AcceptInvitationRequest;
import software.amazon.awssdk.services.detective.model.AcceptInvitationResponse;
import software.amazon.awssdk.services.detective.model.AccessDeniedException;
import software.amazon.awssdk.services.detective.model.BatchGetGraphMemberDatasourcesRequest;
import software.amazon.awssdk.services.detective.model.BatchGetGraphMemberDatasourcesResponse;
import software.amazon.awssdk.services.detective.model.BatchGetMembershipDatasourcesRequest;
import software.amazon.awssdk.services.detective.model.BatchGetMembershipDatasourcesResponse;
import software.amazon.awssdk.services.detective.model.ConflictException;
import software.amazon.awssdk.services.detective.model.CreateGraphRequest;
import software.amazon.awssdk.services.detective.model.CreateGraphResponse;
import software.amazon.awssdk.services.detective.model.CreateMembersRequest;
import software.amazon.awssdk.services.detective.model.CreateMembersResponse;
import software.amazon.awssdk.services.detective.model.DeleteGraphRequest;
import software.amazon.awssdk.services.detective.model.DeleteGraphResponse;
import software.amazon.awssdk.services.detective.model.DeleteMembersRequest;
import software.amazon.awssdk.services.detective.model.DeleteMembersResponse;
import software.amazon.awssdk.services.detective.model.DescribeOrganizationConfigurationRequest;
import software.amazon.awssdk.services.detective.model.DescribeOrganizationConfigurationResponse;
import software.amazon.awssdk.services.detective.model.DetectiveException;
import software.amazon.awssdk.services.detective.model.DisableOrganizationAdminAccountRequest;
import software.amazon.awssdk.services.detective.model.DisableOrganizationAdminAccountResponse;
import software.amazon.awssdk.services.detective.model.DisassociateMembershipRequest;
import software.amazon.awssdk.services.detective.model.DisassociateMembershipResponse;
import software.amazon.awssdk.services.detective.model.EnableOrganizationAdminAccountRequest;
import software.amazon.awssdk.services.detective.model.EnableOrganizationAdminAccountResponse;
import software.amazon.awssdk.services.detective.model.GetInvestigationRequest;
import software.amazon.awssdk.services.detective.model.GetInvestigationResponse;
import software.amazon.awssdk.services.detective.model.GetMembersRequest;
import software.amazon.awssdk.services.detective.model.GetMembersResponse;
import software.amazon.awssdk.services.detective.model.InternalServerException;
import software.amazon.awssdk.services.detective.model.ListDatasourcePackagesRequest;
import software.amazon.awssdk.services.detective.model.ListDatasourcePackagesResponse;
import software.amazon.awssdk.services.detective.model.ListGraphsRequest;
import software.amazon.awssdk.services.detective.model.ListGraphsResponse;
import software.amazon.awssdk.services.detective.model.ListIndicatorsRequest;
import software.amazon.awssdk.services.detective.model.ListIndicatorsResponse;
import software.amazon.awssdk.services.detective.model.ListInvestigationsRequest;
import software.amazon.awssdk.services.detective.model.ListInvestigationsResponse;
import software.amazon.awssdk.services.detective.model.ListInvitationsRequest;
import software.amazon.awssdk.services.detective.model.ListInvitationsResponse;
import software.amazon.awssdk.services.detective.model.ListMembersRequest;
import software.amazon.awssdk.services.detective.model.ListMembersResponse;
import software.amazon.awssdk.services.detective.model.ListOrganizationAdminAccountsRequest;
import software.amazon.awssdk.services.detective.model.ListOrganizationAdminAccountsResponse;
import software.amazon.awssdk.services.detective.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.detective.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.detective.model.RejectInvitationRequest;
import software.amazon.awssdk.services.detective.model.RejectInvitationResponse;
import software.amazon.awssdk.services.detective.model.ResourceNotFoundException;
import software.amazon.awssdk.services.detective.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.detective.model.StartInvestigationRequest;
import software.amazon.awssdk.services.detective.model.StartInvestigationResponse;
import software.amazon.awssdk.services.detective.model.StartMonitoringMemberRequest;
import software.amazon.awssdk.services.detective.model.StartMonitoringMemberResponse;
import software.amazon.awssdk.services.detective.model.TagResourceRequest;
import software.amazon.awssdk.services.detective.model.TagResourceResponse;
import software.amazon.awssdk.services.detective.model.TooManyRequestsException;
import software.amazon.awssdk.services.detective.model.UntagResourceRequest;
import software.amazon.awssdk.services.detective.model.UntagResourceResponse;
import software.amazon.awssdk.services.detective.model.UpdateDatasourcePackagesRequest;
import software.amazon.awssdk.services.detective.model.UpdateDatasourcePackagesResponse;
import software.amazon.awssdk.services.detective.model.UpdateInvestigationStateRequest;
import software.amazon.awssdk.services.detective.model.UpdateInvestigationStateResponse;
import software.amazon.awssdk.services.detective.model.UpdateOrganizationConfigurationRequest;
import software.amazon.awssdk.services.detective.model.UpdateOrganizationConfigurationResponse;
import software.amazon.awssdk.services.detective.model.ValidationException;
import software.amazon.awssdk.services.detective.transform.AcceptInvitationRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.BatchGetGraphMemberDatasourcesRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.BatchGetMembershipDatasourcesRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.CreateGraphRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.CreateMembersRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.DeleteGraphRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.DeleteMembersRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.DescribeOrganizationConfigurationRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.DisableOrganizationAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.DisassociateMembershipRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.EnableOrganizationAdminAccountRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.GetInvestigationRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.GetMembersRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.ListDatasourcePackagesRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.ListGraphsRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.ListIndicatorsRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.ListInvestigationsRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.ListInvitationsRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.ListMembersRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.ListOrganizationAdminAccountsRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.RejectInvitationRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.StartInvestigationRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.StartMonitoringMemberRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.UpdateDatasourcePackagesRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.UpdateInvestigationStateRequestMarshaller;
import software.amazon.awssdk.services.detective.transform.UpdateOrganizationConfigurationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link DetectiveAsyncClient}.
 *
 * @see DetectiveAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDetectiveAsyncClient implements DetectiveAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultDetectiveAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultDetectiveAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Accepts an invitation for the member account to contribute data to a behavior graph. This operation can only be
     * called by an invited member account.
     * </p>
     * <p>
     * The request provides the ARN of behavior graph.
     * </p>
     * <p>
     * The member account status in the graph must be <code>INVITED</code>.
     * </p>
     *
     * @param acceptInvitationRequest
     * @return A Java Future containing the result of the AcceptInvitation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>ConflictException The request attempted an invalid action.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.AcceptInvitation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/AcceptInvitation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptInvitationResponse> acceptInvitation(AcceptInvitationRequest acceptInvitationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(acceptInvitationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptInvitationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptInvitation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcceptInvitationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptInvitationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptInvitationRequest, AcceptInvitationResponse>()
                            .withOperationName("AcceptInvitation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AcceptInvitationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(acceptInvitationRequest));
            CompletableFuture<AcceptInvitationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets data source package information for the behavior graph.
     * </p>
     *
     * @param batchGetGraphMemberDatasourcesRequest
     * @return A Java Future containing the result of the BatchGetGraphMemberDatasources operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.BatchGetGraphMemberDatasources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/BatchGetGraphMemberDatasources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetGraphMemberDatasourcesResponse> batchGetGraphMemberDatasources(
            BatchGetGraphMemberDatasourcesRequest batchGetGraphMemberDatasourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetGraphMemberDatasourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchGetGraphMemberDatasourcesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetGraphMemberDatasources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetGraphMemberDatasourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetGraphMemberDatasourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetGraphMemberDatasourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetGraphMemberDatasourcesRequest, BatchGetGraphMemberDatasourcesResponse>()
                            .withOperationName("BatchGetGraphMemberDatasources").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchGetGraphMemberDatasourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchGetGraphMemberDatasourcesRequest));
            CompletableFuture<BatchGetGraphMemberDatasourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information on the data source package history for an account.
     * </p>
     *
     * @param batchGetMembershipDatasourcesRequest
     * @return A Java Future containing the result of the BatchGetMembershipDatasources operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.BatchGetMembershipDatasources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/BatchGetMembershipDatasources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetMembershipDatasourcesResponse> batchGetMembershipDatasources(
            BatchGetMembershipDatasourcesRequest batchGetMembershipDatasourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetMembershipDatasourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchGetMembershipDatasourcesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetMembershipDatasources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetMembershipDatasourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetMembershipDatasourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetMembershipDatasourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetMembershipDatasourcesRequest, BatchGetMembershipDatasourcesResponse>()
                            .withOperationName("BatchGetMembershipDatasources").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchGetMembershipDatasourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchGetMembershipDatasourcesRequest));
            CompletableFuture<BatchGetMembershipDatasourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new behavior graph for the calling account, and sets that account as the administrator account. This
     * operation is called by the account that is enabling Detective.
     * </p>
     * <p>
     * Before you try to enable Detective, make sure that your account has been enrolled in Amazon GuardDuty for at
     * least 48 hours. If you do not meet this requirement, you cannot enable Detective. If you do meet the GuardDuty
     * prerequisite, then when you make the request to enable Detective, it checks whether your data volume is within
     * the Detective quota. If it exceeds the quota, then you cannot enable Detective.
     * </p>
     * <p>
     * The operation also enables Detective for the calling account in the currently selected Region. It returns the ARN
     * of the new behavior graph.
     * </p>
     * <p>
     * <code>CreateGraph</code> triggers a process to create the corresponding data tables for the new behavior graph.
     * </p>
     * <p>
     * An account can only be the administrator account for one behavior graph within a Region. If the same account
     * calls <code>CreateGraph</code> with the same administrator account, it always returns the same behavior graph
     * ARN. It does not create a new behavior graph.
     * </p>
     *
     * @param createGraphRequest
     * @return A Java Future containing the result of the CreateGraph operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>ConflictException The request attempted an invalid action.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ServiceQuotaExceededException This request cannot be completed for one of the following reasons.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The request would cause the number of member accounts in the behavior graph to exceed the maximum
     *         allowed. A behavior graph cannot have more than 1200 member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The request would cause the data rate for the behavior graph to exceed the maximum allowed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Detective is unable to verify the data rate for the member account. This is usually because the member
     *         account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.CreateGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/CreateGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGraphResponse> createGraph(CreateGraphRequest createGraphRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGraphRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGraph");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateGraphResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGraphResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGraphRequest, CreateGraphResponse>()
                            .withOperationName("CreateGraph").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateGraphRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createGraphRequest));
            CompletableFuture<CreateGraphResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <code>CreateMembers</code> is used to send invitations to accounts. For the organization behavior graph, the
     * Detective administrator account uses <code>CreateMembers</code> to enable organization accounts as member
     * accounts.
     * </p>
     * <p>
     * For invited accounts, <code>CreateMembers</code> sends a request to invite the specified Amazon Web Services
     * accounts to be member accounts in the behavior graph. This operation can only be called by the administrator
     * account for a behavior graph.
     * </p>
     * <p>
     * <code>CreateMembers</code> verifies the accounts and then invites the verified accounts. The administrator can
     * optionally specify to not send invitation emails to the member accounts. This would be used when the
     * administrator manages their member accounts centrally.
     * </p>
     * <p>
     * For organization accounts in the organization behavior graph, <code>CreateMembers</code> attempts to enable the
     * accounts. The organization accounts do not receive invitations.
     * </p>
     * <p>
     * The request provides the behavior graph ARN and the list of accounts to invite or to enable.
     * </p>
     * <p>
     * The response separates the requested accounts into two lists:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The accounts that <code>CreateMembers</code> was able to process. For invited accounts, includes member accounts
     * that are being verified, that have passed verification and are to be invited, and that have failed verification.
     * For organization accounts in the organization behavior graph, includes accounts that can be enabled and that
     * cannot be enabled.
     * </p>
     * </li>
     * <li>
     * <p>
     * The accounts that <code>CreateMembers</code> was unable to process. This list includes accounts that were already
     * invited to be member accounts in the behavior graph.
     * </p>
     * </li>
     * </ul>
     *
     * @param createMembersRequest
     * @return A Java Future containing the result of the CreateMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>ServiceQuotaExceededException This request cannot be completed for one of the following reasons.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The request would cause the number of member accounts in the behavior graph to exceed the maximum
     *         allowed. A behavior graph cannot have more than 1200 member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The request would cause the data rate for the behavior graph to exceed the maximum allowed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Detective is unable to verify the data rate for the member account. This is usually because the member
     *         account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.CreateMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/CreateMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMembersResponse> createMembers(CreateMembersRequest createMembersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createMembersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMembersRequest, CreateMembersResponse>()
                            .withOperationName("CreateMembers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createMembersRequest));
            CompletableFuture<CreateMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables the specified behavior graph and queues it to be deleted. This operation removes the behavior graph from
     * each member account's list of behavior graphs.
     * </p>
     * <p>
     * <code>DeleteGraph</code> can only be called by the administrator account for a behavior graph.
     * </p>
     *
     * @param deleteGraphRequest
     * @return A Java Future containing the result of the DeleteGraph operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.DeleteGraph
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DeleteGraph" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGraphResponse> deleteGraph(DeleteGraphRequest deleteGraphRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteGraphRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGraphRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGraph");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGraphResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteGraphResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGraphResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGraphRequest, DeleteGraphResponse>()
                            .withOperationName("DeleteGraph").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteGraphRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteGraphRequest));
            CompletableFuture<DeleteGraphResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified member accounts from the behavior graph. The removed accounts no longer contribute data to
     * the behavior graph. This operation can only be called by the administrator account for the behavior graph.
     * </p>
     * <p>
     * For invited accounts, the removed accounts are deleted from the list of accounts in the behavior graph. To
     * restore the account, the administrator account must send another invitation.
     * </p>
     * <p>
     * For organization accounts in the organization behavior graph, the Detective administrator account can always
     * enable the organization account again. Organization accounts that are not enabled as member accounts are not
     * included in the <code>ListMembers</code> results for the organization behavior graph.
     * </p>
     * <p>
     * An administrator account cannot use <code>DeleteMembers</code> to remove their own account from the behavior
     * graph. To disable a behavior graph, the administrator account uses the <code>DeleteGraph</code> API method.
     * </p>
     *
     * @param deleteMembersRequest
     * @return A Java Future containing the result of the DeleteMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>ConflictException The request attempted an invalid action.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.DeleteMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DeleteMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMembersResponse> deleteMembers(DeleteMembersRequest deleteMembersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMembersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMembersRequest, DeleteMembersResponse>()
                            .withOperationName("DeleteMembers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMembersRequest));
            CompletableFuture<DeleteMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the configuration for the organization behavior graph. Currently indicates whether to
     * automatically enable new organization accounts as member accounts.
     * </p>
     * <p>
     * Can only be called by the Detective administrator account for the organization.
     * </p>
     *
     * @param describeOrganizationConfigurationRequest
     * @return A Java Future containing the result of the DescribeOrganizationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>TooManyRequestsException The request cannot be completed because too many other requests are
     *         occurring at the same time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.DescribeOrganizationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DescribeOrganizationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeOrganizationConfigurationResponse> describeOrganizationConfiguration(
            DescribeOrganizationConfigurationRequest describeOrganizationConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeOrganizationConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeOrganizationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeOrganizationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeOrganizationConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeOrganizationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeOrganizationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeOrganizationConfigurationRequest, DescribeOrganizationConfigurationResponse>()
                            .withOperationName("DescribeOrganizationConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeOrganizationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeOrganizationConfigurationRequest));
            CompletableFuture<DescribeOrganizationConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the Detective administrator account in the current Region. Deletes the organization behavior graph.
     * </p>
     * <p>
     * Can only be called by the organization management account.
     * </p>
     * <p>
     * Removing the Detective administrator account does not affect the delegated administrator account for Detective in
     * Organizations.
     * </p>
     * <p>
     * To remove the delegated administrator account in Organizations, use the Organizations API. Removing the delegated
     * administrator account also removes the Detective administrator account in all Regions, except for Regions where
     * the Detective administrator account is the organization management account.
     * </p>
     *
     * @param disableOrganizationAdminAccountRequest
     * @return A Java Future containing the result of the DisableOrganizationAdminAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>TooManyRequestsException The request cannot be completed because too many other requests are
     *         occurring at the same time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.DisableOrganizationAdminAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DisableOrganizationAdminAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableOrganizationAdminAccountResponse> disableOrganizationAdminAccount(
            DisableOrganizationAdminAccountRequest disableOrganizationAdminAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableOrganizationAdminAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disableOrganizationAdminAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableOrganizationAdminAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableOrganizationAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisableOrganizationAdminAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableOrganizationAdminAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableOrganizationAdminAccountRequest, DisableOrganizationAdminAccountResponse>()
                            .withOperationName("DisableOrganizationAdminAccount").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisableOrganizationAdminAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disableOrganizationAdminAccountRequest));
            CompletableFuture<DisableOrganizationAdminAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the member account from the specified behavior graph. This operation can only be called by an invited
     * member account that has the <code>ENABLED</code> status.
     * </p>
     * <p>
     * <code>DisassociateMembership</code> cannot be called by an organization account in the organization behavior
     * graph. For the organization behavior graph, the Detective administrator account determines which organization
     * accounts to enable or disable as member accounts.
     * </p>
     *
     * @param disassociateMembershipRequest
     * @return A Java Future containing the result of the DisassociateMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>ConflictException The request attempted an invalid action.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.DisassociateMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/DisassociateMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateMembershipResponse> disassociateMembership(
            DisassociateMembershipRequest disassociateMembershipRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateMembershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateMembershipRequest, DisassociateMembershipResponse>()
                            .withOperationName("DisassociateMembership").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateMembershipRequest));
            CompletableFuture<DisassociateMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Designates the Detective administrator account for the organization in the current Region.
     * </p>
     * <p>
     * If the account does not have Detective enabled, then enables Detective for that account and creates a new
     * behavior graph.
     * </p>
     * <p>
     * Can only be called by the organization management account.
     * </p>
     * <p>
     * If the organization has a delegated administrator account in Organizations, then the Detective administrator
     * account must be either the delegated administrator account or the organization management account.
     * </p>
     * <p>
     * If the organization does not have a delegated administrator account in Organizations, then you can choose any
     * account in the organization. If you choose an account other than the organization management account, Detective
     * calls Organizations to make that account the delegated administrator account for Detective. The organization
     * management account cannot be the delegated administrator account.
     * </p>
     *
     * @param enableOrganizationAdminAccountRequest
     * @return A Java Future containing the result of the EnableOrganizationAdminAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>TooManyRequestsException The request cannot be completed because too many other requests are
     *         occurring at the same time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.EnableOrganizationAdminAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/EnableOrganizationAdminAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableOrganizationAdminAccountResponse> enableOrganizationAdminAccount(
            EnableOrganizationAdminAccountRequest enableOrganizationAdminAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableOrganizationAdminAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                enableOrganizationAdminAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableOrganizationAdminAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableOrganizationAdminAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, EnableOrganizationAdminAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableOrganizationAdminAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableOrganizationAdminAccountRequest, EnableOrganizationAdminAccountResponse>()
                            .withOperationName("EnableOrganizationAdminAccount").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new EnableOrganizationAdminAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(enableOrganizationAdminAccountRequest));
            CompletableFuture<EnableOrganizationAdminAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the investigation results of an investigation for a behavior graph.
     * </p>
     *
     * @param getInvestigationRequest
     * @return A Java Future containing the result of the GetInvestigation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>TooManyRequestsException The request cannot be completed because too many other requests are
     *         occurring at the same time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.GetInvestigation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/GetInvestigation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetInvestigationResponse> getInvestigation(GetInvestigationRequest getInvestigationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getInvestigationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInvestigationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInvestigation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInvestigationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetInvestigationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetInvestigationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInvestigationRequest, GetInvestigationResponse>()
                            .withOperationName("GetInvestigation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetInvestigationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getInvestigationRequest));
            CompletableFuture<GetInvestigationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the membership details for specified member accounts for a behavior graph.
     * </p>
     *
     * @param getMembersRequest
     * @return A Java Future containing the result of the GetMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.GetMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/GetMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetMembersResponse> getMembers(GetMembersRequest getMembersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMembersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMembersRequest, GetMembersResponse>().withOperationName("GetMembers")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMembersRequest));
            CompletableFuture<GetMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists data source packages in the behavior graph.
     * </p>
     *
     * @param listDatasourcePackagesRequest
     * @return A Java Future containing the result of the ListDatasourcePackages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.ListDatasourcePackages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListDatasourcePackages"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatasourcePackagesResponse> listDatasourcePackages(
            ListDatasourcePackagesRequest listDatasourcePackagesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDatasourcePackagesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatasourcePackagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatasourcePackages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatasourcePackagesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDatasourcePackagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatasourcePackagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatasourcePackagesRequest, ListDatasourcePackagesResponse>()
                            .withOperationName("ListDatasourcePackages").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDatasourcePackagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDatasourcePackagesRequest));
            CompletableFuture<ListDatasourcePackagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the list of behavior graphs that the calling account is an administrator account of. This operation can
     * only be called by an administrator account.
     * </p>
     * <p>
     * Because an account can currently only be the administrator of one behavior graph within a Region, the results
     * always contain a single behavior graph.
     * </p>
     *
     * @param listGraphsRequest
     * @return A Java Future containing the result of the ListGraphs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.ListGraphs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListGraphs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListGraphsResponse> listGraphs(ListGraphsRequest listGraphsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGraphsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGraphsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGraphs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGraphsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListGraphsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGraphsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGraphsRequest, ListGraphsResponse>().withOperationName("ListGraphs")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListGraphsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listGraphsRequest));
            CompletableFuture<ListGraphsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get the indicators from an investigation
     * </p>
     *
     * @param listIndicatorsRequest
     * @return A Java Future containing the result of the ListIndicators operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>TooManyRequestsException The request cannot be completed because too many other requests are
     *         occurring at the same time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.ListIndicators
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListIndicators" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListIndicatorsResponse> listIndicators(ListIndicatorsRequest listIndicatorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listIndicatorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIndicatorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIndicators");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListIndicatorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListIndicatorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListIndicatorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListIndicatorsRequest, ListIndicatorsResponse>()
                            .withOperationName("ListIndicators").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListIndicatorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listIndicatorsRequest));
            CompletableFuture<ListIndicatorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all Investigations.
     * </p>
     *
     * @param listInvestigationsRequest
     * @return A Java Future containing the result of the ListInvestigations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>TooManyRequestsException The request cannot be completed because too many other requests are
     *         occurring at the same time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.ListInvestigations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListInvestigations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInvestigationsResponse> listInvestigations(ListInvestigationsRequest listInvestigationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInvestigationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInvestigationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInvestigations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInvestigationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInvestigationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInvestigationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInvestigationsRequest, ListInvestigationsResponse>()
                            .withOperationName("ListInvestigations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListInvestigationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listInvestigationsRequest));
            CompletableFuture<ListInvestigationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the list of open and accepted behavior graph invitations for the member account. This operation can
     * only be called by an invited member account.
     * </p>
     * <p>
     * Open invitations are invitations that the member account has not responded to.
     * </p>
     * <p>
     * The results do not include behavior graphs for which the member account declined the invitation. The results also
     * do not include behavior graphs that the member account resigned from or was removed from.
     * </p>
     *
     * @param listInvitationsRequest
     * @return A Java Future containing the result of the ListInvitations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.ListInvitations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListInvitations" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListInvitationsResponse> listInvitations(ListInvitationsRequest listInvitationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInvitationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInvitationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInvitations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInvitationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInvitationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInvitationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInvitationsRequest, ListInvitationsResponse>()
                            .withOperationName("ListInvitations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListInvitationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listInvitationsRequest));
            CompletableFuture<ListInvitationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the list of member accounts for a behavior graph.
     * </p>
     * <p>
     * For invited accounts, the results do not include member accounts that were removed from the behavior graph.
     * </p>
     * <p>
     * For the organization behavior graph, the results do not include organization accounts that the Detective
     * administrator account has not enabled as member accounts.
     * </p>
     *
     * @param listMembersRequest
     * @return A Java Future containing the result of the ListMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.ListMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListMembersResponse> listMembers(ListMembersRequest listMembersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMembersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMembersRequest, ListMembersResponse>()
                            .withOperationName("ListMembers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMembersRequest));
            CompletableFuture<ListMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the Detective administrator account for an organization. Can only be called by the
     * organization management account.
     * </p>
     *
     * @param listOrganizationAdminAccountsRequest
     * @return A Java Future containing the result of the ListOrganizationAdminAccounts operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>TooManyRequestsException The request cannot be completed because too many other requests are
     *         occurring at the same time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.ListOrganizationAdminAccounts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListOrganizationAdminAccounts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOrganizationAdminAccountsResponse> listOrganizationAdminAccounts(
            ListOrganizationAdminAccountsRequest listOrganizationAdminAccountsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listOrganizationAdminAccountsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listOrganizationAdminAccountsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOrganizationAdminAccounts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOrganizationAdminAccountsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListOrganizationAdminAccountsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOrganizationAdminAccountsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOrganizationAdminAccountsRequest, ListOrganizationAdminAccountsResponse>()
                            .withOperationName("ListOrganizationAdminAccounts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListOrganizationAdminAccountsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listOrganizationAdminAccountsRequest));
            CompletableFuture<ListOrganizationAdminAccountsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the tag values that are assigned to a behavior graph.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Rejects an invitation to contribute the account data to a behavior graph. This operation must be called by an
     * invited member account that has the <code>INVITED</code> status.
     * </p>
     * <p>
     * <code>RejectInvitation</code> cannot be called by an organization account in the organization behavior graph. In
     * the organization behavior graph, organization accounts do not receive an invitation.
     * </p>
     *
     * @param rejectInvitationRequest
     * @return A Java Future containing the result of the RejectInvitation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>ConflictException The request attempted an invalid action.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.RejectInvitation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/RejectInvitation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RejectInvitationResponse> rejectInvitation(RejectInvitationRequest rejectInvitationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rejectInvitationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rejectInvitationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectInvitation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RejectInvitationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RejectInvitationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RejectInvitationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RejectInvitationRequest, RejectInvitationResponse>()
                            .withOperationName("RejectInvitation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RejectInvitationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(rejectInvitationRequest));
            CompletableFuture<RejectInvitationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * initiate an investigation on an entity in a graph
     * </p>
     *
     * @param startInvestigationRequest
     * @return A Java Future containing the result of the StartInvestigation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>TooManyRequestsException The request cannot be completed because too many other requests are
     *         occurring at the same time.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.StartInvestigation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/StartInvestigation" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StartInvestigationResponse> startInvestigation(StartInvestigationRequest startInvestigationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startInvestigationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startInvestigationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartInvestigation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartInvestigationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartInvestigationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartInvestigationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartInvestigationRequest, StartInvestigationResponse>()
                            .withOperationName("StartInvestigation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartInvestigationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startInvestigationRequest));
            CompletableFuture<StartInvestigationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends a request to enable data ingest for a member account that has a status of
     * <code>ACCEPTED_BUT_DISABLED</code>.
     * </p>
     * <p>
     * For valid member accounts, the status is updated as follows.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If Detective enabled the member account, then the new status is <code>ENABLED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If Detective cannot enable the member account, the status remains <code>ACCEPTED_BUT_DISABLED</code>.
     * </p>
     * </li>
     * </ul>
     *
     * @param startMonitoringMemberRequest
     * @return A Java Future containing the result of the StartMonitoringMember operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>ConflictException The request attempted an invalid action.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ServiceQuotaExceededException This request cannot be completed for one of the following reasons.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The request would cause the number of member accounts in the behavior graph to exceed the maximum
     *         allowed. A behavior graph cannot have more than 1200 member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The request would cause the data rate for the behavior graph to exceed the maximum allowed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Detective is unable to verify the data rate for the member account. This is usually because the member
     *         account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li></li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.StartMonitoringMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/StartMonitoringMember"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartMonitoringMemberResponse> startMonitoringMember(
            StartMonitoringMemberRequest startMonitoringMemberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startMonitoringMemberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMonitoringMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMonitoringMember");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartMonitoringMemberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartMonitoringMemberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartMonitoringMemberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartMonitoringMemberRequest, StartMonitoringMemberResponse>()
                            .withOperationName("StartMonitoringMember").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartMonitoringMemberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startMonitoringMemberRequest));
            CompletableFuture<StartMonitoringMemberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Applies tag values to a behavior graph.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from a behavior graph.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a data source packages for the behavior graph.
     * </p>
     *
     * @param updateDatasourcePackagesRequest
     * @return A Java Future containing the result of the UpdateDatasourcePackages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>ServiceQuotaExceededException This request cannot be completed for one of the following reasons.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The request would cause the number of member accounts in the behavior graph to exceed the maximum
     *         allowed. A behavior graph cannot have more than 1200 member accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The request would cause the data rate for the behavior graph to exceed the maximum allowed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Detective is unable to verify the data rate for the member account. This is usually because the member
     *         account is not enrolled in Amazon GuardDuty.
     *         </p>
     *         </li></li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.UpdateDatasourcePackages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/UpdateDatasourcePackages"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDatasourcePackagesResponse> updateDatasourcePackages(
            UpdateDatasourcePackagesRequest updateDatasourcePackagesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDatasourcePackagesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDatasourcePackagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDatasourcePackages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDatasourcePackagesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDatasourcePackagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDatasourcePackagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDatasourcePackagesRequest, UpdateDatasourcePackagesResponse>()
                            .withOperationName("UpdateDatasourcePackages").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDatasourcePackagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDatasourcePackagesRequest));
            CompletableFuture<UpdateDatasourcePackagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update the state of an investigation.
     * </p>
     *
     * @param updateInvestigationStateRequest
     * @return A Java Future containing the result of the UpdateInvestigationState operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>ResourceNotFoundException The request refers to a nonexistent resource.</li>
     *         <li>TooManyRequestsException The request cannot be completed because too many other requests are
     *         occurring at the same time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.UpdateInvestigationState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/UpdateInvestigationState"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateInvestigationStateResponse> updateInvestigationState(
            UpdateInvestigationStateRequest updateInvestigationStateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateInvestigationStateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateInvestigationStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateInvestigationState");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateInvestigationStateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateInvestigationStateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateInvestigationStateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateInvestigationStateRequest, UpdateInvestigationStateResponse>()
                            .withOperationName("UpdateInvestigationState").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateInvestigationStateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateInvestigationStateRequest));
            CompletableFuture<UpdateInvestigationStateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the configuration for the Organizations integration in the current Region. Can only be called by the
     * Detective administrator account for the organization.
     * </p>
     *
     * @param updateOrganizationConfigurationRequest
     * @return A Java Future containing the result of the UpdateOrganizationConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException The request issuer does not have permission to access this resource or perform
     *         this operation.</li>
     *         <li>InternalServerException The request was valid but failed because of a problem with the service.</li>
     *         <li>ValidationException The request parameters are invalid.</li>
     *         <li>TooManyRequestsException The request cannot be completed because too many other requests are
     *         occurring at the same time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DetectiveException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DetectiveAsyncClient.UpdateOrganizationConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/detective-2018-10-26/UpdateOrganizationConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateOrganizationConfigurationResponse> updateOrganizationConfiguration(
            UpdateOrganizationConfigurationRequest updateOrganizationConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateOrganizationConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateOrganizationConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Detective");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateOrganizationConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateOrganizationConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateOrganizationConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateOrganizationConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateOrganizationConfigurationRequest, UpdateOrganizationConfigurationResponse>()
                            .withOperationName("UpdateOrganizationConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateOrganizationConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateOrganizationConfigurationRequest));
            CompletableFuture<UpdateOrganizationConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final DetectiveServiceClientConfiguration serviceClientConfiguration() {
        return new DetectiveServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DetectiveException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        DetectiveServiceClientConfigurationBuilder serviceConfigBuilder = new DetectiveServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
