/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.detective.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateMembersRequest extends DetectiveRequest implements
        ToCopyableBuilder<CreateMembersRequest.Builder, CreateMembersRequest> {
    private static final SdkField<String> GRAPH_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GraphArn").getter(getter(CreateMembersRequest::graphArn)).setter(setter(Builder::graphArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GraphArn").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Message")
            .getter(getter(CreateMembersRequest::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Message").build()).build();

    private static final SdkField<Boolean> DISABLE_EMAIL_NOTIFICATION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DisableEmailNotification").getter(getter(CreateMembersRequest::disableEmailNotification))
            .setter(setter(Builder::disableEmailNotification))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisableEmailNotification").build())
            .build();

    private static final SdkField<List<Account>> ACCOUNTS_FIELD = SdkField
            .<List<Account>> builder(MarshallingType.LIST)
            .memberName("Accounts")
            .getter(getter(CreateMembersRequest::accounts))
            .setter(setter(Builder::accounts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Accounts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Account> builder(MarshallingType.SDK_POJO)
                                            .constructor(Account::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(GRAPH_ARN_FIELD,
            MESSAGE_FIELD, DISABLE_EMAIL_NOTIFICATION_FIELD, ACCOUNTS_FIELD));

    private final String graphArn;

    private final String message;

    private final Boolean disableEmailNotification;

    private final List<Account> accounts;

    private CreateMembersRequest(BuilderImpl builder) {
        super(builder);
        this.graphArn = builder.graphArn;
        this.message = builder.message;
        this.disableEmailNotification = builder.disableEmailNotification;
        this.accounts = builder.accounts;
    }

    /**
     * <p>
     * The ARN of the behavior graph.
     * </p>
     * 
     * @return The ARN of the behavior graph.
     */
    public final String graphArn() {
        return graphArn;
    }

    /**
     * <p>
     * Customized message text to include in the invitation email message to the invited member accounts.
     * </p>
     * 
     * @return Customized message text to include in the invitation email message to the invited member accounts.
     */
    public final String message() {
        return message;
    }

    /**
     * <p>
     * if set to <code>true</code>, then the invited accounts do not receive email notifications. By default, this is
     * set to <code>false</code>, and the invited accounts receive email notifications.
     * </p>
     * <p>
     * Organization accounts in the organization behavior graph do not receive email notifications.
     * </p>
     * 
     * @return if set to <code>true</code>, then the invited accounts do not receive email notifications. By default,
     *         this is set to <code>false</code>, and the invited accounts receive email notifications.</p>
     *         <p>
     *         Organization accounts in the organization behavior graph do not receive email notifications.
     */
    public final Boolean disableEmailNotification() {
        return disableEmailNotification;
    }

    /**
     * For responses, this returns true if the service returned a value for the Accounts property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAccounts() {
        return accounts != null && !(accounts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of Amazon Web Services accounts to invite or to enable. You can invite or enable up to 50 accounts at a
     * time. For each invited account, the account list contains the account identifier and the Amazon Web Services
     * account root user email address. For organization accounts in the organization behavior graph, the email address
     * is not required.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAccounts} method.
     * </p>
     * 
     * @return The list of Amazon Web Services accounts to invite or to enable. You can invite or enable up to 50
     *         accounts at a time. For each invited account, the account list contains the account identifier and the
     *         Amazon Web Services account root user email address. For organization accounts in the organization
     *         behavior graph, the email address is not required.
     */
    public final List<Account> accounts() {
        return accounts;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(graphArn());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(disableEmailNotification());
        hashCode = 31 * hashCode + Objects.hashCode(hasAccounts() ? accounts() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateMembersRequest)) {
            return false;
        }
        CreateMembersRequest other = (CreateMembersRequest) obj;
        return Objects.equals(graphArn(), other.graphArn()) && Objects.equals(message(), other.message())
                && Objects.equals(disableEmailNotification(), other.disableEmailNotification())
                && hasAccounts() == other.hasAccounts() && Objects.equals(accounts(), other.accounts());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateMembersRequest").add("GraphArn", graphArn())
                .add("Message", message() == null ? null : "*** Sensitive Data Redacted ***")
                .add("DisableEmailNotification", disableEmailNotification()).add("Accounts", hasAccounts() ? accounts() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "GraphArn":
            return Optional.ofNullable(clazz.cast(graphArn()));
        case "Message":
            return Optional.ofNullable(clazz.cast(message()));
        case "DisableEmailNotification":
            return Optional.ofNullable(clazz.cast(disableEmailNotification()));
        case "Accounts":
            return Optional.ofNullable(clazz.cast(accounts()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateMembersRequest, T> g) {
        return obj -> g.apply((CreateMembersRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DetectiveRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateMembersRequest> {
        /**
         * <p>
         * The ARN of the behavior graph.
         * </p>
         * 
         * @param graphArn
         *        The ARN of the behavior graph.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder graphArn(String graphArn);

        /**
         * <p>
         * Customized message text to include in the invitation email message to the invited member accounts.
         * </p>
         * 
         * @param message
         *        Customized message text to include in the invitation email message to the invited member accounts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * if set to <code>true</code>, then the invited accounts do not receive email notifications. By default, this
         * is set to <code>false</code>, and the invited accounts receive email notifications.
         * </p>
         * <p>
         * Organization accounts in the organization behavior graph do not receive email notifications.
         * </p>
         * 
         * @param disableEmailNotification
         *        if set to <code>true</code>, then the invited accounts do not receive email notifications. By default,
         *        this is set to <code>false</code>, and the invited accounts receive email notifications.</p>
         *        <p>
         *        Organization accounts in the organization behavior graph do not receive email notifications.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disableEmailNotification(Boolean disableEmailNotification);

        /**
         * <p>
         * The list of Amazon Web Services accounts to invite or to enable. You can invite or enable up to 50 accounts
         * at a time. For each invited account, the account list contains the account identifier and the Amazon Web
         * Services account root user email address. For organization accounts in the organization behavior graph, the
         * email address is not required.
         * </p>
         * 
         * @param accounts
         *        The list of Amazon Web Services accounts to invite or to enable. You can invite or enable up to 50
         *        accounts at a time. For each invited account, the account list contains the account identifier and the
         *        Amazon Web Services account root user email address. For organization accounts in the organization
         *        behavior graph, the email address is not required.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accounts(Collection<Account> accounts);

        /**
         * <p>
         * The list of Amazon Web Services accounts to invite or to enable. You can invite or enable up to 50 accounts
         * at a time. For each invited account, the account list contains the account identifier and the Amazon Web
         * Services account root user email address. For organization accounts in the organization behavior graph, the
         * email address is not required.
         * </p>
         * 
         * @param accounts
         *        The list of Amazon Web Services accounts to invite or to enable. You can invite or enable up to 50
         *        accounts at a time. For each invited account, the account list contains the account identifier and the
         *        Amazon Web Services account root user email address. For organization accounts in the organization
         *        behavior graph, the email address is not required.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accounts(Account... accounts);

        /**
         * <p>
         * The list of Amazon Web Services accounts to invite or to enable. You can invite or enable up to 50 accounts
         * at a time. For each invited account, the account list contains the account identifier and the Amazon Web
         * Services account root user email address. For organization accounts in the organization behavior graph, the
         * email address is not required.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.detective.model.Account.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.detective.model.Account#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.detective.model.Account.Builder#build()} is called immediately and its
         * result is passed to {@link #accounts(List<Account>)}.
         * 
         * @param accounts
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.detective.model.Account.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accounts(java.util.Collection<Account>)
         */
        Builder accounts(Consumer<Account.Builder>... accounts);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DetectiveRequest.BuilderImpl implements Builder {
        private String graphArn;

        private String message;

        private Boolean disableEmailNotification;

        private List<Account> accounts = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateMembersRequest model) {
            super(model);
            graphArn(model.graphArn);
            message(model.message);
            disableEmailNotification(model.disableEmailNotification);
            accounts(model.accounts);
        }

        public final String getGraphArn() {
            return graphArn;
        }

        public final void setGraphArn(String graphArn) {
            this.graphArn = graphArn;
        }

        @Override
        public final Builder graphArn(String graphArn) {
            this.graphArn = graphArn;
            return this;
        }

        public final String getMessage() {
            return message;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final Boolean getDisableEmailNotification() {
            return disableEmailNotification;
        }

        public final void setDisableEmailNotification(Boolean disableEmailNotification) {
            this.disableEmailNotification = disableEmailNotification;
        }

        @Override
        public final Builder disableEmailNotification(Boolean disableEmailNotification) {
            this.disableEmailNotification = disableEmailNotification;
            return this;
        }

        public final List<Account.Builder> getAccounts() {
            List<Account.Builder> result = AccountListCopier.copyToBuilder(this.accounts);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAccounts(Collection<Account.BuilderImpl> accounts) {
            this.accounts = AccountListCopier.copyFromBuilder(accounts);
        }

        @Override
        public final Builder accounts(Collection<Account> accounts) {
            this.accounts = AccountListCopier.copy(accounts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accounts(Account... accounts) {
            accounts(Arrays.asList(accounts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accounts(Consumer<Account.Builder>... accounts) {
            accounts(Stream.of(accounts).map(c -> Account.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateMembersRequest build() {
            return new CreateMembersRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
