/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.devicefarm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.devicefarm.transform.DevicePoolCompatibilityResultMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a device pool compatibility result.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DevicePoolCompatibilityResult implements StructuredPojo,
        ToCopyableBuilder<DevicePoolCompatibilityResult.Builder, DevicePoolCompatibilityResult> {
    private final Device device;

    private final Boolean compatible;

    private final List<IncompatibilityMessage> incompatibilityMessages;

    private DevicePoolCompatibilityResult(BuilderImpl builder) {
        this.device = builder.device;
        this.compatible = builder.compatible;
        this.incompatibilityMessages = builder.incompatibilityMessages;
    }

    /**
     * <p>
     * The device (phone or tablet) that you wish to return information about.
     * </p>
     * 
     * @return The device (phone or tablet) that you wish to return information about.
     */
    public Device device() {
        return device;
    }

    /**
     * <p>
     * Whether the result was compatible with the device pool.
     * </p>
     * 
     * @return Whether the result was compatible with the device pool.
     */
    public Boolean compatible() {
        return compatible;
    }

    /**
     * <p>
     * Information about the compatibility.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Information about the compatibility.
     */
    public List<IncompatibilityMessage> incompatibilityMessages() {
        return incompatibilityMessages;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(device());
        hashCode = 31 * hashCode + Objects.hashCode(compatible());
        hashCode = 31 * hashCode + Objects.hashCode(incompatibilityMessages());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DevicePoolCompatibilityResult)) {
            return false;
        }
        DevicePoolCompatibilityResult other = (DevicePoolCompatibilityResult) obj;
        return Objects.equals(device(), other.device()) && Objects.equals(compatible(), other.compatible())
                && Objects.equals(incompatibilityMessages(), other.incompatibilityMessages());
    }

    @Override
    public String toString() {
        return ToString.builder("DevicePoolCompatibilityResult").add("Device", device()).add("Compatible", compatible())
                .add("IncompatibilityMessages", incompatibilityMessages()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "device":
            return Optional.ofNullable(clazz.cast(device()));
        case "compatible":
            return Optional.ofNullable(clazz.cast(compatible()));
        case "incompatibilityMessages":
            return Optional.ofNullable(clazz.cast(incompatibilityMessages()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        DevicePoolCompatibilityResultMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, DevicePoolCompatibilityResult> {
        /**
         * <p>
         * The device (phone or tablet) that you wish to return information about.
         * </p>
         * 
         * @param device
         *        The device (phone or tablet) that you wish to return information about.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder device(Device device);

        /**
         * <p>
         * The device (phone or tablet) that you wish to return information about.
         * </p>
         * This is a convenience that creates an instance of the {@link Device.Builder} avoiding the need to create one
         * manually via {@link Device#builder()}.
         *
         * When the {@link Consumer} completes, {@link Device.Builder#build()} is called immediately and its result is
         * passed to {@link #device(Device)}.
         * 
         * @param device
         *        a consumer that will call methods on {@link Device.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #device(Device)
         */
        default Builder device(Consumer<Device.Builder> device) {
            return device(Device.builder().applyMutation(device).build());
        }

        /**
         * <p>
         * Whether the result was compatible with the device pool.
         * </p>
         * 
         * @param compatible
         *        Whether the result was compatible with the device pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder compatible(Boolean compatible);

        /**
         * <p>
         * Information about the compatibility.
         * </p>
         * 
         * @param incompatibilityMessages
         *        Information about the compatibility.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder incompatibilityMessages(Collection<IncompatibilityMessage> incompatibilityMessages);

        /**
         * <p>
         * Information about the compatibility.
         * </p>
         * 
         * @param incompatibilityMessages
         *        Information about the compatibility.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder incompatibilityMessages(IncompatibilityMessage... incompatibilityMessages);

        /**
         * <p>
         * Information about the compatibility.
         * </p>
         * This is a convenience that creates an instance of the {@link List<IncompatibilityMessage>.Builder} avoiding
         * the need to create one manually via {@link List<IncompatibilityMessage>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<IncompatibilityMessage>.Builder#build()} is called
         * immediately and its result is passed to {@link #incompatibilityMessages(List<IncompatibilityMessage>)}.
         * 
         * @param incompatibilityMessages
         *        a consumer that will call methods on {@link List<IncompatibilityMessage>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #incompatibilityMessages(List<IncompatibilityMessage>)
         */
        Builder incompatibilityMessages(Consumer<IncompatibilityMessage.Builder>... incompatibilityMessages);
    }

    static final class BuilderImpl implements Builder {
        private Device device;

        private Boolean compatible;

        private List<IncompatibilityMessage> incompatibilityMessages = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DevicePoolCompatibilityResult model) {
            device(model.device);
            compatible(model.compatible);
            incompatibilityMessages(model.incompatibilityMessages);
        }

        public final Device.Builder getDevice() {
            return device != null ? device.toBuilder() : null;
        }

        @Override
        public final Builder device(Device device) {
            this.device = device;
            return this;
        }

        public final void setDevice(Device.BuilderImpl device) {
            this.device = device != null ? device.build() : null;
        }

        public final Boolean getCompatible() {
            return compatible;
        }

        @Override
        public final Builder compatible(Boolean compatible) {
            this.compatible = compatible;
            return this;
        }

        public final void setCompatible(Boolean compatible) {
            this.compatible = compatible;
        }

        public final Collection<IncompatibilityMessage.Builder> getIncompatibilityMessages() {
            return incompatibilityMessages != null ? incompatibilityMessages.stream().map(IncompatibilityMessage::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder incompatibilityMessages(Collection<IncompatibilityMessage> incompatibilityMessages) {
            this.incompatibilityMessages = IncompatibilityMessagesCopier.copy(incompatibilityMessages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder incompatibilityMessages(IncompatibilityMessage... incompatibilityMessages) {
            incompatibilityMessages(Arrays.asList(incompatibilityMessages));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder incompatibilityMessages(Consumer<IncompatibilityMessage.Builder>... incompatibilityMessages) {
            incompatibilityMessages(Stream.of(incompatibilityMessages)
                    .map(c -> IncompatibilityMessage.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setIncompatibilityMessages(Collection<IncompatibilityMessage.BuilderImpl> incompatibilityMessages) {
            this.incompatibilityMessages = IncompatibilityMessagesCopier.copyFromBuilder(incompatibilityMessages);
        }

        @Override
        public DevicePoolCompatibilityResult build() {
            return new DevicePoolCompatibilityResult(this);
        }
    }
}
