/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.devicefarm.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import org.reactivestreams.Subscriber;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.async.SdkPublisher;
import software.amazon.awssdk.core.pagination.async.AsyncPageFetcher;
import software.amazon.awssdk.core.pagination.async.EmptySubscription;
import software.amazon.awssdk.core.pagination.async.PaginatedItemsPublisher;
import software.amazon.awssdk.core.pagination.async.ResponsesSubscription;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.devicefarm.DeviceFarmAsyncClient;
import software.amazon.awssdk.services.devicefarm.model.ListOfferingsRequest;
import software.amazon.awssdk.services.devicefarm.model.ListOfferingsResponse;
import software.amazon.awssdk.services.devicefarm.model.Offering;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.devicefarm.DeviceFarmAsyncClient#listOfferingsPaginator(software.amazon.awssdk.services.devicefarm.model.ListOfferingsRequest)}
 * operation which is a paginated operation. This class is a type of {@link org.reactivestreams.Publisher} which can be
 * used to provide a sequence of {@link software.amazon.awssdk.services.devicefarm.model.ListOfferingsResponse} response
 * pages as per demand from the subscriber.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. If there are errors in your request, you will see the failures
 * only after you start streaming the data. The subscribe method should be called as a request to start streaming data.
 * For more info, see {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the
 * subscribe method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data
 * from the starting request.
 * </p>
 *
 * <p>
 * The following are few ways to use the response class:
 * </p>
 * 1) Using the forEach helper method
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.devicefarm.paginators.ListOfferingsPublisher publisher = client.listOfferingsPaginator(request);
 * CompletableFuture<Void> future = publisher.forEach(res -> { // Do something with the response });
 * future.get();
 * }
 * </pre>
 *
 * 2) Using a custom subscriber
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.devicefarm.paginators.ListOfferingsPublisher publisher = client.listOfferingsPaginator(request);
 * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.devicefarm.model.ListOfferingsResponse>() {
 * 
 * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
 * 
 * 
 * public void onNext(software.amazon.awssdk.services.devicefarm.model.ListOfferingsResponse response) { //... };
 * });}
 * </pre>
 * 
 * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listOfferings(software.amazon.awssdk.services.devicefarm.model.ListOfferingsRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListOfferingsPublisher implements SdkPublisher<ListOfferingsResponse> {
    private final DeviceFarmAsyncClient client;

    private final ListOfferingsRequest firstRequest;

    private final AsyncPageFetcher nextPageFetcher;

    private boolean isLastPage;

    public ListOfferingsPublisher(DeviceFarmAsyncClient client, ListOfferingsRequest firstRequest) {
        this(client, firstRequest, false);
    }

    private ListOfferingsPublisher(DeviceFarmAsyncClient client, ListOfferingsRequest firstRequest, boolean isLastPage) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.isLastPage = isLastPage;
        this.nextPageFetcher = new ListOfferingsResponseFetcher();
    }

    @Override
    public void subscribe(Subscriber<? super ListOfferingsResponse> subscriber) {
        subscriber.onSubscribe(ResponsesSubscription.builder().subscriber(subscriber).nextPageFetcher(nextPageFetcher).build());
    }

    /**
     * Returns a publisher that can be used to get a stream of data. You need to subscribe to the publisher to request
     * the stream of data. The publisher has a helper forEach method that takes in a {@link java.util.function.Consumer}
     * and then applies that consumer to each response returned by the service.
     */
    public final SdkPublisher<Offering> offerings() {
        Function<ListOfferingsResponse, Iterator<Offering>> getIterator = response -> {
            if (response != null && response.offerings() != null) {
                return response.offerings().iterator();
            }
            return Collections.emptyIterator();
        };
        return PaginatedItemsPublisher.builder().nextPageFetcher(new ListOfferingsResponseFetcher())
                .iteratorFunction(getIterator).isLastPage(isLastPage).build();
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link ListOfferingsPublisher} that can be used to retrieve the
     * consecutive pages that follows the input page.
     * </p>
     */
    private final ListOfferingsPublisher resume(ListOfferingsResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new ListOfferingsPublisher(client, firstRequest.toBuilder().nextToken(lastSuccessfulPage.nextToken()).build());
        }
        return new ListOfferingsPublisher(client, firstRequest, true) {
            @Override
            public void subscribe(Subscriber<? super ListOfferingsResponse> subscriber) {
                subscriber.onSubscribe(new EmptySubscription(subscriber));
            }
        };
    }

    private class ListOfferingsResponseFetcher implements AsyncPageFetcher<ListOfferingsResponse> {
        @Override
        public boolean hasNextPage(final ListOfferingsResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public CompletableFuture<ListOfferingsResponse> nextPage(final ListOfferingsResponse previousPage) {
            if (previousPage == null) {
                return client.listOfferings(firstRequest);
            }
            return client.listOfferings(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
