/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.devicefarm.model;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.services.devicefarm.transform.AccountSettingsMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A container for account-level settings within AWS Device Farm.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class AccountSettings implements StructuredPojo, ToCopyableBuilder<AccountSettings.Builder, AccountSettings> {
    private final String awsAccountNumber;

    private final Map<String, Integer> unmeteredDevices;

    private final Map<String, Integer> unmeteredRemoteAccessDevices;

    private final Integer maxJobTimeoutMinutes;

    private final TrialMinutes trialMinutes;

    private final Map<String, Integer> maxSlots;

    private final Integer defaultJobTimeoutMinutes;

    private AccountSettings(BuilderImpl builder) {
        this.awsAccountNumber = builder.awsAccountNumber;
        this.unmeteredDevices = builder.unmeteredDevices;
        this.unmeteredRemoteAccessDevices = builder.unmeteredRemoteAccessDevices;
        this.maxJobTimeoutMinutes = builder.maxJobTimeoutMinutes;
        this.trialMinutes = builder.trialMinutes;
        this.maxSlots = builder.maxSlots;
        this.defaultJobTimeoutMinutes = builder.defaultJobTimeoutMinutes;
    }

    /**
     * <p>
     * The AWS account number specified in the <code>AccountSettings</code> container.
     * </p>
     * 
     * @return The AWS account number specified in the <code>AccountSettings</code> container.
     */
    public String awsAccountNumber() {
        return awsAccountNumber;
    }

    /**
     * <p>
     * Returns the unmetered devices you have purchased or want to purchase.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Returns the unmetered devices you have purchased or want to purchase.
     */
    public Map<DevicePlatform, Integer> unmeteredDevices() {
        return TypeConverter.convert(unmeteredDevices, DevicePlatform::fromValue, Function.identity(),
                (k, v) -> !Objects.equals(k, DevicePlatform.UNKNOWN_TO_SDK_VERSION));
    }

    /**
     * <p>
     * Returns the unmetered devices you have purchased or want to purchase.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Returns the unmetered devices you have purchased or want to purchase.
     */
    public Map<String, Integer> unmeteredDevicesStrings() {
        return unmeteredDevices;
    }

    /**
     * <p>
     * Returns the unmetered remote access devices you have purchased or want to purchase.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Returns the unmetered remote access devices you have purchased or want to purchase.
     */
    public Map<DevicePlatform, Integer> unmeteredRemoteAccessDevices() {
        return TypeConverter.convert(unmeteredRemoteAccessDevices, DevicePlatform::fromValue, Function.identity(),
                (k, v) -> !Objects.equals(k, DevicePlatform.UNKNOWN_TO_SDK_VERSION));
    }

    /**
     * <p>
     * Returns the unmetered remote access devices you have purchased or want to purchase.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Returns the unmetered remote access devices you have purchased or want to purchase.
     */
    public Map<String, Integer> unmeteredRemoteAccessDevicesStrings() {
        return unmeteredRemoteAccessDevices;
    }

    /**
     * <p>
     * The maximum number of minutes a test run will execute before it times out.
     * </p>
     * 
     * @return The maximum number of minutes a test run will execute before it times out.
     */
    public Integer maxJobTimeoutMinutes() {
        return maxJobTimeoutMinutes;
    }

    /**
     * <p>
     * Information about an AWS account's usage of free trial device minutes.
     * </p>
     * 
     * @return Information about an AWS account's usage of free trial device minutes.
     */
    public TrialMinutes trialMinutes() {
        return trialMinutes;
    }

    /**
     * <p>
     * The maximum number of device slots that the AWS account can purchase. Each maximum is expressed as an
     * <code>offering-id:number</code> pair, where the <code>offering-id</code> represents one of the IDs returned by
     * the <code>ListOfferings</code> command.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The maximum number of device slots that the AWS account can purchase. Each maximum is expressed as an
     *         <code>offering-id:number</code> pair, where the <code>offering-id</code> represents one of the IDs
     *         returned by the <code>ListOfferings</code> command.
     */
    public Map<String, Integer> maxSlots() {
        return maxSlots;
    }

    /**
     * <p>
     * The default number of minutes (at the account level) a test run will execute before it times out. Default value
     * is 60 minutes.
     * </p>
     * 
     * @return The default number of minutes (at the account level) a test run will execute before it times out. Default
     *         value is 60 minutes.
     */
    public Integer defaultJobTimeoutMinutes() {
        return defaultJobTimeoutMinutes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(awsAccountNumber());
        hashCode = 31 * hashCode + Objects.hashCode(unmeteredDevicesStrings());
        hashCode = 31 * hashCode + Objects.hashCode(unmeteredRemoteAccessDevicesStrings());
        hashCode = 31 * hashCode + Objects.hashCode(maxJobTimeoutMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(trialMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(maxSlots());
        hashCode = 31 * hashCode + Objects.hashCode(defaultJobTimeoutMinutes());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AccountSettings)) {
            return false;
        }
        AccountSettings other = (AccountSettings) obj;
        return Objects.equals(awsAccountNumber(), other.awsAccountNumber())
                && Objects.equals(unmeteredDevicesStrings(), other.unmeteredDevicesStrings())
                && Objects.equals(unmeteredRemoteAccessDevicesStrings(), other.unmeteredRemoteAccessDevicesStrings())
                && Objects.equals(maxJobTimeoutMinutes(), other.maxJobTimeoutMinutes())
                && Objects.equals(trialMinutes(), other.trialMinutes()) && Objects.equals(maxSlots(), other.maxSlots())
                && Objects.equals(defaultJobTimeoutMinutes(), other.defaultJobTimeoutMinutes());
    }

    @Override
    public String toString() {
        return ToString.builder("AccountSettings").add("AwsAccountNumber", awsAccountNumber())
                .add("UnmeteredDevices", unmeteredDevicesStrings())
                .add("UnmeteredRemoteAccessDevices", unmeteredRemoteAccessDevicesStrings())
                .add("MaxJobTimeoutMinutes", maxJobTimeoutMinutes()).add("TrialMinutes", trialMinutes())
                .add("MaxSlots", maxSlots()).add("DefaultJobTimeoutMinutes", defaultJobTimeoutMinutes()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "awsAccountNumber":
            return Optional.of(clazz.cast(awsAccountNumber()));
        case "unmeteredDevices":
            return Optional.of(clazz.cast(unmeteredDevicesStrings()));
        case "unmeteredRemoteAccessDevices":
            return Optional.of(clazz.cast(unmeteredRemoteAccessDevicesStrings()));
        case "maxJobTimeoutMinutes":
            return Optional.of(clazz.cast(maxJobTimeoutMinutes()));
        case "trialMinutes":
            return Optional.of(clazz.cast(trialMinutes()));
        case "maxSlots":
            return Optional.of(clazz.cast(maxSlots()));
        case "defaultJobTimeoutMinutes":
            return Optional.of(clazz.cast(defaultJobTimeoutMinutes()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        AccountSettingsMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, AccountSettings> {
        /**
         * <p>
         * The AWS account number specified in the <code>AccountSettings</code> container.
         * </p>
         * 
         * @param awsAccountNumber
         *        The AWS account number specified in the <code>AccountSettings</code> container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsAccountNumber(String awsAccountNumber);

        /**
         * <p>
         * Returns the unmetered devices you have purchased or want to purchase.
         * </p>
         * 
         * @param unmeteredDevices
         *        Returns the unmetered devices you have purchased or want to purchase.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unmeteredDevices(Map<String, Integer> unmeteredDevices);

        /**
         * <p>
         * Returns the unmetered remote access devices you have purchased or want to purchase.
         * </p>
         * 
         * @param unmeteredRemoteAccessDevices
         *        Returns the unmetered remote access devices you have purchased or want to purchase.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unmeteredRemoteAccessDevices(Map<String, Integer> unmeteredRemoteAccessDevices);

        /**
         * <p>
         * The maximum number of minutes a test run will execute before it times out.
         * </p>
         * 
         * @param maxJobTimeoutMinutes
         *        The maximum number of minutes a test run will execute before it times out.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxJobTimeoutMinutes(Integer maxJobTimeoutMinutes);

        /**
         * <p>
         * Information about an AWS account's usage of free trial device minutes.
         * </p>
         * 
         * @param trialMinutes
         *        Information about an AWS account's usage of free trial device minutes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trialMinutes(TrialMinutes trialMinutes);

        /**
         * <p>
         * Information about an AWS account's usage of free trial device minutes.
         * </p>
         * This is a convenience that creates an instance of the {@link TrialMinutes.Builder} avoiding the need to
         * create one manually via {@link TrialMinutes#builder()}.
         *
         * When the {@link Consumer} completes, {@link TrialMinutes.Builder#build()} is called immediately and its
         * result is passed to {@link #trialMinutes(TrialMinutes)}.
         * 
         * @param trialMinutes
         *        a consumer that will call methods on {@link TrialMinutes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #trialMinutes(TrialMinutes)
         */
        default Builder trialMinutes(Consumer<TrialMinutes.Builder> trialMinutes) {
            return trialMinutes(TrialMinutes.builder().apply(trialMinutes).build());
        }

        /**
         * <p>
         * The maximum number of device slots that the AWS account can purchase. Each maximum is expressed as an
         * <code>offering-id:number</code> pair, where the <code>offering-id</code> represents one of the IDs returned
         * by the <code>ListOfferings</code> command.
         * </p>
         * 
         * @param maxSlots
         *        The maximum number of device slots that the AWS account can purchase. Each maximum is expressed as an
         *        <code>offering-id:number</code> pair, where the <code>offering-id</code> represents one of the IDs
         *        returned by the <code>ListOfferings</code> command.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxSlots(Map<String, Integer> maxSlots);

        /**
         * <p>
         * The default number of minutes (at the account level) a test run will execute before it times out. Default
         * value is 60 minutes.
         * </p>
         * 
         * @param defaultJobTimeoutMinutes
         *        The default number of minutes (at the account level) a test run will execute before it times out.
         *        Default value is 60 minutes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultJobTimeoutMinutes(Integer defaultJobTimeoutMinutes);
    }

    static final class BuilderImpl implements Builder {
        private String awsAccountNumber;

        private Map<String, Integer> unmeteredDevices;

        private Map<String, Integer> unmeteredRemoteAccessDevices;

        private Integer maxJobTimeoutMinutes;

        private TrialMinutes trialMinutes;

        private Map<String, Integer> maxSlots;

        private Integer defaultJobTimeoutMinutes;

        private BuilderImpl() {
        }

        private BuilderImpl(AccountSettings model) {
            awsAccountNumber(model.awsAccountNumber);
            unmeteredDevices(model.unmeteredDevices);
            unmeteredRemoteAccessDevices(model.unmeteredRemoteAccessDevices);
            maxJobTimeoutMinutes(model.maxJobTimeoutMinutes);
            trialMinutes(model.trialMinutes);
            maxSlots(model.maxSlots);
            defaultJobTimeoutMinutes(model.defaultJobTimeoutMinutes);
        }

        public final String getAwsAccountNumber() {
            return awsAccountNumber;
        }

        @Override
        public final Builder awsAccountNumber(String awsAccountNumber) {
            this.awsAccountNumber = awsAccountNumber;
            return this;
        }

        public final void setAwsAccountNumber(String awsAccountNumber) {
            this.awsAccountNumber = awsAccountNumber;
        }

        public final Map<String, Integer> getUnmeteredDevices() {
            return unmeteredDevices;
        }

        @Override
        public final Builder unmeteredDevices(Map<String, Integer> unmeteredDevices) {
            this.unmeteredDevices = PurchasedDevicesMapCopier.copy(unmeteredDevices);
            return this;
        }

        public final void setUnmeteredDevices(Map<String, Integer> unmeteredDevices) {
            this.unmeteredDevices = PurchasedDevicesMapCopier.copy(unmeteredDevices);
        }

        public final Map<String, Integer> getUnmeteredRemoteAccessDevices() {
            return unmeteredRemoteAccessDevices;
        }

        @Override
        public final Builder unmeteredRemoteAccessDevices(Map<String, Integer> unmeteredRemoteAccessDevices) {
            this.unmeteredRemoteAccessDevices = PurchasedDevicesMapCopier.copy(unmeteredRemoteAccessDevices);
            return this;
        }

        public final void setUnmeteredRemoteAccessDevices(Map<String, Integer> unmeteredRemoteAccessDevices) {
            this.unmeteredRemoteAccessDevices = PurchasedDevicesMapCopier.copy(unmeteredRemoteAccessDevices);
        }

        public final Integer getMaxJobTimeoutMinutes() {
            return maxJobTimeoutMinutes;
        }

        @Override
        public final Builder maxJobTimeoutMinutes(Integer maxJobTimeoutMinutes) {
            this.maxJobTimeoutMinutes = maxJobTimeoutMinutes;
            return this;
        }

        public final void setMaxJobTimeoutMinutes(Integer maxJobTimeoutMinutes) {
            this.maxJobTimeoutMinutes = maxJobTimeoutMinutes;
        }

        public final TrialMinutes.Builder getTrialMinutes() {
            return trialMinutes != null ? trialMinutes.toBuilder() : null;
        }

        @Override
        public final Builder trialMinutes(TrialMinutes trialMinutes) {
            this.trialMinutes = trialMinutes;
            return this;
        }

        public final void setTrialMinutes(TrialMinutes.BuilderImpl trialMinutes) {
            this.trialMinutes = trialMinutes != null ? trialMinutes.build() : null;
        }

        public final Map<String, Integer> getMaxSlots() {
            return maxSlots;
        }

        @Override
        public final Builder maxSlots(Map<String, Integer> maxSlots) {
            this.maxSlots = MaxSlotMapCopier.copy(maxSlots);
            return this;
        }

        public final void setMaxSlots(Map<String, Integer> maxSlots) {
            this.maxSlots = MaxSlotMapCopier.copy(maxSlots);
        }

        public final Integer getDefaultJobTimeoutMinutes() {
            return defaultJobTimeoutMinutes;
        }

        @Override
        public final Builder defaultJobTimeoutMinutes(Integer defaultJobTimeoutMinutes) {
            this.defaultJobTimeoutMinutes = defaultJobTimeoutMinutes;
            return this;
        }

        public final void setDefaultJobTimeoutMinutes(Integer defaultJobTimeoutMinutes) {
            this.defaultJobTimeoutMinutes = defaultJobTimeoutMinutes;
        }

        @Override
        public AccountSettings build() {
            return new AccountSettings(this);
        }
    }
}
