/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.devicefarm.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import javax.annotation.Generated;
import software.amazon.awssdk.core.pagination.PaginatedItemsIterable;
import software.amazon.awssdk.core.pagination.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.SdkIterable;
import software.amazon.awssdk.core.pagination.SyncPageFetcher;
import software.amazon.awssdk.services.devicefarm.DeviceFarmClient;
import software.amazon.awssdk.services.devicefarm.model.ListUniqueProblemsRequest;
import software.amazon.awssdk.services.devicefarm.model.ListUniqueProblemsResponse;
import software.amazon.awssdk.services.devicefarm.model.UniqueProblem;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.devicefarm.DeviceFarmClient#listUniqueProblemsPaginator(software.amazon.awssdk.services.devicefarm.model.ListUniqueProblemsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.devicefarm.model.ListUniqueProblemsResponse} that can be used to iterate
 * through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.devicefarm.paginators.ListUniqueProblemsIterable responses = client.listUniqueProblemsPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.devicefarm.paginators.ListUniqueProblemsIterable responses = client
 *             .listUniqueProblemsPaginator(request);
 *     for (software.amazon.awssdk.services.devicefarm.model.ListUniqueProblemsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.devicefarm.paginators.ListUniqueProblemsIterable responses = client.listUniqueProblemsPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listUniqueProblems(software.amazon.awssdk.services.devicefarm.model.ListUniqueProblemsRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListUniqueProblemsIterable implements SdkIterable<ListUniqueProblemsResponse> {
    private final DeviceFarmClient client;

    private final ListUniqueProblemsRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListUniqueProblemsIterable(final DeviceFarmClient client, final ListUniqueProblemsRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new ListUniqueProblemsResponseFetcher();
    }

    @Override
    public Iterator<ListUniqueProblemsResponse> iterator() {
        return new PaginatedResponsesIterator(nextPageFetcher);
    }

    /**
     * Returns an iterable to iterate through the paginated {@link ListUniqueProblemsResponse#uniqueProblemsStrings()}
     * member. The returned iterable is used to iterate through the results across all response pages and not a single
     * page.
     *
     * This method is useful if you are interested in iterating over the paginated member in the response pages instead
     * of the top level pages. Similar to iteration over pages, this method internally makes service calls to get the
     * next list of results until the iteration stops or there are no more results.
     */
    public final SdkIterable<Map.Entry<String, List<UniqueProblem>>> uniqueProblemsStrings() {
        Function<ListUniqueProblemsResponse, Iterator<Map.Entry<String, List<UniqueProblem>>>> getIterator = response -> {
            if (response != null && response.uniqueProblemsStrings() != null) {
                return response.uniqueProblemsStrings().entrySet().iterator();
            }
            return Collections.emptyIterator();
        };
        return new PaginatedItemsIterable(this, getIterator);
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link ListUniqueProblemsIterable} that can be used to retrieve the
     * consecutive pages that follows the input page.
     * </p>
     */
    public final ListUniqueProblemsIterable resume(final ListUniqueProblemsResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new ListUniqueProblemsIterable(client, firstRequest.toBuilder().nextToken(lastSuccessfulPage.nextToken())
                    .build());
        }
        return new ListUniqueProblemsIterable(client, firstRequest) {
            @Override
            public Iterator<ListUniqueProblemsResponse> iterator() {
                return Collections.emptyIterator();
            }
        };
    }

    private class ListUniqueProblemsResponseFetcher implements SyncPageFetcher<ListUniqueProblemsResponse> {
        @Override
        public boolean hasNextPage(ListUniqueProblemsResponse previousPage) {
            return previousPage.nextToken() != null;
        }

        @Override
        public ListUniqueProblemsResponse nextPage(ListUniqueProblemsResponse previousPage) {
            if (previousPage == null) {
                return client.listUniqueProblems(firstRequest);
            }
            return client.listUniqueProblems(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
