/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.devicefarm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a device filter used to select a set of devices to be included in a test run. This data structure is
 * passed in as the "deviceSelectionConfiguration" parameter to ScheduleRun. For an example of the JSON request syntax,
 * see <a>ScheduleRun</a>.
 * </p>
 * <p>
 * It is also passed in as the "filters" parameter to ListDevices. For an example of the JSON request syntax, see
 * <a>ListDevices</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeviceFilter implements SdkPojo, Serializable, ToCopyableBuilder<DeviceFilter.Builder, DeviceFilter> {
    private static final SdkField<String> ATTRIBUTE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DeviceFilter::attributeAsString)).setter(setter(Builder::attribute))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("attribute").build()).build();

    private static final SdkField<String> OPERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DeviceFilter::operatorAsString)).setter(setter(Builder::operator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("operator").build()).build();

    private static final SdkField<List<String>> VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DeviceFilter::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ATTRIBUTE_FIELD,
            OPERATOR_FIELD, VALUES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String attribute;

    private final String operator;

    private final List<String> values;

    private DeviceFilter(BuilderImpl builder) {
        this.attribute = builder.attribute;
        this.operator = builder.operator;
        this.values = builder.values;
    }

    /**
     * <p>
     * The aspect of a device such as platform or model used as the selection criteria in a device filter.
     * </p>
     * <p>
     * Allowed values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * ARN: The Amazon Resource Name (ARN) of the device. For example,
     * "arn:aws:devicefarm:us-west-2::device:12345Example".
     * </p>
     * </li>
     * <li>
     * <p>
     * PLATFORM: The device platform. Valid values are "ANDROID" or "IOS".
     * </p>
     * </li>
     * <li>
     * <p>
     * OS_VERSION: The operating system version. For example, "10.3.2".
     * </p>
     * </li>
     * <li>
     * <p>
     * MODEL: The device model. For example, "iPad 5th Gen".
     * </p>
     * </li>
     * <li>
     * <p>
     * AVAILABILITY: The current availability of the device. Valid values are "AVAILABLE", "HIGHLY_AVAILABLE", "BUSY",
     * or "TEMPORARY_NOT_AVAILABLE".
     * </p>
     * </li>
     * <li>
     * <p>
     * FORM_FACTOR: The device form factor. Valid values are "PHONE" or "TABLET".
     * </p>
     * </li>
     * <li>
     * <p>
     * MANUFACTURER: The device manufacturer. For example, "Apple".
     * </p>
     * </li>
     * <li>
     * <p>
     * REMOTE_ACCESS_ENABLED: Whether the device is enabled for remote access.
     * </p>
     * </li>
     * <li>
     * <p>
     * REMOTE_DEBUG_ENABLED: Whether the device is enabled for remote debugging.
     * </p>
     * </li>
     * <li>
     * <p>
     * INSTANCE_ARN: The Amazon Resource Name (ARN) of the device instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * INSTANCE_LABELS: The label of the device instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_TYPE: The fleet type. Valid values are "PUBLIC" or "PRIVATE".
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #attribute} will
     * return {@link DeviceFilterAttribute#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #attributeAsString}.
     * </p>
     * 
     * @return The aspect of a device such as platform or model used as the selection criteria in a device filter.</p>
     *         <p>
     *         Allowed values include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         ARN: The Amazon Resource Name (ARN) of the device. For example,
     *         "arn:aws:devicefarm:us-west-2::device:12345Example".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PLATFORM: The device platform. Valid values are "ANDROID" or "IOS".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         OS_VERSION: The operating system version. For example, "10.3.2".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MODEL: The device model. For example, "iPad 5th Gen".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AVAILABILITY: The current availability of the device. Valid values are "AVAILABLE", "HIGHLY_AVAILABLE",
     *         "BUSY", or "TEMPORARY_NOT_AVAILABLE".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FORM_FACTOR: The device form factor. Valid values are "PHONE" or "TABLET".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MANUFACTURER: The device manufacturer. For example, "Apple".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         REMOTE_ACCESS_ENABLED: Whether the device is enabled for remote access.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         REMOTE_DEBUG_ENABLED: Whether the device is enabled for remote debugging.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INSTANCE_ARN: The Amazon Resource Name (ARN) of the device instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INSTANCE_LABELS: The label of the device instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_TYPE: The fleet type. Valid values are "PUBLIC" or "PRIVATE".
     *         </p>
     *         </li>
     * @see DeviceFilterAttribute
     */
    public DeviceFilterAttribute attribute() {
        return DeviceFilterAttribute.fromValue(attribute);
    }

    /**
     * <p>
     * The aspect of a device such as platform or model used as the selection criteria in a device filter.
     * </p>
     * <p>
     * Allowed values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * ARN: The Amazon Resource Name (ARN) of the device. For example,
     * "arn:aws:devicefarm:us-west-2::device:12345Example".
     * </p>
     * </li>
     * <li>
     * <p>
     * PLATFORM: The device platform. Valid values are "ANDROID" or "IOS".
     * </p>
     * </li>
     * <li>
     * <p>
     * OS_VERSION: The operating system version. For example, "10.3.2".
     * </p>
     * </li>
     * <li>
     * <p>
     * MODEL: The device model. For example, "iPad 5th Gen".
     * </p>
     * </li>
     * <li>
     * <p>
     * AVAILABILITY: The current availability of the device. Valid values are "AVAILABLE", "HIGHLY_AVAILABLE", "BUSY",
     * or "TEMPORARY_NOT_AVAILABLE".
     * </p>
     * </li>
     * <li>
     * <p>
     * FORM_FACTOR: The device form factor. Valid values are "PHONE" or "TABLET".
     * </p>
     * </li>
     * <li>
     * <p>
     * MANUFACTURER: The device manufacturer. For example, "Apple".
     * </p>
     * </li>
     * <li>
     * <p>
     * REMOTE_ACCESS_ENABLED: Whether the device is enabled for remote access.
     * </p>
     * </li>
     * <li>
     * <p>
     * REMOTE_DEBUG_ENABLED: Whether the device is enabled for remote debugging.
     * </p>
     * </li>
     * <li>
     * <p>
     * INSTANCE_ARN: The Amazon Resource Name (ARN) of the device instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * INSTANCE_LABELS: The label of the device instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_TYPE: The fleet type. Valid values are "PUBLIC" or "PRIVATE".
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #attribute} will
     * return {@link DeviceFilterAttribute#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #attributeAsString}.
     * </p>
     * 
     * @return The aspect of a device such as platform or model used as the selection criteria in a device filter.</p>
     *         <p>
     *         Allowed values include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         ARN: The Amazon Resource Name (ARN) of the device. For example,
     *         "arn:aws:devicefarm:us-west-2::device:12345Example".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PLATFORM: The device platform. Valid values are "ANDROID" or "IOS".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         OS_VERSION: The operating system version. For example, "10.3.2".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MODEL: The device model. For example, "iPad 5th Gen".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AVAILABILITY: The current availability of the device. Valid values are "AVAILABLE", "HIGHLY_AVAILABLE",
     *         "BUSY", or "TEMPORARY_NOT_AVAILABLE".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FORM_FACTOR: The device form factor. Valid values are "PHONE" or "TABLET".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MANUFACTURER: The device manufacturer. For example, "Apple".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         REMOTE_ACCESS_ENABLED: Whether the device is enabled for remote access.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         REMOTE_DEBUG_ENABLED: Whether the device is enabled for remote debugging.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INSTANCE_ARN: The Amazon Resource Name (ARN) of the device instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INSTANCE_LABELS: The label of the device instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_TYPE: The fleet type. Valid values are "PUBLIC" or "PRIVATE".
     *         </p>
     *         </li>
     * @see DeviceFilterAttribute
     */
    public String attributeAsString() {
        return attribute;
    }

    /**
     * <p>
     * The filter operator.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The EQUALS operator is available for every attribute except INSTANCE_LABELS.
     * </p>
     * </li>
     * <li>
     * <p>
     * The CONTAINS operator is available for the INSTANCE_LABELS and MODEL attributes.
     * </p>
     * </li>
     * <li>
     * <p>
     * The IN and NOT_IN operators are available for the ARN, OS_VERSION, MODEL, MANUFACTURER, and INSTANCE_ARN
     * attributes.
     * </p>
     * </li>
     * <li>
     * <p>
     * The LESS_THAN, GREATER_THAN, LESS_THAN_OR_EQUALS, and GREATER_THAN_OR_EQUALS operators are also available for the
     * OS_VERSION attribute.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operator} will
     * return {@link DeviceFilterOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatorAsString}.
     * </p>
     * 
     * @return The filter operator.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The EQUALS operator is available for every attribute except INSTANCE_LABELS.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The CONTAINS operator is available for the INSTANCE_LABELS and MODEL attributes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The IN and NOT_IN operators are available for the ARN, OS_VERSION, MODEL, MANUFACTURER, and INSTANCE_ARN
     *         attributes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The LESS_THAN, GREATER_THAN, LESS_THAN_OR_EQUALS, and GREATER_THAN_OR_EQUALS operators are also available
     *         for the OS_VERSION attribute.
     *         </p>
     *         </li>
     * @see DeviceFilterOperator
     */
    public DeviceFilterOperator operator() {
        return DeviceFilterOperator.fromValue(operator);
    }

    /**
     * <p>
     * The filter operator.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The EQUALS operator is available for every attribute except INSTANCE_LABELS.
     * </p>
     * </li>
     * <li>
     * <p>
     * The CONTAINS operator is available for the INSTANCE_LABELS and MODEL attributes.
     * </p>
     * </li>
     * <li>
     * <p>
     * The IN and NOT_IN operators are available for the ARN, OS_VERSION, MODEL, MANUFACTURER, and INSTANCE_ARN
     * attributes.
     * </p>
     * </li>
     * <li>
     * <p>
     * The LESS_THAN, GREATER_THAN, LESS_THAN_OR_EQUALS, and GREATER_THAN_OR_EQUALS operators are also available for the
     * OS_VERSION attribute.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operator} will
     * return {@link DeviceFilterOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatorAsString}.
     * </p>
     * 
     * @return The filter operator.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The EQUALS operator is available for every attribute except INSTANCE_LABELS.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The CONTAINS operator is available for the INSTANCE_LABELS and MODEL attributes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The IN and NOT_IN operators are available for the ARN, OS_VERSION, MODEL, MANUFACTURER, and INSTANCE_ARN
     *         attributes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The LESS_THAN, GREATER_THAN, LESS_THAN_OR_EQUALS, and GREATER_THAN_OR_EQUALS operators are also available
     *         for the OS_VERSION attribute.
     *         </p>
     *         </li>
     * @see DeviceFilterOperator
     */
    public String operatorAsString() {
        return operator;
    }

    /**
     * <p>
     * An array of one or more filter values used in a device filter.
     * </p>
     * <p class="title">
     * <b>Operator Values</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The IN and NOT operators can take a values array that has more than one element.
     * </p>
     * </li>
     * <li>
     * <p>
     * The other operators require an array with a single element.
     * </p>
     * </li>
     * </ul>
     * <p class="title">
     * <b>Attribute Values</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The PLATFORM attribute can be set to "ANDROID" or "IOS".
     * </p>
     * </li>
     * <li>
     * <p>
     * The AVAILABILITY attribute can be set to "AVAILABLE", "HIGHLY_AVAILABLE", "BUSY", or "TEMPORARY_NOT_AVAILABLE".
     * </p>
     * </li>
     * <li>
     * <p>
     * The FORM_FACTOR attribute can be set to "PHONE" or "TABLET".
     * </p>
     * </li>
     * <li>
     * <p>
     * The FLEET_TYPE attribute can be set to "PUBLIC" or "PRIVATE".
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of one or more filter values used in a device filter.</p>
     *         <p class="title">
     *         <b>Operator Values</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The IN and NOT operators can take a values array that has more than one element.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The other operators require an array with a single element.
     *         </p>
     *         </li>
     *         </ul>
     *         <p class="title">
     *         <b>Attribute Values</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The PLATFORM attribute can be set to "ANDROID" or "IOS".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The AVAILABILITY attribute can be set to "AVAILABLE", "HIGHLY_AVAILABLE", "BUSY", or
     *         "TEMPORARY_NOT_AVAILABLE".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The FORM_FACTOR attribute can be set to "PHONE" or "TABLET".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The FLEET_TYPE attribute can be set to "PUBLIC" or "PRIVATE".
     *         </p>
     *         </li>
     */
    public List<String> values() {
        return values;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(attributeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(operatorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(values());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeviceFilter)) {
            return false;
        }
        DeviceFilter other = (DeviceFilter) obj;
        return Objects.equals(attributeAsString(), other.attributeAsString())
                && Objects.equals(operatorAsString(), other.operatorAsString()) && Objects.equals(values(), other.values());
    }

    @Override
    public String toString() {
        return ToString.builder("DeviceFilter").add("Attribute", attributeAsString()).add("Operator", operatorAsString())
                .add("Values", values()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "attribute":
            return Optional.ofNullable(clazz.cast(attributeAsString()));
        case "operator":
            return Optional.ofNullable(clazz.cast(operatorAsString()));
        case "values":
            return Optional.ofNullable(clazz.cast(values()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DeviceFilter, T> g) {
        return obj -> g.apply((DeviceFilter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeviceFilter> {
        /**
         * <p>
         * The aspect of a device such as platform or model used as the selection criteria in a device filter.
         * </p>
         * <p>
         * Allowed values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * ARN: The Amazon Resource Name (ARN) of the device. For example,
         * "arn:aws:devicefarm:us-west-2::device:12345Example".
         * </p>
         * </li>
         * <li>
         * <p>
         * PLATFORM: The device platform. Valid values are "ANDROID" or "IOS".
         * </p>
         * </li>
         * <li>
         * <p>
         * OS_VERSION: The operating system version. For example, "10.3.2".
         * </p>
         * </li>
         * <li>
         * <p>
         * MODEL: The device model. For example, "iPad 5th Gen".
         * </p>
         * </li>
         * <li>
         * <p>
         * AVAILABILITY: The current availability of the device. Valid values are "AVAILABLE", "HIGHLY_AVAILABLE",
         * "BUSY", or "TEMPORARY_NOT_AVAILABLE".
         * </p>
         * </li>
         * <li>
         * <p>
         * FORM_FACTOR: The device form factor. Valid values are "PHONE" or "TABLET".
         * </p>
         * </li>
         * <li>
         * <p>
         * MANUFACTURER: The device manufacturer. For example, "Apple".
         * </p>
         * </li>
         * <li>
         * <p>
         * REMOTE_ACCESS_ENABLED: Whether the device is enabled for remote access.
         * </p>
         * </li>
         * <li>
         * <p>
         * REMOTE_DEBUG_ENABLED: Whether the device is enabled for remote debugging.
         * </p>
         * </li>
         * <li>
         * <p>
         * INSTANCE_ARN: The Amazon Resource Name (ARN) of the device instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * INSTANCE_LABELS: The label of the device instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_TYPE: The fleet type. Valid values are "PUBLIC" or "PRIVATE".
         * </p>
         * </li>
         * </ul>
         * 
         * @param attribute
         *        The aspect of a device such as platform or model used as the selection criteria in a device
         *        filter.</p>
         *        <p>
         *        Allowed values include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        ARN: The Amazon Resource Name (ARN) of the device. For example,
         *        "arn:aws:devicefarm:us-west-2::device:12345Example".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PLATFORM: The device platform. Valid values are "ANDROID" or "IOS".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        OS_VERSION: The operating system version. For example, "10.3.2".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MODEL: The device model. For example, "iPad 5th Gen".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AVAILABILITY: The current availability of the device. Valid values are "AVAILABLE",
         *        "HIGHLY_AVAILABLE", "BUSY", or "TEMPORARY_NOT_AVAILABLE".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FORM_FACTOR: The device form factor. Valid values are "PHONE" or "TABLET".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MANUFACTURER: The device manufacturer. For example, "Apple".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        REMOTE_ACCESS_ENABLED: Whether the device is enabled for remote access.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        REMOTE_DEBUG_ENABLED: Whether the device is enabled for remote debugging.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INSTANCE_ARN: The Amazon Resource Name (ARN) of the device instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INSTANCE_LABELS: The label of the device instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_TYPE: The fleet type. Valid values are "PUBLIC" or "PRIVATE".
         *        </p>
         *        </li>
         * @see DeviceFilterAttribute
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeviceFilterAttribute
         */
        Builder attribute(String attribute);

        /**
         * <p>
         * The aspect of a device such as platform or model used as the selection criteria in a device filter.
         * </p>
         * <p>
         * Allowed values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * ARN: The Amazon Resource Name (ARN) of the device. For example,
         * "arn:aws:devicefarm:us-west-2::device:12345Example".
         * </p>
         * </li>
         * <li>
         * <p>
         * PLATFORM: The device platform. Valid values are "ANDROID" or "IOS".
         * </p>
         * </li>
         * <li>
         * <p>
         * OS_VERSION: The operating system version. For example, "10.3.2".
         * </p>
         * </li>
         * <li>
         * <p>
         * MODEL: The device model. For example, "iPad 5th Gen".
         * </p>
         * </li>
         * <li>
         * <p>
         * AVAILABILITY: The current availability of the device. Valid values are "AVAILABLE", "HIGHLY_AVAILABLE",
         * "BUSY", or "TEMPORARY_NOT_AVAILABLE".
         * </p>
         * </li>
         * <li>
         * <p>
         * FORM_FACTOR: The device form factor. Valid values are "PHONE" or "TABLET".
         * </p>
         * </li>
         * <li>
         * <p>
         * MANUFACTURER: The device manufacturer. For example, "Apple".
         * </p>
         * </li>
         * <li>
         * <p>
         * REMOTE_ACCESS_ENABLED: Whether the device is enabled for remote access.
         * </p>
         * </li>
         * <li>
         * <p>
         * REMOTE_DEBUG_ENABLED: Whether the device is enabled for remote debugging.
         * </p>
         * </li>
         * <li>
         * <p>
         * INSTANCE_ARN: The Amazon Resource Name (ARN) of the device instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * INSTANCE_LABELS: The label of the device instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_TYPE: The fleet type. Valid values are "PUBLIC" or "PRIVATE".
         * </p>
         * </li>
         * </ul>
         * 
         * @param attribute
         *        The aspect of a device such as platform or model used as the selection criteria in a device
         *        filter.</p>
         *        <p>
         *        Allowed values include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        ARN: The Amazon Resource Name (ARN) of the device. For example,
         *        "arn:aws:devicefarm:us-west-2::device:12345Example".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PLATFORM: The device platform. Valid values are "ANDROID" or "IOS".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        OS_VERSION: The operating system version. For example, "10.3.2".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MODEL: The device model. For example, "iPad 5th Gen".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AVAILABILITY: The current availability of the device. Valid values are "AVAILABLE",
         *        "HIGHLY_AVAILABLE", "BUSY", or "TEMPORARY_NOT_AVAILABLE".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FORM_FACTOR: The device form factor. Valid values are "PHONE" or "TABLET".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MANUFACTURER: The device manufacturer. For example, "Apple".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        REMOTE_ACCESS_ENABLED: Whether the device is enabled for remote access.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        REMOTE_DEBUG_ENABLED: Whether the device is enabled for remote debugging.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INSTANCE_ARN: The Amazon Resource Name (ARN) of the device instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INSTANCE_LABELS: The label of the device instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_TYPE: The fleet type. Valid values are "PUBLIC" or "PRIVATE".
         *        </p>
         *        </li>
         * @see DeviceFilterAttribute
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeviceFilterAttribute
         */
        Builder attribute(DeviceFilterAttribute attribute);

        /**
         * <p>
         * The filter operator.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The EQUALS operator is available for every attribute except INSTANCE_LABELS.
         * </p>
         * </li>
         * <li>
         * <p>
         * The CONTAINS operator is available for the INSTANCE_LABELS and MODEL attributes.
         * </p>
         * </li>
         * <li>
         * <p>
         * The IN and NOT_IN operators are available for the ARN, OS_VERSION, MODEL, MANUFACTURER, and INSTANCE_ARN
         * attributes.
         * </p>
         * </li>
         * <li>
         * <p>
         * The LESS_THAN, GREATER_THAN, LESS_THAN_OR_EQUALS, and GREATER_THAN_OR_EQUALS operators are also available for
         * the OS_VERSION attribute.
         * </p>
         * </li>
         * </ul>
         * 
         * @param operator
         *        The filter operator.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The EQUALS operator is available for every attribute except INSTANCE_LABELS.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The CONTAINS operator is available for the INSTANCE_LABELS and MODEL attributes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The IN and NOT_IN operators are available for the ARN, OS_VERSION, MODEL, MANUFACTURER, and
         *        INSTANCE_ARN attributes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The LESS_THAN, GREATER_THAN, LESS_THAN_OR_EQUALS, and GREATER_THAN_OR_EQUALS operators are also
         *        available for the OS_VERSION attribute.
         *        </p>
         *        </li>
         * @see DeviceFilterOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeviceFilterOperator
         */
        Builder operator(String operator);

        /**
         * <p>
         * The filter operator.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The EQUALS operator is available for every attribute except INSTANCE_LABELS.
         * </p>
         * </li>
         * <li>
         * <p>
         * The CONTAINS operator is available for the INSTANCE_LABELS and MODEL attributes.
         * </p>
         * </li>
         * <li>
         * <p>
         * The IN and NOT_IN operators are available for the ARN, OS_VERSION, MODEL, MANUFACTURER, and INSTANCE_ARN
         * attributes.
         * </p>
         * </li>
         * <li>
         * <p>
         * The LESS_THAN, GREATER_THAN, LESS_THAN_OR_EQUALS, and GREATER_THAN_OR_EQUALS operators are also available for
         * the OS_VERSION attribute.
         * </p>
         * </li>
         * </ul>
         * 
         * @param operator
         *        The filter operator.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The EQUALS operator is available for every attribute except INSTANCE_LABELS.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The CONTAINS operator is available for the INSTANCE_LABELS and MODEL attributes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The IN and NOT_IN operators are available for the ARN, OS_VERSION, MODEL, MANUFACTURER, and
         *        INSTANCE_ARN attributes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The LESS_THAN, GREATER_THAN, LESS_THAN_OR_EQUALS, and GREATER_THAN_OR_EQUALS operators are also
         *        available for the OS_VERSION attribute.
         *        </p>
         *        </li>
         * @see DeviceFilterOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeviceFilterOperator
         */
        Builder operator(DeviceFilterOperator operator);

        /**
         * <p>
         * An array of one or more filter values used in a device filter.
         * </p>
         * <p class="title">
         * <b>Operator Values</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * The IN and NOT operators can take a values array that has more than one element.
         * </p>
         * </li>
         * <li>
         * <p>
         * The other operators require an array with a single element.
         * </p>
         * </li>
         * </ul>
         * <p class="title">
         * <b>Attribute Values</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * The PLATFORM attribute can be set to "ANDROID" or "IOS".
         * </p>
         * </li>
         * <li>
         * <p>
         * The AVAILABILITY attribute can be set to "AVAILABLE", "HIGHLY_AVAILABLE", "BUSY", or
         * "TEMPORARY_NOT_AVAILABLE".
         * </p>
         * </li>
         * <li>
         * <p>
         * The FORM_FACTOR attribute can be set to "PHONE" or "TABLET".
         * </p>
         * </li>
         * <li>
         * <p>
         * The FLEET_TYPE attribute can be set to "PUBLIC" or "PRIVATE".
         * </p>
         * </li>
         * </ul>
         * 
         * @param values
         *        An array of one or more filter values used in a device filter.</p>
         *        <p class="title">
         *        <b>Operator Values</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The IN and NOT operators can take a values array that has more than one element.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The other operators require an array with a single element.
         *        </p>
         *        </li>
         *        </ul>
         *        <p class="title">
         *        <b>Attribute Values</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The PLATFORM attribute can be set to "ANDROID" or "IOS".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The AVAILABILITY attribute can be set to "AVAILABLE", "HIGHLY_AVAILABLE", "BUSY", or
         *        "TEMPORARY_NOT_AVAILABLE".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The FORM_FACTOR attribute can be set to "PHONE" or "TABLET".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The FLEET_TYPE attribute can be set to "PUBLIC" or "PRIVATE".
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<String> values);

        /**
         * <p>
         * An array of one or more filter values used in a device filter.
         * </p>
         * <p class="title">
         * <b>Operator Values</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * The IN and NOT operators can take a values array that has more than one element.
         * </p>
         * </li>
         * <li>
         * <p>
         * The other operators require an array with a single element.
         * </p>
         * </li>
         * </ul>
         * <p class="title">
         * <b>Attribute Values</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * The PLATFORM attribute can be set to "ANDROID" or "IOS".
         * </p>
         * </li>
         * <li>
         * <p>
         * The AVAILABILITY attribute can be set to "AVAILABLE", "HIGHLY_AVAILABLE", "BUSY", or
         * "TEMPORARY_NOT_AVAILABLE".
         * </p>
         * </li>
         * <li>
         * <p>
         * The FORM_FACTOR attribute can be set to "PHONE" or "TABLET".
         * </p>
         * </li>
         * <li>
         * <p>
         * The FLEET_TYPE attribute can be set to "PUBLIC" or "PRIVATE".
         * </p>
         * </li>
         * </ul>
         * 
         * @param values
         *        An array of one or more filter values used in a device filter.</p>
         *        <p class="title">
         *        <b>Operator Values</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The IN and NOT operators can take a values array that has more than one element.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The other operators require an array with a single element.
         *        </p>
         *        </li>
         *        </ul>
         *        <p class="title">
         *        <b>Attribute Values</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The PLATFORM attribute can be set to "ANDROID" or "IOS".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The AVAILABILITY attribute can be set to "AVAILABLE", "HIGHLY_AVAILABLE", "BUSY", or
         *        "TEMPORARY_NOT_AVAILABLE".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The FORM_FACTOR attribute can be set to "PHONE" or "TABLET".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The FLEET_TYPE attribute can be set to "PUBLIC" or "PRIVATE".
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(String... values);
    }

    static final class BuilderImpl implements Builder {
        private String attribute;

        private String operator;

        private List<String> values = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DeviceFilter model) {
            attribute(model.attribute);
            operator(model.operator);
            values(model.values);
        }

        public final String getAttribute() {
            return attribute;
        }

        @Override
        public final Builder attribute(String attribute) {
            this.attribute = attribute;
            return this;
        }

        @Override
        public final Builder attribute(DeviceFilterAttribute attribute) {
            this.attribute(attribute.toString());
            return this;
        }

        public final void setAttribute(String attribute) {
            this.attribute = attribute;
        }

        public final String getOperator() {
            return operator;
        }

        @Override
        public final Builder operator(String operator) {
            this.operator = operator;
            return this;
        }

        @Override
        public final Builder operator(DeviceFilterOperator operator) {
            this.operator(operator.toString());
            return this;
        }

        public final void setOperator(String operator) {
            this.operator = operator;
        }

        public final Collection<String> getValues() {
            return values;
        }

        @Override
        public final Builder values(Collection<String> values) {
            this.values = DeviceFilterValuesCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(String... values) {
            values(Arrays.asList(values));
            return this;
        }

        public final void setValues(Collection<String> values) {
            this.values = DeviceFilterValuesCopier.copy(values);
        }

        @Override
        public DeviceFilter build() {
            return new DeviceFilter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
