/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.devicefarm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the device filters used in a test run as well as the maximum number of devices to be included in the run.
 * It is passed in as the <code>deviceSelectionConfiguration</code> request parameter in <a>ScheduleRun</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeviceSelectionConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<DeviceSelectionConfiguration.Builder, DeviceSelectionConfiguration> {
    private static final SdkField<List<DeviceFilter>> FILTERS_FIELD = SdkField
            .<List<DeviceFilter>> builder(MarshallingType.LIST)
            .getter(getter(DeviceSelectionConfiguration::filters))
            .setter(setter(Builder::filters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("filters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DeviceFilter> builder(MarshallingType.SDK_POJO)
                                            .constructor(DeviceFilter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> MAX_DEVICES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(DeviceSelectionConfiguration::maxDevices)).setter(setter(Builder::maxDevices))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxDevices").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FILTERS_FIELD,
            MAX_DEVICES_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<DeviceFilter> filters;

    private final Integer maxDevices;

    private DeviceSelectionConfiguration(BuilderImpl builder) {
        this.filters = builder.filters;
        this.maxDevices = builder.maxDevices;
    }

    /**
     * Returns true if the Filters property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasFilters() {
        return filters != null && !(filters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Used to dynamically select a set of devices for a test run. A filter is made up of an attribute, an operator, and
     * one or more values.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Attribute</b>
     * </p>
     * <p>
     * The aspect of a device such as platform or model used as the selection criteria in a device filter.
     * </p>
     * <p>
     * Allowed values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * ARN: The Amazon Resource Name (ARN) of the device. For example,
     * "arn:aws:devicefarm:us-west-2::device:12345Example".
     * </p>
     * </li>
     * <li>
     * <p>
     * PLATFORM: The device platform. Valid values are "ANDROID" or "IOS".
     * </p>
     * </li>
     * <li>
     * <p>
     * OS_VERSION: The operating system version. For example, "10.3.2".
     * </p>
     * </li>
     * <li>
     * <p>
     * MODEL: The device model. For example, "iPad 5th Gen".
     * </p>
     * </li>
     * <li>
     * <p>
     * AVAILABILITY: The current availability of the device. Valid values are "AVAILABLE", "HIGHLY_AVAILABLE", "BUSY",
     * or "TEMPORARY_NOT_AVAILABLE".
     * </p>
     * </li>
     * <li>
     * <p>
     * FORM_FACTOR: The device form factor. Valid values are "PHONE" or "TABLET".
     * </p>
     * </li>
     * <li>
     * <p>
     * MANUFACTURER: The device manufacturer. For example, "Apple".
     * </p>
     * </li>
     * <li>
     * <p>
     * REMOTE_ACCESS_ENABLED: Whether the device is enabled for remote access. Valid values are "TRUE" or "FALSE".
     * </p>
     * </li>
     * <li>
     * <p>
     * REMOTE_DEBUG_ENABLED: Whether the device is enabled for remote debugging. Valid values are "TRUE" or "FALSE".
     * <i>This filter will be ignored, as remote debugging is <a
     * href="https://docs.aws.amazon.com/devicefarm/latest/developerguide/history.html">no longer supported</a>.</i>
     * </p>
     * </li>
     * <li>
     * <p>
     * INSTANCE_ARN: The Amazon Resource Name (ARN) of the device instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * INSTANCE_LABELS: The label of the device instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * FLEET_TYPE: The fleet type. Valid values are "PUBLIC" or "PRIVATE".
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <b>Operator</b>
     * </p>
     * <p>
     * The filter operator.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The EQUALS operator is available for every attribute except INSTANCE_LABELS.
     * </p>
     * </li>
     * <li>
     * <p>
     * The CONTAINS operator is available for the INSTANCE_LABELS and MODEL attributes.
     * </p>
     * </li>
     * <li>
     * <p>
     * The IN and NOT_IN operators are available for the ARN, OS_VERSION, MODEL, MANUFACTURER, and INSTANCE_ARN
     * attributes.
     * </p>
     * </li>
     * <li>
     * <p>
     * The LESS_THAN, GREATER_THAN, LESS_THAN_OR_EQUALS, and GREATER_THAN_OR_EQUALS operators are also available for the
     * OS_VERSION attribute.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <b>Values</b>
     * </p>
     * <p>
     * An array of one or more filter values.
     * </p>
     * <p class="title">
     * <b>Operator Values</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The IN and NOT_IN operators can take a values array that has more than one element.
     * </p>
     * </li>
     * <li>
     * <p>
     * The other operators require an array with a single element.
     * </p>
     * </li>
     * </ul>
     * <p class="title">
     * <b>Attribute Values</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The PLATFORM attribute can be set to "ANDROID" or "IOS".
     * </p>
     * </li>
     * <li>
     * <p>
     * The AVAILABILITY attribute can be set to "AVAILABLE", "HIGHLY_AVAILABLE", "BUSY", or "TEMPORARY_NOT_AVAILABLE".
     * </p>
     * </li>
     * <li>
     * <p>
     * The FORM_FACTOR attribute can be set to "PHONE" or "TABLET".
     * </p>
     * </li>
     * <li>
     * <p>
     * The FLEET_TYPE attribute can be set to "PUBLIC" or "PRIVATE".
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasFilters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Used to dynamically select a set of devices for a test run. A filter is made up of an attribute, an
     *         operator, and one or more values.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>Attribute</b>
     *         </p>
     *         <p>
     *         The aspect of a device such as platform or model used as the selection criteria in a device filter.
     *         </p>
     *         <p>
     *         Allowed values include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         ARN: The Amazon Resource Name (ARN) of the device. For example,
     *         "arn:aws:devicefarm:us-west-2::device:12345Example".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PLATFORM: The device platform. Valid values are "ANDROID" or "IOS".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         OS_VERSION: The operating system version. For example, "10.3.2".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MODEL: The device model. For example, "iPad 5th Gen".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AVAILABILITY: The current availability of the device. Valid values are "AVAILABLE", "HIGHLY_AVAILABLE",
     *         "BUSY", or "TEMPORARY_NOT_AVAILABLE".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FORM_FACTOR: The device form factor. Valid values are "PHONE" or "TABLET".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MANUFACTURER: The device manufacturer. For example, "Apple".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         REMOTE_ACCESS_ENABLED: Whether the device is enabled for remote access. Valid values are "TRUE" or
     *         "FALSE".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         REMOTE_DEBUG_ENABLED: Whether the device is enabled for remote debugging. Valid values are "TRUE" or
     *         "FALSE". <i>This filter will be ignored, as remote debugging is <a
     *         href="https://docs.aws.amazon.com/devicefarm/latest/developerguide/history.html">no longer
     *         supported</a>.</i>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INSTANCE_ARN: The Amazon Resource Name (ARN) of the device instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INSTANCE_LABELS: The label of the device instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FLEET_TYPE: The fleet type. Valid values are "PUBLIC" or "PRIVATE".
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Operator</b>
     *         </p>
     *         <p>
     *         The filter operator.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The EQUALS operator is available for every attribute except INSTANCE_LABELS.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The CONTAINS operator is available for the INSTANCE_LABELS and MODEL attributes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The IN and NOT_IN operators are available for the ARN, OS_VERSION, MODEL, MANUFACTURER, and INSTANCE_ARN
     *         attributes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The LESS_THAN, GREATER_THAN, LESS_THAN_OR_EQUALS, and GREATER_THAN_OR_EQUALS operators are also available
     *         for the OS_VERSION attribute.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <b>Values</b>
     *         </p>
     *         <p>
     *         An array of one or more filter values.
     *         </p>
     *         <p class="title">
     *         <b>Operator Values</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The IN and NOT_IN operators can take a values array that has more than one element.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The other operators require an array with a single element.
     *         </p>
     *         </li>
     *         </ul>
     *         <p class="title">
     *         <b>Attribute Values</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The PLATFORM attribute can be set to "ANDROID" or "IOS".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The AVAILABILITY attribute can be set to "AVAILABLE", "HIGHLY_AVAILABLE", "BUSY", or
     *         "TEMPORARY_NOT_AVAILABLE".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The FORM_FACTOR attribute can be set to "PHONE" or "TABLET".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The FLEET_TYPE attribute can be set to "PUBLIC" or "PRIVATE".
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     */
    public List<DeviceFilter> filters() {
        return filters;
    }

    /**
     * <p>
     * The maximum number of devices to be included in a test run.
     * </p>
     * 
     * @return The maximum number of devices to be included in a test run.
     */
    public Integer maxDevices() {
        return maxDevices;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(filters());
        hashCode = 31 * hashCode + Objects.hashCode(maxDevices());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeviceSelectionConfiguration)) {
            return false;
        }
        DeviceSelectionConfiguration other = (DeviceSelectionConfiguration) obj;
        return Objects.equals(filters(), other.filters()) && Objects.equals(maxDevices(), other.maxDevices());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DeviceSelectionConfiguration").add("Filters", filters()).add("MaxDevices", maxDevices()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "filters":
            return Optional.ofNullable(clazz.cast(filters()));
        case "maxDevices":
            return Optional.ofNullable(clazz.cast(maxDevices()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DeviceSelectionConfiguration, T> g) {
        return obj -> g.apply((DeviceSelectionConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeviceSelectionConfiguration> {
        /**
         * <p>
         * Used to dynamically select a set of devices for a test run. A filter is made up of an attribute, an operator,
         * and one or more values.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>Attribute</b>
         * </p>
         * <p>
         * The aspect of a device such as platform or model used as the selection criteria in a device filter.
         * </p>
         * <p>
         * Allowed values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * ARN: The Amazon Resource Name (ARN) of the device. For example,
         * "arn:aws:devicefarm:us-west-2::device:12345Example".
         * </p>
         * </li>
         * <li>
         * <p>
         * PLATFORM: The device platform. Valid values are "ANDROID" or "IOS".
         * </p>
         * </li>
         * <li>
         * <p>
         * OS_VERSION: The operating system version. For example, "10.3.2".
         * </p>
         * </li>
         * <li>
         * <p>
         * MODEL: The device model. For example, "iPad 5th Gen".
         * </p>
         * </li>
         * <li>
         * <p>
         * AVAILABILITY: The current availability of the device. Valid values are "AVAILABLE", "HIGHLY_AVAILABLE",
         * "BUSY", or "TEMPORARY_NOT_AVAILABLE".
         * </p>
         * </li>
         * <li>
         * <p>
         * FORM_FACTOR: The device form factor. Valid values are "PHONE" or "TABLET".
         * </p>
         * </li>
         * <li>
         * <p>
         * MANUFACTURER: The device manufacturer. For example, "Apple".
         * </p>
         * </li>
         * <li>
         * <p>
         * REMOTE_ACCESS_ENABLED: Whether the device is enabled for remote access. Valid values are "TRUE" or "FALSE".
         * </p>
         * </li>
         * <li>
         * <p>
         * REMOTE_DEBUG_ENABLED: Whether the device is enabled for remote debugging. Valid values are "TRUE" or "FALSE".
         * <i>This filter will be ignored, as remote debugging is <a
         * href="https://docs.aws.amazon.com/devicefarm/latest/developerguide/history.html">no longer supported</a>.</i>
         * </p>
         * </li>
         * <li>
         * <p>
         * INSTANCE_ARN: The Amazon Resource Name (ARN) of the device instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * INSTANCE_LABELS: The label of the device instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_TYPE: The fleet type. Valid values are "PUBLIC" or "PRIVATE".
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <b>Operator</b>
         * </p>
         * <p>
         * The filter operator.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The EQUALS operator is available for every attribute except INSTANCE_LABELS.
         * </p>
         * </li>
         * <li>
         * <p>
         * The CONTAINS operator is available for the INSTANCE_LABELS and MODEL attributes.
         * </p>
         * </li>
         * <li>
         * <p>
         * The IN and NOT_IN operators are available for the ARN, OS_VERSION, MODEL, MANUFACTURER, and INSTANCE_ARN
         * attributes.
         * </p>
         * </li>
         * <li>
         * <p>
         * The LESS_THAN, GREATER_THAN, LESS_THAN_OR_EQUALS, and GREATER_THAN_OR_EQUALS operators are also available for
         * the OS_VERSION attribute.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <b>Values</b>
         * </p>
         * <p>
         * An array of one or more filter values.
         * </p>
         * <p class="title">
         * <b>Operator Values</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * The IN and NOT_IN operators can take a values array that has more than one element.
         * </p>
         * </li>
         * <li>
         * <p>
         * The other operators require an array with a single element.
         * </p>
         * </li>
         * </ul>
         * <p class="title">
         * <b>Attribute Values</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * The PLATFORM attribute can be set to "ANDROID" or "IOS".
         * </p>
         * </li>
         * <li>
         * <p>
         * The AVAILABILITY attribute can be set to "AVAILABLE", "HIGHLY_AVAILABLE", "BUSY", or
         * "TEMPORARY_NOT_AVAILABLE".
         * </p>
         * </li>
         * <li>
         * <p>
         * The FORM_FACTOR attribute can be set to "PHONE" or "TABLET".
         * </p>
         * </li>
         * <li>
         * <p>
         * The FLEET_TYPE attribute can be set to "PUBLIC" or "PRIVATE".
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param filters
         *        Used to dynamically select a set of devices for a test run. A filter is made up of an attribute, an
         *        operator, and one or more values.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>Attribute</b>
         *        </p>
         *        <p>
         *        The aspect of a device such as platform or model used as the selection criteria in a device filter.
         *        </p>
         *        <p>
         *        Allowed values include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        ARN: The Amazon Resource Name (ARN) of the device. For example,
         *        "arn:aws:devicefarm:us-west-2::device:12345Example".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PLATFORM: The device platform. Valid values are "ANDROID" or "IOS".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        OS_VERSION: The operating system version. For example, "10.3.2".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MODEL: The device model. For example, "iPad 5th Gen".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AVAILABILITY: The current availability of the device. Valid values are "AVAILABLE",
         *        "HIGHLY_AVAILABLE", "BUSY", or "TEMPORARY_NOT_AVAILABLE".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FORM_FACTOR: The device form factor. Valid values are "PHONE" or "TABLET".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MANUFACTURER: The device manufacturer. For example, "Apple".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        REMOTE_ACCESS_ENABLED: Whether the device is enabled for remote access. Valid values are "TRUE" or
         *        "FALSE".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        REMOTE_DEBUG_ENABLED: Whether the device is enabled for remote debugging. Valid values are "TRUE" or
         *        "FALSE". <i>This filter will be ignored, as remote debugging is <a
         *        href="https://docs.aws.amazon.com/devicefarm/latest/developerguide/history.html">no longer
         *        supported</a>.</i>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INSTANCE_ARN: The Amazon Resource Name (ARN) of the device instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INSTANCE_LABELS: The label of the device instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_TYPE: The fleet type. Valid values are "PUBLIC" or "PRIVATE".
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Operator</b>
         *        </p>
         *        <p>
         *        The filter operator.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The EQUALS operator is available for every attribute except INSTANCE_LABELS.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The CONTAINS operator is available for the INSTANCE_LABELS and MODEL attributes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The IN and NOT_IN operators are available for the ARN, OS_VERSION, MODEL, MANUFACTURER, and
         *        INSTANCE_ARN attributes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The LESS_THAN, GREATER_THAN, LESS_THAN_OR_EQUALS, and GREATER_THAN_OR_EQUALS operators are also
         *        available for the OS_VERSION attribute.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Values</b>
         *        </p>
         *        <p>
         *        An array of one or more filter values.
         *        </p>
         *        <p class="title">
         *        <b>Operator Values</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The IN and NOT_IN operators can take a values array that has more than one element.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The other operators require an array with a single element.
         *        </p>
         *        </li>
         *        </ul>
         *        <p class="title">
         *        <b>Attribute Values</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The PLATFORM attribute can be set to "ANDROID" or "IOS".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The AVAILABILITY attribute can be set to "AVAILABLE", "HIGHLY_AVAILABLE", "BUSY", or
         *        "TEMPORARY_NOT_AVAILABLE".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The FORM_FACTOR attribute can be set to "PHONE" or "TABLET".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The FLEET_TYPE attribute can be set to "PUBLIC" or "PRIVATE".
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(Collection<DeviceFilter> filters);

        /**
         * <p>
         * Used to dynamically select a set of devices for a test run. A filter is made up of an attribute, an operator,
         * and one or more values.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>Attribute</b>
         * </p>
         * <p>
         * The aspect of a device such as platform or model used as the selection criteria in a device filter.
         * </p>
         * <p>
         * Allowed values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * ARN: The Amazon Resource Name (ARN) of the device. For example,
         * "arn:aws:devicefarm:us-west-2::device:12345Example".
         * </p>
         * </li>
         * <li>
         * <p>
         * PLATFORM: The device platform. Valid values are "ANDROID" or "IOS".
         * </p>
         * </li>
         * <li>
         * <p>
         * OS_VERSION: The operating system version. For example, "10.3.2".
         * </p>
         * </li>
         * <li>
         * <p>
         * MODEL: The device model. For example, "iPad 5th Gen".
         * </p>
         * </li>
         * <li>
         * <p>
         * AVAILABILITY: The current availability of the device. Valid values are "AVAILABLE", "HIGHLY_AVAILABLE",
         * "BUSY", or "TEMPORARY_NOT_AVAILABLE".
         * </p>
         * </li>
         * <li>
         * <p>
         * FORM_FACTOR: The device form factor. Valid values are "PHONE" or "TABLET".
         * </p>
         * </li>
         * <li>
         * <p>
         * MANUFACTURER: The device manufacturer. For example, "Apple".
         * </p>
         * </li>
         * <li>
         * <p>
         * REMOTE_ACCESS_ENABLED: Whether the device is enabled for remote access. Valid values are "TRUE" or "FALSE".
         * </p>
         * </li>
         * <li>
         * <p>
         * REMOTE_DEBUG_ENABLED: Whether the device is enabled for remote debugging. Valid values are "TRUE" or "FALSE".
         * <i>This filter will be ignored, as remote debugging is <a
         * href="https://docs.aws.amazon.com/devicefarm/latest/developerguide/history.html">no longer supported</a>.</i>
         * </p>
         * </li>
         * <li>
         * <p>
         * INSTANCE_ARN: The Amazon Resource Name (ARN) of the device instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * INSTANCE_LABELS: The label of the device instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_TYPE: The fleet type. Valid values are "PUBLIC" or "PRIVATE".
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <b>Operator</b>
         * </p>
         * <p>
         * The filter operator.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The EQUALS operator is available for every attribute except INSTANCE_LABELS.
         * </p>
         * </li>
         * <li>
         * <p>
         * The CONTAINS operator is available for the INSTANCE_LABELS and MODEL attributes.
         * </p>
         * </li>
         * <li>
         * <p>
         * The IN and NOT_IN operators are available for the ARN, OS_VERSION, MODEL, MANUFACTURER, and INSTANCE_ARN
         * attributes.
         * </p>
         * </li>
         * <li>
         * <p>
         * The LESS_THAN, GREATER_THAN, LESS_THAN_OR_EQUALS, and GREATER_THAN_OR_EQUALS operators are also available for
         * the OS_VERSION attribute.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <b>Values</b>
         * </p>
         * <p>
         * An array of one or more filter values.
         * </p>
         * <p class="title">
         * <b>Operator Values</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * The IN and NOT_IN operators can take a values array that has more than one element.
         * </p>
         * </li>
         * <li>
         * <p>
         * The other operators require an array with a single element.
         * </p>
         * </li>
         * </ul>
         * <p class="title">
         * <b>Attribute Values</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * The PLATFORM attribute can be set to "ANDROID" or "IOS".
         * </p>
         * </li>
         * <li>
         * <p>
         * The AVAILABILITY attribute can be set to "AVAILABLE", "HIGHLY_AVAILABLE", "BUSY", or
         * "TEMPORARY_NOT_AVAILABLE".
         * </p>
         * </li>
         * <li>
         * <p>
         * The FORM_FACTOR attribute can be set to "PHONE" or "TABLET".
         * </p>
         * </li>
         * <li>
         * <p>
         * The FLEET_TYPE attribute can be set to "PUBLIC" or "PRIVATE".
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param filters
         *        Used to dynamically select a set of devices for a test run. A filter is made up of an attribute, an
         *        operator, and one or more values.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>Attribute</b>
         *        </p>
         *        <p>
         *        The aspect of a device such as platform or model used as the selection criteria in a device filter.
         *        </p>
         *        <p>
         *        Allowed values include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        ARN: The Amazon Resource Name (ARN) of the device. For example,
         *        "arn:aws:devicefarm:us-west-2::device:12345Example".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PLATFORM: The device platform. Valid values are "ANDROID" or "IOS".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        OS_VERSION: The operating system version. For example, "10.3.2".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MODEL: The device model. For example, "iPad 5th Gen".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AVAILABILITY: The current availability of the device. Valid values are "AVAILABLE",
         *        "HIGHLY_AVAILABLE", "BUSY", or "TEMPORARY_NOT_AVAILABLE".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FORM_FACTOR: The device form factor. Valid values are "PHONE" or "TABLET".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MANUFACTURER: The device manufacturer. For example, "Apple".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        REMOTE_ACCESS_ENABLED: Whether the device is enabled for remote access. Valid values are "TRUE" or
         *        "FALSE".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        REMOTE_DEBUG_ENABLED: Whether the device is enabled for remote debugging. Valid values are "TRUE" or
         *        "FALSE". <i>This filter will be ignored, as remote debugging is <a
         *        href="https://docs.aws.amazon.com/devicefarm/latest/developerguide/history.html">no longer
         *        supported</a>.</i>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INSTANCE_ARN: The Amazon Resource Name (ARN) of the device instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INSTANCE_LABELS: The label of the device instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FLEET_TYPE: The fleet type. Valid values are "PUBLIC" or "PRIVATE".
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Operator</b>
         *        </p>
         *        <p>
         *        The filter operator.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The EQUALS operator is available for every attribute except INSTANCE_LABELS.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The CONTAINS operator is available for the INSTANCE_LABELS and MODEL attributes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The IN and NOT_IN operators are available for the ARN, OS_VERSION, MODEL, MANUFACTURER, and
         *        INSTANCE_ARN attributes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The LESS_THAN, GREATER_THAN, LESS_THAN_OR_EQUALS, and GREATER_THAN_OR_EQUALS operators are also
         *        available for the OS_VERSION attribute.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <b>Values</b>
         *        </p>
         *        <p>
         *        An array of one or more filter values.
         *        </p>
         *        <p class="title">
         *        <b>Operator Values</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The IN and NOT_IN operators can take a values array that has more than one element.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The other operators require an array with a single element.
         *        </p>
         *        </li>
         *        </ul>
         *        <p class="title">
         *        <b>Attribute Values</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The PLATFORM attribute can be set to "ANDROID" or "IOS".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The AVAILABILITY attribute can be set to "AVAILABLE", "HIGHLY_AVAILABLE", "BUSY", or
         *        "TEMPORARY_NOT_AVAILABLE".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The FORM_FACTOR attribute can be set to "PHONE" or "TABLET".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The FLEET_TYPE attribute can be set to "PUBLIC" or "PRIVATE".
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(DeviceFilter... filters);

        /**
         * <p>
         * Used to dynamically select a set of devices for a test run. A filter is made up of an attribute, an operator,
         * and one or more values.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>Attribute</b>
         * </p>
         * <p>
         * The aspect of a device such as platform or model used as the selection criteria in a device filter.
         * </p>
         * <p>
         * Allowed values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * ARN: The Amazon Resource Name (ARN) of the device. For example,
         * "arn:aws:devicefarm:us-west-2::device:12345Example".
         * </p>
         * </li>
         * <li>
         * <p>
         * PLATFORM: The device platform. Valid values are "ANDROID" or "IOS".
         * </p>
         * </li>
         * <li>
         * <p>
         * OS_VERSION: The operating system version. For example, "10.3.2".
         * </p>
         * </li>
         * <li>
         * <p>
         * MODEL: The device model. For example, "iPad 5th Gen".
         * </p>
         * </li>
         * <li>
         * <p>
         * AVAILABILITY: The current availability of the device. Valid values are "AVAILABLE", "HIGHLY_AVAILABLE",
         * "BUSY", or "TEMPORARY_NOT_AVAILABLE".
         * </p>
         * </li>
         * <li>
         * <p>
         * FORM_FACTOR: The device form factor. Valid values are "PHONE" or "TABLET".
         * </p>
         * </li>
         * <li>
         * <p>
         * MANUFACTURER: The device manufacturer. For example, "Apple".
         * </p>
         * </li>
         * <li>
         * <p>
         * REMOTE_ACCESS_ENABLED: Whether the device is enabled for remote access. Valid values are "TRUE" or "FALSE".
         * </p>
         * </li>
         * <li>
         * <p>
         * REMOTE_DEBUG_ENABLED: Whether the device is enabled for remote debugging. Valid values are "TRUE" or "FALSE".
         * <i>This filter will be ignored, as remote debugging is <a
         * href="https://docs.aws.amazon.com/devicefarm/latest/developerguide/history.html">no longer supported</a>.</i>
         * </p>
         * </li>
         * <li>
         * <p>
         * INSTANCE_ARN: The Amazon Resource Name (ARN) of the device instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * INSTANCE_LABELS: The label of the device instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * FLEET_TYPE: The fleet type. Valid values are "PUBLIC" or "PRIVATE".
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <b>Operator</b>
         * </p>
         * <p>
         * The filter operator.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The EQUALS operator is available for every attribute except INSTANCE_LABELS.
         * </p>
         * </li>
         * <li>
         * <p>
         * The CONTAINS operator is available for the INSTANCE_LABELS and MODEL attributes.
         * </p>
         * </li>
         * <li>
         * <p>
         * The IN and NOT_IN operators are available for the ARN, OS_VERSION, MODEL, MANUFACTURER, and INSTANCE_ARN
         * attributes.
         * </p>
         * </li>
         * <li>
         * <p>
         * The LESS_THAN, GREATER_THAN, LESS_THAN_OR_EQUALS, and GREATER_THAN_OR_EQUALS operators are also available for
         * the OS_VERSION attribute.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <b>Values</b>
         * </p>
         * <p>
         * An array of one or more filter values.
         * </p>
         * <p class="title">
         * <b>Operator Values</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * The IN and NOT_IN operators can take a values array that has more than one element.
         * </p>
         * </li>
         * <li>
         * <p>
         * The other operators require an array with a single element.
         * </p>
         * </li>
         * </ul>
         * <p class="title">
         * <b>Attribute Values</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * The PLATFORM attribute can be set to "ANDROID" or "IOS".
         * </p>
         * </li>
         * <li>
         * <p>
         * The AVAILABILITY attribute can be set to "AVAILABLE", "HIGHLY_AVAILABLE", "BUSY", or
         * "TEMPORARY_NOT_AVAILABLE".
         * </p>
         * </li>
         * <li>
         * <p>
         * The FORM_FACTOR attribute can be set to "PHONE" or "TABLET".
         * </p>
         * </li>
         * <li>
         * <p>
         * The FLEET_TYPE attribute can be set to "PUBLIC" or "PRIVATE".
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link List<DeviceFilter>.Builder} avoiding the need to
         * create one manually via {@link List<DeviceFilter>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DeviceFilter>.Builder#build()} is called immediately and its
         * result is passed to {@link #filters(List<DeviceFilter>)}.
         * 
         * @param filters
         *        a consumer that will call methods on {@link List<DeviceFilter>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filters(List<DeviceFilter>)
         */
        Builder filters(Consumer<DeviceFilter.Builder>... filters);

        /**
         * <p>
         * The maximum number of devices to be included in a test run.
         * </p>
         * 
         * @param maxDevices
         *        The maximum number of devices to be included in a test run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxDevices(Integer maxDevices);
    }

    static final class BuilderImpl implements Builder {
        private List<DeviceFilter> filters = DefaultSdkAutoConstructList.getInstance();

        private Integer maxDevices;

        private BuilderImpl() {
        }

        private BuilderImpl(DeviceSelectionConfiguration model) {
            filters(model.filters);
            maxDevices(model.maxDevices);
        }

        public final Collection<DeviceFilter.Builder> getFilters() {
            return filters != null ? filters.stream().map(DeviceFilter::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder filters(Collection<DeviceFilter> filters) {
            this.filters = DeviceFiltersCopier.copy(filters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(DeviceFilter... filters) {
            filters(Arrays.asList(filters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(Consumer<DeviceFilter.Builder>... filters) {
            filters(Stream.of(filters).map(c -> DeviceFilter.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setFilters(Collection<DeviceFilter.BuilderImpl> filters) {
            this.filters = DeviceFiltersCopier.copyFromBuilder(filters);
        }

        public final Integer getMaxDevices() {
            return maxDevices;
        }

        @Override
        public final Builder maxDevices(Integer maxDevices) {
            this.maxDevices = maxDevices;
            return this;
        }

        public final void setMaxDevices(Integer maxDevices) {
            this.maxDevices = maxDevices;
        }

        @Override
        public DeviceSelectionConfiguration build() {
            return new DeviceSelectionConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
