/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directory.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * AWS Directory Service for Microsoft Active Directory allows you to configure trust relationships. For example, you
 * can establish a trust between your AWS Managed Microsoft AD directory, and your existing on-premises Microsoft Active
 * Directory. This would allow you to provide users and groups access to resources in either domain, with a single set
 * of credentials.
 * </p>
 * <p>
 * This action initiates the creation of the AWS side of a trust relationship between an AWS Managed Microsoft AD
 * directory and an external domain.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateTrustRequest extends DirectoryRequest implements
        ToCopyableBuilder<CreateTrustRequest.Builder, CreateTrustRequest> {
    private static final SdkField<String> DIRECTORY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateTrustRequest::directoryId)).setter(setter(Builder::directoryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DirectoryId").build()).build();

    private static final SdkField<String> REMOTE_DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateTrustRequest::remoteDomainName)).setter(setter(Builder::remoteDomainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RemoteDomainName").build()).build();

    private static final SdkField<String> TRUST_PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateTrustRequest::trustPassword)).setter(setter(Builder::trustPassword))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrustPassword").build()).build();

    private static final SdkField<String> TRUST_DIRECTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateTrustRequest::trustDirectionAsString)).setter(setter(Builder::trustDirection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrustDirection").build()).build();

    private static final SdkField<String> TRUST_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateTrustRequest::trustTypeAsString)).setter(setter(Builder::trustType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrustType").build()).build();

    private static final SdkField<List<String>> CONDITIONAL_FORWARDER_IP_ADDRS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(CreateTrustRequest::conditionalForwarderIpAddrs))
            .setter(setter(Builder::conditionalForwarderIpAddrs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConditionalForwarderIpAddrs")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SELECTIVE_AUTH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateTrustRequest::selectiveAuthAsString)).setter(setter(Builder::selectiveAuth))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SelectiveAuth").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DIRECTORY_ID_FIELD,
            REMOTE_DOMAIN_NAME_FIELD, TRUST_PASSWORD_FIELD, TRUST_DIRECTION_FIELD, TRUST_TYPE_FIELD,
            CONDITIONAL_FORWARDER_IP_ADDRS_FIELD, SELECTIVE_AUTH_FIELD));

    private final String directoryId;

    private final String remoteDomainName;

    private final String trustPassword;

    private final String trustDirection;

    private final String trustType;

    private final List<String> conditionalForwarderIpAddrs;

    private final String selectiveAuth;

    private CreateTrustRequest(BuilderImpl builder) {
        super(builder);
        this.directoryId = builder.directoryId;
        this.remoteDomainName = builder.remoteDomainName;
        this.trustPassword = builder.trustPassword;
        this.trustDirection = builder.trustDirection;
        this.trustType = builder.trustType;
        this.conditionalForwarderIpAddrs = builder.conditionalForwarderIpAddrs;
        this.selectiveAuth = builder.selectiveAuth;
    }

    /**
     * <p>
     * The Directory ID of the AWS Managed Microsoft AD directory for which to establish the trust relationship.
     * </p>
     * 
     * @return The Directory ID of the AWS Managed Microsoft AD directory for which to establish the trust relationship.
     */
    public String directoryId() {
        return directoryId;
    }

    /**
     * <p>
     * The Fully Qualified Domain Name (FQDN) of the external domain for which to create the trust relationship.
     * </p>
     * 
     * @return The Fully Qualified Domain Name (FQDN) of the external domain for which to create the trust relationship.
     */
    public String remoteDomainName() {
        return remoteDomainName;
    }

    /**
     * <p>
     * The trust password. The must be the same password that was used when creating the trust relationship on the
     * external domain.
     * </p>
     * 
     * @return The trust password. The must be the same password that was used when creating the trust relationship on
     *         the external domain.
     */
    public String trustPassword() {
        return trustPassword;
    }

    /**
     * <p>
     * The direction of the trust relationship.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #trustDirection}
     * will return {@link TrustDirection#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #trustDirectionAsString}.
     * </p>
     * 
     * @return The direction of the trust relationship.
     * @see TrustDirection
     */
    public TrustDirection trustDirection() {
        return TrustDirection.fromValue(trustDirection);
    }

    /**
     * <p>
     * The direction of the trust relationship.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #trustDirection}
     * will return {@link TrustDirection#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #trustDirectionAsString}.
     * </p>
     * 
     * @return The direction of the trust relationship.
     * @see TrustDirection
     */
    public String trustDirectionAsString() {
        return trustDirection;
    }

    /**
     * <p>
     * The trust relationship type. <code>Forest</code> is the default.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #trustType} will
     * return {@link TrustType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #trustTypeAsString}.
     * </p>
     * 
     * @return The trust relationship type. <code>Forest</code> is the default.
     * @see TrustType
     */
    public TrustType trustType() {
        return TrustType.fromValue(trustType);
    }

    /**
     * <p>
     * The trust relationship type. <code>Forest</code> is the default.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #trustType} will
     * return {@link TrustType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #trustTypeAsString}.
     * </p>
     * 
     * @return The trust relationship type. <code>Forest</code> is the default.
     * @see TrustType
     */
    public String trustTypeAsString() {
        return trustType;
    }

    /**
     * Returns true if the ConditionalForwarderIpAddrs property was specified by the sender (it may be empty), or false
     * if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the
     * AWS service.
     */
    public boolean hasConditionalForwarderIpAddrs() {
        return conditionalForwarderIpAddrs != null && !(conditionalForwarderIpAddrs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IP addresses of the remote DNS server associated with RemoteDomainName.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasConditionalForwarderIpAddrs()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The IP addresses of the remote DNS server associated with RemoteDomainName.
     */
    public List<String> conditionalForwarderIpAddrs() {
        return conditionalForwarderIpAddrs;
    }

    /**
     * <p>
     * Optional parameter to enable selective authentication for the trust.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #selectiveAuth}
     * will return {@link SelectiveAuth#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #selectiveAuthAsString}.
     * </p>
     * 
     * @return Optional parameter to enable selective authentication for the trust.
     * @see SelectiveAuth
     */
    public SelectiveAuth selectiveAuth() {
        return SelectiveAuth.fromValue(selectiveAuth);
    }

    /**
     * <p>
     * Optional parameter to enable selective authentication for the trust.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #selectiveAuth}
     * will return {@link SelectiveAuth#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #selectiveAuthAsString}.
     * </p>
     * 
     * @return Optional parameter to enable selective authentication for the trust.
     * @see SelectiveAuth
     */
    public String selectiveAuthAsString() {
        return selectiveAuth;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(directoryId());
        hashCode = 31 * hashCode + Objects.hashCode(remoteDomainName());
        hashCode = 31 * hashCode + Objects.hashCode(trustPassword());
        hashCode = 31 * hashCode + Objects.hashCode(trustDirectionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(trustTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(conditionalForwarderIpAddrs());
        hashCode = 31 * hashCode + Objects.hashCode(selectiveAuthAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateTrustRequest)) {
            return false;
        }
        CreateTrustRequest other = (CreateTrustRequest) obj;
        return Objects.equals(directoryId(), other.directoryId()) && Objects.equals(remoteDomainName(), other.remoteDomainName())
                && Objects.equals(trustPassword(), other.trustPassword())
                && Objects.equals(trustDirectionAsString(), other.trustDirectionAsString())
                && Objects.equals(trustTypeAsString(), other.trustTypeAsString())
                && Objects.equals(conditionalForwarderIpAddrs(), other.conditionalForwarderIpAddrs())
                && Objects.equals(selectiveAuthAsString(), other.selectiveAuthAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateTrustRequest").add("DirectoryId", directoryId())
                .add("RemoteDomainName", remoteDomainName())
                .add("TrustPassword", trustPassword() == null ? null : "*** Sensitive Data Redacted ***")
                .add("TrustDirection", trustDirectionAsString()).add("TrustType", trustTypeAsString())
                .add("ConditionalForwarderIpAddrs", conditionalForwarderIpAddrs()).add("SelectiveAuth", selectiveAuthAsString())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DirectoryId":
            return Optional.ofNullable(clazz.cast(directoryId()));
        case "RemoteDomainName":
            return Optional.ofNullable(clazz.cast(remoteDomainName()));
        case "TrustPassword":
            return Optional.ofNullable(clazz.cast(trustPassword()));
        case "TrustDirection":
            return Optional.ofNullable(clazz.cast(trustDirectionAsString()));
        case "TrustType":
            return Optional.ofNullable(clazz.cast(trustTypeAsString()));
        case "ConditionalForwarderIpAddrs":
            return Optional.ofNullable(clazz.cast(conditionalForwarderIpAddrs()));
        case "SelectiveAuth":
            return Optional.ofNullable(clazz.cast(selectiveAuthAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateTrustRequest, T> g) {
        return obj -> g.apply((CreateTrustRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DirectoryRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateTrustRequest> {
        /**
         * <p>
         * The Directory ID of the AWS Managed Microsoft AD directory for which to establish the trust relationship.
         * </p>
         * 
         * @param directoryId
         *        The Directory ID of the AWS Managed Microsoft AD directory for which to establish the trust
         *        relationship.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder directoryId(String directoryId);

        /**
         * <p>
         * The Fully Qualified Domain Name (FQDN) of the external domain for which to create the trust relationship.
         * </p>
         * 
         * @param remoteDomainName
         *        The Fully Qualified Domain Name (FQDN) of the external domain for which to create the trust
         *        relationship.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder remoteDomainName(String remoteDomainName);

        /**
         * <p>
         * The trust password. The must be the same password that was used when creating the trust relationship on the
         * external domain.
         * </p>
         * 
         * @param trustPassword
         *        The trust password. The must be the same password that was used when creating the trust relationship
         *        on the external domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trustPassword(String trustPassword);

        /**
         * <p>
         * The direction of the trust relationship.
         * </p>
         * 
         * @param trustDirection
         *        The direction of the trust relationship.
         * @see TrustDirection
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrustDirection
         */
        Builder trustDirection(String trustDirection);

        /**
         * <p>
         * The direction of the trust relationship.
         * </p>
         * 
         * @param trustDirection
         *        The direction of the trust relationship.
         * @see TrustDirection
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrustDirection
         */
        Builder trustDirection(TrustDirection trustDirection);

        /**
         * <p>
         * The trust relationship type. <code>Forest</code> is the default.
         * </p>
         * 
         * @param trustType
         *        The trust relationship type. <code>Forest</code> is the default.
         * @see TrustType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrustType
         */
        Builder trustType(String trustType);

        /**
         * <p>
         * The trust relationship type. <code>Forest</code> is the default.
         * </p>
         * 
         * @param trustType
         *        The trust relationship type. <code>Forest</code> is the default.
         * @see TrustType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrustType
         */
        Builder trustType(TrustType trustType);

        /**
         * <p>
         * The IP addresses of the remote DNS server associated with RemoteDomainName.
         * </p>
         * 
         * @param conditionalForwarderIpAddrs
         *        The IP addresses of the remote DNS server associated with RemoteDomainName.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conditionalForwarderIpAddrs(Collection<String> conditionalForwarderIpAddrs);

        /**
         * <p>
         * The IP addresses of the remote DNS server associated with RemoteDomainName.
         * </p>
         * 
         * @param conditionalForwarderIpAddrs
         *        The IP addresses of the remote DNS server associated with RemoteDomainName.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conditionalForwarderIpAddrs(String... conditionalForwarderIpAddrs);

        /**
         * <p>
         * Optional parameter to enable selective authentication for the trust.
         * </p>
         * 
         * @param selectiveAuth
         *        Optional parameter to enable selective authentication for the trust.
         * @see SelectiveAuth
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SelectiveAuth
         */
        Builder selectiveAuth(String selectiveAuth);

        /**
         * <p>
         * Optional parameter to enable selective authentication for the trust.
         * </p>
         * 
         * @param selectiveAuth
         *        Optional parameter to enable selective authentication for the trust.
         * @see SelectiveAuth
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SelectiveAuth
         */
        Builder selectiveAuth(SelectiveAuth selectiveAuth);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DirectoryRequest.BuilderImpl implements Builder {
        private String directoryId;

        private String remoteDomainName;

        private String trustPassword;

        private String trustDirection;

        private String trustType;

        private List<String> conditionalForwarderIpAddrs = DefaultSdkAutoConstructList.getInstance();

        private String selectiveAuth;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateTrustRequest model) {
            super(model);
            directoryId(model.directoryId);
            remoteDomainName(model.remoteDomainName);
            trustPassword(model.trustPassword);
            trustDirection(model.trustDirection);
            trustType(model.trustType);
            conditionalForwarderIpAddrs(model.conditionalForwarderIpAddrs);
            selectiveAuth(model.selectiveAuth);
        }

        public final String getDirectoryId() {
            return directoryId;
        }

        @Override
        public final Builder directoryId(String directoryId) {
            this.directoryId = directoryId;
            return this;
        }

        public final void setDirectoryId(String directoryId) {
            this.directoryId = directoryId;
        }

        public final String getRemoteDomainName() {
            return remoteDomainName;
        }

        @Override
        public final Builder remoteDomainName(String remoteDomainName) {
            this.remoteDomainName = remoteDomainName;
            return this;
        }

        public final void setRemoteDomainName(String remoteDomainName) {
            this.remoteDomainName = remoteDomainName;
        }

        public final String getTrustPassword() {
            return trustPassword;
        }

        @Override
        public final Builder trustPassword(String trustPassword) {
            this.trustPassword = trustPassword;
            return this;
        }

        public final void setTrustPassword(String trustPassword) {
            this.trustPassword = trustPassword;
        }

        public final String getTrustDirection() {
            return trustDirection;
        }

        @Override
        public final Builder trustDirection(String trustDirection) {
            this.trustDirection = trustDirection;
            return this;
        }

        @Override
        public final Builder trustDirection(TrustDirection trustDirection) {
            this.trustDirection(trustDirection == null ? null : trustDirection.toString());
            return this;
        }

        public final void setTrustDirection(String trustDirection) {
            this.trustDirection = trustDirection;
        }

        public final String getTrustType() {
            return trustType;
        }

        @Override
        public final Builder trustType(String trustType) {
            this.trustType = trustType;
            return this;
        }

        @Override
        public final Builder trustType(TrustType trustType) {
            this.trustType(trustType == null ? null : trustType.toString());
            return this;
        }

        public final void setTrustType(String trustType) {
            this.trustType = trustType;
        }

        public final Collection<String> getConditionalForwarderIpAddrs() {
            return conditionalForwarderIpAddrs;
        }

        @Override
        public final Builder conditionalForwarderIpAddrs(Collection<String> conditionalForwarderIpAddrs) {
            this.conditionalForwarderIpAddrs = DnsIpAddrsCopier.copy(conditionalForwarderIpAddrs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder conditionalForwarderIpAddrs(String... conditionalForwarderIpAddrs) {
            conditionalForwarderIpAddrs(Arrays.asList(conditionalForwarderIpAddrs));
            return this;
        }

        public final void setConditionalForwarderIpAddrs(Collection<String> conditionalForwarderIpAddrs) {
            this.conditionalForwarderIpAddrs = DnsIpAddrsCopier.copy(conditionalForwarderIpAddrs);
        }

        public final String getSelectiveAuth() {
            return selectiveAuth;
        }

        @Override
        public final Builder selectiveAuth(String selectiveAuth) {
            this.selectiveAuth = selectiveAuth;
            return this;
        }

        @Override
        public final Builder selectiveAuth(SelectiveAuth selectiveAuth) {
            this.selectiveAuth(selectiveAuth == null ? null : selectiveAuth.toString());
            return this;
        }

        public final void setSelectiveAuth(String selectiveAuth) {
            this.selectiveAuth = selectiveAuth;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateTrustRequest build() {
            return new CreateTrustRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
