/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directory;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.directory.model.AcceptSharedDirectoryRequest;
import software.amazon.awssdk.services.directory.model.AcceptSharedDirectoryResponse;
import software.amazon.awssdk.services.directory.model.AccessDeniedException;
import software.amazon.awssdk.services.directory.model.AddIpRoutesRequest;
import software.amazon.awssdk.services.directory.model.AddIpRoutesResponse;
import software.amazon.awssdk.services.directory.model.AddRegionRequest;
import software.amazon.awssdk.services.directory.model.AddRegionResponse;
import software.amazon.awssdk.services.directory.model.AddTagsToResourceRequest;
import software.amazon.awssdk.services.directory.model.AddTagsToResourceResponse;
import software.amazon.awssdk.services.directory.model.AuthenticationFailedException;
import software.amazon.awssdk.services.directory.model.CancelSchemaExtensionRequest;
import software.amazon.awssdk.services.directory.model.CancelSchemaExtensionResponse;
import software.amazon.awssdk.services.directory.model.CertificateAlreadyExistsException;
import software.amazon.awssdk.services.directory.model.CertificateDoesNotExistException;
import software.amazon.awssdk.services.directory.model.CertificateInUseException;
import software.amazon.awssdk.services.directory.model.CertificateLimitExceededException;
import software.amazon.awssdk.services.directory.model.ClientException;
import software.amazon.awssdk.services.directory.model.ConnectDirectoryRequest;
import software.amazon.awssdk.services.directory.model.ConnectDirectoryResponse;
import software.amazon.awssdk.services.directory.model.CreateAliasRequest;
import software.amazon.awssdk.services.directory.model.CreateAliasResponse;
import software.amazon.awssdk.services.directory.model.CreateComputerRequest;
import software.amazon.awssdk.services.directory.model.CreateComputerResponse;
import software.amazon.awssdk.services.directory.model.CreateConditionalForwarderRequest;
import software.amazon.awssdk.services.directory.model.CreateConditionalForwarderResponse;
import software.amazon.awssdk.services.directory.model.CreateDirectoryRequest;
import software.amazon.awssdk.services.directory.model.CreateDirectoryResponse;
import software.amazon.awssdk.services.directory.model.CreateLogSubscriptionRequest;
import software.amazon.awssdk.services.directory.model.CreateLogSubscriptionResponse;
import software.amazon.awssdk.services.directory.model.CreateMicrosoftAdRequest;
import software.amazon.awssdk.services.directory.model.CreateMicrosoftAdResponse;
import software.amazon.awssdk.services.directory.model.CreateSnapshotRequest;
import software.amazon.awssdk.services.directory.model.CreateSnapshotResponse;
import software.amazon.awssdk.services.directory.model.CreateTrustRequest;
import software.amazon.awssdk.services.directory.model.CreateTrustResponse;
import software.amazon.awssdk.services.directory.model.DeleteConditionalForwarderRequest;
import software.amazon.awssdk.services.directory.model.DeleteConditionalForwarderResponse;
import software.amazon.awssdk.services.directory.model.DeleteDirectoryRequest;
import software.amazon.awssdk.services.directory.model.DeleteDirectoryResponse;
import software.amazon.awssdk.services.directory.model.DeleteLogSubscriptionRequest;
import software.amazon.awssdk.services.directory.model.DeleteLogSubscriptionResponse;
import software.amazon.awssdk.services.directory.model.DeleteSnapshotRequest;
import software.amazon.awssdk.services.directory.model.DeleteSnapshotResponse;
import software.amazon.awssdk.services.directory.model.DeleteTrustRequest;
import software.amazon.awssdk.services.directory.model.DeleteTrustResponse;
import software.amazon.awssdk.services.directory.model.DeregisterCertificateRequest;
import software.amazon.awssdk.services.directory.model.DeregisterCertificateResponse;
import software.amazon.awssdk.services.directory.model.DeregisterEventTopicRequest;
import software.amazon.awssdk.services.directory.model.DeregisterEventTopicResponse;
import software.amazon.awssdk.services.directory.model.DescribeCertificateRequest;
import software.amazon.awssdk.services.directory.model.DescribeCertificateResponse;
import software.amazon.awssdk.services.directory.model.DescribeConditionalForwardersRequest;
import software.amazon.awssdk.services.directory.model.DescribeConditionalForwardersResponse;
import software.amazon.awssdk.services.directory.model.DescribeDirectoriesRequest;
import software.amazon.awssdk.services.directory.model.DescribeDirectoriesResponse;
import software.amazon.awssdk.services.directory.model.DescribeDomainControllersRequest;
import software.amazon.awssdk.services.directory.model.DescribeDomainControllersResponse;
import software.amazon.awssdk.services.directory.model.DescribeEventTopicsRequest;
import software.amazon.awssdk.services.directory.model.DescribeEventTopicsResponse;
import software.amazon.awssdk.services.directory.model.DescribeLdapsSettingsRequest;
import software.amazon.awssdk.services.directory.model.DescribeLdapsSettingsResponse;
import software.amazon.awssdk.services.directory.model.DescribeRegionsRequest;
import software.amazon.awssdk.services.directory.model.DescribeRegionsResponse;
import software.amazon.awssdk.services.directory.model.DescribeSharedDirectoriesRequest;
import software.amazon.awssdk.services.directory.model.DescribeSharedDirectoriesResponse;
import software.amazon.awssdk.services.directory.model.DescribeSnapshotsRequest;
import software.amazon.awssdk.services.directory.model.DescribeSnapshotsResponse;
import software.amazon.awssdk.services.directory.model.DescribeTrustsRequest;
import software.amazon.awssdk.services.directory.model.DescribeTrustsResponse;
import software.amazon.awssdk.services.directory.model.DirectoryAlreadyInRegionException;
import software.amazon.awssdk.services.directory.model.DirectoryAlreadySharedException;
import software.amazon.awssdk.services.directory.model.DirectoryDoesNotExistException;
import software.amazon.awssdk.services.directory.model.DirectoryException;
import software.amazon.awssdk.services.directory.model.DirectoryLimitExceededException;
import software.amazon.awssdk.services.directory.model.DirectoryNotSharedException;
import software.amazon.awssdk.services.directory.model.DirectoryRequest;
import software.amazon.awssdk.services.directory.model.DirectoryUnavailableException;
import software.amazon.awssdk.services.directory.model.DisableClientAuthenticationRequest;
import software.amazon.awssdk.services.directory.model.DisableClientAuthenticationResponse;
import software.amazon.awssdk.services.directory.model.DisableLdapsRequest;
import software.amazon.awssdk.services.directory.model.DisableLdapsResponse;
import software.amazon.awssdk.services.directory.model.DisableRadiusRequest;
import software.amazon.awssdk.services.directory.model.DisableRadiusResponse;
import software.amazon.awssdk.services.directory.model.DisableSsoRequest;
import software.amazon.awssdk.services.directory.model.DisableSsoResponse;
import software.amazon.awssdk.services.directory.model.DomainControllerLimitExceededException;
import software.amazon.awssdk.services.directory.model.EnableClientAuthenticationRequest;
import software.amazon.awssdk.services.directory.model.EnableClientAuthenticationResponse;
import software.amazon.awssdk.services.directory.model.EnableLdapsRequest;
import software.amazon.awssdk.services.directory.model.EnableLdapsResponse;
import software.amazon.awssdk.services.directory.model.EnableRadiusRequest;
import software.amazon.awssdk.services.directory.model.EnableRadiusResponse;
import software.amazon.awssdk.services.directory.model.EnableSsoRequest;
import software.amazon.awssdk.services.directory.model.EnableSsoResponse;
import software.amazon.awssdk.services.directory.model.EntityAlreadyExistsException;
import software.amazon.awssdk.services.directory.model.EntityDoesNotExistException;
import software.amazon.awssdk.services.directory.model.GetDirectoryLimitsRequest;
import software.amazon.awssdk.services.directory.model.GetDirectoryLimitsResponse;
import software.amazon.awssdk.services.directory.model.GetSnapshotLimitsRequest;
import software.amazon.awssdk.services.directory.model.GetSnapshotLimitsResponse;
import software.amazon.awssdk.services.directory.model.InsufficientPermissionsException;
import software.amazon.awssdk.services.directory.model.InvalidCertificateException;
import software.amazon.awssdk.services.directory.model.InvalidClientAuthStatusException;
import software.amazon.awssdk.services.directory.model.InvalidLdapsStatusException;
import software.amazon.awssdk.services.directory.model.InvalidNextTokenException;
import software.amazon.awssdk.services.directory.model.InvalidParameterException;
import software.amazon.awssdk.services.directory.model.InvalidPasswordException;
import software.amazon.awssdk.services.directory.model.InvalidTargetException;
import software.amazon.awssdk.services.directory.model.IpRouteLimitExceededException;
import software.amazon.awssdk.services.directory.model.ListCertificatesRequest;
import software.amazon.awssdk.services.directory.model.ListCertificatesResponse;
import software.amazon.awssdk.services.directory.model.ListIpRoutesRequest;
import software.amazon.awssdk.services.directory.model.ListIpRoutesResponse;
import software.amazon.awssdk.services.directory.model.ListLogSubscriptionsRequest;
import software.amazon.awssdk.services.directory.model.ListLogSubscriptionsResponse;
import software.amazon.awssdk.services.directory.model.ListSchemaExtensionsRequest;
import software.amazon.awssdk.services.directory.model.ListSchemaExtensionsResponse;
import software.amazon.awssdk.services.directory.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.directory.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.directory.model.NoAvailableCertificateException;
import software.amazon.awssdk.services.directory.model.OrganizationsException;
import software.amazon.awssdk.services.directory.model.RegionLimitExceededException;
import software.amazon.awssdk.services.directory.model.RegisterCertificateRequest;
import software.amazon.awssdk.services.directory.model.RegisterCertificateResponse;
import software.amazon.awssdk.services.directory.model.RegisterEventTopicRequest;
import software.amazon.awssdk.services.directory.model.RegisterEventTopicResponse;
import software.amazon.awssdk.services.directory.model.RejectSharedDirectoryRequest;
import software.amazon.awssdk.services.directory.model.RejectSharedDirectoryResponse;
import software.amazon.awssdk.services.directory.model.RemoveIpRoutesRequest;
import software.amazon.awssdk.services.directory.model.RemoveIpRoutesResponse;
import software.amazon.awssdk.services.directory.model.RemoveRegionRequest;
import software.amazon.awssdk.services.directory.model.RemoveRegionResponse;
import software.amazon.awssdk.services.directory.model.RemoveTagsFromResourceRequest;
import software.amazon.awssdk.services.directory.model.RemoveTagsFromResourceResponse;
import software.amazon.awssdk.services.directory.model.ResetUserPasswordRequest;
import software.amazon.awssdk.services.directory.model.ResetUserPasswordResponse;
import software.amazon.awssdk.services.directory.model.RestoreFromSnapshotRequest;
import software.amazon.awssdk.services.directory.model.RestoreFromSnapshotResponse;
import software.amazon.awssdk.services.directory.model.ServiceException;
import software.amazon.awssdk.services.directory.model.ShareDirectoryRequest;
import software.amazon.awssdk.services.directory.model.ShareDirectoryResponse;
import software.amazon.awssdk.services.directory.model.ShareLimitExceededException;
import software.amazon.awssdk.services.directory.model.SnapshotLimitExceededException;
import software.amazon.awssdk.services.directory.model.StartSchemaExtensionRequest;
import software.amazon.awssdk.services.directory.model.StartSchemaExtensionResponse;
import software.amazon.awssdk.services.directory.model.TagLimitExceededException;
import software.amazon.awssdk.services.directory.model.UnshareDirectoryRequest;
import software.amazon.awssdk.services.directory.model.UnshareDirectoryResponse;
import software.amazon.awssdk.services.directory.model.UnsupportedOperationException;
import software.amazon.awssdk.services.directory.model.UpdateConditionalForwarderRequest;
import software.amazon.awssdk.services.directory.model.UpdateConditionalForwarderResponse;
import software.amazon.awssdk.services.directory.model.UpdateNumberOfDomainControllersRequest;
import software.amazon.awssdk.services.directory.model.UpdateNumberOfDomainControllersResponse;
import software.amazon.awssdk.services.directory.model.UpdateRadiusRequest;
import software.amazon.awssdk.services.directory.model.UpdateRadiusResponse;
import software.amazon.awssdk.services.directory.model.UpdateTrustRequest;
import software.amazon.awssdk.services.directory.model.UpdateTrustResponse;
import software.amazon.awssdk.services.directory.model.UserDoesNotExistException;
import software.amazon.awssdk.services.directory.model.VerifyTrustRequest;
import software.amazon.awssdk.services.directory.model.VerifyTrustResponse;
import software.amazon.awssdk.services.directory.paginators.DescribeDomainControllersPublisher;
import software.amazon.awssdk.services.directory.transform.AcceptSharedDirectoryRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.AddIpRoutesRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.AddRegionRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.AddTagsToResourceRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.CancelSchemaExtensionRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.ConnectDirectoryRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.CreateAliasRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.CreateComputerRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.CreateConditionalForwarderRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.CreateDirectoryRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.CreateLogSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.CreateMicrosoftAdRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.CreateSnapshotRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.CreateTrustRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DeleteConditionalForwarderRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DeleteDirectoryRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DeleteLogSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DeleteSnapshotRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DeleteTrustRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DeregisterCertificateRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DeregisterEventTopicRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DescribeCertificateRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DescribeConditionalForwardersRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DescribeDirectoriesRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DescribeDomainControllersRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DescribeEventTopicsRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DescribeLdapsSettingsRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DescribeRegionsRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DescribeSharedDirectoriesRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DescribeSnapshotsRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DescribeTrustsRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DisableClientAuthenticationRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DisableLdapsRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DisableRadiusRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.DisableSsoRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.EnableClientAuthenticationRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.EnableLdapsRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.EnableRadiusRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.EnableSsoRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.GetDirectoryLimitsRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.GetSnapshotLimitsRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.ListCertificatesRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.ListIpRoutesRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.ListLogSubscriptionsRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.ListSchemaExtensionsRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.RegisterCertificateRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.RegisterEventTopicRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.RejectSharedDirectoryRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.RemoveIpRoutesRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.RemoveRegionRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.RemoveTagsFromResourceRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.ResetUserPasswordRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.RestoreFromSnapshotRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.ShareDirectoryRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.StartSchemaExtensionRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.UnshareDirectoryRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.UpdateConditionalForwarderRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.UpdateNumberOfDomainControllersRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.UpdateRadiusRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.UpdateTrustRequestMarshaller;
import software.amazon.awssdk.services.directory.transform.VerifyTrustRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link DirectoryAsyncClient}.
 *
 * @see DirectoryAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDirectoryAsyncClient implements DirectoryAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultDirectoryAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultDirectoryAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Accepts a directory sharing request that was sent from the directory owner account.
     * </p>
     *
     * @param acceptSharedDirectoryRequest
     * @return A Java Future containing the result of the AcceptSharedDirectory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>DirectoryAlreadySharedException The specified directory has already been shared with this AWS
     *         account.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.AcceptSharedDirectory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/AcceptSharedDirectory" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptSharedDirectoryResponse> acceptSharedDirectory(
            AcceptSharedDirectoryRequest acceptSharedDirectoryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptSharedDirectoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptSharedDirectory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptSharedDirectoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AcceptSharedDirectoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptSharedDirectoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptSharedDirectoryRequest, AcceptSharedDirectoryResponse>()
                            .withOperationName("AcceptSharedDirectory")
                            .withMarshaller(new AcceptSharedDirectoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(acceptSharedDirectoryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = acceptSharedDirectoryRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<AcceptSharedDirectoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * If the DNS server for your on-premises domain uses a publicly addressable IP address, you must add a CIDR address
     * block to correctly route traffic to and from your Microsoft AD on Amazon Web Services. <i>AddIpRoutes</i> adds
     * this address block. You can also use <i>AddIpRoutes</i> to facilitate routing traffic that uses public IP ranges
     * from your Microsoft AD on AWS to a peer VPC.
     * </p>
     * <p>
     * Before you call <i>AddIpRoutes</i>, ensure that all of the required permissions have been explicitly granted
     * through a policy. For details about what permissions are required to run the <i>AddIpRoutes</i> operation, see <a
     * href
     * ="http://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html">AWS
     * Directory Service API Permissions: Actions, Resources, and Conditions Reference</a>.
     * </p>
     *
     * @param addIpRoutesRequest
     * @return A Java Future containing the result of the AddIpRoutes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>EntityAlreadyExistsException The specified entity already exists.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DirectoryUnavailableException The specified directory is unavailable or could not be found.</li>
     *         <li>IpRouteLimitExceededException The maximum allowed number of IP addresses was exceeded. The default
     *         limit is 100 IP address blocks.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.AddIpRoutes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/AddIpRoutes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AddIpRoutesResponse> addIpRoutes(AddIpRoutesRequest addIpRoutesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addIpRoutesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddIpRoutes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddIpRoutesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AddIpRoutesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddIpRoutesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddIpRoutesRequest, AddIpRoutesResponse>()
                            .withOperationName("AddIpRoutes").withMarshaller(new AddIpRoutesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addIpRoutesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = addIpRoutesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<AddIpRoutesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds two domain controllers in the specified Region for the specified directory.
     * </p>
     *
     * @param addRegionRequest
     * @return A Java Future containing the result of the AddRegion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryUnavailableException The specified directory is unavailable or could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>DirectoryAlreadyInRegionException The Region you specified is the same Region where the AWS Managed
     *         Microsoft AD directory was created. Specify a different Region and try again.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>DirectoryDoesNotExistException The specified directory does not exist in the system.</li>
     *         <li>RegionLimitExceededException You have reached the limit for maximum number of simultaneous Region
     *         replications per directory.</li>
     *         <li>AccessDeniedException Client authentication is not available in this region at this time.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.AddRegion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/AddRegion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AddRegionResponse> addRegion(AddRegionRequest addRegionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addRegionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddRegion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddRegionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AddRegionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddRegionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddRegionRequest, AddRegionResponse>().withOperationName("AddRegion")
                            .withMarshaller(new AddRegionRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(addRegionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = addRegionRequest.overrideConfiguration().orElse(null);
            CompletableFuture<AddRegionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or overwrites one or more tags for the specified directory. Each directory can have a maximum of 50 tags.
     * Each tag consists of a key and optional value. Tag keys must be unique to each resource.
     * </p>
     *
     * @param addTagsToResourceRequest
     * @return A Java Future containing the result of the AddTagsToResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>TagLimitExceededException The maximum allowed number of tags was exceeded.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.AddTagsToResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/AddTagsToResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AddTagsToResourceResponse> addTagsToResource(AddTagsToResourceRequest addTagsToResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addTagsToResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddTagsToResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddTagsToResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddTagsToResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddTagsToResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddTagsToResourceRequest, AddTagsToResourceResponse>()
                            .withOperationName("AddTagsToResource")
                            .withMarshaller(new AddTagsToResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addTagsToResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = addTagsToResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<AddTagsToResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels an in-progress schema extension to a Microsoft AD directory. Once a schema extension has started
     * replicating to all domain controllers, the task can no longer be canceled. A schema extension can be canceled
     * during any of the following states; <code>Initializing</code>, <code>CreatingSnapshot</code>, and
     * <code>UpdatingSchema</code>.
     * </p>
     *
     * @param cancelSchemaExtensionRequest
     * @return A Java Future containing the result of the CancelSchemaExtension operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.CancelSchemaExtension
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/CancelSchemaExtension" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelSchemaExtensionResponse> cancelSchemaExtension(
            CancelSchemaExtensionRequest cancelSchemaExtensionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelSchemaExtensionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelSchemaExtension");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelSchemaExtensionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelSchemaExtensionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelSchemaExtensionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelSchemaExtensionRequest, CancelSchemaExtensionResponse>()
                            .withOperationName("CancelSchemaExtension")
                            .withMarshaller(new CancelSchemaExtensionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelSchemaExtensionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = cancelSchemaExtensionRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CancelSchemaExtensionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an AD Connector to connect to an on-premises directory.
     * </p>
     * <p>
     * Before you call <code>ConnectDirectory</code>, ensure that all of the required permissions have been explicitly
     * granted through a policy. For details about what permissions are required to run the
     * <code>ConnectDirectory</code> operation, see <a
     * href="http://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html"
     * >AWS Directory Service API Permissions: Actions, Resources, and Conditions Reference</a>.
     * </p>
     *
     * @param connectDirectoryRequest
     *        Contains the inputs for the <a>ConnectDirectory</a> operation.
     * @return A Java Future containing the result of the ConnectDirectory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryLimitExceededException The maximum number of directories in the region has been reached. You
     *         can use the <a>GetDirectoryLimits</a> operation to determine your directory limits in the region.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.ConnectDirectory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/ConnectDirectory" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ConnectDirectoryResponse> connectDirectory(ConnectDirectoryRequest connectDirectoryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, connectDirectoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ConnectDirectory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ConnectDirectoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ConnectDirectoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ConnectDirectoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ConnectDirectoryRequest, ConnectDirectoryResponse>()
                            .withOperationName("ConnectDirectory")
                            .withMarshaller(new ConnectDirectoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(connectDirectoryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = connectDirectoryRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ConnectDirectoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an alias for a directory and assigns the alias to the directory. The alias is used to construct the
     * access URL for the directory, such as <code>http://&lt;alias&gt;.awsapps.com</code>.
     * </p>
     * <important>
     * <p>
     * After an alias has been created, it cannot be deleted or reused, so this operation should only be used when
     * absolutely necessary.
     * </p>
     * </important>
     *
     * @param createAliasRequest
     *        Contains the inputs for the <a>CreateAlias</a> operation.
     * @return A Java Future containing the result of the CreateAlias operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityAlreadyExistsException The specified entity already exists.</li>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.CreateAlias
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/CreateAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAliasResponse> createAlias(CreateAliasRequest createAliasRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAlias");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateAliasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAliasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAliasRequest, CreateAliasResponse>()
                            .withOperationName("CreateAlias").withMarshaller(new CreateAliasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAliasRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createAliasRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateAliasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Active Directory computer object in the specified directory.
     * </p>
     *
     * @param createComputerRequest
     *        Contains the inputs for the <a>CreateComputer</a> operation.
     * @return A Java Future containing the result of the CreateComputer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AuthenticationFailedException An authentication error occurred.</li>
     *         <li>DirectoryUnavailableException The specified directory is unavailable or could not be found.</li>
     *         <li>EntityAlreadyExistsException The specified entity already exists.</li>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.CreateComputer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/CreateComputer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateComputerResponse> createComputer(CreateComputerRequest createComputerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createComputerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateComputer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateComputerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateComputerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateComputerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateComputerRequest, CreateComputerResponse>()
                            .withOperationName("CreateComputer")
                            .withMarshaller(new CreateComputerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createComputerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createComputerRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateComputerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a conditional forwarder associated with your AWS directory. Conditional forwarders are required in order
     * to set up a trust relationship with another domain. The conditional forwarder points to the trusted domain.
     * </p>
     *
     * @param createConditionalForwarderRequest
     *        Initiates the creation of a conditional forwarder for your AWS Directory Service for Microsoft Active
     *        Directory. Conditional forwarders are required in order to set up a trust relationship with another
     *        domain.
     * @return A Java Future containing the result of the CreateConditionalForwarder operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityAlreadyExistsException The specified entity already exists.</li>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>DirectoryUnavailableException The specified directory is unavailable or could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.CreateConditionalForwarder
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/CreateConditionalForwarder" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConditionalForwarderResponse> createConditionalForwarder(
            CreateConditionalForwarderRequest createConditionalForwarderRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConditionalForwarderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConditionalForwarder");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConditionalForwarderResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateConditionalForwarderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConditionalForwarderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConditionalForwarderRequest, CreateConditionalForwarderResponse>()
                            .withOperationName("CreateConditionalForwarder")
                            .withMarshaller(new CreateConditionalForwarderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createConditionalForwarderRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createConditionalForwarderRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateConditionalForwarderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a Simple AD directory. For more information, see <a
     * href="https://docs.aws.amazon.com/directoryservice/latest/admin-guide/directory_simple_ad.html">Simple Active
     * Directory</a> in the <i>AWS Directory Service Admin Guide</i>.
     * </p>
     * <p>
     * Before you call <code>CreateDirectory</code>, ensure that all of the required permissions have been explicitly
     * granted through a policy. For details about what permissions are required to run the <code>CreateDirectory</code>
     * operation, see <a
     * href="http://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html"
     * >AWS Directory Service API Permissions: Actions, Resources, and Conditions Reference</a>.
     * </p>
     *
     * @param createDirectoryRequest
     *        Contains the inputs for the <a>CreateDirectory</a> operation.
     * @return A Java Future containing the result of the CreateDirectory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryLimitExceededException The maximum number of directories in the region has been reached. You
     *         can use the <a>GetDirectoryLimits</a> operation to determine your directory limits in the region.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.CreateDirectory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/CreateDirectory" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDirectoryResponse> createDirectory(CreateDirectoryRequest createDirectoryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDirectoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDirectory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDirectoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDirectoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDirectoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDirectoryRequest, CreateDirectoryResponse>()
                            .withOperationName("CreateDirectory")
                            .withMarshaller(new CreateDirectoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDirectoryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createDirectoryRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateDirectoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a subscription to forward real-time Directory Service domain controller security logs to the specified
     * Amazon CloudWatch log group in your AWS account.
     * </p>
     *
     * @param createLogSubscriptionRequest
     * @return A Java Future containing the result of the CreateLogSubscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityAlreadyExistsException The specified entity already exists.</li>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>InsufficientPermissionsException The account does not have sufficient permission to perform the
     *         operation.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.CreateLogSubscription
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/CreateLogSubscription" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLogSubscriptionResponse> createLogSubscription(
            CreateLogSubscriptionRequest createLogSubscriptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLogSubscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLogSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLogSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLogSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLogSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLogSubscriptionRequest, CreateLogSubscriptionResponse>()
                            .withOperationName("CreateLogSubscription")
                            .withMarshaller(new CreateLogSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLogSubscriptionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createLogSubscriptionRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreateLogSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a Microsoft AD directory in the AWS Cloud. For more information, see <a
     * href="https://docs.aws.amazon.com/directoryservice/latest/admin-guide/directory_microsoft_ad.html">AWS Managed
     * Microsoft AD</a> in the <i>AWS Directory Service Admin Guide</i>.
     * </p>
     * <p>
     * Before you call <i>CreateMicrosoftAD</i>, ensure that all of the required permissions have been explicitly
     * granted through a policy. For details about what permissions are required to run the <i>CreateMicrosoftAD</i>
     * operation, see <a
     * href="http://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html"
     * >AWS Directory Service API Permissions: Actions, Resources, and Conditions Reference</a>.
     * </p>
     *
     * @param createMicrosoftAdRequest
     *        Creates an AWS Managed Microsoft AD directory.
     * @return A Java Future containing the result of the CreateMicrosoftAD operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryLimitExceededException The maximum number of directories in the region has been reached. You
     *         can use the <a>GetDirectoryLimits</a> operation to determine your directory limits in the region.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.CreateMicrosoftAD
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/CreateMicrosoftAD" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMicrosoftAdResponse> createMicrosoftAD(CreateMicrosoftAdRequest createMicrosoftAdRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMicrosoftAdRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMicrosoftAD");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMicrosoftAdResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateMicrosoftAdResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMicrosoftAdResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMicrosoftAdRequest, CreateMicrosoftAdResponse>()
                            .withOperationName("CreateMicrosoftAD")
                            .withMarshaller(new CreateMicrosoftAdRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createMicrosoftAdRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createMicrosoftAdRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateMicrosoftAdResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a snapshot of a Simple AD or Microsoft AD directory in the AWS cloud.
     * </p>
     * <note>
     * <p>
     * You cannot take snapshots of AD Connector directories.
     * </p>
     * </note>
     *
     * @param createSnapshotRequest
     *        Contains the inputs for the <a>CreateSnapshot</a> operation.
     * @return A Java Future containing the result of the CreateSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>SnapshotLimitExceededException The maximum number of manual snapshots for the directory has been
     *         reached. You can use the <a>GetSnapshotLimits</a> operation to determine the snapshot limits for a
     *         directory.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.CreateSnapshot
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/CreateSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSnapshotResponse> createSnapshot(CreateSnapshotRequest createSnapshotRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSnapshotRequest, CreateSnapshotResponse>()
                            .withOperationName("CreateSnapshot")
                            .withMarshaller(new CreateSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSnapshotRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createSnapshotRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * AWS Directory Service for Microsoft Active Directory allows you to configure trust relationships. For example,
     * you can establish a trust between your AWS Managed Microsoft AD directory, and your existing on-premises
     * Microsoft Active Directory. This would allow you to provide users and groups access to resources in either
     * domain, with a single set of credentials.
     * </p>
     * <p>
     * This action initiates the creation of the AWS side of a trust relationship between an AWS Managed Microsoft AD
     * directory and an external domain. You can create either a forest trust or an external trust.
     * </p>
     *
     * @param createTrustRequest
     *        AWS Directory Service for Microsoft Active Directory allows you to configure trust relationships. For
     *        example, you can establish a trust between your AWS Managed Microsoft AD directory, and your existing
     *        on-premises Microsoft Active Directory. This would allow you to provide users and groups access to
     *        resources in either domain, with a single set of credentials.</p>
     *        <p>
     *        This action initiates the creation of the AWS side of a trust relationship between an AWS Managed
     *        Microsoft AD directory and an external domain.
     * @return A Java Future containing the result of the CreateTrust operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityAlreadyExistsException The specified entity already exists.</li> <li>
     *         EntityDoesNotExistException The specified entity could not be found.</li> <li>InvalidParameterException
     *         One or more parameters are not valid.</li> <li>ClientException A client exception has occurred.</li> <li>
     *         ServiceException An exception has occurred in AWS Directory Service.</li> <li>
     *         UnsupportedOperationException The operation is not supported.</li> <li>SdkException Base class for all
     *         exceptions that can be thrown by the SDK (both service and client). Can be used for catch all scenarios.
     *         </li> <li>SdkClientException If any client side error occurs such as an IO related failure, failure to
     *         get credentials, etc.</li> <li>DirectoryException Base class for all service exceptions. Unknown
     *         exceptions will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.CreateTrust
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/CreateTrust" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTrustResponse> createTrust(CreateTrustRequest createTrustRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTrustRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTrust");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTrustResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTrustResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTrustResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTrustRequest, CreateTrustResponse>()
                            .withOperationName("CreateTrust").withMarshaller(new CreateTrustRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createTrustRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createTrustRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateTrustResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a conditional forwarder that has been set up for your AWS directory.
     * </p>
     *
     * @param deleteConditionalForwarderRequest
     *        Deletes a conditional forwarder.
     * @return A Java Future containing the result of the DeleteConditionalForwarder operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>DirectoryUnavailableException The specified directory is unavailable or could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DeleteConditionalForwarder
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DeleteConditionalForwarder" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConditionalForwarderResponse> deleteConditionalForwarder(
            DeleteConditionalForwarderRequest deleteConditionalForwarderRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConditionalForwarderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConditionalForwarder");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConditionalForwarderResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteConditionalForwarderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConditionalForwarderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConditionalForwarderRequest, DeleteConditionalForwarderResponse>()
                            .withOperationName("DeleteConditionalForwarder")
                            .withMarshaller(new DeleteConditionalForwarderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteConditionalForwarderRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteConditionalForwarderRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteConditionalForwarderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an AWS Directory Service directory.
     * </p>
     * <p>
     * Before you call <code>DeleteDirectory</code>, ensure that all of the required permissions have been explicitly
     * granted through a policy. For details about what permissions are required to run the <code>DeleteDirectory</code>
     * operation, see <a
     * href="http://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html"
     * >AWS Directory Service API Permissions: Actions, Resources, and Conditions Reference</a>.
     * </p>
     *
     * @param deleteDirectoryRequest
     *        Contains the inputs for the <a>DeleteDirectory</a> operation.
     * @return A Java Future containing the result of the DeleteDirectory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DeleteDirectory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DeleteDirectory" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDirectoryResponse> deleteDirectory(DeleteDirectoryRequest deleteDirectoryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDirectoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDirectory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDirectoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDirectoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDirectoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDirectoryRequest, DeleteDirectoryResponse>()
                            .withOperationName("DeleteDirectory")
                            .withMarshaller(new DeleteDirectoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDirectoryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteDirectoryRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteDirectoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified log subscription.
     * </p>
     *
     * @param deleteLogSubscriptionRequest
     * @return A Java Future containing the result of the DeleteLogSubscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DeleteLogSubscription
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DeleteLogSubscription" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLogSubscriptionResponse> deleteLogSubscription(
            DeleteLogSubscriptionRequest deleteLogSubscriptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLogSubscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLogSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLogSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLogSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLogSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLogSubscriptionRequest, DeleteLogSubscriptionResponse>()
                            .withOperationName("DeleteLogSubscription")
                            .withMarshaller(new DeleteLogSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteLogSubscriptionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteLogSubscriptionRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteLogSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a directory snapshot.
     * </p>
     *
     * @param deleteSnapshotRequest
     *        Contains the inputs for the <a>DeleteSnapshot</a> operation.
     * @return A Java Future containing the result of the DeleteSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DeleteSnapshot
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DeleteSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSnapshotResponse> deleteSnapshot(DeleteSnapshotRequest deleteSnapshotRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSnapshotRequest, DeleteSnapshotResponse>()
                            .withOperationName("DeleteSnapshot")
                            .withMarshaller(new DeleteSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSnapshotRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteSnapshotRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing trust relationship between your AWS Managed Microsoft AD directory and an external domain.
     * </p>
     *
     * @param deleteTrustRequest
     *        Deletes the local side of an existing trust relationship between the AWS Managed Microsoft AD directory
     *        and the external domain.
     * @return A Java Future containing the result of the DeleteTrust operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DeleteTrust
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DeleteTrust" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTrustResponse> deleteTrust(DeleteTrustRequest deleteTrustRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTrustRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTrust");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTrustResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTrustResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTrustResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTrustRequest, DeleteTrustResponse>()
                            .withOperationName("DeleteTrust").withMarshaller(new DeleteTrustRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteTrustRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteTrustRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteTrustResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes from the system the certificate that was registered for secure LDAP or client certificate authentication.
     * </p>
     *
     * @param deregisterCertificateRequest
     * @return A Java Future containing the result of the DeregisterCertificate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryUnavailableException The specified directory is unavailable or could not be found.</li>
     *         <li>DirectoryDoesNotExistException The specified directory does not exist in the system.</li>
     *         <li>CertificateDoesNotExistException The certificate is not present in the system for describe or
     *         deregister activities.</li>
     *         <li>CertificateInUseException The certificate is being used for the LDAP security connection and cannot
     *         be removed without disabling LDAP security.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DeregisterCertificate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DeregisterCertificate" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterCertificateResponse> deregisterCertificate(
            DeregisterCertificateRequest deregisterCertificateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterCertificateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterCertificate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterCertificateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterCertificateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterCertificateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterCertificateRequest, DeregisterCertificateResponse>()
                            .withOperationName("DeregisterCertificate")
                            .withMarshaller(new DeregisterCertificateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deregisterCertificateRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deregisterCertificateRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeregisterCertificateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified directory as a publisher to the specified SNS topic.
     * </p>
     *
     * @param deregisterEventTopicRequest
     *        Removes the specified directory as a publisher to the specified SNS topic.
     * @return A Java Future containing the result of the DeregisterEventTopic operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DeregisterEventTopic
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DeregisterEventTopic" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterEventTopicResponse> deregisterEventTopic(
            DeregisterEventTopicRequest deregisterEventTopicRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterEventTopicRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterEventTopic");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterEventTopicResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterEventTopicResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterEventTopicResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterEventTopicRequest, DeregisterEventTopicResponse>()
                            .withOperationName("DeregisterEventTopic")
                            .withMarshaller(new DeregisterEventTopicRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deregisterEventTopicRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deregisterEventTopicRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeregisterEventTopicResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays information about the certificate registered for secure LDAP or client certificate authentication.
     * </p>
     *
     * @param describeCertificateRequest
     * @return A Java Future containing the result of the DescribeCertificate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryDoesNotExistException The specified directory does not exist in the system.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>CertificateDoesNotExistException The certificate is not present in the system for describe or
     *         deregister activities.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DescribeCertificate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DescribeCertificate" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeCertificateResponse> describeCertificate(
            DescribeCertificateRequest describeCertificateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeCertificateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCertificate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeCertificateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeCertificateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeCertificateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeCertificateRequest, DescribeCertificateResponse>()
                            .withOperationName("DescribeCertificate")
                            .withMarshaller(new DescribeCertificateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeCertificateRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeCertificateRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeCertificateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Obtains information about the conditional forwarders for this account.
     * </p>
     * <p>
     * If no input parameters are provided for RemoteDomainNames, this request describes all conditional forwarders for
     * the specified directory ID.
     * </p>
     *
     * @param describeConditionalForwardersRequest
     *        Describes a conditional forwarder.
     * @return A Java Future containing the result of the DescribeConditionalForwarders operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>DirectoryUnavailableException The specified directory is unavailable or could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DescribeConditionalForwarders
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DescribeConditionalForwarders"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeConditionalForwardersResponse> describeConditionalForwarders(
            DescribeConditionalForwardersRequest describeConditionalForwardersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeConditionalForwardersRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConditionalForwarders");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeConditionalForwardersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeConditionalForwardersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeConditionalForwardersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeConditionalForwardersRequest, DescribeConditionalForwardersResponse>()
                            .withOperationName("DescribeConditionalForwarders")
                            .withMarshaller(new DescribeConditionalForwardersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeConditionalForwardersRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeConditionalForwardersRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeConditionalForwardersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Obtains information about the directories that belong to this account.
     * </p>
     * <p>
     * You can retrieve information about specific directories by passing the directory identifiers in the
     * <code>DirectoryIds</code> parameter. Otherwise, all directories that belong to the current account are returned.
     * </p>
     * <p>
     * This operation supports pagination with the use of the <code>NextToken</code> request and response parameters. If
     * more results are available, the <code>DescribeDirectoriesResult.NextToken</code> member contains a token that you
     * pass in the next call to <a>DescribeDirectories</a> to retrieve the next set of items.
     * </p>
     * <p>
     * You can also specify a maximum number of return results with the <code>Limit</code> parameter.
     * </p>
     *
     * @param describeDirectoriesRequest
     *        Contains the inputs for the <a>DescribeDirectories</a> operation.
     * @return A Java Future containing the result of the DescribeDirectories operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DescribeDirectories
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DescribeDirectories" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDirectoriesResponse> describeDirectories(
            DescribeDirectoriesRequest describeDirectoriesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDirectoriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDirectories");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDirectoriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDirectoriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDirectoriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDirectoriesRequest, DescribeDirectoriesResponse>()
                            .withOperationName("DescribeDirectories")
                            .withMarshaller(new DescribeDirectoriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDirectoriesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeDirectoriesRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeDirectoriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about any domain controllers in your directory.
     * </p>
     *
     * @param describeDomainControllersRequest
     * @return A Java Future containing the result of the DescribeDomainControllers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DescribeDomainControllers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DescribeDomainControllers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDomainControllersResponse> describeDomainControllers(
            DescribeDomainControllersRequest describeDomainControllersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDomainControllersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDomainControllers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDomainControllersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDomainControllersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDomainControllersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDomainControllersRequest, DescribeDomainControllersResponse>()
                            .withOperationName("DescribeDomainControllers")
                            .withMarshaller(new DescribeDomainControllersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDomainControllersRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeDomainControllersRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeDomainControllersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about any domain controllers in your directory.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeDomainControllers(software.amazon.awssdk.services.directory.model.DescribeDomainControllersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.directory.paginators.DescribeDomainControllersPublisher publisher = client.describeDomainControllersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.directory.paginators.DescribeDomainControllersPublisher publisher = client.describeDomainControllersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.directory.model.DescribeDomainControllersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.directory.model.DescribeDomainControllersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeDomainControllers(software.amazon.awssdk.services.directory.model.DescribeDomainControllersRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeDomainControllersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DescribeDomainControllers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DescribeDomainControllers" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeDomainControllersPublisher describeDomainControllersPaginator(
            DescribeDomainControllersRequest describeDomainControllersRequest) {
        return new DescribeDomainControllersPublisher(this, applyPaginatorUserAgent(describeDomainControllersRequest));
    }

    /**
     * <p>
     * Obtains information about which SNS topics receive status messages from the specified directory.
     * </p>
     * <p>
     * If no input parameters are provided, such as DirectoryId or TopicName, this request describes all of the
     * associations in the account.
     * </p>
     *
     * @param describeEventTopicsRequest
     *        Describes event topics.
     * @return A Java Future containing the result of the DescribeEventTopics operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DescribeEventTopics
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DescribeEventTopics" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEventTopicsResponse> describeEventTopics(
            DescribeEventTopicsRequest describeEventTopicsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEventTopicsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEventTopics");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEventTopicsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEventTopicsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEventTopicsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEventTopicsRequest, DescribeEventTopicsResponse>()
                            .withOperationName("DescribeEventTopics")
                            .withMarshaller(new DescribeEventTopicsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeEventTopicsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeEventTopicsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeEventTopicsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the status of LDAP security for the specified directory.
     * </p>
     *
     * @param describeLdapsSettingsRequest
     * @return A Java Future containing the result of the DescribeLDAPSSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryDoesNotExistException The specified directory does not exist in the system.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DescribeLDAPSSettings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DescribeLDAPSSettings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLdapsSettingsResponse> describeLDAPSSettings(
            DescribeLdapsSettingsRequest describeLdapsSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLdapsSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLDAPSSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLdapsSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLdapsSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLdapsSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLdapsSettingsRequest, DescribeLdapsSettingsResponse>()
                            .withOperationName("DescribeLDAPSSettings")
                            .withMarshaller(new DescribeLdapsSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeLdapsSettingsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeLdapsSettingsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeLdapsSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about the Regions that are configured for multi-Region replication.
     * </p>
     *
     * @param describeRegionsRequest
     * @return A Java Future containing the result of the DescribeRegions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DirectoryDoesNotExistException The specified directory does not exist in the system.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>AccessDeniedException Client authentication is not available in this region at this time.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DescribeRegions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DescribeRegions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRegionsResponse> describeRegions(DescribeRegionsRequest describeRegionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRegionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRegions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRegionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeRegionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRegionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRegionsRequest, DescribeRegionsResponse>()
                            .withOperationName("DescribeRegions")
                            .withMarshaller(new DescribeRegionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeRegionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeRegionsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeRegionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the shared directories in your account.
     * </p>
     *
     * @param describeSharedDirectoriesRequest
     * @return A Java Future containing the result of the DescribeSharedDirectories operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DescribeSharedDirectories
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DescribeSharedDirectories" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSharedDirectoriesResponse> describeSharedDirectories(
            DescribeSharedDirectoriesRequest describeSharedDirectoriesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSharedDirectoriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSharedDirectories");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSharedDirectoriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSharedDirectoriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSharedDirectoriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSharedDirectoriesRequest, DescribeSharedDirectoriesResponse>()
                            .withOperationName("DescribeSharedDirectories")
                            .withMarshaller(new DescribeSharedDirectoriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeSharedDirectoriesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeSharedDirectoriesRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeSharedDirectoriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Obtains information about the directory snapshots that belong to this account.
     * </p>
     * <p>
     * This operation supports pagination with the use of the <i>NextToken</i> request and response parameters. If more
     * results are available, the <i>DescribeSnapshots.NextToken</i> member contains a token that you pass in the next
     * call to <a>DescribeSnapshots</a> to retrieve the next set of items.
     * </p>
     * <p>
     * You can also specify a maximum number of return results with the <i>Limit</i> parameter.
     * </p>
     *
     * @param describeSnapshotsRequest
     *        Contains the inputs for the <a>DescribeSnapshots</a> operation.
     * @return A Java Future containing the result of the DescribeSnapshots operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DescribeSnapshots
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DescribeSnapshots" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSnapshotsResponse> describeSnapshots(DescribeSnapshotsRequest describeSnapshotsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSnapshotsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSnapshots");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSnapshotsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSnapshotsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSnapshotsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSnapshotsRequest, DescribeSnapshotsResponse>()
                            .withOperationName("DescribeSnapshots")
                            .withMarshaller(new DescribeSnapshotsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeSnapshotsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeSnapshotsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeSnapshotsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Obtains information about the trust relationships for this account.
     * </p>
     * <p>
     * If no input parameters are provided, such as DirectoryId or TrustIds, this request describes all the trust
     * relationships belonging to the account.
     * </p>
     *
     * @param describeTrustsRequest
     *        Describes the trust relationships for a particular AWS Managed Microsoft AD directory. If no input
     *        parameters are are provided, such as directory ID or trust ID, this request describes all the trust
     *        relationships.
     * @return A Java Future containing the result of the DescribeTrusts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DescribeTrusts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DescribeTrusts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTrustsResponse> describeTrusts(DescribeTrustsRequest describeTrustsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTrustsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTrusts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTrustsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeTrustsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTrustsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTrustsRequest, DescribeTrustsResponse>()
                            .withOperationName("DescribeTrusts")
                            .withMarshaller(new DescribeTrustsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeTrustsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeTrustsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeTrustsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables alternative client authentication methods for the specified directory.
     * </p>
     *
     * @param disableClientAuthenticationRequest
     * @return A Java Future containing the result of the DisableClientAuthentication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryDoesNotExistException The specified directory does not exist in the system.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>InvalidClientAuthStatusException Client authentication is already enabled.</li>
     *         <li>AccessDeniedException Client authentication is not available in this region at this time.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DisableClientAuthentication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DisableClientAuthentication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableClientAuthenticationResponse> disableClientAuthentication(
            DisableClientAuthenticationRequest disableClientAuthenticationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableClientAuthenticationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableClientAuthentication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableClientAuthenticationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisableClientAuthenticationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableClientAuthenticationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableClientAuthenticationRequest, DisableClientAuthenticationResponse>()
                            .withOperationName("DisableClientAuthentication")
                            .withMarshaller(new DisableClientAuthenticationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disableClientAuthenticationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = disableClientAuthenticationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DisableClientAuthenticationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deactivates LDAP secure calls for the specified directory.
     * </p>
     *
     * @param disableLdapsRequest
     * @return A Java Future containing the result of the DisableLDAPS operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryUnavailableException The specified directory is unavailable or could not be found.</li>
     *         <li>DirectoryDoesNotExistException The specified directory does not exist in the system.</li>
     *         <li>InvalidLdapsStatusException The LDAP activities could not be performed because they are limited by
     *         the LDAPS status.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DisableLDAPS
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DisableLDAPS" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DisableLdapsResponse> disableLDAPS(DisableLdapsRequest disableLdapsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableLdapsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableLDAPS");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableLdapsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DisableLdapsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableLdapsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableLdapsRequest, DisableLdapsResponse>()
                            .withOperationName("DisableLDAPS").withMarshaller(new DisableLdapsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disableLdapsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = disableLdapsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DisableLdapsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server
     * for an AD Connector or Microsoft AD directory.
     * </p>
     *
     * @param disableRadiusRequest
     *        Contains the inputs for the <a>DisableRadius</a> operation.
     * @return A Java Future containing the result of the DisableRadius operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DisableRadius
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DisableRadius" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DisableRadiusResponse> disableRadius(DisableRadiusRequest disableRadiusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableRadiusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableRadius");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableRadiusResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DisableRadiusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableRadiusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableRadiusRequest, DisableRadiusResponse>()
                            .withOperationName("DisableRadius")
                            .withMarshaller(new DisableRadiusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disableRadiusRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = disableRadiusRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DisableRadiusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables single-sign on for a directory.
     * </p>
     *
     * @param disableSsoRequest
     *        Contains the inputs for the <a>DisableSso</a> operation.
     * @return A Java Future containing the result of the DisableSso operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InsufficientPermissionsException The account does not have sufficient permission to perform the
     *         operation.</li>
     *         <li>AuthenticationFailedException An authentication error occurred.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.DisableSso
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/DisableSso" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DisableSsoResponse> disableSso(DisableSsoRequest disableSsoRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableSsoRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableSso");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableSsoResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DisableSsoResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableSsoResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableSsoRequest, DisableSsoResponse>().withOperationName("DisableSso")
                            .withMarshaller(new DisableSsoRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disableSsoRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = disableSsoRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DisableSsoResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables alternative client authentication methods for the specified directory.
     * </p>
     *
     * @param enableClientAuthenticationRequest
     * @return A Java Future containing the result of the EnableClientAuthentication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryDoesNotExistException The specified directory does not exist in the system.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>InvalidClientAuthStatusException Client authentication is already enabled.</li>
     *         <li>AccessDeniedException Client authentication is not available in this region at this time.</li>
     *         <li>NoAvailableCertificateException Client authentication setup could not be completed because at least
     *         one valid certificate must be registered in the system.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.EnableClientAuthentication
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/EnableClientAuthentication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableClientAuthenticationResponse> enableClientAuthentication(
            EnableClientAuthenticationRequest enableClientAuthenticationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableClientAuthenticationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableClientAuthentication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableClientAuthenticationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, EnableClientAuthenticationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableClientAuthenticationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableClientAuthenticationRequest, EnableClientAuthenticationResponse>()
                            .withOperationName("EnableClientAuthentication")
                            .withMarshaller(new EnableClientAuthenticationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(enableClientAuthenticationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = enableClientAuthenticationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<EnableClientAuthenticationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Activates the switch for the specific directory to always use LDAP secure calls.
     * </p>
     *
     * @param enableLdapsRequest
     * @return A Java Future containing the result of the EnableLDAPS operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryUnavailableException The specified directory is unavailable or could not be found.</li>
     *         <li>DirectoryDoesNotExistException The specified directory does not exist in the system.</li>
     *         <li>NoAvailableCertificateException Client authentication setup could not be completed because at least
     *         one valid certificate must be registered in the system.</li>
     *         <li>InvalidLdapsStatusException The LDAP activities could not be performed because they are limited by
     *         the LDAPS status.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.EnableLDAPS
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/EnableLDAPS" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<EnableLdapsResponse> enableLDAPS(EnableLdapsRequest enableLdapsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableLdapsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableLDAPS");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableLdapsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    EnableLdapsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableLdapsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableLdapsRequest, EnableLdapsResponse>()
                            .withOperationName("EnableLDAPS").withMarshaller(new EnableLdapsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(enableLdapsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = enableLdapsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<EnableLdapsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for
     * an AD Connector or Microsoft AD directory.
     * </p>
     *
     * @param enableRadiusRequest
     *        Contains the inputs for the <a>EnableRadius</a> operation.
     * @return A Java Future containing the result of the EnableRadius operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>EntityAlreadyExistsException The specified entity already exists.</li>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.EnableRadius
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/EnableRadius" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<EnableRadiusResponse> enableRadius(EnableRadiusRequest enableRadiusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableRadiusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableRadius");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableRadiusResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    EnableRadiusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableRadiusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableRadiusRequest, EnableRadiusResponse>()
                            .withOperationName("EnableRadius").withMarshaller(new EnableRadiusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(enableRadiusRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = enableRadiusRequest.overrideConfiguration().orElse(null);
            CompletableFuture<EnableRadiusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables single sign-on for a directory. Single sign-on allows users in your directory to access certain AWS
     * services from a computer joined to the directory without having to enter their credentials separately.
     * </p>
     *
     * @param enableSsoRequest
     *        Contains the inputs for the <a>EnableSso</a> operation.
     * @return A Java Future containing the result of the EnableSso operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InsufficientPermissionsException The account does not have sufficient permission to perform the
     *         operation.</li>
     *         <li>AuthenticationFailedException An authentication error occurred.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.EnableSso
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/EnableSso" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<EnableSsoResponse> enableSso(EnableSsoRequest enableSsoRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableSsoRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableSso");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableSsoResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    EnableSsoResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableSsoResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableSsoRequest, EnableSsoResponse>().withOperationName("EnableSso")
                            .withMarshaller(new EnableSsoRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(enableSsoRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = enableSsoRequest.overrideConfiguration().orElse(null);
            CompletableFuture<EnableSsoResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Obtains directory limit information for the current Region.
     * </p>
     *
     * @param getDirectoryLimitsRequest
     *        Contains the inputs for the <a>GetDirectoryLimits</a> operation.
     * @return A Java Future containing the result of the GetDirectoryLimits operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.GetDirectoryLimits
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/GetDirectoryLimits" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDirectoryLimitsResponse> getDirectoryLimits(GetDirectoryLimitsRequest getDirectoryLimitsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDirectoryLimitsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDirectoryLimits");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDirectoryLimitsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDirectoryLimitsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDirectoryLimitsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDirectoryLimitsRequest, GetDirectoryLimitsResponse>()
                            .withOperationName("GetDirectoryLimits")
                            .withMarshaller(new GetDirectoryLimitsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDirectoryLimitsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getDirectoryLimitsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetDirectoryLimitsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Obtains the manual snapshot limits for a directory.
     * </p>
     *
     * @param getSnapshotLimitsRequest
     *        Contains the inputs for the <a>GetSnapshotLimits</a> operation.
     * @return A Java Future containing the result of the GetSnapshotLimits operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.GetSnapshotLimits
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/GetSnapshotLimits" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSnapshotLimitsResponse> getSnapshotLimits(GetSnapshotLimitsRequest getSnapshotLimitsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSnapshotLimitsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSnapshotLimits");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSnapshotLimitsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSnapshotLimitsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSnapshotLimitsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSnapshotLimitsRequest, GetSnapshotLimitsResponse>()
                            .withOperationName("GetSnapshotLimits")
                            .withMarshaller(new GetSnapshotLimitsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSnapshotLimitsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getSnapshotLimitsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetSnapshotLimitsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For the specified directory, lists all the certificates registered for a secure LDAP or client certificate
     * authentication.
     * </p>
     *
     * @param listCertificatesRequest
     * @return A Java Future containing the result of the ListCertificates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryDoesNotExistException The specified directory does not exist in the system.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.ListCertificates
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/ListCertificates" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListCertificatesResponse> listCertificates(ListCertificatesRequest listCertificatesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCertificatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCertificates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCertificatesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCertificatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCertificatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCertificatesRequest, ListCertificatesResponse>()
                            .withOperationName("ListCertificates")
                            .withMarshaller(new ListCertificatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listCertificatesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listCertificatesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListCertificatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the address blocks that you have added to a directory.
     * </p>
     *
     * @param listIpRoutesRequest
     * @return A Java Future containing the result of the ListIpRoutes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.ListIpRoutes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/ListIpRoutes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListIpRoutesResponse> listIpRoutes(ListIpRoutesRequest listIpRoutesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIpRoutesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIpRoutes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListIpRoutesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListIpRoutesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListIpRoutesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListIpRoutesRequest, ListIpRoutesResponse>()
                            .withOperationName("ListIpRoutes").withMarshaller(new ListIpRoutesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listIpRoutesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listIpRoutesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListIpRoutesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the active log subscriptions for the AWS account.
     * </p>
     *
     * @param listLogSubscriptionsRequest
     * @return A Java Future containing the result of the ListLogSubscriptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.ListLogSubscriptions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/ListLogSubscriptions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListLogSubscriptionsResponse> listLogSubscriptions(
            ListLogSubscriptionsRequest listLogSubscriptionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLogSubscriptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLogSubscriptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLogSubscriptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLogSubscriptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLogSubscriptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLogSubscriptionsRequest, ListLogSubscriptionsResponse>()
                            .withOperationName("ListLogSubscriptions")
                            .withMarshaller(new ListLogSubscriptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLogSubscriptionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listLogSubscriptionsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListLogSubscriptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all schema extensions applied to a Microsoft AD Directory.
     * </p>
     *
     * @param listSchemaExtensionsRequest
     * @return A Java Future containing the result of the ListSchemaExtensions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.ListSchemaExtensions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/ListSchemaExtensions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSchemaExtensionsResponse> listSchemaExtensions(
            ListSchemaExtensionsRequest listSchemaExtensionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSchemaExtensionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSchemaExtensions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSchemaExtensionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSchemaExtensionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSchemaExtensionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSchemaExtensionsRequest, ListSchemaExtensionsResponse>()
                            .withOperationName("ListSchemaExtensions")
                            .withMarshaller(new ListSchemaExtensionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSchemaExtensionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listSchemaExtensionsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListSchemaExtensionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all tags on a directory.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidNextTokenException The <code>NextToken</code> value is not valid.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers a certificate for a secure LDAP or client certificate authentication.
     * </p>
     *
     * @param registerCertificateRequest
     * @return A Java Future containing the result of the RegisterCertificate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryUnavailableException The specified directory is unavailable or could not be found.</li>
     *         <li>DirectoryDoesNotExistException The specified directory does not exist in the system.</li>
     *         <li>InvalidCertificateException The certificate PEM that was provided has incorrect encoding.</li>
     *         <li>CertificateLimitExceededException The certificate could not be added because the certificate limit
     *         has been reached.</li>
     *         <li>CertificateAlreadyExistsException The certificate has already been registered into the system.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.RegisterCertificate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/RegisterCertificate" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterCertificateResponse> registerCertificate(
            RegisterCertificateRequest registerCertificateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerCertificateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterCertificate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterCertificateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterCertificateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterCertificateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterCertificateRequest, RegisterCertificateResponse>()
                            .withOperationName("RegisterCertificate")
                            .withMarshaller(new RegisterCertificateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(registerCertificateRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = registerCertificateRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<RegisterCertificateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates a directory with an SNS topic. This establishes the directory as a publisher to the specified SNS
     * topic. You can then receive email or text (SMS) messages when the status of your directory changes. You get
     * notified if your directory goes from an Active status to an Impaired or Inoperable status. You also receive a
     * notification when the directory returns to an Active status.
     * </p>
     *
     * @param registerEventTopicRequest
     *        Registers a new event topic.
     * @return A Java Future containing the result of the RegisterEventTopic operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.RegisterEventTopic
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/RegisterEventTopic" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterEventTopicResponse> registerEventTopic(RegisterEventTopicRequest registerEventTopicRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerEventTopicRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterEventTopic");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterEventTopicResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterEventTopicResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterEventTopicResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterEventTopicRequest, RegisterEventTopicResponse>()
                            .withOperationName("RegisterEventTopic")
                            .withMarshaller(new RegisterEventTopicRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(registerEventTopicRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = registerEventTopicRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<RegisterEventTopicResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Rejects a directory sharing request that was sent from the directory owner account.
     * </p>
     *
     * @param rejectSharedDirectoryRequest
     * @return A Java Future containing the result of the RejectSharedDirectory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>DirectoryAlreadySharedException The specified directory has already been shared with this AWS
     *         account.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.RejectSharedDirectory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/RejectSharedDirectory" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RejectSharedDirectoryResponse> rejectSharedDirectory(
            RejectSharedDirectoryRequest rejectSharedDirectoryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rejectSharedDirectoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectSharedDirectory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RejectSharedDirectoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RejectSharedDirectoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RejectSharedDirectoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RejectSharedDirectoryRequest, RejectSharedDirectoryResponse>()
                            .withOperationName("RejectSharedDirectory")
                            .withMarshaller(new RejectSharedDirectoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(rejectSharedDirectoryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = rejectSharedDirectoryRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<RejectSharedDirectoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes IP address blocks from a directory.
     * </p>
     *
     * @param removeIpRoutesRequest
     * @return A Java Future containing the result of the RemoveIpRoutes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DirectoryUnavailableException The specified directory is unavailable or could not be found.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.RemoveIpRoutes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/RemoveIpRoutes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveIpRoutesResponse> removeIpRoutes(RemoveIpRoutesRequest removeIpRoutesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeIpRoutesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveIpRoutes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveIpRoutesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveIpRoutesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveIpRoutesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveIpRoutesRequest, RemoveIpRoutesResponse>()
                            .withOperationName("RemoveIpRoutes")
                            .withMarshaller(new RemoveIpRoutesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeIpRoutesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = removeIpRoutesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<RemoveIpRoutesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops all replication and removes the domain controllers from the specified Region. You cannot remove the primary
     * Region with this operation. Instead, use the <code>DeleteDirectory</code> API.
     * </p>
     *
     * @param removeRegionRequest
     * @return A Java Future containing the result of the RemoveRegion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryUnavailableException The specified directory is unavailable or could not be found.</li>
     *         <li>DirectoryDoesNotExistException The specified directory does not exist in the system.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>AccessDeniedException Client authentication is not available in this region at this time.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.RemoveRegion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/RemoveRegion" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveRegionResponse> removeRegion(RemoveRegionRequest removeRegionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeRegionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveRegion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveRegionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RemoveRegionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveRegionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveRegionRequest, RemoveRegionResponse>()
                            .withOperationName("RemoveRegion").withMarshaller(new RemoveRegionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeRegionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = removeRegionRequest.overrideConfiguration().orElse(null);
            CompletableFuture<RemoveRegionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from a directory.
     * </p>
     *
     * @param removeTagsFromResourceRequest
     * @return A Java Future containing the result of the RemoveTagsFromResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.RemoveTagsFromResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/RemoveTagsFromResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveTagsFromResourceResponse> removeTagsFromResource(
            RemoveTagsFromResourceRequest removeTagsFromResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeTagsFromResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveTagsFromResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveTagsFromResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveTagsFromResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveTagsFromResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveTagsFromResourceRequest, RemoveTagsFromResourceResponse>()
                            .withOperationName("RemoveTagsFromResource")
                            .withMarshaller(new RemoveTagsFromResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeTagsFromResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = removeTagsFromResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<RemoveTagsFromResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Resets the password for any user in your AWS Managed Microsoft AD or Simple AD directory.
     * </p>
     * <p>
     * You can reset the password for any user in your directory with the following exceptions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For Simple AD, you cannot reset the password for any user that is a member of either the <b>Domain Admins</b> or
     * <b>Enterprise Admins</b> group except for the administrator user.
     * </p>
     * </li>
     * <li>
     * <p>
     * For AWS Managed Microsoft AD, you can only reset the password for a user that is in an OU based off of the
     * NetBIOS name that you typed when you created your directory. For example, you cannot reset the password for a
     * user in the <b>AWS Reserved</b> OU. For more information about the OU structure for an AWS Managed Microsoft AD
     * directory, see <a href=
     * "https://docs.aws.amazon.com/directoryservice/latest/admin-guide/ms_ad_getting_started_what_gets_created.html"
     * >What Gets Created</a> in the <i>AWS Directory Service Administration Guide</i>.
     * </p>
     * </li>
     * </ul>
     *
     * @param resetUserPasswordRequest
     * @return A Java Future containing the result of the ResetUserPassword operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryUnavailableException The specified directory is unavailable or could not be found.</li>
     *         <li>UserDoesNotExistException The user provided a username that does not exist in your directory.</li>
     *         <li>InvalidPasswordException The new password provided by the user does not meet the password complexity
     *         requirements defined in your directory.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.ResetUserPassword
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/ResetUserPassword" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ResetUserPasswordResponse> resetUserPassword(ResetUserPasswordRequest resetUserPasswordRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resetUserPasswordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResetUserPassword");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ResetUserPasswordResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ResetUserPasswordResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ResetUserPasswordResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ResetUserPasswordRequest, ResetUserPasswordResponse>()
                            .withOperationName("ResetUserPassword")
                            .withMarshaller(new ResetUserPasswordRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(resetUserPasswordRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = resetUserPasswordRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ResetUserPasswordResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Restores a directory using an existing directory snapshot.
     * </p>
     * <p>
     * When you restore a directory from a snapshot, any changes made to the directory after the snapshot date are
     * overwritten.
     * </p>
     * <p>
     * This action returns as soon as the restore operation is initiated. You can monitor the progress of the restore
     * operation by calling the <a>DescribeDirectories</a> operation with the directory identifier. When the
     * <b>DirectoryDescription.Stage</b> value changes to <code>Active</code>, the restore operation is complete.
     * </p>
     *
     * @param restoreFromSnapshotRequest
     *        An object representing the inputs for the <a>RestoreFromSnapshot</a> operation.
     * @return A Java Future containing the result of the RestoreFromSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.RestoreFromSnapshot
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/RestoreFromSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RestoreFromSnapshotResponse> restoreFromSnapshot(
            RestoreFromSnapshotRequest restoreFromSnapshotRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreFromSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreFromSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestoreFromSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RestoreFromSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RestoreFromSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestoreFromSnapshotRequest, RestoreFromSnapshotResponse>()
                            .withOperationName("RestoreFromSnapshot")
                            .withMarshaller(new RestoreFromSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(restoreFromSnapshotRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = restoreFromSnapshotRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<RestoreFromSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Shares a specified directory (<code>DirectoryId</code>) in your AWS account (directory owner) with another AWS
     * account (directory consumer). With this operation you can use your directory from any AWS account and from any
     * Amazon VPC within an AWS Region.
     * </p>
     * <p>
     * When you share your AWS Managed Microsoft AD directory, AWS Directory Service creates a shared directory in the
     * directory consumer account. This shared directory contains the metadata to provide access to the directory within
     * the directory owner account. The shared directory is visible in all VPCs in the directory consumer account.
     * </p>
     * <p>
     * The <code>ShareMethod</code> parameter determines whether the specified directory can be shared between AWS
     * accounts inside the same AWS organization (<code>ORGANIZATIONS</code>). It also determines whether you can share
     * the directory with any other AWS account either inside or outside of the organization (<code>HANDSHAKE</code>).
     * </p>
     * <p>
     * The <code>ShareNotes</code> parameter is only used when <code>HANDSHAKE</code> is called, which sends a directory
     * sharing request to the directory consumer.
     * </p>
     *
     * @param shareDirectoryRequest
     * @return A Java Future containing the result of the ShareDirectory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryAlreadySharedException The specified directory has already been shared with this AWS
     *         account.</li>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidTargetException The specified shared target is not valid.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ShareLimitExceededException The maximum number of AWS accounts that you can share with this directory
     *         has been reached.</li>
     *         <li>OrganizationsException Exception encountered while trying to access your AWS organization.</li>
     *         <li>AccessDeniedException Client authentication is not available in this region at this time.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.ShareDirectory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/ShareDirectory" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ShareDirectoryResponse> shareDirectory(ShareDirectoryRequest shareDirectoryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, shareDirectoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ShareDirectory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ShareDirectoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ShareDirectoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ShareDirectoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ShareDirectoryRequest, ShareDirectoryResponse>()
                            .withOperationName("ShareDirectory")
                            .withMarshaller(new ShareDirectoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(shareDirectoryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = shareDirectoryRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ShareDirectoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Applies a schema extension to a Microsoft AD directory.
     * </p>
     *
     * @param startSchemaExtensionRequest
     * @return A Java Future containing the result of the StartSchemaExtension operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DirectoryUnavailableException The specified directory is unavailable or could not be found.</li>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>SnapshotLimitExceededException The maximum number of manual snapshots for the directory has been
     *         reached. You can use the <a>GetSnapshotLimits</a> operation to determine the snapshot limits for a
     *         directory.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.StartSchemaExtension
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/StartSchemaExtension" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartSchemaExtensionResponse> startSchemaExtension(
            StartSchemaExtensionRequest startSchemaExtensionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startSchemaExtensionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartSchemaExtension");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartSchemaExtensionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartSchemaExtensionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartSchemaExtensionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartSchemaExtensionRequest, StartSchemaExtensionResponse>()
                            .withOperationName("StartSchemaExtension")
                            .withMarshaller(new StartSchemaExtensionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startSchemaExtensionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startSchemaExtensionRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<StartSchemaExtensionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops the directory sharing between the directory owner and consumer accounts.
     * </p>
     *
     * @param unshareDirectoryRequest
     * @return A Java Future containing the result of the UnshareDirectory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidTargetException The specified shared target is not valid.</li>
     *         <li>DirectoryNotSharedException The specified directory has not been shared with this AWS account.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.UnshareDirectory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/UnshareDirectory" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UnshareDirectoryResponse> unshareDirectory(UnshareDirectoryRequest unshareDirectoryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, unshareDirectoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UnshareDirectory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UnshareDirectoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UnshareDirectoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UnshareDirectoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UnshareDirectoryRequest, UnshareDirectoryResponse>()
                            .withOperationName("UnshareDirectory")
                            .withMarshaller(new UnshareDirectoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(unshareDirectoryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = unshareDirectoryRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UnshareDirectoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a conditional forwarder that has been set up for your AWS directory.
     * </p>
     *
     * @param updateConditionalForwarderRequest
     *        Updates a conditional forwarder.
     * @return A Java Future containing the result of the UpdateConditionalForwarder operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>DirectoryUnavailableException The specified directory is unavailable or could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.UpdateConditionalForwarder
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/UpdateConditionalForwarder" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConditionalForwarderResponse> updateConditionalForwarder(
            UpdateConditionalForwarderRequest updateConditionalForwarderRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConditionalForwarderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConditionalForwarder");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConditionalForwarderResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateConditionalForwarderResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConditionalForwarderResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConditionalForwarderRequest, UpdateConditionalForwarderResponse>()
                            .withOperationName("UpdateConditionalForwarder")
                            .withMarshaller(new UpdateConditionalForwarderRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateConditionalForwarderRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateConditionalForwarderRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateConditionalForwarderResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or removes domain controllers to or from the directory. Based on the difference between current value and
     * new value (provided through this API call), domain controllers will be added or removed. It may take up to 45
     * minutes for any new domain controllers to become fully active once the requested number of domain controllers is
     * updated. During this time, you cannot make another update request.
     * </p>
     *
     * @param updateNumberOfDomainControllersRequest
     * @return A Java Future containing the result of the UpdateNumberOfDomainControllers operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>DirectoryUnavailableException The specified directory is unavailable or could not be found.</li>
     *         <li>DomainControllerLimitExceededException The maximum allowed number of domain controllers per directory
     *         was exceeded. The default limit per directory is 20 domain controllers.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.UpdateNumberOfDomainControllers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/UpdateNumberOfDomainControllers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateNumberOfDomainControllersResponse> updateNumberOfDomainControllers(
            UpdateNumberOfDomainControllersRequest updateNumberOfDomainControllersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateNumberOfDomainControllersRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNumberOfDomainControllers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateNumberOfDomainControllersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateNumberOfDomainControllersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateNumberOfDomainControllersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateNumberOfDomainControllersRequest, UpdateNumberOfDomainControllersResponse>()
                            .withOperationName("UpdateNumberOfDomainControllers")
                            .withMarshaller(new UpdateNumberOfDomainControllersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateNumberOfDomainControllersRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateNumberOfDomainControllersRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<UpdateNumberOfDomainControllersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the Remote Authentication Dial In User Service (RADIUS) server information for an AD Connector or
     * Microsoft AD directory.
     * </p>
     *
     * @param updateRadiusRequest
     *        Contains the inputs for the <a>UpdateRadius</a> operation.
     * @return A Java Future containing the result of the UpdateRadius operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.UpdateRadius
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/UpdateRadius" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRadiusResponse> updateRadius(UpdateRadiusRequest updateRadiusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRadiusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRadius");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRadiusResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateRadiusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRadiusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRadiusRequest, UpdateRadiusResponse>()
                            .withOperationName("UpdateRadius").withMarshaller(new UpdateRadiusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateRadiusRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateRadiusRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateRadiusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the trust that has been set up between your AWS Managed Microsoft AD directory and an on-premises Active
     * Directory.
     * </p>
     *
     * @param updateTrustRequest
     * @return A Java Future containing the result of the UpdateTrust operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.UpdateTrust
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/UpdateTrust" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTrustResponse> updateTrust(UpdateTrustRequest updateTrustRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTrustRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTrust");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTrustResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateTrustResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTrustResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTrustRequest, UpdateTrustResponse>()
                            .withOperationName("UpdateTrust").withMarshaller(new UpdateTrustRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateTrustRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateTrustRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateTrustResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * AWS Directory Service for Microsoft Active Directory allows you to configure and verify trust relationships.
     * </p>
     * <p>
     * This action verifies a trust relationship between your AWS Managed Microsoft AD directory and an external domain.
     * </p>
     *
     * @param verifyTrustRequest
     *        Initiates the verification of an existing trust relationship between an AWS Managed Microsoft AD directory
     *        and an external domain.
     * @return A Java Future containing the result of the VerifyTrust operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EntityDoesNotExistException The specified entity could not be found.</li>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>ClientException A client exception has occurred.</li>
     *         <li>ServiceException An exception has occurred in AWS Directory Service.</li>
     *         <li>UnsupportedOperationException The operation is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DirectoryAsyncClient.VerifyTrust
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ds-2015-04-16/VerifyTrust" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<VerifyTrustResponse> verifyTrust(VerifyTrustRequest verifyTrustRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, verifyTrustRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "VerifyTrust");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<VerifyTrustResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    VerifyTrustResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<VerifyTrustResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<VerifyTrustRequest, VerifyTrustResponse>()
                            .withOperationName("VerifyTrust").withMarshaller(new VerifyTrustRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(verifyTrustRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = verifyTrustRequest.overrideConfiguration().orElse(null);
            CompletableFuture<VerifyTrustResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DirectoryException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DirectoryAlreadySharedException")
                                .exceptionBuilderSupplier(DirectoryAlreadySharedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DirectoryUnavailableException")
                                .exceptionBuilderSupplier(DirectoryUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCertificateException")
                                .exceptionBuilderSupplier(InvalidCertificateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DirectoryNotSharedException")
                                .exceptionBuilderSupplier(DirectoryNotSharedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RegionLimitExceededException")
                                .exceptionBuilderSupplier(RegionLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EntityDoesNotExistException")
                                .exceptionBuilderSupplier(EntityDoesNotExistException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DirectoryLimitExceededException")
                                .exceptionBuilderSupplier(DirectoryLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DirectoryDoesNotExistException")
                                .exceptionBuilderSupplier(DirectoryDoesNotExistException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CertificateAlreadyExistsException")
                                .exceptionBuilderSupplier(CertificateAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DirectoryAlreadyInRegionException")
                                .exceptionBuilderSupplier(DirectoryAlreadyInRegionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DomainControllerLimitExceededException")
                                .exceptionBuilderSupplier(DomainControllerLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UserDoesNotExistException")
                                .exceptionBuilderSupplier(UserDoesNotExistException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPasswordException")
                                .exceptionBuilderSupplier(InvalidPasswordException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedOperationException")
                                .exceptionBuilderSupplier(UnsupportedOperationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClientException")
                                .exceptionBuilderSupplier(ClientException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AuthenticationFailedException")
                                .exceptionBuilderSupplier(AuthenticationFailedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InsufficientPermissionsException")
                                .exceptionBuilderSupplier(InsufficientPermissionsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CertificateLimitExceededException")
                                .exceptionBuilderSupplier(CertificateLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OrganizationsException")
                                .exceptionBuilderSupplier(OrganizationsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLDAPSStatusException")
                                .exceptionBuilderSupplier(InvalidLdapsStatusException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CertificateDoesNotExistException")
                                .exceptionBuilderSupplier(CertificateDoesNotExistException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceException")
                                .exceptionBuilderSupplier(ServiceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SnapshotLimitExceededException")
                                .exceptionBuilderSupplier(SnapshotLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EntityAlreadyExistsException")
                                .exceptionBuilderSupplier(EntityAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidClientAuthStatusException")
                                .exceptionBuilderSupplier(InvalidClientAuthStatusException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoAvailableCertificateException")
                                .exceptionBuilderSupplier(NoAvailableCertificateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTargetException")
                                .exceptionBuilderSupplier(InvalidTargetException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IpRouteLimitExceededException")
                                .exceptionBuilderSupplier(IpRouteLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagLimitExceededException")
                                .exceptionBuilderSupplier(TagLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ShareLimitExceededException")
                                .exceptionBuilderSupplier(ShareLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CertificateInUseException")
                                .exceptionBuilderSupplier(CertificateInUseException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends DirectoryRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
