/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directory.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the directory owner account details that have been shared to the directory consumer account.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OwnerDirectoryDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<OwnerDirectoryDescription.Builder, OwnerDirectoryDescription> {
    private static final SdkField<String> DIRECTORY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DirectoryId").getter(getter(OwnerDirectoryDescription::directoryId))
            .setter(setter(Builder::directoryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DirectoryId").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountId").getter(getter(OwnerDirectoryDescription::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountId").build()).build();

    private static final SdkField<List<String>> DNS_IP_ADDRS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DnsIpAddrs")
            .getter(getter(OwnerDirectoryDescription::dnsIpAddrs))
            .setter(setter(Builder::dnsIpAddrs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsIpAddrs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DirectoryVpcSettingsDescription> VPC_SETTINGS_FIELD = SdkField
            .<DirectoryVpcSettingsDescription> builder(MarshallingType.SDK_POJO).memberName("VpcSettings")
            .getter(getter(OwnerDirectoryDescription::vpcSettings)).setter(setter(Builder::vpcSettings))
            .constructor(DirectoryVpcSettingsDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcSettings").build()).build();

    private static final SdkField<RadiusSettings> RADIUS_SETTINGS_FIELD = SdkField
            .<RadiusSettings> builder(MarshallingType.SDK_POJO).memberName("RadiusSettings")
            .getter(getter(OwnerDirectoryDescription::radiusSettings)).setter(setter(Builder::radiusSettings))
            .constructor(RadiusSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RadiusSettings").build()).build();

    private static final SdkField<String> RADIUS_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RadiusStatus").getter(getter(OwnerDirectoryDescription::radiusStatusAsString))
            .setter(setter(Builder::radiusStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RadiusStatus").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DIRECTORY_ID_FIELD,
            ACCOUNT_ID_FIELD, DNS_IP_ADDRS_FIELD, VPC_SETTINGS_FIELD, RADIUS_SETTINGS_FIELD, RADIUS_STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String directoryId;

    private final String accountId;

    private final List<String> dnsIpAddrs;

    private final DirectoryVpcSettingsDescription vpcSettings;

    private final RadiusSettings radiusSettings;

    private final String radiusStatus;

    private OwnerDirectoryDescription(BuilderImpl builder) {
        this.directoryId = builder.directoryId;
        this.accountId = builder.accountId;
        this.dnsIpAddrs = builder.dnsIpAddrs;
        this.vpcSettings = builder.vpcSettings;
        this.radiusSettings = builder.radiusSettings;
        this.radiusStatus = builder.radiusStatus;
    }

    /**
     * <p>
     * Identifier of the AWS Managed Microsoft AD directory in the directory owner account.
     * </p>
     * 
     * @return Identifier of the AWS Managed Microsoft AD directory in the directory owner account.
     */
    public String directoryId() {
        return directoryId;
    }

    /**
     * <p>
     * Identifier of the directory owner account.
     * </p>
     * 
     * @return Identifier of the directory owner account.
     */
    public String accountId() {
        return accountId;
    }

    /**
     * Returns true if the DnsIpAddrs property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDnsIpAddrs() {
        return dnsIpAddrs != null && !(dnsIpAddrs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * IP address of the directory’s domain controllers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDnsIpAddrs()} to see if a value was sent in this field.
     * </p>
     * 
     * @return IP address of the directory’s domain controllers.
     */
    public List<String> dnsIpAddrs() {
        return dnsIpAddrs;
    }

    /**
     * <p>
     * Information about the VPC settings for the directory.
     * </p>
     * 
     * @return Information about the VPC settings for the directory.
     */
    public DirectoryVpcSettingsDescription vpcSettings() {
        return vpcSettings;
    }

    /**
     * <p>
     * A <a>RadiusSettings</a> object that contains information about the RADIUS server.
     * </p>
     * 
     * @return A <a>RadiusSettings</a> object that contains information about the RADIUS server.
     */
    public RadiusSettings radiusSettings() {
        return radiusSettings;
    }

    /**
     * <p>
     * Information about the status of the RADIUS server.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #radiusStatus} will
     * return {@link RadiusStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #radiusStatusAsString}.
     * </p>
     * 
     * @return Information about the status of the RADIUS server.
     * @see RadiusStatus
     */
    public RadiusStatus radiusStatus() {
        return RadiusStatus.fromValue(radiusStatus);
    }

    /**
     * <p>
     * Information about the status of the RADIUS server.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #radiusStatus} will
     * return {@link RadiusStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #radiusStatusAsString}.
     * </p>
     * 
     * @return Information about the status of the RADIUS server.
     * @see RadiusStatus
     */
    public String radiusStatusAsString() {
        return radiusStatus;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(directoryId());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(hasDnsIpAddrs() ? dnsIpAddrs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(vpcSettings());
        hashCode = 31 * hashCode + Objects.hashCode(radiusSettings());
        hashCode = 31 * hashCode + Objects.hashCode(radiusStatusAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OwnerDirectoryDescription)) {
            return false;
        }
        OwnerDirectoryDescription other = (OwnerDirectoryDescription) obj;
        return Objects.equals(directoryId(), other.directoryId()) && Objects.equals(accountId(), other.accountId())
                && hasDnsIpAddrs() == other.hasDnsIpAddrs() && Objects.equals(dnsIpAddrs(), other.dnsIpAddrs())
                && Objects.equals(vpcSettings(), other.vpcSettings()) && Objects.equals(radiusSettings(), other.radiusSettings())
                && Objects.equals(radiusStatusAsString(), other.radiusStatusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("OwnerDirectoryDescription").add("DirectoryId", directoryId()).add("AccountId", accountId())
                .add("DnsIpAddrs", hasDnsIpAddrs() ? dnsIpAddrs() : null).add("VpcSettings", vpcSettings())
                .add("RadiusSettings", radiusSettings()).add("RadiusStatus", radiusStatusAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DirectoryId":
            return Optional.ofNullable(clazz.cast(directoryId()));
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "DnsIpAddrs":
            return Optional.ofNullable(clazz.cast(dnsIpAddrs()));
        case "VpcSettings":
            return Optional.ofNullable(clazz.cast(vpcSettings()));
        case "RadiusSettings":
            return Optional.ofNullable(clazz.cast(radiusSettings()));
        case "RadiusStatus":
            return Optional.ofNullable(clazz.cast(radiusStatusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OwnerDirectoryDescription, T> g) {
        return obj -> g.apply((OwnerDirectoryDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OwnerDirectoryDescription> {
        /**
         * <p>
         * Identifier of the AWS Managed Microsoft AD directory in the directory owner account.
         * </p>
         * 
         * @param directoryId
         *        Identifier of the AWS Managed Microsoft AD directory in the directory owner account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder directoryId(String directoryId);

        /**
         * <p>
         * Identifier of the directory owner account.
         * </p>
         * 
         * @param accountId
         *        Identifier of the directory owner account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * IP address of the directory’s domain controllers.
         * </p>
         * 
         * @param dnsIpAddrs
         *        IP address of the directory’s domain controllers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIpAddrs(Collection<String> dnsIpAddrs);

        /**
         * <p>
         * IP address of the directory’s domain controllers.
         * </p>
         * 
         * @param dnsIpAddrs
         *        IP address of the directory’s domain controllers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIpAddrs(String... dnsIpAddrs);

        /**
         * <p>
         * Information about the VPC settings for the directory.
         * </p>
         * 
         * @param vpcSettings
         *        Information about the VPC settings for the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSettings(DirectoryVpcSettingsDescription vpcSettings);

        /**
         * <p>
         * Information about the VPC settings for the directory.
         * </p>
         * This is a convenience that creates an instance of the {@link DirectoryVpcSettingsDescription.Builder}
         * avoiding the need to create one manually via {@link DirectoryVpcSettingsDescription#builder()}.
         *
         * When the {@link Consumer} completes, {@link DirectoryVpcSettingsDescription.Builder#build()} is called
         * immediately and its result is passed to {@link #vpcSettings(DirectoryVpcSettingsDescription)}.
         * 
         * @param vpcSettings
         *        a consumer that will call methods on {@link DirectoryVpcSettingsDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcSettings(DirectoryVpcSettingsDescription)
         */
        default Builder vpcSettings(Consumer<DirectoryVpcSettingsDescription.Builder> vpcSettings) {
            return vpcSettings(DirectoryVpcSettingsDescription.builder().applyMutation(vpcSettings).build());
        }

        /**
         * <p>
         * A <a>RadiusSettings</a> object that contains information about the RADIUS server.
         * </p>
         * 
         * @param radiusSettings
         *        A <a>RadiusSettings</a> object that contains information about the RADIUS server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder radiusSettings(RadiusSettings radiusSettings);

        /**
         * <p>
         * A <a>RadiusSettings</a> object that contains information about the RADIUS server.
         * </p>
         * This is a convenience that creates an instance of the {@link RadiusSettings.Builder} avoiding the need to
         * create one manually via {@link RadiusSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link RadiusSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #radiusSettings(RadiusSettings)}.
         * 
         * @param radiusSettings
         *        a consumer that will call methods on {@link RadiusSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #radiusSettings(RadiusSettings)
         */
        default Builder radiusSettings(Consumer<RadiusSettings.Builder> radiusSettings) {
            return radiusSettings(RadiusSettings.builder().applyMutation(radiusSettings).build());
        }

        /**
         * <p>
         * Information about the status of the RADIUS server.
         * </p>
         * 
         * @param radiusStatus
         *        Information about the status of the RADIUS server.
         * @see RadiusStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RadiusStatus
         */
        Builder radiusStatus(String radiusStatus);

        /**
         * <p>
         * Information about the status of the RADIUS server.
         * </p>
         * 
         * @param radiusStatus
         *        Information about the status of the RADIUS server.
         * @see RadiusStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RadiusStatus
         */
        Builder radiusStatus(RadiusStatus radiusStatus);
    }

    static final class BuilderImpl implements Builder {
        private String directoryId;

        private String accountId;

        private List<String> dnsIpAddrs = DefaultSdkAutoConstructList.getInstance();

        private DirectoryVpcSettingsDescription vpcSettings;

        private RadiusSettings radiusSettings;

        private String radiusStatus;

        private BuilderImpl() {
        }

        private BuilderImpl(OwnerDirectoryDescription model) {
            directoryId(model.directoryId);
            accountId(model.accountId);
            dnsIpAddrs(model.dnsIpAddrs);
            vpcSettings(model.vpcSettings);
            radiusSettings(model.radiusSettings);
            radiusStatus(model.radiusStatus);
        }

        public final String getDirectoryId() {
            return directoryId;
        }

        @Override
        public final Builder directoryId(String directoryId) {
            this.directoryId = directoryId;
            return this;
        }

        public final void setDirectoryId(String directoryId) {
            this.directoryId = directoryId;
        }

        public final String getAccountId() {
            return accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        public final Collection<String> getDnsIpAddrs() {
            if (dnsIpAddrs instanceof SdkAutoConstructList) {
                return null;
            }
            return dnsIpAddrs;
        }

        @Override
        public final Builder dnsIpAddrs(Collection<String> dnsIpAddrs) {
            this.dnsIpAddrs = DnsIpAddrsCopier.copy(dnsIpAddrs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsIpAddrs(String... dnsIpAddrs) {
            dnsIpAddrs(Arrays.asList(dnsIpAddrs));
            return this;
        }

        public final void setDnsIpAddrs(Collection<String> dnsIpAddrs) {
            this.dnsIpAddrs = DnsIpAddrsCopier.copy(dnsIpAddrs);
        }

        public final DirectoryVpcSettingsDescription.Builder getVpcSettings() {
            return vpcSettings != null ? vpcSettings.toBuilder() : null;
        }

        @Override
        public final Builder vpcSettings(DirectoryVpcSettingsDescription vpcSettings) {
            this.vpcSettings = vpcSettings;
            return this;
        }

        public final void setVpcSettings(DirectoryVpcSettingsDescription.BuilderImpl vpcSettings) {
            this.vpcSettings = vpcSettings != null ? vpcSettings.build() : null;
        }

        public final RadiusSettings.Builder getRadiusSettings() {
            return radiusSettings != null ? radiusSettings.toBuilder() : null;
        }

        @Override
        public final Builder radiusSettings(RadiusSettings radiusSettings) {
            this.radiusSettings = radiusSettings;
            return this;
        }

        public final void setRadiusSettings(RadiusSettings.BuilderImpl radiusSettings) {
            this.radiusSettings = radiusSettings != null ? radiusSettings.build() : null;
        }

        public final String getRadiusStatus() {
            return radiusStatus;
        }

        @Override
        public final Builder radiusStatus(String radiusStatus) {
            this.radiusStatus = radiusStatus;
            return this;
        }

        @Override
        public final Builder radiusStatus(RadiusStatus radiusStatus) {
            this.radiusStatus(radiusStatus == null ? null : radiusStatus.toString());
            return this;
        }

        public final void setRadiusStatus(String radiusStatus) {
            this.radiusStatus = radiusStatus;
        }

        @Override
        public OwnerDirectoryDescription build() {
            return new OwnerDirectoryDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
