/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directory.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the trust relationships for a particular AWS Managed Microsoft AD directory. If no input parameters are are
 * provided, such as directory ID or trust ID, this request describes all the trust relationships.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribeTrustsRequest extends DirectoryRequest implements
        ToCopyableBuilder<DescribeTrustsRequest.Builder, DescribeTrustsRequest> {
    private static final SdkField<String> DIRECTORY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DirectoryId").getter(getter(DescribeTrustsRequest::directoryId)).setter(setter(Builder::directoryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DirectoryId").build()).build();

    private static final SdkField<List<String>> TRUST_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TrustIds")
            .getter(getter(DescribeTrustsRequest::trustIds))
            .setter(setter(Builder::trustIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrustIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NEXT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NextToken").getter(getter(DescribeTrustsRequest::nextToken)).setter(setter(Builder::nextToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NextToken").build()).build();

    private static final SdkField<Integer> LIMIT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Limit")
            .getter(getter(DescribeTrustsRequest::limit)).setter(setter(Builder::limit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Limit").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DIRECTORY_ID_FIELD,
            TRUST_IDS_FIELD, NEXT_TOKEN_FIELD, LIMIT_FIELD));

    private final String directoryId;

    private final List<String> trustIds;

    private final String nextToken;

    private final Integer limit;

    private DescribeTrustsRequest(BuilderImpl builder) {
        super(builder);
        this.directoryId = builder.directoryId;
        this.trustIds = builder.trustIds;
        this.nextToken = builder.nextToken;
        this.limit = builder.limit;
    }

    /**
     * <p>
     * The Directory ID of the AWS directory that is a part of the requested trust relationship.
     * </p>
     * 
     * @return The Directory ID of the AWS directory that is a part of the requested trust relationship.
     */
    public final String directoryId() {
        return directoryId;
    }

    /**
     * Returns true if the TrustIds property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTrustIds() {
        return trustIds != null && !(trustIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of identifiers of the trust relationships for which to obtain the information. If this member is null, all
     * trust relationships that belong to the current account are returned.
     * </p>
     * <p>
     * An empty list results in an <code>InvalidParameterException</code> being thrown.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTrustIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of identifiers of the trust relationships for which to obtain the information. If this member is
     *         null, all trust relationships that belong to the current account are returned.</p>
     *         <p>
     *         An empty list results in an <code>InvalidParameterException</code> being thrown.
     */
    public final List<String> trustIds() {
        return trustIds;
    }

    /**
     * <p>
     * The <i>DescribeTrustsResult.NextToken</i> value from a previous call to <a>DescribeTrusts</a>. Pass null if this
     * is the first call.
     * </p>
     * 
     * @return The <i>DescribeTrustsResult.NextToken</i> value from a previous call to <a>DescribeTrusts</a>. Pass null
     *         if this is the first call.
     */
    public final String nextToken() {
        return nextToken;
    }

    /**
     * <p>
     * The maximum number of objects to return.
     * </p>
     * 
     * @return The maximum number of objects to return.
     */
    public final Integer limit() {
        return limit;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(directoryId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTrustIds() ? trustIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(nextToken());
        hashCode = 31 * hashCode + Objects.hashCode(limit());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DescribeTrustsRequest)) {
            return false;
        }
        DescribeTrustsRequest other = (DescribeTrustsRequest) obj;
        return Objects.equals(directoryId(), other.directoryId()) && hasTrustIds() == other.hasTrustIds()
                && Objects.equals(trustIds(), other.trustIds()) && Objects.equals(nextToken(), other.nextToken())
                && Objects.equals(limit(), other.limit());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DescribeTrustsRequest").add("DirectoryId", directoryId())
                .add("TrustIds", hasTrustIds() ? trustIds() : null).add("NextToken", nextToken()).add("Limit", limit()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DirectoryId":
            return Optional.ofNullable(clazz.cast(directoryId()));
        case "TrustIds":
            return Optional.ofNullable(clazz.cast(trustIds()));
        case "NextToken":
            return Optional.ofNullable(clazz.cast(nextToken()));
        case "Limit":
            return Optional.ofNullable(clazz.cast(limit()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DescribeTrustsRequest, T> g) {
        return obj -> g.apply((DescribeTrustsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DirectoryRequest.Builder, SdkPojo, CopyableBuilder<Builder, DescribeTrustsRequest> {
        /**
         * <p>
         * The Directory ID of the AWS directory that is a part of the requested trust relationship.
         * </p>
         * 
         * @param directoryId
         *        The Directory ID of the AWS directory that is a part of the requested trust relationship.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder directoryId(String directoryId);

        /**
         * <p>
         * A list of identifiers of the trust relationships for which to obtain the information. If this member is null,
         * all trust relationships that belong to the current account are returned.
         * </p>
         * <p>
         * An empty list results in an <code>InvalidParameterException</code> being thrown.
         * </p>
         * 
         * @param trustIds
         *        A list of identifiers of the trust relationships for which to obtain the information. If this member
         *        is null, all trust relationships that belong to the current account are returned.</p>
         *        <p>
         *        An empty list results in an <code>InvalidParameterException</code> being thrown.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trustIds(Collection<String> trustIds);

        /**
         * <p>
         * A list of identifiers of the trust relationships for which to obtain the information. If this member is null,
         * all trust relationships that belong to the current account are returned.
         * </p>
         * <p>
         * An empty list results in an <code>InvalidParameterException</code> being thrown.
         * </p>
         * 
         * @param trustIds
         *        A list of identifiers of the trust relationships for which to obtain the information. If this member
         *        is null, all trust relationships that belong to the current account are returned.</p>
         *        <p>
         *        An empty list results in an <code>InvalidParameterException</code> being thrown.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trustIds(String... trustIds);

        /**
         * <p>
         * The <i>DescribeTrustsResult.NextToken</i> value from a previous call to <a>DescribeTrusts</a>. Pass null if
         * this is the first call.
         * </p>
         * 
         * @param nextToken
         *        The <i>DescribeTrustsResult.NextToken</i> value from a previous call to <a>DescribeTrusts</a>. Pass
         *        null if this is the first call.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextToken(String nextToken);

        /**
         * <p>
         * The maximum number of objects to return.
         * </p>
         * 
         * @param limit
         *        The maximum number of objects to return.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder limit(Integer limit);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DirectoryRequest.BuilderImpl implements Builder {
        private String directoryId;

        private List<String> trustIds = DefaultSdkAutoConstructList.getInstance();

        private String nextToken;

        private Integer limit;

        private BuilderImpl() {
        }

        private BuilderImpl(DescribeTrustsRequest model) {
            super(model);
            directoryId(model.directoryId);
            trustIds(model.trustIds);
            nextToken(model.nextToken);
            limit(model.limit);
        }

        public final String getDirectoryId() {
            return directoryId;
        }

        @Override
        public final Builder directoryId(String directoryId) {
            this.directoryId = directoryId;
            return this;
        }

        public final void setDirectoryId(String directoryId) {
            this.directoryId = directoryId;
        }

        public final Collection<String> getTrustIds() {
            if (trustIds instanceof SdkAutoConstructList) {
                return null;
            }
            return trustIds;
        }

        @Override
        public final Builder trustIds(Collection<String> trustIds) {
            this.trustIds = TrustIdsCopier.copy(trustIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder trustIds(String... trustIds) {
            trustIds(Arrays.asList(trustIds));
            return this;
        }

        public final void setTrustIds(Collection<String> trustIds) {
            this.trustIds = TrustIdsCopier.copy(trustIds);
        }

        public final String getNextToken() {
            return nextToken;
        }

        @Override
        public final Builder nextToken(String nextToken) {
            this.nextToken = nextToken;
            return this;
        }

        public final void setNextToken(String nextToken) {
            this.nextToken = nextToken;
        }

        public final Integer getLimit() {
            return limit;
        }

        @Override
        public final Builder limit(Integer limit) {
            this.limit = limit;
            return this;
        }

        public final void setLimit(Integer limit) {
            this.limit = limit;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public DescribeTrustsRequest build() {
            return new DescribeTrustsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
