/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directory.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about an Directory Service directory.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DirectoryDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<DirectoryDescription.Builder, DirectoryDescription> {
    private static final SdkField<String> DIRECTORY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DirectoryId").getter(getter(DirectoryDescription::directoryId)).setter(setter(Builder::directoryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DirectoryId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(DirectoryDescription::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> SHORT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ShortName").getter(getter(DirectoryDescription::shortName)).setter(setter(Builder::shortName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ShortName").build()).build();

    private static final SdkField<String> SIZE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Size")
            .getter(getter(DirectoryDescription::sizeAsString)).setter(setter(Builder::size))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Size").build()).build();

    private static final SdkField<String> EDITION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Edition")
            .getter(getter(DirectoryDescription::editionAsString)).setter(setter(Builder::edition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Edition").build()).build();

    private static final SdkField<String> ALIAS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Alias")
            .getter(getter(DirectoryDescription::alias)).setter(setter(Builder::alias))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Alias").build()).build();

    private static final SdkField<String> ACCESS_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccessUrl").getter(getter(DirectoryDescription::accessUrl)).setter(setter(Builder::accessUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessUrl").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(DirectoryDescription::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<String>> DNS_IP_ADDRS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DnsIpAddrs")
            .getter(getter(DirectoryDescription::dnsIpAddrs))
            .setter(setter(Builder::dnsIpAddrs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsIpAddrs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Stage")
            .getter(getter(DirectoryDescription::stageAsString)).setter(setter(Builder::stage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Stage").build()).build();

    private static final SdkField<String> SHARE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ShareStatus").getter(getter(DirectoryDescription::shareStatusAsString))
            .setter(setter(Builder::shareStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ShareStatus").build()).build();

    private static final SdkField<String> SHARE_METHOD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ShareMethod").getter(getter(DirectoryDescription::shareMethodAsString))
            .setter(setter(Builder::shareMethod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ShareMethod").build()).build();

    private static final SdkField<String> SHARE_NOTES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ShareNotes").getter(getter(DirectoryDescription::shareNotes)).setter(setter(Builder::shareNotes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ShareNotes").build()).build();

    private static final SdkField<Instant> LAUNCH_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LaunchTime").getter(getter(DirectoryDescription::launchTime)).setter(setter(Builder::launchTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LaunchTime").build()).build();

    private static final SdkField<Instant> STAGE_LAST_UPDATED_DATE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT).memberName("StageLastUpdatedDateTime")
            .getter(getter(DirectoryDescription::stageLastUpdatedDateTime)).setter(setter(Builder::stageLastUpdatedDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StageLastUpdatedDateTime").build())
            .build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(DirectoryDescription::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<DirectoryVpcSettingsDescription> VPC_SETTINGS_FIELD = SdkField
            .<DirectoryVpcSettingsDescription> builder(MarshallingType.SDK_POJO).memberName("VpcSettings")
            .getter(getter(DirectoryDescription::vpcSettings)).setter(setter(Builder::vpcSettings))
            .constructor(DirectoryVpcSettingsDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcSettings").build()).build();

    private static final SdkField<DirectoryConnectSettingsDescription> CONNECT_SETTINGS_FIELD = SdkField
            .<DirectoryConnectSettingsDescription> builder(MarshallingType.SDK_POJO).memberName("ConnectSettings")
            .getter(getter(DirectoryDescription::connectSettings)).setter(setter(Builder::connectSettings))
            .constructor(DirectoryConnectSettingsDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectSettings").build()).build();

    private static final SdkField<RadiusSettings> RADIUS_SETTINGS_FIELD = SdkField
            .<RadiusSettings> builder(MarshallingType.SDK_POJO).memberName("RadiusSettings")
            .getter(getter(DirectoryDescription::radiusSettings)).setter(setter(Builder::radiusSettings))
            .constructor(RadiusSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RadiusSettings").build()).build();

    private static final SdkField<String> RADIUS_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RadiusStatus").getter(getter(DirectoryDescription::radiusStatusAsString))
            .setter(setter(Builder::radiusStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RadiusStatus").build()).build();

    private static final SdkField<String> STAGE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StageReason").getter(getter(DirectoryDescription::stageReason)).setter(setter(Builder::stageReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StageReason").build()).build();

    private static final SdkField<Boolean> SSO_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SsoEnabled").getter(getter(DirectoryDescription::ssoEnabled)).setter(setter(Builder::ssoEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SsoEnabled").build()).build();

    private static final SdkField<Integer> DESIRED_NUMBER_OF_DOMAIN_CONTROLLERS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("DesiredNumberOfDomainControllers")
            .getter(getter(DirectoryDescription::desiredNumberOfDomainControllers))
            .setter(setter(Builder::desiredNumberOfDomainControllers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DesiredNumberOfDomainControllers")
                    .build()).build();

    private static final SdkField<OwnerDirectoryDescription> OWNER_DIRECTORY_DESCRIPTION_FIELD = SdkField
            .<OwnerDirectoryDescription> builder(MarshallingType.SDK_POJO).memberName("OwnerDirectoryDescription")
            .getter(getter(DirectoryDescription::ownerDirectoryDescription)).setter(setter(Builder::ownerDirectoryDescription))
            .constructor(OwnerDirectoryDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerDirectoryDescription").build())
            .build();

    private static final SdkField<RegionsInfo> REGIONS_INFO_FIELD = SdkField.<RegionsInfo> builder(MarshallingType.SDK_POJO)
            .memberName("RegionsInfo").getter(getter(DirectoryDescription::regionsInfo)).setter(setter(Builder::regionsInfo))
            .constructor(RegionsInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RegionsInfo").build()).build();

    private static final SdkField<String> OS_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OsVersion").getter(getter(DirectoryDescription::osVersionAsString)).setter(setter(Builder::osVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OsVersion").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DIRECTORY_ID_FIELD,
            NAME_FIELD, SHORT_NAME_FIELD, SIZE_FIELD, EDITION_FIELD, ALIAS_FIELD, ACCESS_URL_FIELD, DESCRIPTION_FIELD,
            DNS_IP_ADDRS_FIELD, STAGE_FIELD, SHARE_STATUS_FIELD, SHARE_METHOD_FIELD, SHARE_NOTES_FIELD, LAUNCH_TIME_FIELD,
            STAGE_LAST_UPDATED_DATE_TIME_FIELD, TYPE_FIELD, VPC_SETTINGS_FIELD, CONNECT_SETTINGS_FIELD, RADIUS_SETTINGS_FIELD,
            RADIUS_STATUS_FIELD, STAGE_REASON_FIELD, SSO_ENABLED_FIELD, DESIRED_NUMBER_OF_DOMAIN_CONTROLLERS_FIELD,
            OWNER_DIRECTORY_DESCRIPTION_FIELD, REGIONS_INFO_FIELD, OS_VERSION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String directoryId;

    private final String name;

    private final String shortName;

    private final String size;

    private final String edition;

    private final String alias;

    private final String accessUrl;

    private final String description;

    private final List<String> dnsIpAddrs;

    private final String stage;

    private final String shareStatus;

    private final String shareMethod;

    private final String shareNotes;

    private final Instant launchTime;

    private final Instant stageLastUpdatedDateTime;

    private final String type;

    private final DirectoryVpcSettingsDescription vpcSettings;

    private final DirectoryConnectSettingsDescription connectSettings;

    private final RadiusSettings radiusSettings;

    private final String radiusStatus;

    private final String stageReason;

    private final Boolean ssoEnabled;

    private final Integer desiredNumberOfDomainControllers;

    private final OwnerDirectoryDescription ownerDirectoryDescription;

    private final RegionsInfo regionsInfo;

    private final String osVersion;

    private DirectoryDescription(BuilderImpl builder) {
        this.directoryId = builder.directoryId;
        this.name = builder.name;
        this.shortName = builder.shortName;
        this.size = builder.size;
        this.edition = builder.edition;
        this.alias = builder.alias;
        this.accessUrl = builder.accessUrl;
        this.description = builder.description;
        this.dnsIpAddrs = builder.dnsIpAddrs;
        this.stage = builder.stage;
        this.shareStatus = builder.shareStatus;
        this.shareMethod = builder.shareMethod;
        this.shareNotes = builder.shareNotes;
        this.launchTime = builder.launchTime;
        this.stageLastUpdatedDateTime = builder.stageLastUpdatedDateTime;
        this.type = builder.type;
        this.vpcSettings = builder.vpcSettings;
        this.connectSettings = builder.connectSettings;
        this.radiusSettings = builder.radiusSettings;
        this.radiusStatus = builder.radiusStatus;
        this.stageReason = builder.stageReason;
        this.ssoEnabled = builder.ssoEnabled;
        this.desiredNumberOfDomainControllers = builder.desiredNumberOfDomainControllers;
        this.ownerDirectoryDescription = builder.ownerDirectoryDescription;
        this.regionsInfo = builder.regionsInfo;
        this.osVersion = builder.osVersion;
    }

    /**
     * <p>
     * The directory identifier.
     * </p>
     * 
     * @return The directory identifier.
     */
    public final String directoryId() {
        return directoryId;
    }

    /**
     * <p>
     * The fully qualified name of the directory.
     * </p>
     * 
     * @return The fully qualified name of the directory.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The short name of the directory.
     * </p>
     * 
     * @return The short name of the directory.
     */
    public final String shortName() {
        return shortName;
    }

    /**
     * <p>
     * The directory size.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #size} will return
     * {@link DirectorySize#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sizeAsString}.
     * </p>
     * 
     * @return The directory size.
     * @see DirectorySize
     */
    public final DirectorySize size() {
        return DirectorySize.fromValue(size);
    }

    /**
     * <p>
     * The directory size.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #size} will return
     * {@link DirectorySize#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sizeAsString}.
     * </p>
     * 
     * @return The directory size.
     * @see DirectorySize
     */
    public final String sizeAsString() {
        return size;
    }

    /**
     * <p>
     * The edition associated with this directory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #edition} will
     * return {@link DirectoryEdition#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #editionAsString}.
     * </p>
     * 
     * @return The edition associated with this directory.
     * @see DirectoryEdition
     */
    public final DirectoryEdition edition() {
        return DirectoryEdition.fromValue(edition);
    }

    /**
     * <p>
     * The edition associated with this directory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #edition} will
     * return {@link DirectoryEdition#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #editionAsString}.
     * </p>
     * 
     * @return The edition associated with this directory.
     * @see DirectoryEdition
     */
    public final String editionAsString() {
        return edition;
    }

    /**
     * <p>
     * The alias for the directory. If no alias has been created for the directory, the alias is the directory
     * identifier, such as <code>d-XXXXXXXXXX</code>.
     * </p>
     * 
     * @return The alias for the directory. If no alias has been created for the directory, the alias is the directory
     *         identifier, such as <code>d-XXXXXXXXXX</code>.
     */
    public final String alias() {
        return alias;
    }

    /**
     * <p>
     * The access URL for the directory, such as <code>http://&lt;alias&gt;.awsapps.com</code>. If no alias has been
     * created for the directory, <code>&lt;alias&gt;</code> is the directory identifier, such as
     * <code>d-XXXXXXXXXX</code>.
     * </p>
     * 
     * @return The access URL for the directory, such as <code>http://&lt;alias&gt;.awsapps.com</code>. If no alias has
     *         been created for the directory, <code>&lt;alias&gt;</code> is the directory identifier, such as
     *         <code>d-XXXXXXXXXX</code>.
     */
    public final String accessUrl() {
        return accessUrl;
    }

    /**
     * <p>
     * The description for the directory.
     * </p>
     * 
     * @return The description for the directory.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the DnsIpAddrs property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDnsIpAddrs() {
        return dnsIpAddrs != null && !(dnsIpAddrs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IP addresses of the DNS servers for the directory. For a Simple AD or Microsoft AD directory, these are the
     * IP addresses of the Simple AD or Microsoft AD directory servers. For an AD Connector directory, these are the IP
     * addresses of the DNS servers or domain controllers in your self-managed directory to which the AD Connector is
     * connected.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDnsIpAddrs} method.
     * </p>
     * 
     * @return The IP addresses of the DNS servers for the directory. For a Simple AD or Microsoft AD directory, these
     *         are the IP addresses of the Simple AD or Microsoft AD directory servers. For an AD Connector directory,
     *         these are the IP addresses of the DNS servers or domain controllers in your self-managed directory to
     *         which the AD Connector is connected.
     */
    public final List<String> dnsIpAddrs() {
        return dnsIpAddrs;
    }

    /**
     * <p>
     * The current stage of the directory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stage} will return
     * {@link DirectoryStage#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stageAsString}.
     * </p>
     * 
     * @return The current stage of the directory.
     * @see DirectoryStage
     */
    public final DirectoryStage stage() {
        return DirectoryStage.fromValue(stage);
    }

    /**
     * <p>
     * The current stage of the directory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stage} will return
     * {@link DirectoryStage#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stageAsString}.
     * </p>
     * 
     * @return The current stage of the directory.
     * @see DirectoryStage
     */
    public final String stageAsString() {
        return stage;
    }

    /**
     * <p>
     * Current directory status of the shared Managed Microsoft AD directory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #shareStatus} will
     * return {@link ShareStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #shareStatusAsString}.
     * </p>
     * 
     * @return Current directory status of the shared Managed Microsoft AD directory.
     * @see ShareStatus
     */
    public final ShareStatus shareStatus() {
        return ShareStatus.fromValue(shareStatus);
    }

    /**
     * <p>
     * Current directory status of the shared Managed Microsoft AD directory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #shareStatus} will
     * return {@link ShareStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #shareStatusAsString}.
     * </p>
     * 
     * @return Current directory status of the shared Managed Microsoft AD directory.
     * @see ShareStatus
     */
    public final String shareStatusAsString() {
        return shareStatus;
    }

    /**
     * <p>
     * The method used when sharing a directory to determine whether the directory should be shared within your Amazon
     * Web Services organization (<code>ORGANIZATIONS</code>) or with any Amazon Web Services account by sending a
     * shared directory request (<code>HANDSHAKE</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #shareMethod} will
     * return {@link ShareMethod#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #shareMethodAsString}.
     * </p>
     * 
     * @return The method used when sharing a directory to determine whether the directory should be shared within your
     *         Amazon Web Services organization (<code>ORGANIZATIONS</code>) or with any Amazon Web Services account by
     *         sending a shared directory request (<code>HANDSHAKE</code>).
     * @see ShareMethod
     */
    public final ShareMethod shareMethod() {
        return ShareMethod.fromValue(shareMethod);
    }

    /**
     * <p>
     * The method used when sharing a directory to determine whether the directory should be shared within your Amazon
     * Web Services organization (<code>ORGANIZATIONS</code>) or with any Amazon Web Services account by sending a
     * shared directory request (<code>HANDSHAKE</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #shareMethod} will
     * return {@link ShareMethod#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #shareMethodAsString}.
     * </p>
     * 
     * @return The method used when sharing a directory to determine whether the directory should be shared within your
     *         Amazon Web Services organization (<code>ORGANIZATIONS</code>) or with any Amazon Web Services account by
     *         sending a shared directory request (<code>HANDSHAKE</code>).
     * @see ShareMethod
     */
    public final String shareMethodAsString() {
        return shareMethod;
    }

    /**
     * <p>
     * A directory share request that is sent by the directory owner to the directory consumer. The request includes a
     * typed message to help the directory consumer administrator determine whether to approve or reject the share
     * invitation.
     * </p>
     * 
     * @return A directory share request that is sent by the directory owner to the directory consumer. The request
     *         includes a typed message to help the directory consumer administrator determine whether to approve or
     *         reject the share invitation.
     */
    public final String shareNotes() {
        return shareNotes;
    }

    /**
     * <p>
     * Specifies when the directory was created.
     * </p>
     * 
     * @return Specifies when the directory was created.
     */
    public final Instant launchTime() {
        return launchTime;
    }

    /**
     * <p>
     * The date and time that the stage was last updated.
     * </p>
     * 
     * @return The date and time that the stage was last updated.
     */
    public final Instant stageLastUpdatedDateTime() {
        return stageLastUpdatedDateTime;
    }

    /**
     * <p>
     * The directory size.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DirectoryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The directory size.
     * @see DirectoryType
     */
    public final DirectoryType type() {
        return DirectoryType.fromValue(type);
    }

    /**
     * <p>
     * The directory size.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DirectoryType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The directory size.
     * @see DirectoryType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * A <a>DirectoryVpcSettingsDescription</a> object that contains additional information about a directory. This
     * member is only present if the directory is a Simple AD or Managed Microsoft AD directory.
     * </p>
     * 
     * @return A <a>DirectoryVpcSettingsDescription</a> object that contains additional information about a directory.
     *         This member is only present if the directory is a Simple AD or Managed Microsoft AD directory.
     */
    public final DirectoryVpcSettingsDescription vpcSettings() {
        return vpcSettings;
    }

    /**
     * <p>
     * A <a>DirectoryConnectSettingsDescription</a> object that contains additional information about an AD Connector
     * directory. This member is only present if the directory is an AD Connector directory.
     * </p>
     * 
     * @return A <a>DirectoryConnectSettingsDescription</a> object that contains additional information about an AD
     *         Connector directory. This member is only present if the directory is an AD Connector directory.
     */
    public final DirectoryConnectSettingsDescription connectSettings() {
        return connectSettings;
    }

    /**
     * <p>
     * A <a>RadiusSettings</a> object that contains information about the RADIUS server configured for this directory.
     * </p>
     * 
     * @return A <a>RadiusSettings</a> object that contains information about the RADIUS server configured for this
     *         directory.
     */
    public final RadiusSettings radiusSettings() {
        return radiusSettings;
    }

    /**
     * <p>
     * The status of the RADIUS MFA server connection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #radiusStatus} will
     * return {@link RadiusStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #radiusStatusAsString}.
     * </p>
     * 
     * @return The status of the RADIUS MFA server connection.
     * @see RadiusStatus
     */
    public final RadiusStatus radiusStatus() {
        return RadiusStatus.fromValue(radiusStatus);
    }

    /**
     * <p>
     * The status of the RADIUS MFA server connection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #radiusStatus} will
     * return {@link RadiusStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #radiusStatusAsString}.
     * </p>
     * 
     * @return The status of the RADIUS MFA server connection.
     * @see RadiusStatus
     */
    public final String radiusStatusAsString() {
        return radiusStatus;
    }

    /**
     * <p>
     * Additional information about the directory stage.
     * </p>
     * 
     * @return Additional information about the directory stage.
     */
    public final String stageReason() {
        return stageReason;
    }

    /**
     * <p>
     * Indicates if single sign-on is enabled for the directory. For more information, see <a>EnableSso</a> and
     * <a>DisableSso</a>.
     * </p>
     * 
     * @return Indicates if single sign-on is enabled for the directory. For more information, see <a>EnableSso</a> and
     *         <a>DisableSso</a>.
     */
    public final Boolean ssoEnabled() {
        return ssoEnabled;
    }

    /**
     * <p>
     * The desired number of domain controllers in the directory if the directory is Microsoft AD.
     * </p>
     * 
     * @return The desired number of domain controllers in the directory if the directory is Microsoft AD.
     */
    public final Integer desiredNumberOfDomainControllers() {
        return desiredNumberOfDomainControllers;
    }

    /**
     * <p>
     * Describes the Managed Microsoft AD directory in the directory owner account.
     * </p>
     * 
     * @return Describes the Managed Microsoft AD directory in the directory owner account.
     */
    public final OwnerDirectoryDescription ownerDirectoryDescription() {
        return ownerDirectoryDescription;
    }

    /**
     * <p>
     * Lists the Regions where the directory has replicated.
     * </p>
     * 
     * @return Lists the Regions where the directory has replicated.
     */
    public final RegionsInfo regionsInfo() {
        return regionsInfo;
    }

    /**
     * <p>
     * The operating system (OS) version of the directory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #osVersion} will
     * return {@link OSVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #osVersionAsString}.
     * </p>
     * 
     * @return The operating system (OS) version of the directory.
     * @see OSVersion
     */
    public final OSVersion osVersion() {
        return OSVersion.fromValue(osVersion);
    }

    /**
     * <p>
     * The operating system (OS) version of the directory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #osVersion} will
     * return {@link OSVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #osVersionAsString}.
     * </p>
     * 
     * @return The operating system (OS) version of the directory.
     * @see OSVersion
     */
    public final String osVersionAsString() {
        return osVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(directoryId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(shortName());
        hashCode = 31 * hashCode + Objects.hashCode(sizeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(editionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(alias());
        hashCode = 31 * hashCode + Objects.hashCode(accessUrl());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasDnsIpAddrs() ? dnsIpAddrs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stageAsString());
        hashCode = 31 * hashCode + Objects.hashCode(shareStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(shareMethodAsString());
        hashCode = 31 * hashCode + Objects.hashCode(shareNotes());
        hashCode = 31 * hashCode + Objects.hashCode(launchTime());
        hashCode = 31 * hashCode + Objects.hashCode(stageLastUpdatedDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpcSettings());
        hashCode = 31 * hashCode + Objects.hashCode(connectSettings());
        hashCode = 31 * hashCode + Objects.hashCode(radiusSettings());
        hashCode = 31 * hashCode + Objects.hashCode(radiusStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stageReason());
        hashCode = 31 * hashCode + Objects.hashCode(ssoEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(desiredNumberOfDomainControllers());
        hashCode = 31 * hashCode + Objects.hashCode(ownerDirectoryDescription());
        hashCode = 31 * hashCode + Objects.hashCode(regionsInfo());
        hashCode = 31 * hashCode + Objects.hashCode(osVersionAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DirectoryDescription)) {
            return false;
        }
        DirectoryDescription other = (DirectoryDescription) obj;
        return Objects.equals(directoryId(), other.directoryId()) && Objects.equals(name(), other.name())
                && Objects.equals(shortName(), other.shortName()) && Objects.equals(sizeAsString(), other.sizeAsString())
                && Objects.equals(editionAsString(), other.editionAsString()) && Objects.equals(alias(), other.alias())
                && Objects.equals(accessUrl(), other.accessUrl()) && Objects.equals(description(), other.description())
                && hasDnsIpAddrs() == other.hasDnsIpAddrs() && Objects.equals(dnsIpAddrs(), other.dnsIpAddrs())
                && Objects.equals(stageAsString(), other.stageAsString())
                && Objects.equals(shareStatusAsString(), other.shareStatusAsString())
                && Objects.equals(shareMethodAsString(), other.shareMethodAsString())
                && Objects.equals(shareNotes(), other.shareNotes()) && Objects.equals(launchTime(), other.launchTime())
                && Objects.equals(stageLastUpdatedDateTime(), other.stageLastUpdatedDateTime())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(vpcSettings(), other.vpcSettings())
                && Objects.equals(connectSettings(), other.connectSettings())
                && Objects.equals(radiusSettings(), other.radiusSettings())
                && Objects.equals(radiusStatusAsString(), other.radiusStatusAsString())
                && Objects.equals(stageReason(), other.stageReason()) && Objects.equals(ssoEnabled(), other.ssoEnabled())
                && Objects.equals(desiredNumberOfDomainControllers(), other.desiredNumberOfDomainControllers())
                && Objects.equals(ownerDirectoryDescription(), other.ownerDirectoryDescription())
                && Objects.equals(regionsInfo(), other.regionsInfo())
                && Objects.equals(osVersionAsString(), other.osVersionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DirectoryDescription").add("DirectoryId", directoryId()).add("Name", name())
                .add("ShortName", shortName()).add("Size", sizeAsString()).add("Edition", editionAsString())
                .add("Alias", alias()).add("AccessUrl", accessUrl()).add("Description", description())
                .add("DnsIpAddrs", hasDnsIpAddrs() ? dnsIpAddrs() : null).add("Stage", stageAsString())
                .add("ShareStatus", shareStatusAsString()).add("ShareMethod", shareMethodAsString())
                .add("ShareNotes", shareNotes() == null ? null : "*** Sensitive Data Redacted ***")
                .add("LaunchTime", launchTime()).add("StageLastUpdatedDateTime", stageLastUpdatedDateTime())
                .add("Type", typeAsString()).add("VpcSettings", vpcSettings()).add("ConnectSettings", connectSettings())
                .add("RadiusSettings", radiusSettings()).add("RadiusStatus", radiusStatusAsString())
                .add("StageReason", stageReason()).add("SsoEnabled", ssoEnabled())
                .add("DesiredNumberOfDomainControllers", desiredNumberOfDomainControllers())
                .add("OwnerDirectoryDescription", ownerDirectoryDescription()).add("RegionsInfo", regionsInfo())
                .add("OsVersion", osVersionAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DirectoryId":
            return Optional.ofNullable(clazz.cast(directoryId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ShortName":
            return Optional.ofNullable(clazz.cast(shortName()));
        case "Size":
            return Optional.ofNullable(clazz.cast(sizeAsString()));
        case "Edition":
            return Optional.ofNullable(clazz.cast(editionAsString()));
        case "Alias":
            return Optional.ofNullable(clazz.cast(alias()));
        case "AccessUrl":
            return Optional.ofNullable(clazz.cast(accessUrl()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "DnsIpAddrs":
            return Optional.ofNullable(clazz.cast(dnsIpAddrs()));
        case "Stage":
            return Optional.ofNullable(clazz.cast(stageAsString()));
        case "ShareStatus":
            return Optional.ofNullable(clazz.cast(shareStatusAsString()));
        case "ShareMethod":
            return Optional.ofNullable(clazz.cast(shareMethodAsString()));
        case "ShareNotes":
            return Optional.ofNullable(clazz.cast(shareNotes()));
        case "LaunchTime":
            return Optional.ofNullable(clazz.cast(launchTime()));
        case "StageLastUpdatedDateTime":
            return Optional.ofNullable(clazz.cast(stageLastUpdatedDateTime()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "VpcSettings":
            return Optional.ofNullable(clazz.cast(vpcSettings()));
        case "ConnectSettings":
            return Optional.ofNullable(clazz.cast(connectSettings()));
        case "RadiusSettings":
            return Optional.ofNullable(clazz.cast(radiusSettings()));
        case "RadiusStatus":
            return Optional.ofNullable(clazz.cast(radiusStatusAsString()));
        case "StageReason":
            return Optional.ofNullable(clazz.cast(stageReason()));
        case "SsoEnabled":
            return Optional.ofNullable(clazz.cast(ssoEnabled()));
        case "DesiredNumberOfDomainControllers":
            return Optional.ofNullable(clazz.cast(desiredNumberOfDomainControllers()));
        case "OwnerDirectoryDescription":
            return Optional.ofNullable(clazz.cast(ownerDirectoryDescription()));
        case "RegionsInfo":
            return Optional.ofNullable(clazz.cast(regionsInfo()));
        case "OsVersion":
            return Optional.ofNullable(clazz.cast(osVersionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DirectoryDescription, T> g) {
        return obj -> g.apply((DirectoryDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DirectoryDescription> {
        /**
         * <p>
         * The directory identifier.
         * </p>
         * 
         * @param directoryId
         *        The directory identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder directoryId(String directoryId);

        /**
         * <p>
         * The fully qualified name of the directory.
         * </p>
         * 
         * @param name
         *        The fully qualified name of the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The short name of the directory.
         * </p>
         * 
         * @param shortName
         *        The short name of the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shortName(String shortName);

        /**
         * <p>
         * The directory size.
         * </p>
         * 
         * @param size
         *        The directory size.
         * @see DirectorySize
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DirectorySize
         */
        Builder size(String size);

        /**
         * <p>
         * The directory size.
         * </p>
         * 
         * @param size
         *        The directory size.
         * @see DirectorySize
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DirectorySize
         */
        Builder size(DirectorySize size);

        /**
         * <p>
         * The edition associated with this directory.
         * </p>
         * 
         * @param edition
         *        The edition associated with this directory.
         * @see DirectoryEdition
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DirectoryEdition
         */
        Builder edition(String edition);

        /**
         * <p>
         * The edition associated with this directory.
         * </p>
         * 
         * @param edition
         *        The edition associated with this directory.
         * @see DirectoryEdition
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DirectoryEdition
         */
        Builder edition(DirectoryEdition edition);

        /**
         * <p>
         * The alias for the directory. If no alias has been created for the directory, the alias is the directory
         * identifier, such as <code>d-XXXXXXXXXX</code>.
         * </p>
         * 
         * @param alias
         *        The alias for the directory. If no alias has been created for the directory, the alias is the
         *        directory identifier, such as <code>d-XXXXXXXXXX</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alias(String alias);

        /**
         * <p>
         * The access URL for the directory, such as <code>http://&lt;alias&gt;.awsapps.com</code>. If no alias has been
         * created for the directory, <code>&lt;alias&gt;</code> is the directory identifier, such as
         * <code>d-XXXXXXXXXX</code>.
         * </p>
         * 
         * @param accessUrl
         *        The access URL for the directory, such as <code>http://&lt;alias&gt;.awsapps.com</code>. If no alias
         *        has been created for the directory, <code>&lt;alias&gt;</code> is the directory identifier, such as
         *        <code>d-XXXXXXXXXX</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessUrl(String accessUrl);

        /**
         * <p>
         * The description for the directory.
         * </p>
         * 
         * @param description
         *        The description for the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The IP addresses of the DNS servers for the directory. For a Simple AD or Microsoft AD directory, these are
         * the IP addresses of the Simple AD or Microsoft AD directory servers. For an AD Connector directory, these are
         * the IP addresses of the DNS servers or domain controllers in your self-managed directory to which the AD
         * Connector is connected.
         * </p>
         * 
         * @param dnsIpAddrs
         *        The IP addresses of the DNS servers for the directory. For a Simple AD or Microsoft AD directory,
         *        these are the IP addresses of the Simple AD or Microsoft AD directory servers. For an AD Connector
         *        directory, these are the IP addresses of the DNS servers or domain controllers in your self-managed
         *        directory to which the AD Connector is connected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIpAddrs(Collection<String> dnsIpAddrs);

        /**
         * <p>
         * The IP addresses of the DNS servers for the directory. For a Simple AD or Microsoft AD directory, these are
         * the IP addresses of the Simple AD or Microsoft AD directory servers. For an AD Connector directory, these are
         * the IP addresses of the DNS servers or domain controllers in your self-managed directory to which the AD
         * Connector is connected.
         * </p>
         * 
         * @param dnsIpAddrs
         *        The IP addresses of the DNS servers for the directory. For a Simple AD or Microsoft AD directory,
         *        these are the IP addresses of the Simple AD or Microsoft AD directory servers. For an AD Connector
         *        directory, these are the IP addresses of the DNS servers or domain controllers in your self-managed
         *        directory to which the AD Connector is connected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIpAddrs(String... dnsIpAddrs);

        /**
         * <p>
         * The current stage of the directory.
         * </p>
         * 
         * @param stage
         *        The current stage of the directory.
         * @see DirectoryStage
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DirectoryStage
         */
        Builder stage(String stage);

        /**
         * <p>
         * The current stage of the directory.
         * </p>
         * 
         * @param stage
         *        The current stage of the directory.
         * @see DirectoryStage
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DirectoryStage
         */
        Builder stage(DirectoryStage stage);

        /**
         * <p>
         * Current directory status of the shared Managed Microsoft AD directory.
         * </p>
         * 
         * @param shareStatus
         *        Current directory status of the shared Managed Microsoft AD directory.
         * @see ShareStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ShareStatus
         */
        Builder shareStatus(String shareStatus);

        /**
         * <p>
         * Current directory status of the shared Managed Microsoft AD directory.
         * </p>
         * 
         * @param shareStatus
         *        Current directory status of the shared Managed Microsoft AD directory.
         * @see ShareStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ShareStatus
         */
        Builder shareStatus(ShareStatus shareStatus);

        /**
         * <p>
         * The method used when sharing a directory to determine whether the directory should be shared within your
         * Amazon Web Services organization (<code>ORGANIZATIONS</code>) or with any Amazon Web Services account by
         * sending a shared directory request (<code>HANDSHAKE</code>).
         * </p>
         * 
         * @param shareMethod
         *        The method used when sharing a directory to determine whether the directory should be shared within
         *        your Amazon Web Services organization (<code>ORGANIZATIONS</code>) or with any Amazon Web Services
         *        account by sending a shared directory request (<code>HANDSHAKE</code>).
         * @see ShareMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ShareMethod
         */
        Builder shareMethod(String shareMethod);

        /**
         * <p>
         * The method used when sharing a directory to determine whether the directory should be shared within your
         * Amazon Web Services organization (<code>ORGANIZATIONS</code>) or with any Amazon Web Services account by
         * sending a shared directory request (<code>HANDSHAKE</code>).
         * </p>
         * 
         * @param shareMethod
         *        The method used when sharing a directory to determine whether the directory should be shared within
         *        your Amazon Web Services organization (<code>ORGANIZATIONS</code>) or with any Amazon Web Services
         *        account by sending a shared directory request (<code>HANDSHAKE</code>).
         * @see ShareMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ShareMethod
         */
        Builder shareMethod(ShareMethod shareMethod);

        /**
         * <p>
         * A directory share request that is sent by the directory owner to the directory consumer. The request includes
         * a typed message to help the directory consumer administrator determine whether to approve or reject the share
         * invitation.
         * </p>
         * 
         * @param shareNotes
         *        A directory share request that is sent by the directory owner to the directory consumer. The request
         *        includes a typed message to help the directory consumer administrator determine whether to approve or
         *        reject the share invitation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shareNotes(String shareNotes);

        /**
         * <p>
         * Specifies when the directory was created.
         * </p>
         * 
         * @param launchTime
         *        Specifies when the directory was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchTime(Instant launchTime);

        /**
         * <p>
         * The date and time that the stage was last updated.
         * </p>
         * 
         * @param stageLastUpdatedDateTime
         *        The date and time that the stage was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stageLastUpdatedDateTime(Instant stageLastUpdatedDateTime);

        /**
         * <p>
         * The directory size.
         * </p>
         * 
         * @param type
         *        The directory size.
         * @see DirectoryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DirectoryType
         */
        Builder type(String type);

        /**
         * <p>
         * The directory size.
         * </p>
         * 
         * @param type
         *        The directory size.
         * @see DirectoryType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DirectoryType
         */
        Builder type(DirectoryType type);

        /**
         * <p>
         * A <a>DirectoryVpcSettingsDescription</a> object that contains additional information about a directory. This
         * member is only present if the directory is a Simple AD or Managed Microsoft AD directory.
         * </p>
         * 
         * @param vpcSettings
         *        A <a>DirectoryVpcSettingsDescription</a> object that contains additional information about a
         *        directory. This member is only present if the directory is a Simple AD or Managed Microsoft AD
         *        directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSettings(DirectoryVpcSettingsDescription vpcSettings);

        /**
         * <p>
         * A <a>DirectoryVpcSettingsDescription</a> object that contains additional information about a directory. This
         * member is only present if the directory is a Simple AD or Managed Microsoft AD directory.
         * </p>
         * This is a convenience method that creates an instance of the {@link DirectoryVpcSettingsDescription.Builder}
         * avoiding the need to create one manually via {@link DirectoryVpcSettingsDescription#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DirectoryVpcSettingsDescription.Builder#build()} is called
         * immediately and its result is passed to {@link #vpcSettings(DirectoryVpcSettingsDescription)}.
         * 
         * @param vpcSettings
         *        a consumer that will call methods on {@link DirectoryVpcSettingsDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcSettings(DirectoryVpcSettingsDescription)
         */
        default Builder vpcSettings(Consumer<DirectoryVpcSettingsDescription.Builder> vpcSettings) {
            return vpcSettings(DirectoryVpcSettingsDescription.builder().applyMutation(vpcSettings).build());
        }

        /**
         * <p>
         * A <a>DirectoryConnectSettingsDescription</a> object that contains additional information about an AD
         * Connector directory. This member is only present if the directory is an AD Connector directory.
         * </p>
         * 
         * @param connectSettings
         *        A <a>DirectoryConnectSettingsDescription</a> object that contains additional information about an AD
         *        Connector directory. This member is only present if the directory is an AD Connector directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectSettings(DirectoryConnectSettingsDescription connectSettings);

        /**
         * <p>
         * A <a>DirectoryConnectSettingsDescription</a> object that contains additional information about an AD
         * Connector directory. This member is only present if the directory is an AD Connector directory.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link DirectoryConnectSettingsDescription.Builder} avoiding the need to create one manually via
         * {@link DirectoryConnectSettingsDescription#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DirectoryConnectSettingsDescription.Builder#build()} is called
         * immediately and its result is passed to {@link #connectSettings(DirectoryConnectSettingsDescription)}.
         * 
         * @param connectSettings
         *        a consumer that will call methods on {@link DirectoryConnectSettingsDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #connectSettings(DirectoryConnectSettingsDescription)
         */
        default Builder connectSettings(Consumer<DirectoryConnectSettingsDescription.Builder> connectSettings) {
            return connectSettings(DirectoryConnectSettingsDescription.builder().applyMutation(connectSettings).build());
        }

        /**
         * <p>
         * A <a>RadiusSettings</a> object that contains information about the RADIUS server configured for this
         * directory.
         * </p>
         * 
         * @param radiusSettings
         *        A <a>RadiusSettings</a> object that contains information about the RADIUS server configured for this
         *        directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder radiusSettings(RadiusSettings radiusSettings);

        /**
         * <p>
         * A <a>RadiusSettings</a> object that contains information about the RADIUS server configured for this
         * directory.
         * </p>
         * This is a convenience method that creates an instance of the {@link RadiusSettings.Builder} avoiding the need
         * to create one manually via {@link RadiusSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RadiusSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #radiusSettings(RadiusSettings)}.
         * 
         * @param radiusSettings
         *        a consumer that will call methods on {@link RadiusSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #radiusSettings(RadiusSettings)
         */
        default Builder radiusSettings(Consumer<RadiusSettings.Builder> radiusSettings) {
            return radiusSettings(RadiusSettings.builder().applyMutation(radiusSettings).build());
        }

        /**
         * <p>
         * The status of the RADIUS MFA server connection.
         * </p>
         * 
         * @param radiusStatus
         *        The status of the RADIUS MFA server connection.
         * @see RadiusStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RadiusStatus
         */
        Builder radiusStatus(String radiusStatus);

        /**
         * <p>
         * The status of the RADIUS MFA server connection.
         * </p>
         * 
         * @param radiusStatus
         *        The status of the RADIUS MFA server connection.
         * @see RadiusStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RadiusStatus
         */
        Builder radiusStatus(RadiusStatus radiusStatus);

        /**
         * <p>
         * Additional information about the directory stage.
         * </p>
         * 
         * @param stageReason
         *        Additional information about the directory stage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stageReason(String stageReason);

        /**
         * <p>
         * Indicates if single sign-on is enabled for the directory. For more information, see <a>EnableSso</a> and
         * <a>DisableSso</a>.
         * </p>
         * 
         * @param ssoEnabled
         *        Indicates if single sign-on is enabled for the directory. For more information, see <a>EnableSso</a>
         *        and <a>DisableSso</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ssoEnabled(Boolean ssoEnabled);

        /**
         * <p>
         * The desired number of domain controllers in the directory if the directory is Microsoft AD.
         * </p>
         * 
         * @param desiredNumberOfDomainControllers
         *        The desired number of domain controllers in the directory if the directory is Microsoft AD.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredNumberOfDomainControllers(Integer desiredNumberOfDomainControllers);

        /**
         * <p>
         * Describes the Managed Microsoft AD directory in the directory owner account.
         * </p>
         * 
         * @param ownerDirectoryDescription
         *        Describes the Managed Microsoft AD directory in the directory owner account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerDirectoryDescription(OwnerDirectoryDescription ownerDirectoryDescription);

        /**
         * <p>
         * Describes the Managed Microsoft AD directory in the directory owner account.
         * </p>
         * This is a convenience method that creates an instance of the {@link OwnerDirectoryDescription.Builder}
         * avoiding the need to create one manually via {@link OwnerDirectoryDescription#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OwnerDirectoryDescription.Builder#build()} is called immediately
         * and its result is passed to {@link #ownerDirectoryDescription(OwnerDirectoryDescription)}.
         * 
         * @param ownerDirectoryDescription
         *        a consumer that will call methods on {@link OwnerDirectoryDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ownerDirectoryDescription(OwnerDirectoryDescription)
         */
        default Builder ownerDirectoryDescription(Consumer<OwnerDirectoryDescription.Builder> ownerDirectoryDescription) {
            return ownerDirectoryDescription(OwnerDirectoryDescription.builder().applyMutation(ownerDirectoryDescription).build());
        }

        /**
         * <p>
         * Lists the Regions where the directory has replicated.
         * </p>
         * 
         * @param regionsInfo
         *        Lists the Regions where the directory has replicated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regionsInfo(RegionsInfo regionsInfo);

        /**
         * <p>
         * Lists the Regions where the directory has replicated.
         * </p>
         * This is a convenience method that creates an instance of the {@link RegionsInfo.Builder} avoiding the need to
         * create one manually via {@link RegionsInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RegionsInfo.Builder#build()} is called immediately and its result
         * is passed to {@link #regionsInfo(RegionsInfo)}.
         * 
         * @param regionsInfo
         *        a consumer that will call methods on {@link RegionsInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #regionsInfo(RegionsInfo)
         */
        default Builder regionsInfo(Consumer<RegionsInfo.Builder> regionsInfo) {
            return regionsInfo(RegionsInfo.builder().applyMutation(regionsInfo).build());
        }

        /**
         * <p>
         * The operating system (OS) version of the directory.
         * </p>
         * 
         * @param osVersion
         *        The operating system (OS) version of the directory.
         * @see OSVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OSVersion
         */
        Builder osVersion(String osVersion);

        /**
         * <p>
         * The operating system (OS) version of the directory.
         * </p>
         * 
         * @param osVersion
         *        The operating system (OS) version of the directory.
         * @see OSVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OSVersion
         */
        Builder osVersion(OSVersion osVersion);
    }

    static final class BuilderImpl implements Builder {
        private String directoryId;

        private String name;

        private String shortName;

        private String size;

        private String edition;

        private String alias;

        private String accessUrl;

        private String description;

        private List<String> dnsIpAddrs = DefaultSdkAutoConstructList.getInstance();

        private String stage;

        private String shareStatus;

        private String shareMethod;

        private String shareNotes;

        private Instant launchTime;

        private Instant stageLastUpdatedDateTime;

        private String type;

        private DirectoryVpcSettingsDescription vpcSettings;

        private DirectoryConnectSettingsDescription connectSettings;

        private RadiusSettings radiusSettings;

        private String radiusStatus;

        private String stageReason;

        private Boolean ssoEnabled;

        private Integer desiredNumberOfDomainControllers;

        private OwnerDirectoryDescription ownerDirectoryDescription;

        private RegionsInfo regionsInfo;

        private String osVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(DirectoryDescription model) {
            directoryId(model.directoryId);
            name(model.name);
            shortName(model.shortName);
            size(model.size);
            edition(model.edition);
            alias(model.alias);
            accessUrl(model.accessUrl);
            description(model.description);
            dnsIpAddrs(model.dnsIpAddrs);
            stage(model.stage);
            shareStatus(model.shareStatus);
            shareMethod(model.shareMethod);
            shareNotes(model.shareNotes);
            launchTime(model.launchTime);
            stageLastUpdatedDateTime(model.stageLastUpdatedDateTime);
            type(model.type);
            vpcSettings(model.vpcSettings);
            connectSettings(model.connectSettings);
            radiusSettings(model.radiusSettings);
            radiusStatus(model.radiusStatus);
            stageReason(model.stageReason);
            ssoEnabled(model.ssoEnabled);
            desiredNumberOfDomainControllers(model.desiredNumberOfDomainControllers);
            ownerDirectoryDescription(model.ownerDirectoryDescription);
            regionsInfo(model.regionsInfo);
            osVersion(model.osVersion);
        }

        public final String getDirectoryId() {
            return directoryId;
        }

        public final void setDirectoryId(String directoryId) {
            this.directoryId = directoryId;
        }

        @Override
        public final Builder directoryId(String directoryId) {
            this.directoryId = directoryId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getShortName() {
            return shortName;
        }

        public final void setShortName(String shortName) {
            this.shortName = shortName;
        }

        @Override
        public final Builder shortName(String shortName) {
            this.shortName = shortName;
            return this;
        }

        public final String getSize() {
            return size;
        }

        public final void setSize(String size) {
            this.size = size;
        }

        @Override
        public final Builder size(String size) {
            this.size = size;
            return this;
        }

        @Override
        public final Builder size(DirectorySize size) {
            this.size(size == null ? null : size.toString());
            return this;
        }

        public final String getEdition() {
            return edition;
        }

        public final void setEdition(String edition) {
            this.edition = edition;
        }

        @Override
        public final Builder edition(String edition) {
            this.edition = edition;
            return this;
        }

        @Override
        public final Builder edition(DirectoryEdition edition) {
            this.edition(edition == null ? null : edition.toString());
            return this;
        }

        public final String getAlias() {
            return alias;
        }

        public final void setAlias(String alias) {
            this.alias = alias;
        }

        @Override
        public final Builder alias(String alias) {
            this.alias = alias;
            return this;
        }

        public final String getAccessUrl() {
            return accessUrl;
        }

        public final void setAccessUrl(String accessUrl) {
            this.accessUrl = accessUrl;
        }

        @Override
        public final Builder accessUrl(String accessUrl) {
            this.accessUrl = accessUrl;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Collection<String> getDnsIpAddrs() {
            if (dnsIpAddrs instanceof SdkAutoConstructList) {
                return null;
            }
            return dnsIpAddrs;
        }

        public final void setDnsIpAddrs(Collection<String> dnsIpAddrs) {
            this.dnsIpAddrs = DnsIpAddrsCopier.copy(dnsIpAddrs);
        }

        @Override
        public final Builder dnsIpAddrs(Collection<String> dnsIpAddrs) {
            this.dnsIpAddrs = DnsIpAddrsCopier.copy(dnsIpAddrs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsIpAddrs(String... dnsIpAddrs) {
            dnsIpAddrs(Arrays.asList(dnsIpAddrs));
            return this;
        }

        public final String getStage() {
            return stage;
        }

        public final void setStage(String stage) {
            this.stage = stage;
        }

        @Override
        public final Builder stage(String stage) {
            this.stage = stage;
            return this;
        }

        @Override
        public final Builder stage(DirectoryStage stage) {
            this.stage(stage == null ? null : stage.toString());
            return this;
        }

        public final String getShareStatus() {
            return shareStatus;
        }

        public final void setShareStatus(String shareStatus) {
            this.shareStatus = shareStatus;
        }

        @Override
        public final Builder shareStatus(String shareStatus) {
            this.shareStatus = shareStatus;
            return this;
        }

        @Override
        public final Builder shareStatus(ShareStatus shareStatus) {
            this.shareStatus(shareStatus == null ? null : shareStatus.toString());
            return this;
        }

        public final String getShareMethod() {
            return shareMethod;
        }

        public final void setShareMethod(String shareMethod) {
            this.shareMethod = shareMethod;
        }

        @Override
        public final Builder shareMethod(String shareMethod) {
            this.shareMethod = shareMethod;
            return this;
        }

        @Override
        public final Builder shareMethod(ShareMethod shareMethod) {
            this.shareMethod(shareMethod == null ? null : shareMethod.toString());
            return this;
        }

        public final String getShareNotes() {
            return shareNotes;
        }

        public final void setShareNotes(String shareNotes) {
            this.shareNotes = shareNotes;
        }

        @Override
        public final Builder shareNotes(String shareNotes) {
            this.shareNotes = shareNotes;
            return this;
        }

        public final Instant getLaunchTime() {
            return launchTime;
        }

        public final void setLaunchTime(Instant launchTime) {
            this.launchTime = launchTime;
        }

        @Override
        public final Builder launchTime(Instant launchTime) {
            this.launchTime = launchTime;
            return this;
        }

        public final Instant getStageLastUpdatedDateTime() {
            return stageLastUpdatedDateTime;
        }

        public final void setStageLastUpdatedDateTime(Instant stageLastUpdatedDateTime) {
            this.stageLastUpdatedDateTime = stageLastUpdatedDateTime;
        }

        @Override
        public final Builder stageLastUpdatedDateTime(Instant stageLastUpdatedDateTime) {
            this.stageLastUpdatedDateTime = stageLastUpdatedDateTime;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(DirectoryType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final DirectoryVpcSettingsDescription.Builder getVpcSettings() {
            return vpcSettings != null ? vpcSettings.toBuilder() : null;
        }

        public final void setVpcSettings(DirectoryVpcSettingsDescription.BuilderImpl vpcSettings) {
            this.vpcSettings = vpcSettings != null ? vpcSettings.build() : null;
        }

        @Override
        public final Builder vpcSettings(DirectoryVpcSettingsDescription vpcSettings) {
            this.vpcSettings = vpcSettings;
            return this;
        }

        public final DirectoryConnectSettingsDescription.Builder getConnectSettings() {
            return connectSettings != null ? connectSettings.toBuilder() : null;
        }

        public final void setConnectSettings(DirectoryConnectSettingsDescription.BuilderImpl connectSettings) {
            this.connectSettings = connectSettings != null ? connectSettings.build() : null;
        }

        @Override
        public final Builder connectSettings(DirectoryConnectSettingsDescription connectSettings) {
            this.connectSettings = connectSettings;
            return this;
        }

        public final RadiusSettings.Builder getRadiusSettings() {
            return radiusSettings != null ? radiusSettings.toBuilder() : null;
        }

        public final void setRadiusSettings(RadiusSettings.BuilderImpl radiusSettings) {
            this.radiusSettings = radiusSettings != null ? radiusSettings.build() : null;
        }

        @Override
        public final Builder radiusSettings(RadiusSettings radiusSettings) {
            this.radiusSettings = radiusSettings;
            return this;
        }

        public final String getRadiusStatus() {
            return radiusStatus;
        }

        public final void setRadiusStatus(String radiusStatus) {
            this.radiusStatus = radiusStatus;
        }

        @Override
        public final Builder radiusStatus(String radiusStatus) {
            this.radiusStatus = radiusStatus;
            return this;
        }

        @Override
        public final Builder radiusStatus(RadiusStatus radiusStatus) {
            this.radiusStatus(radiusStatus == null ? null : radiusStatus.toString());
            return this;
        }

        public final String getStageReason() {
            return stageReason;
        }

        public final void setStageReason(String stageReason) {
            this.stageReason = stageReason;
        }

        @Override
        public final Builder stageReason(String stageReason) {
            this.stageReason = stageReason;
            return this;
        }

        public final Boolean getSsoEnabled() {
            return ssoEnabled;
        }

        public final void setSsoEnabled(Boolean ssoEnabled) {
            this.ssoEnabled = ssoEnabled;
        }

        @Override
        public final Builder ssoEnabled(Boolean ssoEnabled) {
            this.ssoEnabled = ssoEnabled;
            return this;
        }

        public final Integer getDesiredNumberOfDomainControllers() {
            return desiredNumberOfDomainControllers;
        }

        public final void setDesiredNumberOfDomainControllers(Integer desiredNumberOfDomainControllers) {
            this.desiredNumberOfDomainControllers = desiredNumberOfDomainControllers;
        }

        @Override
        public final Builder desiredNumberOfDomainControllers(Integer desiredNumberOfDomainControllers) {
            this.desiredNumberOfDomainControllers = desiredNumberOfDomainControllers;
            return this;
        }

        public final OwnerDirectoryDescription.Builder getOwnerDirectoryDescription() {
            return ownerDirectoryDescription != null ? ownerDirectoryDescription.toBuilder() : null;
        }

        public final void setOwnerDirectoryDescription(OwnerDirectoryDescription.BuilderImpl ownerDirectoryDescription) {
            this.ownerDirectoryDescription = ownerDirectoryDescription != null ? ownerDirectoryDescription.build() : null;
        }

        @Override
        public final Builder ownerDirectoryDescription(OwnerDirectoryDescription ownerDirectoryDescription) {
            this.ownerDirectoryDescription = ownerDirectoryDescription;
            return this;
        }

        public final RegionsInfo.Builder getRegionsInfo() {
            return regionsInfo != null ? regionsInfo.toBuilder() : null;
        }

        public final void setRegionsInfo(RegionsInfo.BuilderImpl regionsInfo) {
            this.regionsInfo = regionsInfo != null ? regionsInfo.build() : null;
        }

        @Override
        public final Builder regionsInfo(RegionsInfo regionsInfo) {
            this.regionsInfo = regionsInfo;
            return this;
        }

        public final String getOsVersion() {
            return osVersion;
        }

        public final void setOsVersion(String osVersion) {
            this.osVersion = osVersion;
        }

        @Override
        public final Builder osVersion(String osVersion) {
            this.osVersion = osVersion;
            return this;
        }

        @Override
        public final Builder osVersion(OSVersion osVersion) {
            this.osVersion(osVersion == null ? null : osVersion.toString());
            return this;
        }

        @Override
        public DirectoryDescription build() {
            return new DirectoryDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
