/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directory.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the inputs for the <a>CreateDirectory</a> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateDirectoryRequest extends DirectoryRequest implements
        ToCopyableBuilder<CreateDirectoryRequest.Builder, CreateDirectoryRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateDirectoryRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> SHORT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateDirectoryRequest::shortName)).setter(setter(Builder::shortName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ShortName").build()).build();

    private static final SdkField<String> PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateDirectoryRequest::password)).setter(setter(Builder::password))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Password").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateDirectoryRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> SIZE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateDirectoryRequest::sizeAsString)).setter(setter(Builder::size))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Size").build()).build();

    private static final SdkField<DirectoryVpcSettings> VPC_SETTINGS_FIELD = SdkField
            .<DirectoryVpcSettings> builder(MarshallingType.SDK_POJO).getter(getter(CreateDirectoryRequest::vpcSettings))
            .setter(setter(Builder::vpcSettings)).constructor(DirectoryVpcSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcSettings").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, SHORT_NAME_FIELD,
            PASSWORD_FIELD, DESCRIPTION_FIELD, SIZE_FIELD, VPC_SETTINGS_FIELD));

    private final String name;

    private final String shortName;

    private final String password;

    private final String description;

    private final String size;

    private final DirectoryVpcSettings vpcSettings;

    private CreateDirectoryRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.shortName = builder.shortName;
        this.password = builder.password;
        this.description = builder.description;
        this.size = builder.size;
        this.vpcSettings = builder.vpcSettings;
    }

    /**
     * <p>
     * The fully qualified name for the directory, such as <code>corp.example.com</code>.
     * </p>
     * 
     * @return The fully qualified name for the directory, such as <code>corp.example.com</code>.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The short name of the directory, such as <code>CORP</code>.
     * </p>
     * 
     * @return The short name of the directory, such as <code>CORP</code>.
     */
    public String shortName() {
        return shortName;
    }

    /**
     * <p>
     * The password for the directory administrator. The directory creation process creates a directory administrator
     * account with the user name <code>Administrator</code> and this password.
     * </p>
     * 
     * @return The password for the directory administrator. The directory creation process creates a directory
     *         administrator account with the user name <code>Administrator</code> and this password.
     */
    public String password() {
        return password;
    }

    /**
     * <p>
     * A textual description for the directory.
     * </p>
     * 
     * @return A textual description for the directory.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The size of the directory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #size} will return
     * {@link DirectorySize#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sizeAsString}.
     * </p>
     * 
     * @return The size of the directory.
     * @see DirectorySize
     */
    public DirectorySize size() {
        return DirectorySize.fromValue(size);
    }

    /**
     * <p>
     * The size of the directory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #size} will return
     * {@link DirectorySize#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sizeAsString}.
     * </p>
     * 
     * @return The size of the directory.
     * @see DirectorySize
     */
    public String sizeAsString() {
        return size;
    }

    /**
     * <p>
     * A <a>DirectoryVpcSettings</a> object that contains additional information for the operation.
     * </p>
     * 
     * @return A <a>DirectoryVpcSettings</a> object that contains additional information for the operation.
     */
    public DirectoryVpcSettings vpcSettings() {
        return vpcSettings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(shortName());
        hashCode = 31 * hashCode + Objects.hashCode(password());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(sizeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpcSettings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateDirectoryRequest)) {
            return false;
        }
        CreateDirectoryRequest other = (CreateDirectoryRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(shortName(), other.shortName())
                && Objects.equals(password(), other.password()) && Objects.equals(description(), other.description())
                && Objects.equals(sizeAsString(), other.sizeAsString()) && Objects.equals(vpcSettings(), other.vpcSettings());
    }

    @Override
    public String toString() {
        return ToString.builder("CreateDirectoryRequest").add("Name", name()).add("ShortName", shortName())
                .add("Password", password()).add("Description", description()).add("Size", sizeAsString())
                .add("VpcSettings", vpcSettings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ShortName":
            return Optional.ofNullable(clazz.cast(shortName()));
        case "Password":
            return Optional.ofNullable(clazz.cast(password()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Size":
            return Optional.ofNullable(clazz.cast(sizeAsString()));
        case "VpcSettings":
            return Optional.ofNullable(clazz.cast(vpcSettings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateDirectoryRequest, T> g) {
        return obj -> g.apply((CreateDirectoryRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DirectoryRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateDirectoryRequest> {
        /**
         * <p>
         * The fully qualified name for the directory, such as <code>corp.example.com</code>.
         * </p>
         * 
         * @param name
         *        The fully qualified name for the directory, such as <code>corp.example.com</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The short name of the directory, such as <code>CORP</code>.
         * </p>
         * 
         * @param shortName
         *        The short name of the directory, such as <code>CORP</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shortName(String shortName);

        /**
         * <p>
         * The password for the directory administrator. The directory creation process creates a directory
         * administrator account with the user name <code>Administrator</code> and this password.
         * </p>
         * 
         * @param password
         *        The password for the directory administrator. The directory creation process creates a directory
         *        administrator account with the user name <code>Administrator</code> and this password.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder password(String password);

        /**
         * <p>
         * A textual description for the directory.
         * </p>
         * 
         * @param description
         *        A textual description for the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The size of the directory.
         * </p>
         * 
         * @param size
         *        The size of the directory.
         * @see DirectorySize
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DirectorySize
         */
        Builder size(String size);

        /**
         * <p>
         * The size of the directory.
         * </p>
         * 
         * @param size
         *        The size of the directory.
         * @see DirectorySize
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DirectorySize
         */
        Builder size(DirectorySize size);

        /**
         * <p>
         * A <a>DirectoryVpcSettings</a> object that contains additional information for the operation.
         * </p>
         * 
         * @param vpcSettings
         *        A <a>DirectoryVpcSettings</a> object that contains additional information for the operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSettings(DirectoryVpcSettings vpcSettings);

        /**
         * <p>
         * A <a>DirectoryVpcSettings</a> object that contains additional information for the operation.
         * </p>
         * This is a convenience that creates an instance of the {@link DirectoryVpcSettings.Builder} avoiding the need
         * to create one manually via {@link DirectoryVpcSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link DirectoryVpcSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #vpcSettings(DirectoryVpcSettings)}.
         * 
         * @param vpcSettings
         *        a consumer that will call methods on {@link DirectoryVpcSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcSettings(DirectoryVpcSettings)
         */
        default Builder vpcSettings(Consumer<DirectoryVpcSettings.Builder> vpcSettings) {
            return vpcSettings(DirectoryVpcSettings.builder().applyMutation(vpcSettings).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DirectoryRequest.BuilderImpl implements Builder {
        private String name;

        private String shortName;

        private String password;

        private String description;

        private String size;

        private DirectoryVpcSettings vpcSettings;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateDirectoryRequest model) {
            super(model);
            name(model.name);
            shortName(model.shortName);
            password(model.password);
            description(model.description);
            size(model.size);
            vpcSettings(model.vpcSettings);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getShortName() {
            return shortName;
        }

        @Override
        public final Builder shortName(String shortName) {
            this.shortName = shortName;
            return this;
        }

        public final void setShortName(String shortName) {
            this.shortName = shortName;
        }

        public final String getPassword() {
            return password;
        }

        @Override
        public final Builder password(String password) {
            this.password = password;
            return this;
        }

        public final void setPassword(String password) {
            this.password = password;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getSizeAsString() {
            return size;
        }

        @Override
        public final Builder size(String size) {
            this.size = size;
            return this;
        }

        @Override
        public final Builder size(DirectorySize size) {
            this.size(size.toString());
            return this;
        }

        public final void setSize(String size) {
            this.size = size;
        }

        public final DirectoryVpcSettings.Builder getVpcSettings() {
            return vpcSettings != null ? vpcSettings.toBuilder() : null;
        }

        @Override
        public final Builder vpcSettings(DirectoryVpcSettings vpcSettings) {
            this.vpcSettings = vpcSettings;
            return this;
        }

        public final void setVpcSettings(DirectoryVpcSettings.BuilderImpl vpcSettings) {
            this.vpcSettings = vpcSettings != null ? vpcSettings.build() : null;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateDirectoryRequest build() {
            return new CreateDirectoryRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
