/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directory.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Points to a remote domain with which you are setting up a trust relationship. Conditional forwarders are required in
 * order to set up a trust relationship with another domain.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ConditionalForwarder implements SdkPojo, Serializable,
        ToCopyableBuilder<ConditionalForwarder.Builder, ConditionalForwarder> {
    private static final SdkField<String> REMOTE_DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RemoteDomainName").getter(getter(ConditionalForwarder::remoteDomainName))
            .setter(setter(Builder::remoteDomainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RemoteDomainName").build()).build();

    private static final SdkField<List<String>> DNS_IP_ADDRS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DnsIpAddrs")
            .getter(getter(ConditionalForwarder::dnsIpAddrs))
            .setter(setter(Builder::dnsIpAddrs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsIpAddrs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> REPLICATION_SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReplicationScope").getter(getter(ConditionalForwarder::replicationScopeAsString))
            .setter(setter(Builder::replicationScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationScope").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REMOTE_DOMAIN_NAME_FIELD,
            DNS_IP_ADDRS_FIELD, REPLICATION_SCOPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String remoteDomainName;

    private final List<String> dnsIpAddrs;

    private final String replicationScope;

    private ConditionalForwarder(BuilderImpl builder) {
        this.remoteDomainName = builder.remoteDomainName;
        this.dnsIpAddrs = builder.dnsIpAddrs;
        this.replicationScope = builder.replicationScope;
    }

    /**
     * <p>
     * The fully qualified domain name (FQDN) of the remote domains pointed to by the conditional forwarder.
     * </p>
     * 
     * @return The fully qualified domain name (FQDN) of the remote domains pointed to by the conditional forwarder.
     */
    public final String remoteDomainName() {
        return remoteDomainName;
    }

    /**
     * For responses, this returns true if the service returned a value for the DnsIpAddrs property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDnsIpAddrs() {
        return dnsIpAddrs != null && !(dnsIpAddrs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IP addresses of the remote DNS server associated with RemoteDomainName. This is the IP address of the DNS
     * server that your conditional forwarder points to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDnsIpAddrs} method.
     * </p>
     * 
     * @return The IP addresses of the remote DNS server associated with RemoteDomainName. This is the IP address of the
     *         DNS server that your conditional forwarder points to.
     */
    public final List<String> dnsIpAddrs() {
        return dnsIpAddrs;
    }

    /**
     * <p>
     * The replication scope of the conditional forwarder. The only allowed value is <code>Domain</code>, which will
     * replicate the conditional forwarder to all of the domain controllers for your Amazon Web Services directory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #replicationScope}
     * will return {@link ReplicationScope#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #replicationScopeAsString}.
     * </p>
     * 
     * @return The replication scope of the conditional forwarder. The only allowed value is <code>Domain</code>, which
     *         will replicate the conditional forwarder to all of the domain controllers for your Amazon Web Services
     *         directory.
     * @see ReplicationScope
     */
    public final ReplicationScope replicationScope() {
        return ReplicationScope.fromValue(replicationScope);
    }

    /**
     * <p>
     * The replication scope of the conditional forwarder. The only allowed value is <code>Domain</code>, which will
     * replicate the conditional forwarder to all of the domain controllers for your Amazon Web Services directory.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #replicationScope}
     * will return {@link ReplicationScope#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #replicationScopeAsString}.
     * </p>
     * 
     * @return The replication scope of the conditional forwarder. The only allowed value is <code>Domain</code>, which
     *         will replicate the conditional forwarder to all of the domain controllers for your Amazon Web Services
     *         directory.
     * @see ReplicationScope
     */
    public final String replicationScopeAsString() {
        return replicationScope;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(remoteDomainName());
        hashCode = 31 * hashCode + Objects.hashCode(hasDnsIpAddrs() ? dnsIpAddrs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(replicationScopeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConditionalForwarder)) {
            return false;
        }
        ConditionalForwarder other = (ConditionalForwarder) obj;
        return Objects.equals(remoteDomainName(), other.remoteDomainName()) && hasDnsIpAddrs() == other.hasDnsIpAddrs()
                && Objects.equals(dnsIpAddrs(), other.dnsIpAddrs())
                && Objects.equals(replicationScopeAsString(), other.replicationScopeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ConditionalForwarder").add("RemoteDomainName", remoteDomainName())
                .add("DnsIpAddrs", hasDnsIpAddrs() ? dnsIpAddrs() : null).add("ReplicationScope", replicationScopeAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RemoteDomainName":
            return Optional.ofNullable(clazz.cast(remoteDomainName()));
        case "DnsIpAddrs":
            return Optional.ofNullable(clazz.cast(dnsIpAddrs()));
        case "ReplicationScope":
            return Optional.ofNullable(clazz.cast(replicationScopeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RemoteDomainName", REMOTE_DOMAIN_NAME_FIELD);
        map.put("DnsIpAddrs", DNS_IP_ADDRS_FIELD);
        map.put("ReplicationScope", REPLICATION_SCOPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ConditionalForwarder, T> g) {
        return obj -> g.apply((ConditionalForwarder) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ConditionalForwarder> {
        /**
         * <p>
         * The fully qualified domain name (FQDN) of the remote domains pointed to by the conditional forwarder.
         * </p>
         * 
         * @param remoteDomainName
         *        The fully qualified domain name (FQDN) of the remote domains pointed to by the conditional forwarder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder remoteDomainName(String remoteDomainName);

        /**
         * <p>
         * The IP addresses of the remote DNS server associated with RemoteDomainName. This is the IP address of the DNS
         * server that your conditional forwarder points to.
         * </p>
         * 
         * @param dnsIpAddrs
         *        The IP addresses of the remote DNS server associated with RemoteDomainName. This is the IP address of
         *        the DNS server that your conditional forwarder points to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIpAddrs(Collection<String> dnsIpAddrs);

        /**
         * <p>
         * The IP addresses of the remote DNS server associated with RemoteDomainName. This is the IP address of the DNS
         * server that your conditional forwarder points to.
         * </p>
         * 
         * @param dnsIpAddrs
         *        The IP addresses of the remote DNS server associated with RemoteDomainName. This is the IP address of
         *        the DNS server that your conditional forwarder points to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIpAddrs(String... dnsIpAddrs);

        /**
         * <p>
         * The replication scope of the conditional forwarder. The only allowed value is <code>Domain</code>, which will
         * replicate the conditional forwarder to all of the domain controllers for your Amazon Web Services directory.
         * </p>
         * 
         * @param replicationScope
         *        The replication scope of the conditional forwarder. The only allowed value is <code>Domain</code>,
         *        which will replicate the conditional forwarder to all of the domain controllers for your Amazon Web
         *        Services directory.
         * @see ReplicationScope
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReplicationScope
         */
        Builder replicationScope(String replicationScope);

        /**
         * <p>
         * The replication scope of the conditional forwarder. The only allowed value is <code>Domain</code>, which will
         * replicate the conditional forwarder to all of the domain controllers for your Amazon Web Services directory.
         * </p>
         * 
         * @param replicationScope
         *        The replication scope of the conditional forwarder. The only allowed value is <code>Domain</code>,
         *        which will replicate the conditional forwarder to all of the domain controllers for your Amazon Web
         *        Services directory.
         * @see ReplicationScope
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReplicationScope
         */
        Builder replicationScope(ReplicationScope replicationScope);
    }

    static final class BuilderImpl implements Builder {
        private String remoteDomainName;

        private List<String> dnsIpAddrs = DefaultSdkAutoConstructList.getInstance();

        private String replicationScope;

        private BuilderImpl() {
        }

        private BuilderImpl(ConditionalForwarder model) {
            remoteDomainName(model.remoteDomainName);
            dnsIpAddrs(model.dnsIpAddrs);
            replicationScope(model.replicationScope);
        }

        public final String getRemoteDomainName() {
            return remoteDomainName;
        }

        public final void setRemoteDomainName(String remoteDomainName) {
            this.remoteDomainName = remoteDomainName;
        }

        @Override
        public final Builder remoteDomainName(String remoteDomainName) {
            this.remoteDomainName = remoteDomainName;
            return this;
        }

        public final Collection<String> getDnsIpAddrs() {
            if (dnsIpAddrs instanceof SdkAutoConstructList) {
                return null;
            }
            return dnsIpAddrs;
        }

        public final void setDnsIpAddrs(Collection<String> dnsIpAddrs) {
            this.dnsIpAddrs = DnsIpAddrsCopier.copy(dnsIpAddrs);
        }

        @Override
        public final Builder dnsIpAddrs(Collection<String> dnsIpAddrs) {
            this.dnsIpAddrs = DnsIpAddrsCopier.copy(dnsIpAddrs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsIpAddrs(String... dnsIpAddrs) {
            dnsIpAddrs(Arrays.asList(dnsIpAddrs));
            return this;
        }

        public final String getReplicationScope() {
            return replicationScope;
        }

        public final void setReplicationScope(String replicationScope) {
            this.replicationScope = replicationScope;
        }

        @Override
        public final Builder replicationScope(String replicationScope) {
            this.replicationScope = replicationScope;
            return this;
        }

        @Override
        public final Builder replicationScope(ReplicationScope replicationScope) {
            this.replicationScope(replicationScope == null ? null : replicationScope.toString());
            return this;
        }

        @Override
        public ConditionalForwarder build() {
            return new ConditionalForwarder(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
