/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directory.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains detailed information about a directory assessment, including configuration parameters, status, and
 * validation results.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Assessment implements SdkPojo, Serializable, ToCopyableBuilder<Assessment.Builder, Assessment> {
    private static final SdkField<String> ASSESSMENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AssessmentId").getter(getter(Assessment::assessmentId)).setter(setter(Builder::assessmentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssessmentId").build()).build();

    private static final SdkField<String> DIRECTORY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DirectoryId").getter(getter(Assessment::directoryId)).setter(setter(Builder::directoryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DirectoryId").build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DnsName").getter(getter(Assessment::dnsName)).setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsName").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(Assessment::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATE_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdateDateTime").getter(getter(Assessment::lastUpdateDateTime))
            .setter(setter(Builder::lastUpdateDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdateDateTime").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Assessment::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusCode").getter(getter(Assessment::statusCode)).setter(setter(Builder::statusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusCode").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusReason").getter(getter(Assessment::statusReason)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusReason").build()).build();

    private static final SdkField<List<String>> CUSTOMER_DNS_IPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("CustomerDnsIps")
            .getter(getter(Assessment::customerDnsIps))
            .setter(setter(Builder::customerDnsIps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerDnsIps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(Assessment::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<List<String>> SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SubnetIds")
            .getter(getter(Assessment::subnetIds))
            .setter(setter(Builder::subnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroupIds")
            .getter(getter(Assessment::securityGroupIds))
            .setter(setter(Builder::securityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SELF_MANAGED_INSTANCE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SelfManagedInstanceIds")
            .getter(getter(Assessment::selfManagedInstanceIds))
            .setter(setter(Builder::selfManagedInstanceIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SelfManagedInstanceIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> REPORT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReportType").getter(getter(Assessment::reportType)).setter(setter(Builder::reportType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReportType").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Version")
            .getter(getter(Assessment::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ASSESSMENT_ID_FIELD,
            DIRECTORY_ID_FIELD, DNS_NAME_FIELD, START_TIME_FIELD, LAST_UPDATE_DATE_TIME_FIELD, STATUS_FIELD, STATUS_CODE_FIELD,
            STATUS_REASON_FIELD, CUSTOMER_DNS_IPS_FIELD, VPC_ID_FIELD, SUBNET_IDS_FIELD, SECURITY_GROUP_IDS_FIELD,
            SELF_MANAGED_INSTANCE_IDS_FIELD, REPORT_TYPE_FIELD, VERSION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String assessmentId;

    private final String directoryId;

    private final String dnsName;

    private final Instant startTime;

    private final Instant lastUpdateDateTime;

    private final String status;

    private final String statusCode;

    private final String statusReason;

    private final List<String> customerDnsIps;

    private final String vpcId;

    private final List<String> subnetIds;

    private final List<String> securityGroupIds;

    private final List<String> selfManagedInstanceIds;

    private final String reportType;

    private final String version;

    private Assessment(BuilderImpl builder) {
        this.assessmentId = builder.assessmentId;
        this.directoryId = builder.directoryId;
        this.dnsName = builder.dnsName;
        this.startTime = builder.startTime;
        this.lastUpdateDateTime = builder.lastUpdateDateTime;
        this.status = builder.status;
        this.statusCode = builder.statusCode;
        this.statusReason = builder.statusReason;
        this.customerDnsIps = builder.customerDnsIps;
        this.vpcId = builder.vpcId;
        this.subnetIds = builder.subnetIds;
        this.securityGroupIds = builder.securityGroupIds;
        this.selfManagedInstanceIds = builder.selfManagedInstanceIds;
        this.reportType = builder.reportType;
        this.version = builder.version;
    }

    /**
     * <p>
     * The unique identifier of the directory assessment.
     * </p>
     * 
     * @return The unique identifier of the directory assessment.
     */
    public final String assessmentId() {
        return assessmentId;
    }

    /**
     * <p>
     * The identifier of the directory associated with this assessment.
     * </p>
     * 
     * @return The identifier of the directory associated with this assessment.
     */
    public final String directoryId() {
        return directoryId;
    }

    /**
     * <p>
     * The fully qualified domain name (FQDN) of the Active Directory domain being assessed.
     * </p>
     * 
     * @return The fully qualified domain name (FQDN) of the Active Directory domain being assessed.
     */
    public final String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * The date and time when the assessment was initiated.
     * </p>
     * 
     * @return The date and time when the assessment was initiated.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The date and time when the assessment status was last updated.
     * </p>
     * 
     * @return The date and time when the assessment status was last updated.
     */
    public final Instant lastUpdateDateTime() {
        return lastUpdateDateTime;
    }

    /**
     * <p>
     * The current status of the assessment. Valid values include <code>SUCCESS</code>, <code>FAILED</code>,
     * <code>PENDING</code>, and <code>IN_PROGRESS</code>.
     * </p>
     * 
     * @return The current status of the assessment. Valid values include <code>SUCCESS</code>, <code>FAILED</code>,
     *         <code>PENDING</code>, and <code>IN_PROGRESS</code>.
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * A detailed status code providing additional information about the assessment state.
     * </p>
     * 
     * @return A detailed status code providing additional information about the assessment state.
     */
    public final String statusCode() {
        return statusCode;
    }

    /**
     * <p>
     * A human-readable description of the current assessment status, including any error details or progress
     * information.
     * </p>
     * 
     * @return A human-readable description of the current assessment status, including any error details or progress
     *         information.
     */
    public final String statusReason() {
        return statusReason;
    }

    /**
     * For responses, this returns true if the service returned a value for the CustomerDnsIps property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCustomerDnsIps() {
        return customerDnsIps != null && !(customerDnsIps instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IP addresses of the DNS servers or domain controllers in your self-managed AD environment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCustomerDnsIps} method.
     * </p>
     * 
     * @return The IP addresses of the DNS servers or domain controllers in your self-managed AD environment.
     */
    public final List<String> customerDnsIps() {
        return customerDnsIps;
    }

    /**
     * <p>
     * Contains Amazon VPC information for the <code>StartADAssessment</code> operation.
     * </p>
     * 
     * @return Contains Amazon VPC information for the <code>StartADAssessment</code> operation.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * For responses, this returns true if the service returned a value for the SubnetIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSubnetIds() {
        return subnetIds != null && !(subnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of subnet identifiers in the Amazon VPC in which the hybrid directory is created.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubnetIds} method.
     * </p>
     * 
     * @return A list of subnet identifiers in the Amazon VPC in which the hybrid directory is created.
     */
    public final List<String> subnetIds() {
        return subnetIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroupIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroupIds() {
        return securityGroupIds != null && !(securityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The security groups identifiers attached to the network interfaces.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroupIds} method.
     * </p>
     * 
     * @return The security groups identifiers attached to the network interfaces.
     */
    public final List<String> securityGroupIds() {
        return securityGroupIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the SelfManagedInstanceIds property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSelfManagedInstanceIds() {
        return selfManagedInstanceIds != null && !(selfManagedInstanceIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The identifiers of the self-managed AD instances used to perform the assessment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSelfManagedInstanceIds} method.
     * </p>
     * 
     * @return The identifiers of the self-managed AD instances used to perform the assessment.
     */
    public final List<String> selfManagedInstanceIds() {
        return selfManagedInstanceIds;
    }

    /**
     * <p>
     * The type of assessment report generated. Valid values are <code>CUSTOMER</code> and <code>SYSTEM</code>.
     * </p>
     * 
     * @return The type of assessment report generated. Valid values are <code>CUSTOMER</code> and <code>SYSTEM</code>.
     */
    public final String reportType() {
        return reportType;
    }

    /**
     * <p>
     * The version of the assessment framework used to evaluate your self-managed AD environment.
     * </p>
     * 
     * @return The version of the assessment framework used to evaluate your self-managed AD environment.
     */
    public final String version() {
        return version;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(assessmentId());
        hashCode = 31 * hashCode + Objects.hashCode(directoryId());
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(statusCode());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(hasCustomerDnsIps() ? customerDnsIps() : null);
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnetIds() ? subnetIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroupIds() ? securityGroupIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSelfManagedInstanceIds() ? selfManagedInstanceIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(reportType());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Assessment)) {
            return false;
        }
        Assessment other = (Assessment) obj;
        return Objects.equals(assessmentId(), other.assessmentId()) && Objects.equals(directoryId(), other.directoryId())
                && Objects.equals(dnsName(), other.dnsName()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(lastUpdateDateTime(), other.lastUpdateDateTime()) && Objects.equals(status(), other.status())
                && Objects.equals(statusCode(), other.statusCode()) && Objects.equals(statusReason(), other.statusReason())
                && hasCustomerDnsIps() == other.hasCustomerDnsIps() && Objects.equals(customerDnsIps(), other.customerDnsIps())
                && Objects.equals(vpcId(), other.vpcId()) && hasSubnetIds() == other.hasSubnetIds()
                && Objects.equals(subnetIds(), other.subnetIds()) && hasSecurityGroupIds() == other.hasSecurityGroupIds()
                && Objects.equals(securityGroupIds(), other.securityGroupIds())
                && hasSelfManagedInstanceIds() == other.hasSelfManagedInstanceIds()
                && Objects.equals(selfManagedInstanceIds(), other.selfManagedInstanceIds())
                && Objects.equals(reportType(), other.reportType()) && Objects.equals(version(), other.version());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Assessment").add("AssessmentId", assessmentId()).add("DirectoryId", directoryId())
                .add("DnsName", dnsName()).add("StartTime", startTime()).add("LastUpdateDateTime", lastUpdateDateTime())
                .add("Status", status()).add("StatusCode", statusCode()).add("StatusReason", statusReason())
                .add("CustomerDnsIps", hasCustomerDnsIps() ? customerDnsIps() : null).add("VpcId", vpcId())
                .add("SubnetIds", hasSubnetIds() ? subnetIds() : null)
                .add("SecurityGroupIds", hasSecurityGroupIds() ? securityGroupIds() : null)
                .add("SelfManagedInstanceIds", hasSelfManagedInstanceIds() ? selfManagedInstanceIds() : null)
                .add("ReportType", reportType()).add("Version", version()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AssessmentId":
            return Optional.ofNullable(clazz.cast(assessmentId()));
        case "DirectoryId":
            return Optional.ofNullable(clazz.cast(directoryId()));
        case "DnsName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "LastUpdateDateTime":
            return Optional.ofNullable(clazz.cast(lastUpdateDateTime()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "StatusCode":
            return Optional.ofNullable(clazz.cast(statusCode()));
        case "StatusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "CustomerDnsIps":
            return Optional.ofNullable(clazz.cast(customerDnsIps()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "SubnetIds":
            return Optional.ofNullable(clazz.cast(subnetIds()));
        case "SecurityGroupIds":
            return Optional.ofNullable(clazz.cast(securityGroupIds()));
        case "SelfManagedInstanceIds":
            return Optional.ofNullable(clazz.cast(selfManagedInstanceIds()));
        case "ReportType":
            return Optional.ofNullable(clazz.cast(reportType()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AssessmentId", ASSESSMENT_ID_FIELD);
        map.put("DirectoryId", DIRECTORY_ID_FIELD);
        map.put("DnsName", DNS_NAME_FIELD);
        map.put("StartTime", START_TIME_FIELD);
        map.put("LastUpdateDateTime", LAST_UPDATE_DATE_TIME_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("StatusCode", STATUS_CODE_FIELD);
        map.put("StatusReason", STATUS_REASON_FIELD);
        map.put("CustomerDnsIps", CUSTOMER_DNS_IPS_FIELD);
        map.put("VpcId", VPC_ID_FIELD);
        map.put("SubnetIds", SUBNET_IDS_FIELD);
        map.put("SecurityGroupIds", SECURITY_GROUP_IDS_FIELD);
        map.put("SelfManagedInstanceIds", SELF_MANAGED_INSTANCE_IDS_FIELD);
        map.put("ReportType", REPORT_TYPE_FIELD);
        map.put("Version", VERSION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Assessment, T> g) {
        return obj -> g.apply((Assessment) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Assessment> {
        /**
         * <p>
         * The unique identifier of the directory assessment.
         * </p>
         * 
         * @param assessmentId
         *        The unique identifier of the directory assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assessmentId(String assessmentId);

        /**
         * <p>
         * The identifier of the directory associated with this assessment.
         * </p>
         * 
         * @param directoryId
         *        The identifier of the directory associated with this assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder directoryId(String directoryId);

        /**
         * <p>
         * The fully qualified domain name (FQDN) of the Active Directory domain being assessed.
         * </p>
         * 
         * @param dnsName
         *        The fully qualified domain name (FQDN) of the Active Directory domain being assessed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * The date and time when the assessment was initiated.
         * </p>
         * 
         * @param startTime
         *        The date and time when the assessment was initiated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The date and time when the assessment status was last updated.
         * </p>
         * 
         * @param lastUpdateDateTime
         *        The date and time when the assessment status was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateDateTime(Instant lastUpdateDateTime);

        /**
         * <p>
         * The current status of the assessment. Valid values include <code>SUCCESS</code>, <code>FAILED</code>,
         * <code>PENDING</code>, and <code>IN_PROGRESS</code>.
         * </p>
         * 
         * @param status
         *        The current status of the assessment. Valid values include <code>SUCCESS</code>, <code>FAILED</code>,
         *        <code>PENDING</code>, and <code>IN_PROGRESS</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * A detailed status code providing additional information about the assessment state.
         * </p>
         * 
         * @param statusCode
         *        A detailed status code providing additional information about the assessment state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusCode(String statusCode);

        /**
         * <p>
         * A human-readable description of the current assessment status, including any error details or progress
         * information.
         * </p>
         * 
         * @param statusReason
         *        A human-readable description of the current assessment status, including any error details or progress
         *        information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * The IP addresses of the DNS servers or domain controllers in your self-managed AD environment.
         * </p>
         * 
         * @param customerDnsIps
         *        The IP addresses of the DNS servers or domain controllers in your self-managed AD environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerDnsIps(Collection<String> customerDnsIps);

        /**
         * <p>
         * The IP addresses of the DNS servers or domain controllers in your self-managed AD environment.
         * </p>
         * 
         * @param customerDnsIps
         *        The IP addresses of the DNS servers or domain controllers in your self-managed AD environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerDnsIps(String... customerDnsIps);

        /**
         * <p>
         * Contains Amazon VPC information for the <code>StartADAssessment</code> operation.
         * </p>
         * 
         * @param vpcId
         *        Contains Amazon VPC information for the <code>StartADAssessment</code> operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * A list of subnet identifiers in the Amazon VPC in which the hybrid directory is created.
         * </p>
         * 
         * @param subnetIds
         *        A list of subnet identifiers in the Amazon VPC in which the hybrid directory is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(Collection<String> subnetIds);

        /**
         * <p>
         * A list of subnet identifiers in the Amazon VPC in which the hybrid directory is created.
         * </p>
         * 
         * @param subnetIds
         *        A list of subnet identifiers in the Amazon VPC in which the hybrid directory is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(String... subnetIds);

        /**
         * <p>
         * The security groups identifiers attached to the network interfaces.
         * </p>
         * 
         * @param securityGroupIds
         *        The security groups identifiers attached to the network interfaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(Collection<String> securityGroupIds);

        /**
         * <p>
         * The security groups identifiers attached to the network interfaces.
         * </p>
         * 
         * @param securityGroupIds
         *        The security groups identifiers attached to the network interfaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(String... securityGroupIds);

        /**
         * <p>
         * The identifiers of the self-managed AD instances used to perform the assessment.
         * </p>
         * 
         * @param selfManagedInstanceIds
         *        The identifiers of the self-managed AD instances used to perform the assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder selfManagedInstanceIds(Collection<String> selfManagedInstanceIds);

        /**
         * <p>
         * The identifiers of the self-managed AD instances used to perform the assessment.
         * </p>
         * 
         * @param selfManagedInstanceIds
         *        The identifiers of the self-managed AD instances used to perform the assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder selfManagedInstanceIds(String... selfManagedInstanceIds);

        /**
         * <p>
         * The type of assessment report generated. Valid values are <code>CUSTOMER</code> and <code>SYSTEM</code>.
         * </p>
         * 
         * @param reportType
         *        The type of assessment report generated. Valid values are <code>CUSTOMER</code> and
         *        <code>SYSTEM</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reportType(String reportType);

        /**
         * <p>
         * The version of the assessment framework used to evaluate your self-managed AD environment.
         * </p>
         * 
         * @param version
         *        The version of the assessment framework used to evaluate your self-managed AD environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);
    }

    static final class BuilderImpl implements Builder {
        private String assessmentId;

        private String directoryId;

        private String dnsName;

        private Instant startTime;

        private Instant lastUpdateDateTime;

        private String status;

        private String statusCode;

        private String statusReason;

        private List<String> customerDnsIps = DefaultSdkAutoConstructList.getInstance();

        private String vpcId;

        private List<String> subnetIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> securityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> selfManagedInstanceIds = DefaultSdkAutoConstructList.getInstance();

        private String reportType;

        private String version;

        private BuilderImpl() {
        }

        private BuilderImpl(Assessment model) {
            assessmentId(model.assessmentId);
            directoryId(model.directoryId);
            dnsName(model.dnsName);
            startTime(model.startTime);
            lastUpdateDateTime(model.lastUpdateDateTime);
            status(model.status);
            statusCode(model.statusCode);
            statusReason(model.statusReason);
            customerDnsIps(model.customerDnsIps);
            vpcId(model.vpcId);
            subnetIds(model.subnetIds);
            securityGroupIds(model.securityGroupIds);
            selfManagedInstanceIds(model.selfManagedInstanceIds);
            reportType(model.reportType);
            version(model.version);
        }

        public final String getAssessmentId() {
            return assessmentId;
        }

        public final void setAssessmentId(String assessmentId) {
            this.assessmentId = assessmentId;
        }

        @Override
        public final Builder assessmentId(String assessmentId) {
            this.assessmentId = assessmentId;
            return this;
        }

        public final String getDirectoryId() {
            return directoryId;
        }

        public final void setDirectoryId(String directoryId) {
            this.directoryId = directoryId;
        }

        @Override
        public final Builder directoryId(String directoryId) {
            this.directoryId = directoryId;
            return this;
        }

        public final String getDnsName() {
            return dnsName;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getLastUpdateDateTime() {
            return lastUpdateDateTime;
        }

        public final void setLastUpdateDateTime(Instant lastUpdateDateTime) {
            this.lastUpdateDateTime = lastUpdateDateTime;
        }

        @Override
        public final Builder lastUpdateDateTime(Instant lastUpdateDateTime) {
            this.lastUpdateDateTime = lastUpdateDateTime;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final String getStatusCode() {
            return statusCode;
        }

        public final void setStatusCode(String statusCode) {
            this.statusCode = statusCode;
        }

        @Override
        public final Builder statusCode(String statusCode) {
            this.statusCode = statusCode;
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final Collection<String> getCustomerDnsIps() {
            if (customerDnsIps instanceof SdkAutoConstructList) {
                return null;
            }
            return customerDnsIps;
        }

        public final void setCustomerDnsIps(Collection<String> customerDnsIps) {
            this.customerDnsIps = CustomerDnsIpsCopier.copy(customerDnsIps);
        }

        @Override
        public final Builder customerDnsIps(Collection<String> customerDnsIps) {
            this.customerDnsIps = CustomerDnsIpsCopier.copy(customerDnsIps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder customerDnsIps(String... customerDnsIps) {
            customerDnsIps(Arrays.asList(customerDnsIps));
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final Collection<String> getSubnetIds() {
            if (subnetIds instanceof SdkAutoConstructList) {
                return null;
            }
            return subnetIds;
        }

        public final void setSubnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
        }

        @Override
        public final Builder subnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetIds(String... subnetIds) {
            subnetIds(Arrays.asList(subnetIds));
            return this;
        }

        public final Collection<String> getSecurityGroupIds() {
            if (securityGroupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroupIds;
        }

        public final void setSecurityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdsCopier.copy(securityGroupIds);
        }

        @Override
        public final Builder securityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdsCopier.copy(securityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupIds(String... securityGroupIds) {
            securityGroupIds(Arrays.asList(securityGroupIds));
            return this;
        }

        public final Collection<String> getSelfManagedInstanceIds() {
            if (selfManagedInstanceIds instanceof SdkAutoConstructList) {
                return null;
            }
            return selfManagedInstanceIds;
        }

        public final void setSelfManagedInstanceIds(Collection<String> selfManagedInstanceIds) {
            this.selfManagedInstanceIds = AssessmentInstanceIdsCopier.copy(selfManagedInstanceIds);
        }

        @Override
        public final Builder selfManagedInstanceIds(Collection<String> selfManagedInstanceIds) {
            this.selfManagedInstanceIds = AssessmentInstanceIdsCopier.copy(selfManagedInstanceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder selfManagedInstanceIds(String... selfManagedInstanceIds) {
            selfManagedInstanceIds(Arrays.asList(selfManagedInstanceIds));
            return this;
        }

        public final String getReportType() {
            return reportType;
        }

        public final void setReportType(String reportType) {
            this.reportType = reportType;
        }

        @Override
        public final Builder reportType(String reportType) {
            this.reportType = reportType;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        @Override
        public Assessment build() {
            return new Assessment(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
