/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directory.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains summary information about a directory assessment, providing a high-level overview without detailed
 * validation results.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AssessmentSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<AssessmentSummary.Builder, AssessmentSummary> {
    private static final SdkField<String> ASSESSMENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AssessmentId").getter(getter(AssessmentSummary::assessmentId)).setter(setter(Builder::assessmentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssessmentId").build()).build();

    private static final SdkField<String> DIRECTORY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DirectoryId").getter(getter(AssessmentSummary::directoryId)).setter(setter(Builder::directoryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DirectoryId").build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DnsName").getter(getter(AssessmentSummary::dnsName)).setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsName").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(AssessmentSummary::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATE_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdateDateTime").getter(getter(AssessmentSummary::lastUpdateDateTime))
            .setter(setter(Builder::lastUpdateDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdateDateTime").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(AssessmentSummary::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<List<String>> CUSTOMER_DNS_IPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("CustomerDnsIps")
            .getter(getter(AssessmentSummary::customerDnsIps))
            .setter(setter(Builder::customerDnsIps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerDnsIps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> REPORT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReportType").getter(getter(AssessmentSummary::reportType)).setter(setter(Builder::reportType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReportType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ASSESSMENT_ID_FIELD,
            DIRECTORY_ID_FIELD, DNS_NAME_FIELD, START_TIME_FIELD, LAST_UPDATE_DATE_TIME_FIELD, STATUS_FIELD,
            CUSTOMER_DNS_IPS_FIELD, REPORT_TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String assessmentId;

    private final String directoryId;

    private final String dnsName;

    private final Instant startTime;

    private final Instant lastUpdateDateTime;

    private final String status;

    private final List<String> customerDnsIps;

    private final String reportType;

    private AssessmentSummary(BuilderImpl builder) {
        this.assessmentId = builder.assessmentId;
        this.directoryId = builder.directoryId;
        this.dnsName = builder.dnsName;
        this.startTime = builder.startTime;
        this.lastUpdateDateTime = builder.lastUpdateDateTime;
        this.status = builder.status;
        this.customerDnsIps = builder.customerDnsIps;
        this.reportType = builder.reportType;
    }

    /**
     * <p>
     * The unique identifier of the directory assessment.
     * </p>
     * 
     * @return The unique identifier of the directory assessment.
     */
    public final String assessmentId() {
        return assessmentId;
    }

    /**
     * <p>
     * The identifier of the directory associated with this assessment.
     * </p>
     * 
     * @return The identifier of the directory associated with this assessment.
     */
    public final String directoryId() {
        return directoryId;
    }

    /**
     * <p>
     * The fully qualified domain name (FQDN) of the Active Directory domain being assessed.
     * </p>
     * 
     * @return The fully qualified domain name (FQDN) of the Active Directory domain being assessed.
     */
    public final String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * The date and time when the assessment was initiated.
     * </p>
     * 
     * @return The date and time when the assessment was initiated.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The date and time when the assessment status was last updated.
     * </p>
     * 
     * @return The date and time when the assessment status was last updated.
     */
    public final Instant lastUpdateDateTime() {
        return lastUpdateDateTime;
    }

    /**
     * <p>
     * The current status of the assessment. Valid values include <code>SUCCESS</code>, <code>FAILED</code>,
     * <code>PENDING</code>, and <code>IN_PROGRESS</code>.
     * </p>
     * 
     * @return The current status of the assessment. Valid values include <code>SUCCESS</code>, <code>FAILED</code>,
     *         <code>PENDING</code>, and <code>IN_PROGRESS</code>.
     */
    public final String status() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the CustomerDnsIps property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCustomerDnsIps() {
        return customerDnsIps != null && !(customerDnsIps instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IP addresses of the DNS servers or domain controllers in your self-managed AD environment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCustomerDnsIps} method.
     * </p>
     * 
     * @return The IP addresses of the DNS servers or domain controllers in your self-managed AD environment.
     */
    public final List<String> customerDnsIps() {
        return customerDnsIps;
    }

    /**
     * <p>
     * The type of assessment report generated. Valid values include <code>CUSTOMER</code> and <code>SYSTEM</code>.
     * </p>
     * 
     * @return The type of assessment report generated. Valid values include <code>CUSTOMER</code> and
     *         <code>SYSTEM</code>.
     */
    public final String reportType() {
        return reportType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(assessmentId());
        hashCode = 31 * hashCode + Objects.hashCode(directoryId());
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(hasCustomerDnsIps() ? customerDnsIps() : null);
        hashCode = 31 * hashCode + Objects.hashCode(reportType());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AssessmentSummary)) {
            return false;
        }
        AssessmentSummary other = (AssessmentSummary) obj;
        return Objects.equals(assessmentId(), other.assessmentId()) && Objects.equals(directoryId(), other.directoryId())
                && Objects.equals(dnsName(), other.dnsName()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(lastUpdateDateTime(), other.lastUpdateDateTime()) && Objects.equals(status(), other.status())
                && hasCustomerDnsIps() == other.hasCustomerDnsIps() && Objects.equals(customerDnsIps(), other.customerDnsIps())
                && Objects.equals(reportType(), other.reportType());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AssessmentSummary").add("AssessmentId", assessmentId()).add("DirectoryId", directoryId())
                .add("DnsName", dnsName()).add("StartTime", startTime()).add("LastUpdateDateTime", lastUpdateDateTime())
                .add("Status", status()).add("CustomerDnsIps", hasCustomerDnsIps() ? customerDnsIps() : null)
                .add("ReportType", reportType()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AssessmentId":
            return Optional.ofNullable(clazz.cast(assessmentId()));
        case "DirectoryId":
            return Optional.ofNullable(clazz.cast(directoryId()));
        case "DnsName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "LastUpdateDateTime":
            return Optional.ofNullable(clazz.cast(lastUpdateDateTime()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "CustomerDnsIps":
            return Optional.ofNullable(clazz.cast(customerDnsIps()));
        case "ReportType":
            return Optional.ofNullable(clazz.cast(reportType()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AssessmentId", ASSESSMENT_ID_FIELD);
        map.put("DirectoryId", DIRECTORY_ID_FIELD);
        map.put("DnsName", DNS_NAME_FIELD);
        map.put("StartTime", START_TIME_FIELD);
        map.put("LastUpdateDateTime", LAST_UPDATE_DATE_TIME_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("CustomerDnsIps", CUSTOMER_DNS_IPS_FIELD);
        map.put("ReportType", REPORT_TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AssessmentSummary, T> g) {
        return obj -> g.apply((AssessmentSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AssessmentSummary> {
        /**
         * <p>
         * The unique identifier of the directory assessment.
         * </p>
         * 
         * @param assessmentId
         *        The unique identifier of the directory assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assessmentId(String assessmentId);

        /**
         * <p>
         * The identifier of the directory associated with this assessment.
         * </p>
         * 
         * @param directoryId
         *        The identifier of the directory associated with this assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder directoryId(String directoryId);

        /**
         * <p>
         * The fully qualified domain name (FQDN) of the Active Directory domain being assessed.
         * </p>
         * 
         * @param dnsName
         *        The fully qualified domain name (FQDN) of the Active Directory domain being assessed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * The date and time when the assessment was initiated.
         * </p>
         * 
         * @param startTime
         *        The date and time when the assessment was initiated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The date and time when the assessment status was last updated.
         * </p>
         * 
         * @param lastUpdateDateTime
         *        The date and time when the assessment status was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateDateTime(Instant lastUpdateDateTime);

        /**
         * <p>
         * The current status of the assessment. Valid values include <code>SUCCESS</code>, <code>FAILED</code>,
         * <code>PENDING</code>, and <code>IN_PROGRESS</code>.
         * </p>
         * 
         * @param status
         *        The current status of the assessment. Valid values include <code>SUCCESS</code>, <code>FAILED</code>,
         *        <code>PENDING</code>, and <code>IN_PROGRESS</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * The IP addresses of the DNS servers or domain controllers in your self-managed AD environment.
         * </p>
         * 
         * @param customerDnsIps
         *        The IP addresses of the DNS servers or domain controllers in your self-managed AD environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerDnsIps(Collection<String> customerDnsIps);

        /**
         * <p>
         * The IP addresses of the DNS servers or domain controllers in your self-managed AD environment.
         * </p>
         * 
         * @param customerDnsIps
         *        The IP addresses of the DNS servers or domain controllers in your self-managed AD environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerDnsIps(String... customerDnsIps);

        /**
         * <p>
         * The type of assessment report generated. Valid values include <code>CUSTOMER</code> and <code>SYSTEM</code>.
         * </p>
         * 
         * @param reportType
         *        The type of assessment report generated. Valid values include <code>CUSTOMER</code> and
         *        <code>SYSTEM</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reportType(String reportType);
    }

    static final class BuilderImpl implements Builder {
        private String assessmentId;

        private String directoryId;

        private String dnsName;

        private Instant startTime;

        private Instant lastUpdateDateTime;

        private String status;

        private List<String> customerDnsIps = DefaultSdkAutoConstructList.getInstance();

        private String reportType;

        private BuilderImpl() {
        }

        private BuilderImpl(AssessmentSummary model) {
            assessmentId(model.assessmentId);
            directoryId(model.directoryId);
            dnsName(model.dnsName);
            startTime(model.startTime);
            lastUpdateDateTime(model.lastUpdateDateTime);
            status(model.status);
            customerDnsIps(model.customerDnsIps);
            reportType(model.reportType);
        }

        public final String getAssessmentId() {
            return assessmentId;
        }

        public final void setAssessmentId(String assessmentId) {
            this.assessmentId = assessmentId;
        }

        @Override
        public final Builder assessmentId(String assessmentId) {
            this.assessmentId = assessmentId;
            return this;
        }

        public final String getDirectoryId() {
            return directoryId;
        }

        public final void setDirectoryId(String directoryId) {
            this.directoryId = directoryId;
        }

        @Override
        public final Builder directoryId(String directoryId) {
            this.directoryId = directoryId;
            return this;
        }

        public final String getDnsName() {
            return dnsName;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getLastUpdateDateTime() {
            return lastUpdateDateTime;
        }

        public final void setLastUpdateDateTime(Instant lastUpdateDateTime) {
            this.lastUpdateDateTime = lastUpdateDateTime;
        }

        @Override
        public final Builder lastUpdateDateTime(Instant lastUpdateDateTime) {
            this.lastUpdateDateTime = lastUpdateDateTime;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final Collection<String> getCustomerDnsIps() {
            if (customerDnsIps instanceof SdkAutoConstructList) {
                return null;
            }
            return customerDnsIps;
        }

        public final void setCustomerDnsIps(Collection<String> customerDnsIps) {
            this.customerDnsIps = CustomerDnsIpsCopier.copy(customerDnsIps);
        }

        @Override
        public final Builder customerDnsIps(Collection<String> customerDnsIps) {
            this.customerDnsIps = CustomerDnsIpsCopier.copy(customerDnsIps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder customerDnsIps(String... customerDnsIps) {
            customerDnsIps(Arrays.asList(customerDnsIps));
            return this;
        }

        public final String getReportType() {
            return reportType;
        }

        public final void setReportType(String reportType) {
            this.reportType = reportType;
        }

        @Override
        public final Builder reportType(String reportType) {
            this.reportType = reportType;
            return this;
        }

        @Override
        public AssessmentSummary build() {
            return new AssessmentSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
