/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directory.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the network configuration for directory update operations.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NetworkUpdateSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<NetworkUpdateSettings.Builder, NetworkUpdateSettings> {
    private static final SdkField<String> NETWORK_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NetworkType").getter(getter(NetworkUpdateSettings::networkTypeAsString))
            .setter(setter(Builder::networkType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkType").build()).build();

    private static final SdkField<List<String>> CUSTOMER_DNS_IPS_V6_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("CustomerDnsIpsV6")
            .getter(getter(NetworkUpdateSettings::customerDnsIpsV6))
            .setter(setter(Builder::customerDnsIpsV6))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerDnsIpsV6").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NETWORK_TYPE_FIELD,
            CUSTOMER_DNS_IPS_V6_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String networkType;

    private final List<String> customerDnsIpsV6;

    private NetworkUpdateSettings(BuilderImpl builder) {
        this.networkType = builder.networkType;
        this.customerDnsIpsV6 = builder.customerDnsIpsV6;
    }

    /**
     * <p>
     * The target network type for the directory update.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #networkType} will
     * return {@link NetworkType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #networkTypeAsString}.
     * </p>
     * 
     * @return The target network type for the directory update.
     * @see NetworkType
     */
    public final NetworkType networkType() {
        return NetworkType.fromValue(networkType);
    }

    /**
     * <p>
     * The target network type for the directory update.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #networkType} will
     * return {@link NetworkType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #networkTypeAsString}.
     * </p>
     * 
     * @return The target network type for the directory update.
     * @see NetworkType
     */
    public final String networkTypeAsString() {
        return networkType;
    }

    /**
     * For responses, this returns true if the service returned a value for the CustomerDnsIpsV6 property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCustomerDnsIpsV6() {
        return customerDnsIpsV6 != null && !(customerDnsIpsV6 instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * IPv6 addresses of DNS servers or domain controllers in the self-managed directory. Required only when updating an
     * AD Connector directory.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCustomerDnsIpsV6} method.
     * </p>
     * 
     * @return IPv6 addresses of DNS servers or domain controllers in the self-managed directory. Required only when
     *         updating an AD Connector directory.
     */
    public final List<String> customerDnsIpsV6() {
        return customerDnsIpsV6;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(networkTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasCustomerDnsIpsV6() ? customerDnsIpsV6() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NetworkUpdateSettings)) {
            return false;
        }
        NetworkUpdateSettings other = (NetworkUpdateSettings) obj;
        return Objects.equals(networkTypeAsString(), other.networkTypeAsString())
                && hasCustomerDnsIpsV6() == other.hasCustomerDnsIpsV6()
                && Objects.equals(customerDnsIpsV6(), other.customerDnsIpsV6());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NetworkUpdateSettings").add("NetworkType", networkTypeAsString())
                .add("CustomerDnsIpsV6", hasCustomerDnsIpsV6() ? customerDnsIpsV6() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NetworkType":
            return Optional.ofNullable(clazz.cast(networkTypeAsString()));
        case "CustomerDnsIpsV6":
            return Optional.ofNullable(clazz.cast(customerDnsIpsV6()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("NetworkType", NETWORK_TYPE_FIELD);
        map.put("CustomerDnsIpsV6", CUSTOMER_DNS_IPS_V6_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<NetworkUpdateSettings, T> g) {
        return obj -> g.apply((NetworkUpdateSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NetworkUpdateSettings> {
        /**
         * <p>
         * The target network type for the directory update.
         * </p>
         * 
         * @param networkType
         *        The target network type for the directory update.
         * @see NetworkType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkType
         */
        Builder networkType(String networkType);

        /**
         * <p>
         * The target network type for the directory update.
         * </p>
         * 
         * @param networkType
         *        The target network type for the directory update.
         * @see NetworkType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkType
         */
        Builder networkType(NetworkType networkType);

        /**
         * <p>
         * IPv6 addresses of DNS servers or domain controllers in the self-managed directory. Required only when
         * updating an AD Connector directory.
         * </p>
         * 
         * @param customerDnsIpsV6
         *        IPv6 addresses of DNS servers or domain controllers in the self-managed directory. Required only when
         *        updating an AD Connector directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerDnsIpsV6(Collection<String> customerDnsIpsV6);

        /**
         * <p>
         * IPv6 addresses of DNS servers or domain controllers in the self-managed directory. Required only when
         * updating an AD Connector directory.
         * </p>
         * 
         * @param customerDnsIpsV6
         *        IPv6 addresses of DNS servers or domain controllers in the self-managed directory. Required only when
         *        updating an AD Connector directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerDnsIpsV6(String... customerDnsIpsV6);
    }

    static final class BuilderImpl implements Builder {
        private String networkType;

        private List<String> customerDnsIpsV6 = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(NetworkUpdateSettings model) {
            networkType(model.networkType);
            customerDnsIpsV6(model.customerDnsIpsV6);
        }

        public final String getNetworkType() {
            return networkType;
        }

        public final void setNetworkType(String networkType) {
            this.networkType = networkType;
        }

        @Override
        public final Builder networkType(String networkType) {
            this.networkType = networkType;
            return this;
        }

        @Override
        public final Builder networkType(NetworkType networkType) {
            this.networkType(networkType == null ? null : networkType.toString());
            return this;
        }

        public final Collection<String> getCustomerDnsIpsV6() {
            if (customerDnsIpsV6 instanceof SdkAutoConstructList) {
                return null;
            }
            return customerDnsIpsV6;
        }

        public final void setCustomerDnsIpsV6(Collection<String> customerDnsIpsV6) {
            this.customerDnsIpsV6 = DnsIpv6AddrsCopier.copy(customerDnsIpsV6);
        }

        @Override
        public final Builder customerDnsIpsV6(Collection<String> customerDnsIpsV6) {
            this.customerDnsIpsV6 = DnsIpv6AddrsCopier.copy(customerDnsIpsV6);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder customerDnsIpsV6(String... customerDnsIpsV6) {
            customerDnsIpsV6(Arrays.asList(customerDnsIpsV6));
            return this;
        }

        @Override
        public NetworkUpdateSettings build() {
            return new NetworkUpdateSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
