/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directory.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.directory.DirectoryClient;
import software.amazon.awssdk.services.directory.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.directory.model.DescribeHybridAdUpdateRequest;
import software.amazon.awssdk.services.directory.model.DescribeHybridAdUpdateResponse;
import software.amazon.awssdk.services.directory.model.DirectoryRequest;
import software.amazon.awssdk.services.directory.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultDirectoryWaiter implements DirectoryWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final DirectoryClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeHybridAdUpdateResponse> hybridADUpdatedWaiter;

    private DefaultDirectoryWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = DirectoryClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.hybridADUpdatedWaiter = Waiter.builder(DescribeHybridAdUpdateResponse.class)
                .acceptors(hybridADUpdatedWaiterAcceptors())
                .overrideConfiguration(hybridADUpdatedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeHybridAdUpdateResponse> waitUntilHybridADUpdated(
            DescribeHybridAdUpdateRequest describeHybridAdUpdateRequest) {
        return hybridADUpdatedWaiter.run(() -> client
                .describeHybridADUpdate(applyWaitersUserAgent(describeHybridAdUpdateRequest)));
    }

    @Override
    public WaiterResponse<DescribeHybridAdUpdateResponse> waitUntilHybridADUpdated(
            DescribeHybridAdUpdateRequest describeHybridAdUpdateRequest, WaiterOverrideConfiguration overrideConfig) {
        return hybridADUpdatedWaiter.run(
                () -> client.describeHybridADUpdate(applyWaitersUserAgent(describeHybridAdUpdateRequest)),
                hybridADUpdatedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeHybridAdUpdateResponse>> hybridADUpdatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeHybridAdUpdateResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("UpdateActivities").field("SelfManagedInstances").flatten().field("Status")
                    .values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "Updated"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("UpdateActivities").field("SelfManagedInstances").flatten()
                            .field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "UpdateFailed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (UpdateActivities.SelfManagedInstances[].Status=UpdateFailed) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration hybridADUpdatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(120)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static DirectoryWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends DirectoryRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements DirectoryWaiter.Builder {
        private DirectoryClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public DirectoryWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public DirectoryWaiter.Builder client(DirectoryClient client) {
            this.client = client;
            return this;
        }

        public DirectoryWaiter build() {
            return new DefaultDirectoryWaiter(this);
        }
    }
}
