/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directoryservicedata;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.directoryservicedata.internal.DirectoryServiceDataServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.directoryservicedata.model.AccessDeniedException;
import software.amazon.awssdk.services.directoryservicedata.model.AddGroupMemberRequest;
import software.amazon.awssdk.services.directoryservicedata.model.AddGroupMemberResponse;
import software.amazon.awssdk.services.directoryservicedata.model.ConflictException;
import software.amazon.awssdk.services.directoryservicedata.model.CreateGroupRequest;
import software.amazon.awssdk.services.directoryservicedata.model.CreateGroupResponse;
import software.amazon.awssdk.services.directoryservicedata.model.CreateUserRequest;
import software.amazon.awssdk.services.directoryservicedata.model.CreateUserResponse;
import software.amazon.awssdk.services.directoryservicedata.model.DeleteGroupRequest;
import software.amazon.awssdk.services.directoryservicedata.model.DeleteGroupResponse;
import software.amazon.awssdk.services.directoryservicedata.model.DeleteUserRequest;
import software.amazon.awssdk.services.directoryservicedata.model.DeleteUserResponse;
import software.amazon.awssdk.services.directoryservicedata.model.DescribeGroupRequest;
import software.amazon.awssdk.services.directoryservicedata.model.DescribeGroupResponse;
import software.amazon.awssdk.services.directoryservicedata.model.DescribeUserRequest;
import software.amazon.awssdk.services.directoryservicedata.model.DescribeUserResponse;
import software.amazon.awssdk.services.directoryservicedata.model.DirectoryServiceDataException;
import software.amazon.awssdk.services.directoryservicedata.model.DirectoryUnavailableException;
import software.amazon.awssdk.services.directoryservicedata.model.DisableUserRequest;
import software.amazon.awssdk.services.directoryservicedata.model.DisableUserResponse;
import software.amazon.awssdk.services.directoryservicedata.model.InternalServerException;
import software.amazon.awssdk.services.directoryservicedata.model.ListGroupMembersRequest;
import software.amazon.awssdk.services.directoryservicedata.model.ListGroupMembersResponse;
import software.amazon.awssdk.services.directoryservicedata.model.ListGroupsForMemberRequest;
import software.amazon.awssdk.services.directoryservicedata.model.ListGroupsForMemberResponse;
import software.amazon.awssdk.services.directoryservicedata.model.ListGroupsRequest;
import software.amazon.awssdk.services.directoryservicedata.model.ListGroupsResponse;
import software.amazon.awssdk.services.directoryservicedata.model.ListUsersRequest;
import software.amazon.awssdk.services.directoryservicedata.model.ListUsersResponse;
import software.amazon.awssdk.services.directoryservicedata.model.RemoveGroupMemberRequest;
import software.amazon.awssdk.services.directoryservicedata.model.RemoveGroupMemberResponse;
import software.amazon.awssdk.services.directoryservicedata.model.ResourceNotFoundException;
import software.amazon.awssdk.services.directoryservicedata.model.SearchGroupsRequest;
import software.amazon.awssdk.services.directoryservicedata.model.SearchGroupsResponse;
import software.amazon.awssdk.services.directoryservicedata.model.SearchUsersRequest;
import software.amazon.awssdk.services.directoryservicedata.model.SearchUsersResponse;
import software.amazon.awssdk.services.directoryservicedata.model.ThrottlingException;
import software.amazon.awssdk.services.directoryservicedata.model.UpdateGroupRequest;
import software.amazon.awssdk.services.directoryservicedata.model.UpdateGroupResponse;
import software.amazon.awssdk.services.directoryservicedata.model.UpdateUserRequest;
import software.amazon.awssdk.services.directoryservicedata.model.UpdateUserResponse;
import software.amazon.awssdk.services.directoryservicedata.model.ValidationException;
import software.amazon.awssdk.services.directoryservicedata.transform.AddGroupMemberRequestMarshaller;
import software.amazon.awssdk.services.directoryservicedata.transform.CreateGroupRequestMarshaller;
import software.amazon.awssdk.services.directoryservicedata.transform.CreateUserRequestMarshaller;
import software.amazon.awssdk.services.directoryservicedata.transform.DeleteGroupRequestMarshaller;
import software.amazon.awssdk.services.directoryservicedata.transform.DeleteUserRequestMarshaller;
import software.amazon.awssdk.services.directoryservicedata.transform.DescribeGroupRequestMarshaller;
import software.amazon.awssdk.services.directoryservicedata.transform.DescribeUserRequestMarshaller;
import software.amazon.awssdk.services.directoryservicedata.transform.DisableUserRequestMarshaller;
import software.amazon.awssdk.services.directoryservicedata.transform.ListGroupMembersRequestMarshaller;
import software.amazon.awssdk.services.directoryservicedata.transform.ListGroupsForMemberRequestMarshaller;
import software.amazon.awssdk.services.directoryservicedata.transform.ListGroupsRequestMarshaller;
import software.amazon.awssdk.services.directoryservicedata.transform.ListUsersRequestMarshaller;
import software.amazon.awssdk.services.directoryservicedata.transform.RemoveGroupMemberRequestMarshaller;
import software.amazon.awssdk.services.directoryservicedata.transform.SearchGroupsRequestMarshaller;
import software.amazon.awssdk.services.directoryservicedata.transform.SearchUsersRequestMarshaller;
import software.amazon.awssdk.services.directoryservicedata.transform.UpdateGroupRequestMarshaller;
import software.amazon.awssdk.services.directoryservicedata.transform.UpdateUserRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link DirectoryServiceDataAsyncClient}.
 *
 * @see DirectoryServiceDataAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDirectoryServiceDataAsyncClient implements DirectoryServiceDataAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultDirectoryServiceDataAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultDirectoryServiceDataAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Adds an existing user, group, or computer as a group member.
     * </p>
     *
     * @param addGroupMemberRequest
     * @return A Java Future containing the result of the AddGroupMember operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ConflictException This error will occur when you try to create a resource that conflicts with an
     *         existing object. It can also occur when adding a member to a group that the member is already in.</p>
     *         <p>
     *         This error can be caused by a request sent within the 8-hour idempotency window with the same client
     *         token but different input parameters. Client tokens should not be re-used across different requests.
     *         After 8 hours, any request with the same client token is treated as a new request.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.AddGroupMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/AddGroupMember"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddGroupMemberResponse> addGroupMember(AddGroupMemberRequest addGroupMemberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(addGroupMemberRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addGroupMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddGroupMember");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddGroupMemberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddGroupMemberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddGroupMemberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddGroupMemberRequest, AddGroupMemberResponse>()
                            .withOperationName("AddGroupMember").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AddGroupMemberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(addGroupMemberRequest));
            CompletableFuture<AddGroupMemberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new group.
     * </p>
     *
     * @param createGroupRequest
     * @return A Java Future containing the result of the CreateGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ConflictException This error will occur when you try to create a resource that conflicts with an
     *         existing object. It can also occur when adding a member to a group that the member is already in.</p>
     *         <p>
     *         This error can be caused by a request sent within the 8-hour idempotency window with the same client
     *         token but different input parameters. Client tokens should not be re-used across different requests.
     *         After 8 hours, any request with the same client token is treated as a new request.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.CreateGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/CreateGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGroupResponse> createGroup(CreateGroupRequest createGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGroupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGroupRequest, CreateGroupResponse>()
                            .withOperationName("CreateGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createGroupRequest));
            CompletableFuture<CreateGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new user.
     * </p>
     *
     * @param createUserRequest
     * @return A Java Future containing the result of the CreateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ConflictException This error will occur when you try to create a resource that conflicts with an
     *         existing object. It can also occur when adding a member to a group that the member is already in.</p>
     *         <p>
     *         This error can be caused by a request sent within the 8-hour idempotency window with the same client
     *         token but different input parameters. Client tokens should not be re-used across different requests.
     *         After 8 hours, any request with the same client token is treated as a new request.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.CreateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/CreateUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateUserResponse> createUser(CreateUserRequest createUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateUserRequest, CreateUserResponse>().withOperationName("CreateUser")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createUserRequest));
            CompletableFuture<CreateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a group.
     * </p>
     *
     * @param deleteGroupRequest
     * @return A Java Future containing the result of the DeleteGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ConflictException This error will occur when you try to create a resource that conflicts with an
     *         existing object. It can also occur when adding a member to a group that the member is already in.</p>
     *         <p>
     *         This error can be caused by a request sent within the 8-hour idempotency window with the same client
     *         token but different input parameters. Client tokens should not be re-used across different requests.
     *         After 8 hours, any request with the same client token is treated as a new request.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.DeleteGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/DeleteGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGroupResponse> deleteGroup(DeleteGroupRequest deleteGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteGroupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGroupRequest, DeleteGroupResponse>()
                            .withOperationName("DeleteGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteGroupRequest));
            CompletableFuture<DeleteGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a user.
     * </p>
     *
     * @param deleteUserRequest
     * @return A Java Future containing the result of the DeleteUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ConflictException This error will occur when you try to create a resource that conflicts with an
     *         existing object. It can also occur when adding a member to a group that the member is already in.</p>
     *         <p>
     *         This error can be caused by a request sent within the 8-hour idempotency window with the same client
     *         token but different input parameters. Client tokens should not be re-used across different requests.
     *         After 8 hours, any request with the same client token is treated as a new request.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.DeleteUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/DeleteUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteUserResponse> deleteUser(DeleteUserRequest deleteUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteUserRequest, DeleteUserResponse>().withOperationName("DeleteUser")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteUserRequest));
            CompletableFuture<DeleteUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a specific group.
     * </p>
     *
     * @param describeGroupRequest
     * @return A Java Future containing the result of the DescribeGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.DescribeGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/DescribeGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGroupResponse> describeGroup(DescribeGroupRequest describeGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeGroupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGroupRequest, DescribeGroupResponse>()
                            .withOperationName("DescribeGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeGroupRequest));
            CompletableFuture<DescribeGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a specific user.
     * </p>
     *
     * @param describeUserRequest
     * @return A Java Future containing the result of the DescribeUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.DescribeUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/DescribeUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeUserResponse> describeUser(DescribeUserRequest describeUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeUserRequest, DescribeUserResponse>()
                            .withOperationName("DescribeUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeUserRequest));
            CompletableFuture<DescribeUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deactivates an active user account. For information about how to enable an inactive user account, see <a
     * href="https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ResetUserPassword.html"
     * >ResetUserPassword</a> in the <i>Directory Service API Reference</i>.
     * </p>
     *
     * @param disableUserRequest
     * @return A Java Future containing the result of the DisableUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ConflictException This error will occur when you try to create a resource that conflicts with an
     *         existing object. It can also occur when adding a member to a group that the member is already in.</p>
     *         <p>
     *         This error can be caused by a request sent within the 8-hour idempotency window with the same client
     *         token but different input parameters. Client tokens should not be re-used across different requests.
     *         After 8 hours, any request with the same client token is treated as a new request.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.DisableUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/DisableUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableUserResponse> disableUser(DisableUserRequest disableUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DisableUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableUserRequest, DisableUserResponse>()
                            .withOperationName("DisableUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisableUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disableUserRequest));
            CompletableFuture<DisableUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns member information for the specified group.
     * </p>
     * <p>
     * This operation supports pagination with the use of the <code>NextToken</code> request and response parameters. If
     * more results are available, the <code>ListGroupMembers.NextToken</code> member contains a token that you pass in
     * the next call to <code>ListGroupMembers</code>. This retrieves the next set of items.
     * </p>
     * <p>
     * You can also specify a maximum number of return results with the <code>MaxResults</code> parameter.
     * </p>
     *
     * @param listGroupMembersRequest
     * @return A Java Future containing the result of the ListGroupMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.ListGroupMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/ListGroupMembers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListGroupMembersResponse> listGroupMembers(ListGroupMembersRequest listGroupMembersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGroupMembersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGroupMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroupMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGroupMembersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListGroupMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGroupMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGroupMembersRequest, ListGroupMembersResponse>()
                            .withOperationName("ListGroupMembers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListGroupMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listGroupMembersRequest));
            CompletableFuture<ListGroupMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns group information for the specified directory.
     * </p>
     * <p>
     * This operation supports pagination with the use of the <code>NextToken</code> request and response parameters. If
     * more results are available, the <code>ListGroups.NextToken</code> member contains a token that you pass in the
     * next call to <code>ListGroups</code>. This retrieves the next set of items.
     * </p>
     * <p>
     * You can also specify a maximum number of return results with the <code>MaxResults</code> parameter.
     * </p>
     *
     * @param listGroupsRequest
     * @return A Java Future containing the result of the ListGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.ListGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/ListGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListGroupsResponse> listGroups(ListGroupsRequest listGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGroupsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGroupsRequest, ListGroupsResponse>().withOperationName("ListGroups")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listGroupsRequest));
            CompletableFuture<ListGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns group information for the specified member.
     * </p>
     * <p>
     * This operation supports pagination with the use of the <code>NextToken</code> request and response parameters. If
     * more results are available, the <code>ListGroupsForMember.NextToken</code> member contains a token that you pass
     * in the next call to <code>ListGroupsForMember</code>. This retrieves the next set of items.
     * </p>
     * <p>
     * You can also specify a maximum number of return results with the <code>MaxResults</code> parameter.
     * </p>
     *
     * @param listGroupsForMemberRequest
     * @return A Java Future containing the result of the ListGroupsForMember operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.ListGroupsForMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/ListGroupsForMember"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListGroupsForMemberResponse> listGroupsForMember(
            ListGroupsForMemberRequest listGroupsForMemberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGroupsForMemberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGroupsForMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroupsForMember");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGroupsForMemberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListGroupsForMemberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGroupsForMemberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGroupsForMemberRequest, ListGroupsForMemberResponse>()
                            .withOperationName("ListGroupsForMember").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListGroupsForMemberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listGroupsForMemberRequest));
            CompletableFuture<ListGroupsForMemberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns user information for the specified directory.
     * </p>
     * <p>
     * This operation supports pagination with the use of the <code>NextToken</code> request and response parameters. If
     * more results are available, the <code>ListUsers.NextToken</code> member contains a token that you pass in the
     * next call to <code>ListUsers</code>. This retrieves the next set of items.
     * </p>
     * <p>
     * You can also specify a maximum number of return results with the <code>MaxResults</code> parameter.
     * </p>
     *
     * @param listUsersRequest
     * @return A Java Future containing the result of the ListUsers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/ListUsers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListUsersResponse> listUsers(ListUsersRequest listUsersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listUsersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUsers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListUsersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListUsersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListUsersRequest, ListUsersResponse>().withOperationName("ListUsers")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListUsersRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listUsersRequest));
            CompletableFuture<ListUsersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a member from a group.
     * </p>
     *
     * @param removeGroupMemberRequest
     * @return A Java Future containing the result of the RemoveGroupMember operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ConflictException This error will occur when you try to create a resource that conflicts with an
     *         existing object. It can also occur when adding a member to a group that the member is already in.</p>
     *         <p>
     *         This error can be caused by a request sent within the 8-hour idempotency window with the same client
     *         token but different input parameters. Client tokens should not be re-used across different requests.
     *         After 8 hours, any request with the same client token is treated as a new request.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.RemoveGroupMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/RemoveGroupMember"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveGroupMemberResponse> removeGroupMember(RemoveGroupMemberRequest removeGroupMemberRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(removeGroupMemberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeGroupMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveGroupMember");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveGroupMemberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveGroupMemberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveGroupMemberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveGroupMemberRequest, RemoveGroupMemberResponse>()
                            .withOperationName("RemoveGroupMember").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RemoveGroupMemberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(removeGroupMemberRequest));
            CompletableFuture<RemoveGroupMemberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Searches the specified directory for a group. You can find groups that match the <code>SearchString</code>
     * parameter with the value of their attributes included in the <code>SearchString</code> parameter.
     * </p>
     * <p>
     * This operation supports pagination with the use of the <code>NextToken</code> request and response parameters. If
     * more results are available, the <code>SearchGroups.NextToken</code> member contains a token that you pass in the
     * next call to <code>SearchGroups</code>. This retrieves the next set of items.
     * </p>
     * <p>
     * You can also specify a maximum number of return results with the <code>MaxResults</code> parameter.
     * </p>
     *
     * @param searchGroupsRequest
     * @return A Java Future containing the result of the SearchGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.SearchGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/SearchGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchGroupsResponse> searchGroups(SearchGroupsRequest searchGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchGroupsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchGroupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SearchGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchGroupsRequest, SearchGroupsResponse>()
                            .withOperationName("SearchGroups").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SearchGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(searchGroupsRequest));
            CompletableFuture<SearchGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Searches the specified directory for a user. You can find users that match the <code>SearchString</code>
     * parameter with the value of their attributes included in the <code>SearchString</code> parameter.
     * </p>
     * <p>
     * This operation supports pagination with the use of the <code>NextToken</code> request and response parameters. If
     * more results are available, the <code>SearchUsers.NextToken</code> member contains a token that you pass in the
     * next call to <code>SearchUsers</code>. This retrieves the next set of items.
     * </p>
     * <p>
     * You can also specify a maximum number of return results with the <code>MaxResults</code> parameter.
     * </p>
     *
     * @param searchUsersRequest
     * @return A Java Future containing the result of the SearchUsers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.SearchUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/SearchUsers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchUsersResponse> searchUsers(SearchUsersRequest searchUsersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchUsersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchUsers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SearchUsersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchUsersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchUsersRequest, SearchUsersResponse>()
                            .withOperationName("SearchUsers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SearchUsersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(searchUsersRequest));
            CompletableFuture<SearchUsersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates group information.
     * </p>
     *
     * @param updateGroupRequest
     * @return A Java Future containing the result of the UpdateGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ConflictException This error will occur when you try to create a resource that conflicts with an
     *         existing object. It can also occur when adding a member to a group that the member is already in.</p>
     *         <p>
     *         This error can be caused by a request sent within the 8-hour idempotency window with the same client
     *         token but different input parameters. Client tokens should not be re-used across different requests.
     *         After 8 hours, any request with the same client token is treated as a new request.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.UpdateGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/UpdateGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGroupResponse> updateGroup(UpdateGroupRequest updateGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateGroupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGroupRequest, UpdateGroupResponse>()
                            .withOperationName("UpdateGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateGroupRequest));
            CompletableFuture<UpdateGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates user information.
     * </p>
     *
     * @param updateUserRequest
     * @return A Java Future containing the result of the UpdateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource couldn't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the request or access the directory. It
     *         can also occur when the <code>DirectoryId</code> doesn't exist or the user, member, or group might be
     *         outside of your organizational unit (OU). </p>
     *         <p>
     *         Make sure that you have the authentication and authorization to perform the action. Review the directory
     *         information in the request, and make sure that the object isn't outside of your OU.</li>
     *         <li>InternalServerException The operation didn't succeed because an internal error occurred. Try again
     *         later.</li>
     *         <li>ValidationException The request isn't valid. Review the details in the error message to update the
     *         invalid parameters or values in your request.</li>
     *         <li>DirectoryUnavailableException The request could not be completed due to a problem in the
     *         configuration or current state of the specified directory.</li>
     *         <li>ConflictException This error will occur when you try to create a resource that conflicts with an
     *         existing object. It can also occur when adding a member to a group that the member is already in.</p>
     *         <p>
     *         This error can be caused by a request sent within the 8-hour idempotency window with the same client
     *         token but different input parameters. Client tokens should not be re-used across different requests.
     *         After 8 hours, any request with the same client token is treated as a new request.</li>
     *         <li>ThrottlingException The limit on the number of requests per second has been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DirectoryServiceDataException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample DirectoryServiceDataAsyncClient.UpdateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/directory-service-data-2023-05-31/UpdateUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateUserResponse> updateUser(UpdateUserRequest updateUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Directory Service Data");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateUserRequest, UpdateUserResponse>().withOperationName("UpdateUser")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateUserRequest));
            CompletableFuture<UpdateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final DirectoryServiceDataServiceClientConfiguration serviceClientConfiguration() {
        return new DirectoryServiceDataServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DirectoryServiceDataException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DirectoryUnavailableException")
                                .exceptionBuilderSupplier(DirectoryUnavailableException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        DirectoryServiceDataServiceClientConfigurationBuilder serviceConfigBuilder = new DirectoryServiceDataServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
