/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dlm;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.dlm.model.CreateLifecyclePolicyRequest;
import software.amazon.awssdk.services.dlm.model.CreateLifecyclePolicyResponse;
import software.amazon.awssdk.services.dlm.model.DeleteLifecyclePolicyRequest;
import software.amazon.awssdk.services.dlm.model.DeleteLifecyclePolicyResponse;
import software.amazon.awssdk.services.dlm.model.DlmException;
import software.amazon.awssdk.services.dlm.model.GetLifecyclePoliciesRequest;
import software.amazon.awssdk.services.dlm.model.GetLifecyclePoliciesResponse;
import software.amazon.awssdk.services.dlm.model.GetLifecyclePolicyRequest;
import software.amazon.awssdk.services.dlm.model.GetLifecyclePolicyResponse;
import software.amazon.awssdk.services.dlm.model.InternalServerException;
import software.amazon.awssdk.services.dlm.model.InvalidRequestException;
import software.amazon.awssdk.services.dlm.model.LimitExceededException;
import software.amazon.awssdk.services.dlm.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.dlm.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.dlm.model.ResourceNotFoundException;
import software.amazon.awssdk.services.dlm.model.TagResourceRequest;
import software.amazon.awssdk.services.dlm.model.TagResourceResponse;
import software.amazon.awssdk.services.dlm.model.UntagResourceRequest;
import software.amazon.awssdk.services.dlm.model.UntagResourceResponse;
import software.amazon.awssdk.services.dlm.model.UpdateLifecyclePolicyRequest;
import software.amazon.awssdk.services.dlm.model.UpdateLifecyclePolicyResponse;
import software.amazon.awssdk.services.dlm.transform.CreateLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.dlm.transform.DeleteLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.dlm.transform.GetLifecyclePoliciesRequestMarshaller;
import software.amazon.awssdk.services.dlm.transform.GetLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.dlm.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.dlm.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.dlm.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.dlm.transform.UpdateLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link DlmClient}.
 *
 * @see DlmClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDlmClient implements DlmClient {
    private static final Logger log = Logger.loggerFor(DefaultDlmClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultDlmClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a policy to manage the lifecycle of the specified AWS resources. You can create up to 100 lifecycle
     * policies.
     * </p>
     *
     * @param createLifecyclePolicyRequest
     * @return Result of the CreateLifecyclePolicy operation returned by the service.
     * @throws InvalidRequestException
     *         Bad request. The request is missing required parameters or has invalid parameters.
     * @throws LimitExceededException
     *         The request failed because a limit was exceeded.
     * @throws InternalServerException
     *         The service failed in an unexpected way.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DlmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DlmClient.CreateLifecyclePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dlm-2018-01-12/CreateLifecyclePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateLifecyclePolicyResponse createLifecyclePolicy(CreateLifecyclePolicyRequest createLifecyclePolicyRequest)
            throws InvalidRequestException, LimitExceededException, InternalServerException, AwsServiceException,
            SdkClientException, DlmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateLifecyclePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLifecyclePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DLM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLifecyclePolicy");

            return clientHandler.execute(new ClientExecutionParams<CreateLifecyclePolicyRequest, CreateLifecyclePolicyResponse>()
                    .withOperationName("CreateLifecyclePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createLifecyclePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateLifecyclePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified lifecycle policy and halts the automated operations that the policy specified.
     * </p>
     *
     * @param deleteLifecyclePolicyRequest
     * @return Result of the DeleteLifecyclePolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         A requested resource was not found.
     * @throws InternalServerException
     *         The service failed in an unexpected way.
     * @throws LimitExceededException
     *         The request failed because a limit was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DlmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DlmClient.DeleteLifecyclePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dlm-2018-01-12/DeleteLifecyclePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteLifecyclePolicyResponse deleteLifecyclePolicy(DeleteLifecyclePolicyRequest deleteLifecyclePolicyRequest)
            throws ResourceNotFoundException, InternalServerException, LimitExceededException, AwsServiceException,
            SdkClientException, DlmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteLifecyclePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLifecyclePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DLM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLifecyclePolicy");

            return clientHandler.execute(new ClientExecutionParams<DeleteLifecyclePolicyRequest, DeleteLifecyclePolicyResponse>()
                    .withOperationName("DeleteLifecyclePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteLifecyclePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteLifecyclePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets summary information about all or the specified data lifecycle policies.
     * </p>
     * <p>
     * To get complete information about a policy, use <a>GetLifecyclePolicy</a>.
     * </p>
     *
     * @param getLifecyclePoliciesRequest
     * @return Result of the GetLifecyclePolicies operation returned by the service.
     * @throws ResourceNotFoundException
     *         A requested resource was not found.
     * @throws InvalidRequestException
     *         Bad request. The request is missing required parameters or has invalid parameters.
     * @throws InternalServerException
     *         The service failed in an unexpected way.
     * @throws LimitExceededException
     *         The request failed because a limit was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DlmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DlmClient.GetLifecyclePolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dlm-2018-01-12/GetLifecyclePolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetLifecyclePoliciesResponse getLifecyclePolicies(GetLifecyclePoliciesRequest getLifecyclePoliciesRequest)
            throws ResourceNotFoundException, InvalidRequestException, InternalServerException, LimitExceededException,
            AwsServiceException, SdkClientException, DlmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLifecyclePoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetLifecyclePoliciesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLifecyclePoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DLM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLifecyclePolicies");

            return clientHandler.execute(new ClientExecutionParams<GetLifecyclePoliciesRequest, GetLifecyclePoliciesResponse>()
                    .withOperationName("GetLifecyclePolicies").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getLifecyclePoliciesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetLifecyclePoliciesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets detailed information about the specified lifecycle policy.
     * </p>
     *
     * @param getLifecyclePolicyRequest
     * @return Result of the GetLifecyclePolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         A requested resource was not found.
     * @throws InternalServerException
     *         The service failed in an unexpected way.
     * @throws LimitExceededException
     *         The request failed because a limit was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DlmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DlmClient.GetLifecyclePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dlm-2018-01-12/GetLifecyclePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetLifecyclePolicyResponse getLifecyclePolicy(GetLifecyclePolicyRequest getLifecyclePolicyRequest)
            throws ResourceNotFoundException, InternalServerException, LimitExceededException, AwsServiceException,
            SdkClientException, DlmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetLifecyclePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLifecyclePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DLM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLifecyclePolicy");

            return clientHandler.execute(new ClientExecutionParams<GetLifecyclePolicyRequest, GetLifecyclePolicyResponse>()
                    .withOperationName("GetLifecyclePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getLifecyclePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetLifecyclePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags for the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServerException
     *         The service failed in an unexpected way.
     * @throws InvalidRequestException
     *         Bad request. The request is missing required parameters or has invalid parameters.
     * @throws ResourceNotFoundException
     *         A requested resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DlmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DlmClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dlm-2018-01-12/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServerException, InvalidRequestException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, DlmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DLM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds the specified tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServerException
     *         The service failed in an unexpected way.
     * @throws InvalidRequestException
     *         Bad request. The request is missing required parameters or has invalid parameters.
     * @throws ResourceNotFoundException
     *         A requested resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DlmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DlmClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dlm-2018-01-12/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalServerException,
            InvalidRequestException, ResourceNotFoundException, AwsServiceException, SdkClientException, DlmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DLM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServerException
     *         The service failed in an unexpected way.
     * @throws InvalidRequestException
     *         Bad request. The request is missing required parameters or has invalid parameters.
     * @throws ResourceNotFoundException
     *         A requested resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DlmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DlmClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dlm-2018-01-12/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalServerException,
            InvalidRequestException, ResourceNotFoundException, AwsServiceException, SdkClientException, DlmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DLM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified lifecycle policy.
     * </p>
     *
     * @param updateLifecyclePolicyRequest
     * @return Result of the UpdateLifecyclePolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         A requested resource was not found.
     * @throws InvalidRequestException
     *         Bad request. The request is missing required parameters or has invalid parameters.
     * @throws InternalServerException
     *         The service failed in an unexpected way.
     * @throws LimitExceededException
     *         The request failed because a limit was exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DlmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DlmClient.UpdateLifecyclePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dlm-2018-01-12/UpdateLifecyclePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateLifecyclePolicyResponse updateLifecyclePolicy(UpdateLifecyclePolicyRequest updateLifecyclePolicyRequest)
            throws ResourceNotFoundException, InvalidRequestException, InternalServerException, LimitExceededException,
            AwsServiceException, SdkClientException, DlmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateLifecyclePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLifecyclePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DLM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLifecyclePolicy");

            return clientHandler.execute(new ClientExecutionParams<UpdateLifecyclePolicyRequest, UpdateLifecyclePolicyResponse>()
                    .withOperationName("UpdateLifecyclePolicy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateLifecyclePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateLifecyclePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DlmException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
