/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dlm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <b>[All policy types]</b> Specifies the configuration of a lifecycle policy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PolicyDetails implements SdkPojo, Serializable, ToCopyableBuilder<PolicyDetails.Builder, PolicyDetails> {
    private static final SdkField<String> POLICY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyType").getter(getter(PolicyDetails::policyTypeAsString)).setter(setter(Builder::policyType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyType").build()).build();

    private static final SdkField<List<String>> RESOURCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourceTypes")
            .getter(getter(PolicyDetails::resourceTypesAsStrings))
            .setter(setter(Builder::resourceTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> RESOURCE_LOCATIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourceLocations")
            .getter(getter(PolicyDetails::resourceLocationsAsStrings))
            .setter(setter(Builder::resourceLocationsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceLocations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TARGET_TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("TargetTags")
            .getter(getter(PolicyDetails::targetTags))
            .setter(setter(Builder::targetTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Schedule>> SCHEDULES_FIELD = SdkField
            .<List<Schedule>> builder(MarshallingType.LIST)
            .memberName("Schedules")
            .getter(getter(PolicyDetails::schedules))
            .setter(setter(Builder::schedules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Schedules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Schedule> builder(MarshallingType.SDK_POJO)
                                            .constructor(Schedule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Parameters> PARAMETERS_FIELD = SdkField.<Parameters> builder(MarshallingType.SDK_POJO)
            .memberName("Parameters").getter(getter(PolicyDetails::parameters)).setter(setter(Builder::parameters))
            .constructor(Parameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build()).build();

    private static final SdkField<EventSource> EVENT_SOURCE_FIELD = SdkField.<EventSource> builder(MarshallingType.SDK_POJO)
            .memberName("EventSource").getter(getter(PolicyDetails::eventSource)).setter(setter(Builder::eventSource))
            .constructor(EventSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventSource").build()).build();

    private static final SdkField<List<Action>> ACTIONS_FIELD = SdkField
            .<List<Action>> builder(MarshallingType.LIST)
            .memberName("Actions")
            .getter(getter(PolicyDetails::actions))
            .setter(setter(Builder::actions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Actions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Action> builder(MarshallingType.SDK_POJO)
                                            .constructor(Action::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(POLICY_TYPE_FIELD,
            RESOURCE_TYPES_FIELD, RESOURCE_LOCATIONS_FIELD, TARGET_TAGS_FIELD, SCHEDULES_FIELD, PARAMETERS_FIELD,
            EVENT_SOURCE_FIELD, ACTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String policyType;

    private final List<String> resourceTypes;

    private final List<String> resourceLocations;

    private final List<Tag> targetTags;

    private final List<Schedule> schedules;

    private final Parameters parameters;

    private final EventSource eventSource;

    private final List<Action> actions;

    private PolicyDetails(BuilderImpl builder) {
        this.policyType = builder.policyType;
        this.resourceTypes = builder.resourceTypes;
        this.resourceLocations = builder.resourceLocations;
        this.targetTags = builder.targetTags;
        this.schedules = builder.schedules;
        this.parameters = builder.parameters;
        this.eventSource = builder.eventSource;
        this.actions = builder.actions;
    }

    /**
     * <p>
     * <b>[All policy types]</b> The valid target resource types and actions a policy can manage. Specify
     * <code>EBS_SNAPSHOT_MANAGEMENT</code> to create a lifecycle policy that manages the lifecycle of Amazon EBS
     * snapshots. Specify <code>IMAGE_MANAGEMENT</code> to create a lifecycle policy that manages the lifecycle of
     * EBS-backed AMIs. Specify <code>EVENT_BASED_POLICY </code> to create an event-based policy that performs specific
     * actions when a defined event occurs in your Amazon Web Services account.
     * </p>
     * <p>
     * The default is <code>EBS_SNAPSHOT_MANAGEMENT</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyType} will
     * return {@link PolicyTypeValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #policyTypeAsString}.
     * </p>
     * 
     * @return <b>[All policy types]</b> The valid target resource types and actions a policy can manage. Specify
     *         <code>EBS_SNAPSHOT_MANAGEMENT</code> to create a lifecycle policy that manages the lifecycle of Amazon
     *         EBS snapshots. Specify <code>IMAGE_MANAGEMENT</code> to create a lifecycle policy that manages the
     *         lifecycle of EBS-backed AMIs. Specify <code>EVENT_BASED_POLICY </code> to create an event-based policy
     *         that performs specific actions when a defined event occurs in your Amazon Web Services account.</p>
     *         <p>
     *         The default is <code>EBS_SNAPSHOT_MANAGEMENT</code>.
     * @see PolicyTypeValues
     */
    public final PolicyTypeValues policyType() {
        return PolicyTypeValues.fromValue(policyType);
    }

    /**
     * <p>
     * <b>[All policy types]</b> The valid target resource types and actions a policy can manage. Specify
     * <code>EBS_SNAPSHOT_MANAGEMENT</code> to create a lifecycle policy that manages the lifecycle of Amazon EBS
     * snapshots. Specify <code>IMAGE_MANAGEMENT</code> to create a lifecycle policy that manages the lifecycle of
     * EBS-backed AMIs. Specify <code>EVENT_BASED_POLICY </code> to create an event-based policy that performs specific
     * actions when a defined event occurs in your Amazon Web Services account.
     * </p>
     * <p>
     * The default is <code>EBS_SNAPSHOT_MANAGEMENT</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyType} will
     * return {@link PolicyTypeValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #policyTypeAsString}.
     * </p>
     * 
     * @return <b>[All policy types]</b> The valid target resource types and actions a policy can manage. Specify
     *         <code>EBS_SNAPSHOT_MANAGEMENT</code> to create a lifecycle policy that manages the lifecycle of Amazon
     *         EBS snapshots. Specify <code>IMAGE_MANAGEMENT</code> to create a lifecycle policy that manages the
     *         lifecycle of EBS-backed AMIs. Specify <code>EVENT_BASED_POLICY </code> to create an event-based policy
     *         that performs specific actions when a defined event occurs in your Amazon Web Services account.</p>
     *         <p>
     *         The default is <code>EBS_SNAPSHOT_MANAGEMENT</code>.
     * @see PolicyTypeValues
     */
    public final String policyTypeAsString() {
        return policyType;
    }

    /**
     * <p>
     * <b>[Snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
     * <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create multi-volume
     * snapshots from the volumes for an instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypes} method.
     * </p>
     * 
     * @return <b>[Snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
     *         <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create
     *         multi-volume snapshots from the volumes for an instance.
     */
    public final List<ResourceTypeValues> resourceTypes() {
        return ResourceTypeValuesListCopier.copyStringToEnum(resourceTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTypes() {
        return resourceTypes != null && !(resourceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <b>[Snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
     * <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create multi-volume
     * snapshots from the volumes for an instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypes} method.
     * </p>
     * 
     * @return <b>[Snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
     *         <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create
     *         multi-volume snapshots from the volumes for an instance.
     */
    public final List<String> resourceTypesAsStrings() {
        return resourceTypes;
    }

    /**
     * <p>
     * <b>[Snapshot and AMI policies only]</b> The location of the resources to backup. If the source resources are
     * located in an Amazon Web Services Region, specify <code>CLOUD</code>. If the source resources are located on an
     * Outpost in your account, specify <code>OUTPOST</code>.
     * </p>
     * <p>
     * If you specify <code>OUTPOST</code>, Amazon Data Lifecycle Manager backs up all resources of the specified type
     * with matching target tags across all of the Outposts in your account.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceLocations} method.
     * </p>
     * 
     * @return <b>[Snapshot and AMI policies only]</b> The location of the resources to backup. If the source resources
     *         are located in an Amazon Web Services Region, specify <code>CLOUD</code>. If the source resources are
     *         located on an Outpost in your account, specify <code>OUTPOST</code>.</p>
     *         <p>
     *         If you specify <code>OUTPOST</code>, Amazon Data Lifecycle Manager backs up all resources of the
     *         specified type with matching target tags across all of the Outposts in your account.
     */
    public final List<ResourceLocationValues> resourceLocations() {
        return ResourceLocationListCopier.copyStringToEnum(resourceLocations);
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceLocations property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceLocations() {
        return resourceLocations != null && !(resourceLocations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <b>[Snapshot and AMI policies only]</b> The location of the resources to backup. If the source resources are
     * located in an Amazon Web Services Region, specify <code>CLOUD</code>. If the source resources are located on an
     * Outpost in your account, specify <code>OUTPOST</code>.
     * </p>
     * <p>
     * If you specify <code>OUTPOST</code>, Amazon Data Lifecycle Manager backs up all resources of the specified type
     * with matching target tags across all of the Outposts in your account.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceLocations} method.
     * </p>
     * 
     * @return <b>[Snapshot and AMI policies only]</b> The location of the resources to backup. If the source resources
     *         are located in an Amazon Web Services Region, specify <code>CLOUD</code>. If the source resources are
     *         located on an Outpost in your account, specify <code>OUTPOST</code>.</p>
     *         <p>
     *         If you specify <code>OUTPOST</code>, Amazon Data Lifecycle Manager backs up all resources of the
     *         specified type with matching target tags across all of the Outposts in your account.
     */
    public final List<String> resourceLocationsAsStrings() {
        return resourceLocations;
    }

    /**
     * For responses, this returns true if the service returned a value for the TargetTags property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTargetTags() {
        return targetTags != null && !(targetTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <b>[Snapshot and AMI policies only]</b> The single tag that identifies targeted resources for this policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargetTags} method.
     * </p>
     * 
     * @return <b>[Snapshot and AMI policies only]</b> The single tag that identifies targeted resources for this
     *         policy.
     */
    public final List<Tag> targetTags() {
        return targetTags;
    }

    /**
     * For responses, this returns true if the service returned a value for the Schedules property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSchedules() {
        return schedules != null && !(schedules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <b>[Snapshot and AMI policies only]</b> The schedules of policy-defined actions for snapshot and AMI lifecycle
     * policies. A policy can have up to four schedules—one mandatory schedule and up to three optional schedules.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSchedules} method.
     * </p>
     * 
     * @return <b>[Snapshot and AMI policies only]</b> The schedules of policy-defined actions for snapshot and AMI
     *         lifecycle policies. A policy can have up to four schedules—one mandatory schedule and up to three
     *         optional schedules.
     */
    public final List<Schedule> schedules() {
        return schedules;
    }

    /**
     * <p>
     * <b>[Snapshot and AMI policies only]</b> A set of optional parameters for snapshot and AMI lifecycle policies.
     * </p>
     * <note>
     * <p>
     * If you are modifying a policy that was created or previously modified using the Amazon Data Lifecycle Manager
     * console, then you must include this parameter and specify either the default values or the new values that you
     * require. You can't omit this parameter or set its values to null.
     * </p>
     * </note>
     * 
     * @return <b>[Snapshot and AMI policies only]</b> A set of optional parameters for snapshot and AMI lifecycle
     *         policies. </p> <note>
     *         <p>
     *         If you are modifying a policy that was created or previously modified using the Amazon Data Lifecycle
     *         Manager console, then you must include this parameter and specify either the default values or the new
     *         values that you require. You can't omit this parameter or set its values to null.
     *         </p>
     */
    public final Parameters parameters() {
        return parameters;
    }

    /**
     * <p>
     * <b>[Event-based policies only]</b> The event that activates the event-based policy.
     * </p>
     * 
     * @return <b>[Event-based policies only]</b> The event that activates the event-based policy.
     */
    public final EventSource eventSource() {
        return eventSource;
    }

    /**
     * For responses, this returns true if the service returned a value for the Actions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasActions() {
        return actions != null && !(actions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <b>[Event-based policies only]</b> The actions to be performed when the event-based policy is activated. You can
     * specify only one action per policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasActions} method.
     * </p>
     * 
     * @return <b>[Event-based policies only]</b> The actions to be performed when the event-based policy is activated.
     *         You can specify only one action per policy.
     */
    public final List<Action> actions() {
        return actions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(policyTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTypes() ? resourceTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceLocations() ? resourceLocationsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTargetTags() ? targetTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSchedules() ? schedules() : null);
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(eventSource());
        hashCode = 31 * hashCode + Objects.hashCode(hasActions() ? actions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PolicyDetails)) {
            return false;
        }
        PolicyDetails other = (PolicyDetails) obj;
        return Objects.equals(policyTypeAsString(), other.policyTypeAsString()) && hasResourceTypes() == other.hasResourceTypes()
                && Objects.equals(resourceTypesAsStrings(), other.resourceTypesAsStrings())
                && hasResourceLocations() == other.hasResourceLocations()
                && Objects.equals(resourceLocationsAsStrings(), other.resourceLocationsAsStrings())
                && hasTargetTags() == other.hasTargetTags() && Objects.equals(targetTags(), other.targetTags())
                && hasSchedules() == other.hasSchedules() && Objects.equals(schedules(), other.schedules())
                && Objects.equals(parameters(), other.parameters()) && Objects.equals(eventSource(), other.eventSource())
                && hasActions() == other.hasActions() && Objects.equals(actions(), other.actions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PolicyDetails").add("PolicyType", policyTypeAsString())
                .add("ResourceTypes", hasResourceTypes() ? resourceTypesAsStrings() : null)
                .add("ResourceLocations", hasResourceLocations() ? resourceLocationsAsStrings() : null)
                .add("TargetTags", hasTargetTags() ? targetTags() : null).add("Schedules", hasSchedules() ? schedules() : null)
                .add("Parameters", parameters()).add("EventSource", eventSource())
                .add("Actions", hasActions() ? actions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PolicyType":
            return Optional.ofNullable(clazz.cast(policyTypeAsString()));
        case "ResourceTypes":
            return Optional.ofNullable(clazz.cast(resourceTypesAsStrings()));
        case "ResourceLocations":
            return Optional.ofNullable(clazz.cast(resourceLocationsAsStrings()));
        case "TargetTags":
            return Optional.ofNullable(clazz.cast(targetTags()));
        case "Schedules":
            return Optional.ofNullable(clazz.cast(schedules()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "EventSource":
            return Optional.ofNullable(clazz.cast(eventSource()));
        case "Actions":
            return Optional.ofNullable(clazz.cast(actions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PolicyDetails, T> g) {
        return obj -> g.apply((PolicyDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PolicyDetails> {
        /**
         * <p>
         * <b>[All policy types]</b> The valid target resource types and actions a policy can manage. Specify
         * <code>EBS_SNAPSHOT_MANAGEMENT</code> to create a lifecycle policy that manages the lifecycle of Amazon EBS
         * snapshots. Specify <code>IMAGE_MANAGEMENT</code> to create a lifecycle policy that manages the lifecycle of
         * EBS-backed AMIs. Specify <code>EVENT_BASED_POLICY </code> to create an event-based policy that performs
         * specific actions when a defined event occurs in your Amazon Web Services account.
         * </p>
         * <p>
         * The default is <code>EBS_SNAPSHOT_MANAGEMENT</code>.
         * </p>
         * 
         * @param policyType
         *        <b>[All policy types]</b> The valid target resource types and actions a policy can manage. Specify
         *        <code>EBS_SNAPSHOT_MANAGEMENT</code> to create a lifecycle policy that manages the lifecycle of Amazon
         *        EBS snapshots. Specify <code>IMAGE_MANAGEMENT</code> to create a lifecycle policy that manages the
         *        lifecycle of EBS-backed AMIs. Specify <code>EVENT_BASED_POLICY </code> to create an event-based policy
         *        that performs specific actions when a defined event occurs in your Amazon Web Services account.</p>
         *        <p>
         *        The default is <code>EBS_SNAPSHOT_MANAGEMENT</code>.
         * @see PolicyTypeValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyTypeValues
         */
        Builder policyType(String policyType);

        /**
         * <p>
         * <b>[All policy types]</b> The valid target resource types and actions a policy can manage. Specify
         * <code>EBS_SNAPSHOT_MANAGEMENT</code> to create a lifecycle policy that manages the lifecycle of Amazon EBS
         * snapshots. Specify <code>IMAGE_MANAGEMENT</code> to create a lifecycle policy that manages the lifecycle of
         * EBS-backed AMIs. Specify <code>EVENT_BASED_POLICY </code> to create an event-based policy that performs
         * specific actions when a defined event occurs in your Amazon Web Services account.
         * </p>
         * <p>
         * The default is <code>EBS_SNAPSHOT_MANAGEMENT</code>.
         * </p>
         * 
         * @param policyType
         *        <b>[All policy types]</b> The valid target resource types and actions a policy can manage. Specify
         *        <code>EBS_SNAPSHOT_MANAGEMENT</code> to create a lifecycle policy that manages the lifecycle of Amazon
         *        EBS snapshots. Specify <code>IMAGE_MANAGEMENT</code> to create a lifecycle policy that manages the
         *        lifecycle of EBS-backed AMIs. Specify <code>EVENT_BASED_POLICY </code> to create an event-based policy
         *        that performs specific actions when a defined event occurs in your Amazon Web Services account.</p>
         *        <p>
         *        The default is <code>EBS_SNAPSHOT_MANAGEMENT</code>.
         * @see PolicyTypeValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyTypeValues
         */
        Builder policyType(PolicyTypeValues policyType);

        /**
         * <p>
         * <b>[Snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
         * <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create
         * multi-volume snapshots from the volumes for an instance.
         * </p>
         * 
         * @param resourceTypes
         *        <b>[Snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
         *        <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create
         *        multi-volume snapshots from the volumes for an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesWithStrings(Collection<String> resourceTypes);

        /**
         * <p>
         * <b>[Snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
         * <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create
         * multi-volume snapshots from the volumes for an instance.
         * </p>
         * 
         * @param resourceTypes
         *        <b>[Snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
         *        <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create
         *        multi-volume snapshots from the volumes for an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesWithStrings(String... resourceTypes);

        /**
         * <p>
         * <b>[Snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
         * <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create
         * multi-volume snapshots from the volumes for an instance.
         * </p>
         * 
         * @param resourceTypes
         *        <b>[Snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
         *        <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create
         *        multi-volume snapshots from the volumes for an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(Collection<ResourceTypeValues> resourceTypes);

        /**
         * <p>
         * <b>[Snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
         * <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create
         * multi-volume snapshots from the volumes for an instance.
         * </p>
         * 
         * @param resourceTypes
         *        <b>[Snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
         *        <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create
         *        multi-volume snapshots from the volumes for an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(ResourceTypeValues... resourceTypes);

        /**
         * <p>
         * <b>[Snapshot and AMI policies only]</b> The location of the resources to backup. If the source resources are
         * located in an Amazon Web Services Region, specify <code>CLOUD</code>. If the source resources are located on
         * an Outpost in your account, specify <code>OUTPOST</code>.
         * </p>
         * <p>
         * If you specify <code>OUTPOST</code>, Amazon Data Lifecycle Manager backs up all resources of the specified
         * type with matching target tags across all of the Outposts in your account.
         * </p>
         * 
         * @param resourceLocations
         *        <b>[Snapshot and AMI policies only]</b> The location of the resources to backup. If the source
         *        resources are located in an Amazon Web Services Region, specify <code>CLOUD</code>. If the source
         *        resources are located on an Outpost in your account, specify <code>OUTPOST</code>.</p>
         *        <p>
         *        If you specify <code>OUTPOST</code>, Amazon Data Lifecycle Manager backs up all resources of the
         *        specified type with matching target tags across all of the Outposts in your account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceLocationsWithStrings(Collection<String> resourceLocations);

        /**
         * <p>
         * <b>[Snapshot and AMI policies only]</b> The location of the resources to backup. If the source resources are
         * located in an Amazon Web Services Region, specify <code>CLOUD</code>. If the source resources are located on
         * an Outpost in your account, specify <code>OUTPOST</code>.
         * </p>
         * <p>
         * If you specify <code>OUTPOST</code>, Amazon Data Lifecycle Manager backs up all resources of the specified
         * type with matching target tags across all of the Outposts in your account.
         * </p>
         * 
         * @param resourceLocations
         *        <b>[Snapshot and AMI policies only]</b> The location of the resources to backup. If the source
         *        resources are located in an Amazon Web Services Region, specify <code>CLOUD</code>. If the source
         *        resources are located on an Outpost in your account, specify <code>OUTPOST</code>.</p>
         *        <p>
         *        If you specify <code>OUTPOST</code>, Amazon Data Lifecycle Manager backs up all resources of the
         *        specified type with matching target tags across all of the Outposts in your account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceLocationsWithStrings(String... resourceLocations);

        /**
         * <p>
         * <b>[Snapshot and AMI policies only]</b> The location of the resources to backup. If the source resources are
         * located in an Amazon Web Services Region, specify <code>CLOUD</code>. If the source resources are located on
         * an Outpost in your account, specify <code>OUTPOST</code>.
         * </p>
         * <p>
         * If you specify <code>OUTPOST</code>, Amazon Data Lifecycle Manager backs up all resources of the specified
         * type with matching target tags across all of the Outposts in your account.
         * </p>
         * 
         * @param resourceLocations
         *        <b>[Snapshot and AMI policies only]</b> The location of the resources to backup. If the source
         *        resources are located in an Amazon Web Services Region, specify <code>CLOUD</code>. If the source
         *        resources are located on an Outpost in your account, specify <code>OUTPOST</code>.</p>
         *        <p>
         *        If you specify <code>OUTPOST</code>, Amazon Data Lifecycle Manager backs up all resources of the
         *        specified type with matching target tags across all of the Outposts in your account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceLocations(Collection<ResourceLocationValues> resourceLocations);

        /**
         * <p>
         * <b>[Snapshot and AMI policies only]</b> The location of the resources to backup. If the source resources are
         * located in an Amazon Web Services Region, specify <code>CLOUD</code>. If the source resources are located on
         * an Outpost in your account, specify <code>OUTPOST</code>.
         * </p>
         * <p>
         * If you specify <code>OUTPOST</code>, Amazon Data Lifecycle Manager backs up all resources of the specified
         * type with matching target tags across all of the Outposts in your account.
         * </p>
         * 
         * @param resourceLocations
         *        <b>[Snapshot and AMI policies only]</b> The location of the resources to backup. If the source
         *        resources are located in an Amazon Web Services Region, specify <code>CLOUD</code>. If the source
         *        resources are located on an Outpost in your account, specify <code>OUTPOST</code>.</p>
         *        <p>
         *        If you specify <code>OUTPOST</code>, Amazon Data Lifecycle Manager backs up all resources of the
         *        specified type with matching target tags across all of the Outposts in your account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceLocations(ResourceLocationValues... resourceLocations);

        /**
         * <p>
         * <b>[Snapshot and AMI policies only]</b> The single tag that identifies targeted resources for this policy.
         * </p>
         * 
         * @param targetTags
         *        <b>[Snapshot and AMI policies only]</b> The single tag that identifies targeted resources for this
         *        policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetTags(Collection<Tag> targetTags);

        /**
         * <p>
         * <b>[Snapshot and AMI policies only]</b> The single tag that identifies targeted resources for this policy.
         * </p>
         * 
         * @param targetTags
         *        <b>[Snapshot and AMI policies only]</b> The single tag that identifies targeted resources for this
         *        policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetTags(Tag... targetTags);

        /**
         * <p>
         * <b>[Snapshot and AMI policies only]</b> The single tag that identifies targeted resources for this policy.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Tag>.Builder} avoiding the need to
         * create one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #targetTags(List<Tag>)}.
         * 
         * @param targetTags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetTags(List<Tag>)
         */
        Builder targetTags(Consumer<Tag.Builder>... targetTags);

        /**
         * <p>
         * <b>[Snapshot and AMI policies only]</b> The schedules of policy-defined actions for snapshot and AMI
         * lifecycle policies. A policy can have up to four schedules—one mandatory schedule and up to three optional
         * schedules.
         * </p>
         * 
         * @param schedules
         *        <b>[Snapshot and AMI policies only]</b> The schedules of policy-defined actions for snapshot and AMI
         *        lifecycle policies. A policy can have up to four schedules—one mandatory schedule and up to three
         *        optional schedules.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schedules(Collection<Schedule> schedules);

        /**
         * <p>
         * <b>[Snapshot and AMI policies only]</b> The schedules of policy-defined actions for snapshot and AMI
         * lifecycle policies. A policy can have up to four schedules—one mandatory schedule and up to three optional
         * schedules.
         * </p>
         * 
         * @param schedules
         *        <b>[Snapshot and AMI policies only]</b> The schedules of policy-defined actions for snapshot and AMI
         *        lifecycle policies. A policy can have up to four schedules—one mandatory schedule and up to three
         *        optional schedules.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schedules(Schedule... schedules);

        /**
         * <p>
         * <b>[Snapshot and AMI policies only]</b> The schedules of policy-defined actions for snapshot and AMI
         * lifecycle policies. A policy can have up to four schedules—one mandatory schedule and up to three optional
         * schedules.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Schedule>.Builder} avoiding the need
         * to create one manually via {@link List<Schedule>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Schedule>.Builder#build()} is called immediately and its
         * result is passed to {@link #schedules(List<Schedule>)}.
         * 
         * @param schedules
         *        a consumer that will call methods on {@link List<Schedule>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #schedules(List<Schedule>)
         */
        Builder schedules(Consumer<Schedule.Builder>... schedules);

        /**
         * <p>
         * <b>[Snapshot and AMI policies only]</b> A set of optional parameters for snapshot and AMI lifecycle policies.
         * </p>
         * <note>
         * <p>
         * If you are modifying a policy that was created or previously modified using the Amazon Data Lifecycle Manager
         * console, then you must include this parameter and specify either the default values or the new values that
         * you require. You can't omit this parameter or set its values to null.
         * </p>
         * </note>
         * 
         * @param parameters
         *        <b>[Snapshot and AMI policies only]</b> A set of optional parameters for snapshot and AMI lifecycle
         *        policies. </p> <note>
         *        <p>
         *        If you are modifying a policy that was created or previously modified using the Amazon Data Lifecycle
         *        Manager console, then you must include this parameter and specify either the default values or the new
         *        values that you require. You can't omit this parameter or set its values to null.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Parameters parameters);

        /**
         * <p>
         * <b>[Snapshot and AMI policies only]</b> A set of optional parameters for snapshot and AMI lifecycle policies.
         * </p>
         * <note>
         * <p>
         * If you are modifying a policy that was created or previously modified using the Amazon Data Lifecycle Manager
         * console, then you must include this parameter and specify either the default values or the new values that
         * you require. You can't omit this parameter or set its values to null.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link Parameters.Builder} avoiding the
         * need to create one manually via {@link Parameters#builder()}.
         *
         * When the {@link Consumer} completes, {@link Parameters.Builder#build()} is called immediately and its result
         * is passed to {@link #parameters(Parameters)}.
         * 
         * @param parameters
         *        a consumer that will call methods on {@link Parameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(Parameters)
         */
        default Builder parameters(Consumer<Parameters.Builder> parameters) {
            return parameters(Parameters.builder().applyMutation(parameters).build());
        }

        /**
         * <p>
         * <b>[Event-based policies only]</b> The event that activates the event-based policy.
         * </p>
         * 
         * @param eventSource
         *        <b>[Event-based policies only]</b> The event that activates the event-based policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventSource(EventSource eventSource);

        /**
         * <p>
         * <b>[Event-based policies only]</b> The event that activates the event-based policy.
         * </p>
         * This is a convenience method that creates an instance of the {@link EventSource.Builder} avoiding the need to
         * create one manually via {@link EventSource#builder()}.
         *
         * When the {@link Consumer} completes, {@link EventSource.Builder#build()} is called immediately and its result
         * is passed to {@link #eventSource(EventSource)}.
         * 
         * @param eventSource
         *        a consumer that will call methods on {@link EventSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #eventSource(EventSource)
         */
        default Builder eventSource(Consumer<EventSource.Builder> eventSource) {
            return eventSource(EventSource.builder().applyMutation(eventSource).build());
        }

        /**
         * <p>
         * <b>[Event-based policies only]</b> The actions to be performed when the event-based policy is activated. You
         * can specify only one action per policy.
         * </p>
         * 
         * @param actions
         *        <b>[Event-based policies only]</b> The actions to be performed when the event-based policy is
         *        activated. You can specify only one action per policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(Collection<Action> actions);

        /**
         * <p>
         * <b>[Event-based policies only]</b> The actions to be performed when the event-based policy is activated. You
         * can specify only one action per policy.
         * </p>
         * 
         * @param actions
         *        <b>[Event-based policies only]</b> The actions to be performed when the event-based policy is
         *        activated. You can specify only one action per policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(Action... actions);

        /**
         * <p>
         * <b>[Event-based policies only]</b> The actions to be performed when the event-based policy is activated. You
         * can specify only one action per policy.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Action>.Builder} avoiding the need
         * to create one manually via {@link List<Action>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Action>.Builder#build()} is called immediately and its
         * result is passed to {@link #actions(List<Action>)}.
         * 
         * @param actions
         *        a consumer that will call methods on {@link List<Action>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #actions(List<Action>)
         */
        Builder actions(Consumer<Action.Builder>... actions);
    }

    static final class BuilderImpl implements Builder {
        private String policyType;

        private List<String> resourceTypes = DefaultSdkAutoConstructList.getInstance();

        private List<String> resourceLocations = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> targetTags = DefaultSdkAutoConstructList.getInstance();

        private List<Schedule> schedules = DefaultSdkAutoConstructList.getInstance();

        private Parameters parameters;

        private EventSource eventSource;

        private List<Action> actions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PolicyDetails model) {
            policyType(model.policyType);
            resourceTypesWithStrings(model.resourceTypes);
            resourceLocationsWithStrings(model.resourceLocations);
            targetTags(model.targetTags);
            schedules(model.schedules);
            parameters(model.parameters);
            eventSource(model.eventSource);
            actions(model.actions);
        }

        public final String getPolicyType() {
            return policyType;
        }

        public final void setPolicyType(String policyType) {
            this.policyType = policyType;
        }

        @Override
        public final Builder policyType(String policyType) {
            this.policyType = policyType;
            return this;
        }

        @Override
        public final Builder policyType(PolicyTypeValues policyType) {
            this.policyType(policyType == null ? null : policyType.toString());
            return this;
        }

        public final Collection<String> getResourceTypes() {
            if (resourceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceTypes;
        }

        public final void setResourceTypes(Collection<String> resourceTypes) {
            this.resourceTypes = ResourceTypeValuesListCopier.copy(resourceTypes);
        }

        @Override
        public final Builder resourceTypesWithStrings(Collection<String> resourceTypes) {
            this.resourceTypes = ResourceTypeValuesListCopier.copy(resourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypesWithStrings(String... resourceTypes) {
            resourceTypesWithStrings(Arrays.asList(resourceTypes));
            return this;
        }

        @Override
        public final Builder resourceTypes(Collection<ResourceTypeValues> resourceTypes) {
            this.resourceTypes = ResourceTypeValuesListCopier.copyEnumToString(resourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypes(ResourceTypeValues... resourceTypes) {
            resourceTypes(Arrays.asList(resourceTypes));
            return this;
        }

        public final Collection<String> getResourceLocations() {
            if (resourceLocations instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceLocations;
        }

        public final void setResourceLocations(Collection<String> resourceLocations) {
            this.resourceLocations = ResourceLocationListCopier.copy(resourceLocations);
        }

        @Override
        public final Builder resourceLocationsWithStrings(Collection<String> resourceLocations) {
            this.resourceLocations = ResourceLocationListCopier.copy(resourceLocations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceLocationsWithStrings(String... resourceLocations) {
            resourceLocationsWithStrings(Arrays.asList(resourceLocations));
            return this;
        }

        @Override
        public final Builder resourceLocations(Collection<ResourceLocationValues> resourceLocations) {
            this.resourceLocations = ResourceLocationListCopier.copyEnumToString(resourceLocations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceLocations(ResourceLocationValues... resourceLocations) {
            resourceLocations(Arrays.asList(resourceLocations));
            return this;
        }

        public final List<Tag.Builder> getTargetTags() {
            List<Tag.Builder> result = TargetTagListCopier.copyToBuilder(this.targetTags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTargetTags(Collection<Tag.BuilderImpl> targetTags) {
            this.targetTags = TargetTagListCopier.copyFromBuilder(targetTags);
        }

        @Override
        public final Builder targetTags(Collection<Tag> targetTags) {
            this.targetTags = TargetTagListCopier.copy(targetTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetTags(Tag... targetTags) {
            targetTags(Arrays.asList(targetTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetTags(Consumer<Tag.Builder>... targetTags) {
            targetTags(Stream.of(targetTags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<Schedule.Builder> getSchedules() {
            List<Schedule.Builder> result = ScheduleListCopier.copyToBuilder(this.schedules);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSchedules(Collection<Schedule.BuilderImpl> schedules) {
            this.schedules = ScheduleListCopier.copyFromBuilder(schedules);
        }

        @Override
        public final Builder schedules(Collection<Schedule> schedules) {
            this.schedules = ScheduleListCopier.copy(schedules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder schedules(Schedule... schedules) {
            schedules(Arrays.asList(schedules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder schedules(Consumer<Schedule.Builder>... schedules) {
            schedules(Stream.of(schedules).map(c -> Schedule.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Parameters.Builder getParameters() {
            return parameters != null ? parameters.toBuilder() : null;
        }

        public final void setParameters(Parameters.BuilderImpl parameters) {
            this.parameters = parameters != null ? parameters.build() : null;
        }

        @Override
        public final Builder parameters(Parameters parameters) {
            this.parameters = parameters;
            return this;
        }

        public final EventSource.Builder getEventSource() {
            return eventSource != null ? eventSource.toBuilder() : null;
        }

        public final void setEventSource(EventSource.BuilderImpl eventSource) {
            this.eventSource = eventSource != null ? eventSource.build() : null;
        }

        @Override
        public final Builder eventSource(EventSource eventSource) {
            this.eventSource = eventSource;
            return this;
        }

        public final List<Action.Builder> getActions() {
            List<Action.Builder> result = ActionListCopier.copyToBuilder(this.actions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setActions(Collection<Action.BuilderImpl> actions) {
            this.actions = ActionListCopier.copyFromBuilder(actions);
        }

        @Override
        public final Builder actions(Collection<Action> actions) {
            this.actions = ActionListCopier.copy(actions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(Action... actions) {
            actions(Arrays.asList(actions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(Consumer<Action.Builder>... actions) {
            actions(Stream.of(actions).map(c -> Action.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public PolicyDetails build() {
            return new PolicyDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
