/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dlm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <b>[Snapshot policies only]</b> Specifies a rule for enabling fast snapshot restore for snapshots created by
 * snaspshot policies. You can enable fast snapshot restore based on either a count or a time interval.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FastRestoreRule implements SdkPojo, Serializable, ToCopyableBuilder<FastRestoreRule.Builder, FastRestoreRule> {
    private static final SdkField<Integer> COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Count")
            .getter(getter(FastRestoreRule::count)).setter(setter(Builder::count))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Count").build()).build();

    private static final SdkField<Integer> INTERVAL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Interval").getter(getter(FastRestoreRule::interval)).setter(setter(Builder::interval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Interval").build()).build();

    private static final SdkField<String> INTERVAL_UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IntervalUnit").getter(getter(FastRestoreRule::intervalUnitAsString))
            .setter(setter(Builder::intervalUnit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IntervalUnit").build()).build();

    private static final SdkField<List<String>> AVAILABILITY_ZONES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AvailabilityZones")
            .getter(getter(FastRestoreRule::availabilityZones))
            .setter(setter(Builder::availabilityZones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZones").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COUNT_FIELD, INTERVAL_FIELD,
            INTERVAL_UNIT_FIELD, AVAILABILITY_ZONES_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer count;

    private final Integer interval;

    private final String intervalUnit;

    private final List<String> availabilityZones;

    private FastRestoreRule(BuilderImpl builder) {
        this.count = builder.count;
        this.interval = builder.interval;
        this.intervalUnit = builder.intervalUnit;
        this.availabilityZones = builder.availabilityZones;
    }

    /**
     * <p>
     * The number of snapshots to be enabled with fast snapshot restore.
     * </p>
     * 
     * @return The number of snapshots to be enabled with fast snapshot restore.
     */
    public final Integer count() {
        return count;
    }

    /**
     * <p>
     * The amount of time to enable fast snapshot restore. The maximum is 100 years. This is equivalent to 1200 months,
     * 5200 weeks, or 36500 days.
     * </p>
     * 
     * @return The amount of time to enable fast snapshot restore. The maximum is 100 years. This is equivalent to 1200
     *         months, 5200 weeks, or 36500 days.
     */
    public final Integer interval() {
        return interval;
    }

    /**
     * <p>
     * The unit of time for enabling fast snapshot restore.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #intervalUnit} will
     * return {@link RetentionIntervalUnitValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #intervalUnitAsString}.
     * </p>
     * 
     * @return The unit of time for enabling fast snapshot restore.
     * @see RetentionIntervalUnitValues
     */
    public final RetentionIntervalUnitValues intervalUnit() {
        return RetentionIntervalUnitValues.fromValue(intervalUnit);
    }

    /**
     * <p>
     * The unit of time for enabling fast snapshot restore.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #intervalUnit} will
     * return {@link RetentionIntervalUnitValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #intervalUnitAsString}.
     * </p>
     * 
     * @return The unit of time for enabling fast snapshot restore.
     * @see RetentionIntervalUnitValues
     */
    public final String intervalUnitAsString() {
        return intervalUnit;
    }

    /**
     * For responses, this returns true if the service returned a value for the AvailabilityZones property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAvailabilityZones() {
        return availabilityZones != null && !(availabilityZones instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Availability Zones in which to enable fast snapshot restore.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAvailabilityZones} method.
     * </p>
     * 
     * @return The Availability Zones in which to enable fast snapshot restore.
     */
    public final List<String> availabilityZones() {
        return availabilityZones;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(count());
        hashCode = 31 * hashCode + Objects.hashCode(interval());
        hashCode = 31 * hashCode + Objects.hashCode(intervalUnitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAvailabilityZones() ? availabilityZones() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FastRestoreRule)) {
            return false;
        }
        FastRestoreRule other = (FastRestoreRule) obj;
        return Objects.equals(count(), other.count()) && Objects.equals(interval(), other.interval())
                && Objects.equals(intervalUnitAsString(), other.intervalUnitAsString())
                && hasAvailabilityZones() == other.hasAvailabilityZones()
                && Objects.equals(availabilityZones(), other.availabilityZones());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FastRestoreRule").add("Count", count()).add("Interval", interval())
                .add("IntervalUnit", intervalUnitAsString())
                .add("AvailabilityZones", hasAvailabilityZones() ? availabilityZones() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Count":
            return Optional.ofNullable(clazz.cast(count()));
        case "Interval":
            return Optional.ofNullable(clazz.cast(interval()));
        case "IntervalUnit":
            return Optional.ofNullable(clazz.cast(intervalUnitAsString()));
        case "AvailabilityZones":
            return Optional.ofNullable(clazz.cast(availabilityZones()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FastRestoreRule, T> g) {
        return obj -> g.apply((FastRestoreRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FastRestoreRule> {
        /**
         * <p>
         * The number of snapshots to be enabled with fast snapshot restore.
         * </p>
         * 
         * @param count
         *        The number of snapshots to be enabled with fast snapshot restore.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder count(Integer count);

        /**
         * <p>
         * The amount of time to enable fast snapshot restore. The maximum is 100 years. This is equivalent to 1200
         * months, 5200 weeks, or 36500 days.
         * </p>
         * 
         * @param interval
         *        The amount of time to enable fast snapshot restore. The maximum is 100 years. This is equivalent to
         *        1200 months, 5200 weeks, or 36500 days.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder interval(Integer interval);

        /**
         * <p>
         * The unit of time for enabling fast snapshot restore.
         * </p>
         * 
         * @param intervalUnit
         *        The unit of time for enabling fast snapshot restore.
         * @see RetentionIntervalUnitValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RetentionIntervalUnitValues
         */
        Builder intervalUnit(String intervalUnit);

        /**
         * <p>
         * The unit of time for enabling fast snapshot restore.
         * </p>
         * 
         * @param intervalUnit
         *        The unit of time for enabling fast snapshot restore.
         * @see RetentionIntervalUnitValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RetentionIntervalUnitValues
         */
        Builder intervalUnit(RetentionIntervalUnitValues intervalUnit);

        /**
         * <p>
         * The Availability Zones in which to enable fast snapshot restore.
         * </p>
         * 
         * @param availabilityZones
         *        The Availability Zones in which to enable fast snapshot restore.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(Collection<String> availabilityZones);

        /**
         * <p>
         * The Availability Zones in which to enable fast snapshot restore.
         * </p>
         * 
         * @param availabilityZones
         *        The Availability Zones in which to enable fast snapshot restore.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(String... availabilityZones);
    }

    static final class BuilderImpl implements Builder {
        private Integer count;

        private Integer interval;

        private String intervalUnit;

        private List<String> availabilityZones = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(FastRestoreRule model) {
            count(model.count);
            interval(model.interval);
            intervalUnit(model.intervalUnit);
            availabilityZones(model.availabilityZones);
        }

        public final Integer getCount() {
            return count;
        }

        public final void setCount(Integer count) {
            this.count = count;
        }

        @Override
        public final Builder count(Integer count) {
            this.count = count;
            return this;
        }

        public final Integer getInterval() {
            return interval;
        }

        public final void setInterval(Integer interval) {
            this.interval = interval;
        }

        @Override
        public final Builder interval(Integer interval) {
            this.interval = interval;
            return this;
        }

        public final String getIntervalUnit() {
            return intervalUnit;
        }

        public final void setIntervalUnit(String intervalUnit) {
            this.intervalUnit = intervalUnit;
        }

        @Override
        public final Builder intervalUnit(String intervalUnit) {
            this.intervalUnit = intervalUnit;
            return this;
        }

        @Override
        public final Builder intervalUnit(RetentionIntervalUnitValues intervalUnit) {
            this.intervalUnit(intervalUnit == null ? null : intervalUnit.toString());
            return this;
        }

        public final Collection<String> getAvailabilityZones() {
            if (availabilityZones instanceof SdkAutoConstructList) {
                return null;
            }
            return availabilityZones;
        }

        public final void setAvailabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZoneListCopier.copy(availabilityZones);
        }

        @Override
        public final Builder availabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZoneListCopier.copy(availabilityZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availabilityZones(String... availabilityZones) {
            availabilityZones(Arrays.asList(availabilityZones));
            return this;
        }

        @Override
        public FastRestoreRule build() {
            return new FastRestoreRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
