/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dlm;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.dlm.model.CreateLifecyclePolicyRequest;
import software.amazon.awssdk.services.dlm.model.CreateLifecyclePolicyResponse;
import software.amazon.awssdk.services.dlm.model.DeleteLifecyclePolicyRequest;
import software.amazon.awssdk.services.dlm.model.DeleteLifecyclePolicyResponse;
import software.amazon.awssdk.services.dlm.model.DlmException;
import software.amazon.awssdk.services.dlm.model.GetLifecyclePoliciesRequest;
import software.amazon.awssdk.services.dlm.model.GetLifecyclePoliciesResponse;
import software.amazon.awssdk.services.dlm.model.GetLifecyclePolicyRequest;
import software.amazon.awssdk.services.dlm.model.GetLifecyclePolicyResponse;
import software.amazon.awssdk.services.dlm.model.InternalServerException;
import software.amazon.awssdk.services.dlm.model.InvalidRequestException;
import software.amazon.awssdk.services.dlm.model.LimitExceededException;
import software.amazon.awssdk.services.dlm.model.ResourceNotFoundException;
import software.amazon.awssdk.services.dlm.model.UpdateLifecyclePolicyRequest;
import software.amazon.awssdk.services.dlm.model.UpdateLifecyclePolicyResponse;
import software.amazon.awssdk.services.dlm.transform.CreateLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.dlm.transform.DeleteLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.dlm.transform.GetLifecyclePoliciesRequestMarshaller;
import software.amazon.awssdk.services.dlm.transform.GetLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.dlm.transform.UpdateLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link DlmAsyncClient}.
 *
 * @see DlmAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDlmAsyncClient implements DlmAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultDlmAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultDlmAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a policy to manage the lifecycle of the specified AWS resources. You can create up to 100 lifecycle
     * policies.
     * </p>
     *
     * @param createLifecyclePolicyRequest
     * @return A Java Future containing the result of the CreateLifecyclePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException Bad request. The request is missing required parameters or has invalid
     *         parameters.</li>
     *         <li>LimitExceededException The request failed because a limit was exceeded.</li>
     *         <li>InternalServerException The service failed in an unexpected way.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DlmException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DlmAsyncClient.CreateLifecyclePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dlm-2018-01-12/CreateLifecyclePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLifecyclePolicyResponse> createLifecyclePolicy(
            CreateLifecyclePolicyRequest createLifecyclePolicyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLifecyclePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<CreateLifecyclePolicyRequest, CreateLifecyclePolicyResponse>()
                    .withOperationName("CreateLifecyclePolicy")
                    .withMarshaller(new CreateLifecyclePolicyRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(createLifecyclePolicyRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified lifecycle policy and halts the automated operations that the policy specified.
     * </p>
     *
     * @param deleteLifecyclePolicyRequest
     * @return A Java Future containing the result of the DeleteLifecyclePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A requested resource was not found.</li>
     *         <li>InternalServerException The service failed in an unexpected way.</li>
     *         <li>LimitExceededException The request failed because a limit was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DlmException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DlmAsyncClient.DeleteLifecyclePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dlm-2018-01-12/DeleteLifecyclePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLifecyclePolicyResponse> deleteLifecyclePolicy(
            DeleteLifecyclePolicyRequest deleteLifecyclePolicyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLifecyclePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<DeleteLifecyclePolicyRequest, DeleteLifecyclePolicyResponse>()
                    .withOperationName("DeleteLifecyclePolicy")
                    .withMarshaller(new DeleteLifecyclePolicyRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(deleteLifecyclePolicyRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets summary information about all or the specified data lifecycle policies.
     * </p>
     * <p>
     * To get complete information about a policy, use <a>GetLifecyclePolicy</a>.
     * </p>
     *
     * @param getLifecyclePoliciesRequest
     * @return A Java Future containing the result of the GetLifecyclePolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A requested resource was not found.</li>
     *         <li>InvalidRequestException Bad request. The request is missing required parameters or has invalid
     *         parameters.</li>
     *         <li>InternalServerException The service failed in an unexpected way.</li>
     *         <li>LimitExceededException The request failed because a limit was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DlmException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DlmAsyncClient.GetLifecyclePolicies
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dlm-2018-01-12/GetLifecyclePolicies" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLifecyclePoliciesResponse> getLifecyclePolicies(
            GetLifecyclePoliciesRequest getLifecyclePoliciesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLifecyclePoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLifecyclePoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<GetLifecyclePoliciesRequest, GetLifecyclePoliciesResponse>()
                    .withOperationName("GetLifecyclePolicies")
                    .withMarshaller(new GetLifecyclePoliciesRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(getLifecyclePoliciesRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets detailed information about the specified lifecycle policy.
     * </p>
     *
     * @param getLifecyclePolicyRequest
     * @return A Java Future containing the result of the GetLifecyclePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A requested resource was not found.</li>
     *         <li>InternalServerException The service failed in an unexpected way.</li>
     *         <li>LimitExceededException The request failed because a limit was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DlmException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DlmAsyncClient.GetLifecyclePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dlm-2018-01-12/GetLifecyclePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetLifecyclePolicyResponse> getLifecyclePolicy(GetLifecyclePolicyRequest getLifecyclePolicyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLifecyclePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<GetLifecyclePolicyRequest, GetLifecyclePolicyResponse>()
                    .withOperationName("GetLifecyclePolicy")
                    .withMarshaller(new GetLifecyclePolicyRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(getLifecyclePolicyRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified lifecycle policy.
     * </p>
     *
     * @param updateLifecyclePolicyRequest
     * @return A Java Future containing the result of the UpdateLifecyclePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A requested resource was not found.</li>
     *         <li>InvalidRequestException Bad request. The request is missing required parameters or has invalid
     *         parameters.</li>
     *         <li>InternalServerException The service failed in an unexpected way.</li>
     *         <li>LimitExceededException The request failed because a limit was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DlmException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DlmAsyncClient.UpdateLifecyclePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dlm-2018-01-12/UpdateLifecyclePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLifecyclePolicyResponse> updateLifecyclePolicy(
            UpdateLifecyclePolicyRequest updateLifecyclePolicyRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLifecyclePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<UpdateLifecyclePolicyRequest, UpdateLifecyclePolicyResponse>()
                    .withOperationName("UpdateLifecyclePolicy")
                    .withMarshaller(new UpdateLifecyclePolicyRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(updateLifecyclePolicyRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DlmException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
