/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dlm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <b>[Custom snapshot and AMI policies only]</b> Specifies when the policy should create snapshots or AMIs.
 * </p>
 * <note>
 * <ul>
 * <li>
 * <p>
 * You must specify either <b>CronExpression</b>, or <b>Interval</b>, <b>IntervalUnit</b>, and <b>Times</b>.
 * </p>
 * </li>
 * <li>
 * <p>
 * If you need to specify an <a
 * href="https://docs.aws.amazon.com/dlm/latest/APIReference/API_ArchiveRule.html">ArchiveRule</a> for the schedule,
 * then you must specify a creation frequency of at least 28 days.
 * </p>
 * </li>
 * </ul>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateRule implements SdkPojo, Serializable, ToCopyableBuilder<CreateRule.Builder, CreateRule> {
    private static final SdkField<String> LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Location").getter(getter(CreateRule::locationAsString)).setter(setter(Builder::location))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Location").build()).build();

    private static final SdkField<Integer> INTERVAL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Interval").getter(getter(CreateRule::interval)).setter(setter(Builder::interval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Interval").build()).build();

    private static final SdkField<String> INTERVAL_UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IntervalUnit").getter(getter(CreateRule::intervalUnitAsString)).setter(setter(Builder::intervalUnit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IntervalUnit").build()).build();

    private static final SdkField<List<String>> TIMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Times")
            .getter(getter(CreateRule::times))
            .setter(setter(Builder::times))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Times").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CRON_EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CronExpression").getter(getter(CreateRule::cronExpression)).setter(setter(Builder::cronExpression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CronExpression").build()).build();

    private static final SdkField<List<Script>> SCRIPTS_FIELD = SdkField
            .<List<Script>> builder(MarshallingType.LIST)
            .memberName("Scripts")
            .getter(getter(CreateRule::scripts))
            .setter(setter(Builder::scripts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Scripts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Script> builder(MarshallingType.SDK_POJO)
                                            .constructor(Script::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOCATION_FIELD,
            INTERVAL_FIELD, INTERVAL_UNIT_FIELD, TIMES_FIELD, CRON_EXPRESSION_FIELD, SCRIPTS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String location;

    private final Integer interval;

    private final String intervalUnit;

    private final List<String> times;

    private final String cronExpression;

    private final List<Script> scripts;

    private CreateRule(BuilderImpl builder) {
        this.location = builder.location;
        this.interval = builder.interval;
        this.intervalUnit = builder.intervalUnit;
        this.times = builder.times;
        this.cronExpression = builder.cronExpression;
        this.scripts = builder.scripts;
    }

    /**
     * <p>
     * <b>[Custom snapshot policies only]</b> Specifies the destination for snapshots created by the policy. The allowed
     * destinations depend on the location of the targeted resources.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the policy targets resources in a Region, then you must create snapshots in the same Region as the source
     * resource.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the policy targets resources in a Local Zone, you can create snapshots in the same Local Zone or in its parent
     * Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the policy targets resources on an Outpost, then you can create snapshots on the same Outpost or in its parent
     * Region.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Specify one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To create snapshots in the same Region as the source resource, specify <code>CLOUD</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To create snapshots in the same Local Zone as the source resource, specify <code>LOCAL_ZONE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To create snapshots on the same Outpost as the source resource, specify <code>OUTPOST_LOCAL</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>CLOUD</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #location} will
     * return {@link LocationValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #locationAsString}.
     * </p>
     * 
     * @return <b>[Custom snapshot policies only]</b> Specifies the destination for snapshots created by the policy. The
     *         allowed destinations depend on the location of the targeted resources.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the policy targets resources in a Region, then you must create snapshots in the same Region as the
     *         source resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the policy targets resources in a Local Zone, you can create snapshots in the same Local Zone or in
     *         its parent Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the policy targets resources on an Outpost, then you can create snapshots on the same Outpost or in
     *         its parent Region.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Specify one of the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         To create snapshots in the same Region as the source resource, specify <code>CLOUD</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To create snapshots in the same Local Zone as the source resource, specify <code>LOCAL_ZONE</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To create snapshots on the same Outpost as the source resource, specify <code>OUTPOST_LOCAL</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>CLOUD</code>
     * @see LocationValues
     */
    public final LocationValues location() {
        return LocationValues.fromValue(location);
    }

    /**
     * <p>
     * <b>[Custom snapshot policies only]</b> Specifies the destination for snapshots created by the policy. The allowed
     * destinations depend on the location of the targeted resources.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the policy targets resources in a Region, then you must create snapshots in the same Region as the source
     * resource.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the policy targets resources in a Local Zone, you can create snapshots in the same Local Zone or in its parent
     * Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the policy targets resources on an Outpost, then you can create snapshots on the same Outpost or in its parent
     * Region.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Specify one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To create snapshots in the same Region as the source resource, specify <code>CLOUD</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To create snapshots in the same Local Zone as the source resource, specify <code>LOCAL_ZONE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To create snapshots on the same Outpost as the source resource, specify <code>OUTPOST_LOCAL</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>CLOUD</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #location} will
     * return {@link LocationValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #locationAsString}.
     * </p>
     * 
     * @return <b>[Custom snapshot policies only]</b> Specifies the destination for snapshots created by the policy. The
     *         allowed destinations depend on the location of the targeted resources.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the policy targets resources in a Region, then you must create snapshots in the same Region as the
     *         source resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the policy targets resources in a Local Zone, you can create snapshots in the same Local Zone or in
     *         its parent Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the policy targets resources on an Outpost, then you can create snapshots on the same Outpost or in
     *         its parent Region.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Specify one of the following values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         To create snapshots in the same Region as the source resource, specify <code>CLOUD</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To create snapshots in the same Local Zone as the source resource, specify <code>LOCAL_ZONE</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To create snapshots on the same Outpost as the source resource, specify <code>OUTPOST_LOCAL</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>CLOUD</code>
     * @see LocationValues
     */
    public final String locationAsString() {
        return location;
    }

    /**
     * <p>
     * The interval between snapshots. The supported values are 1, 2, 3, 4, 6, 8, 12, and 24.
     * </p>
     * 
     * @return The interval between snapshots. The supported values are 1, 2, 3, 4, 6, 8, 12, and 24.
     */
    public final Integer interval() {
        return interval;
    }

    /**
     * <p>
     * The interval unit.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #intervalUnit} will
     * return {@link IntervalUnitValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #intervalUnitAsString}.
     * </p>
     * 
     * @return The interval unit.
     * @see IntervalUnitValues
     */
    public final IntervalUnitValues intervalUnit() {
        return IntervalUnitValues.fromValue(intervalUnit);
    }

    /**
     * <p>
     * The interval unit.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #intervalUnit} will
     * return {@link IntervalUnitValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #intervalUnitAsString}.
     * </p>
     * 
     * @return The interval unit.
     * @see IntervalUnitValues
     */
    public final String intervalUnitAsString() {
        return intervalUnit;
    }

    /**
     * For responses, this returns true if the service returned a value for the Times property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTimes() {
        return times != null && !(times instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The time, in UTC, to start the operation. The supported format is hh:mm.
     * </p>
     * <p>
     * The operation occurs within a one-hour window following the specified time. If you do not specify a time, Amazon
     * Data Lifecycle Manager selects a time within the next 24 hours.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTimes} method.
     * </p>
     * 
     * @return The time, in UTC, to start the operation. The supported format is hh:mm.</p>
     *         <p>
     *         The operation occurs within a one-hour window following the specified time. If you do not specify a time,
     *         Amazon Data Lifecycle Manager selects a time within the next 24 hours.
     */
    public final List<String> times() {
        return times;
    }

    /**
     * <p>
     * The schedule, as a Cron expression. The schedule interval must be between 1 hour and 1 year. For more
     * information, see the <a
     * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-cron-expressions.html">Cron expressions
     * reference</a> in the <i>Amazon EventBridge User Guide</i>.
     * </p>
     * 
     * @return The schedule, as a Cron expression. The schedule interval must be between 1 hour and 1 year. For more
     *         information, see the <a
     *         href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-cron-expressions.html">Cron expressions
     *         reference</a> in the <i>Amazon EventBridge User Guide</i>.
     */
    public final String cronExpression() {
        return cronExpression;
    }

    /**
     * For responses, this returns true if the service returned a value for the Scripts property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasScripts() {
        return scripts != null && !(scripts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <b>[Custom snapshot policies that target instances only]</b> Specifies pre and/or post scripts for a snapshot
     * lifecycle policy that targets instances. This is useful for creating application-consistent snapshots, or for
     * performing specific administrative tasks before or after Amazon Data Lifecycle Manager initiates snapshot
     * creation.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/automate-app-consistent-backups.html">Automating
     * application-consistent snapshots with pre and post scripts</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasScripts} method.
     * </p>
     * 
     * @return <b>[Custom snapshot policies that target instances only]</b> Specifies pre and/or post scripts for a
     *         snapshot lifecycle policy that targets instances. This is useful for creating application-consistent
     *         snapshots, or for performing specific administrative tasks before or after Amazon Data Lifecycle Manager
     *         initiates snapshot creation.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/automate-app-consistent-backups.html"
     *         >Automating application-consistent snapshots with pre and post scripts</a>.
     */
    public final List<Script> scripts() {
        return scripts;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(locationAsString());
        hashCode = 31 * hashCode + Objects.hashCode(interval());
        hashCode = 31 * hashCode + Objects.hashCode(intervalUnitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTimes() ? times() : null);
        hashCode = 31 * hashCode + Objects.hashCode(cronExpression());
        hashCode = 31 * hashCode + Objects.hashCode(hasScripts() ? scripts() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateRule)) {
            return false;
        }
        CreateRule other = (CreateRule) obj;
        return Objects.equals(locationAsString(), other.locationAsString()) && Objects.equals(interval(), other.interval())
                && Objects.equals(intervalUnitAsString(), other.intervalUnitAsString()) && hasTimes() == other.hasTimes()
                && Objects.equals(times(), other.times()) && Objects.equals(cronExpression(), other.cronExpression())
                && hasScripts() == other.hasScripts() && Objects.equals(scripts(), other.scripts());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateRule").add("Location", locationAsString()).add("Interval", interval())
                .add("IntervalUnit", intervalUnitAsString()).add("Times", hasTimes() ? times() : null)
                .add("CronExpression", cronExpression()).add("Scripts", hasScripts() ? scripts() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Location":
            return Optional.ofNullable(clazz.cast(locationAsString()));
        case "Interval":
            return Optional.ofNullable(clazz.cast(interval()));
        case "IntervalUnit":
            return Optional.ofNullable(clazz.cast(intervalUnitAsString()));
        case "Times":
            return Optional.ofNullable(clazz.cast(times()));
        case "CronExpression":
            return Optional.ofNullable(clazz.cast(cronExpression()));
        case "Scripts":
            return Optional.ofNullable(clazz.cast(scripts()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Location", LOCATION_FIELD);
        map.put("Interval", INTERVAL_FIELD);
        map.put("IntervalUnit", INTERVAL_UNIT_FIELD);
        map.put("Times", TIMES_FIELD);
        map.put("CronExpression", CRON_EXPRESSION_FIELD);
        map.put("Scripts", SCRIPTS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateRule, T> g) {
        return obj -> g.apply((CreateRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CreateRule> {
        /**
         * <p>
         * <b>[Custom snapshot policies only]</b> Specifies the destination for snapshots created by the policy. The
         * allowed destinations depend on the location of the targeted resources.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the policy targets resources in a Region, then you must create snapshots in the same Region as the source
         * resource.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the policy targets resources in a Local Zone, you can create snapshots in the same Local Zone or in its
         * parent Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the policy targets resources on an Outpost, then you can create snapshots on the same Outpost or in its
         * parent Region.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Specify one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * To create snapshots in the same Region as the source resource, specify <code>CLOUD</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To create snapshots in the same Local Zone as the source resource, specify <code>LOCAL_ZONE</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To create snapshots on the same Outpost as the source resource, specify <code>OUTPOST_LOCAL</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>CLOUD</code>
         * </p>
         * 
         * @param location
         *        <b>[Custom snapshot policies only]</b> Specifies the destination for snapshots created by the policy.
         *        The allowed destinations depend on the location of the targeted resources.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the policy targets resources in a Region, then you must create snapshots in the same Region as the
         *        source resource.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the policy targets resources in a Local Zone, you can create snapshots in the same Local Zone or in
         *        its parent Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the policy targets resources on an Outpost, then you can create snapshots on the same Outpost or in
         *        its parent Region.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Specify one of the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        To create snapshots in the same Region as the source resource, specify <code>CLOUD</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To create snapshots in the same Local Zone as the source resource, specify <code>LOCAL_ZONE</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To create snapshots on the same Outpost as the source resource, specify <code>OUTPOST_LOCAL</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>CLOUD</code>
         * @see LocationValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LocationValues
         */
        Builder location(String location);

        /**
         * <p>
         * <b>[Custom snapshot policies only]</b> Specifies the destination for snapshots created by the policy. The
         * allowed destinations depend on the location of the targeted resources.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the policy targets resources in a Region, then you must create snapshots in the same Region as the source
         * resource.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the policy targets resources in a Local Zone, you can create snapshots in the same Local Zone or in its
         * parent Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the policy targets resources on an Outpost, then you can create snapshots on the same Outpost or in its
         * parent Region.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Specify one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * To create snapshots in the same Region as the source resource, specify <code>CLOUD</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To create snapshots in the same Local Zone as the source resource, specify <code>LOCAL_ZONE</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To create snapshots on the same Outpost as the source resource, specify <code>OUTPOST_LOCAL</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>CLOUD</code>
         * </p>
         * 
         * @param location
         *        <b>[Custom snapshot policies only]</b> Specifies the destination for snapshots created by the policy.
         *        The allowed destinations depend on the location of the targeted resources.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the policy targets resources in a Region, then you must create snapshots in the same Region as the
         *        source resource.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the policy targets resources in a Local Zone, you can create snapshots in the same Local Zone or in
         *        its parent Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the policy targets resources on an Outpost, then you can create snapshots on the same Outpost or in
         *        its parent Region.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Specify one of the following values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        To create snapshots in the same Region as the source resource, specify <code>CLOUD</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To create snapshots in the same Local Zone as the source resource, specify <code>LOCAL_ZONE</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To create snapshots on the same Outpost as the source resource, specify <code>OUTPOST_LOCAL</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>CLOUD</code>
         * @see LocationValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LocationValues
         */
        Builder location(LocationValues location);

        /**
         * <p>
         * The interval between snapshots. The supported values are 1, 2, 3, 4, 6, 8, 12, and 24.
         * </p>
         * 
         * @param interval
         *        The interval between snapshots. The supported values are 1, 2, 3, 4, 6, 8, 12, and 24.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder interval(Integer interval);

        /**
         * <p>
         * The interval unit.
         * </p>
         * 
         * @param intervalUnit
         *        The interval unit.
         * @see IntervalUnitValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IntervalUnitValues
         */
        Builder intervalUnit(String intervalUnit);

        /**
         * <p>
         * The interval unit.
         * </p>
         * 
         * @param intervalUnit
         *        The interval unit.
         * @see IntervalUnitValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IntervalUnitValues
         */
        Builder intervalUnit(IntervalUnitValues intervalUnit);

        /**
         * <p>
         * The time, in UTC, to start the operation. The supported format is hh:mm.
         * </p>
         * <p>
         * The operation occurs within a one-hour window following the specified time. If you do not specify a time,
         * Amazon Data Lifecycle Manager selects a time within the next 24 hours.
         * </p>
         * 
         * @param times
         *        The time, in UTC, to start the operation. The supported format is hh:mm.</p>
         *        <p>
         *        The operation occurs within a one-hour window following the specified time. If you do not specify a
         *        time, Amazon Data Lifecycle Manager selects a time within the next 24 hours.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder times(Collection<String> times);

        /**
         * <p>
         * The time, in UTC, to start the operation. The supported format is hh:mm.
         * </p>
         * <p>
         * The operation occurs within a one-hour window following the specified time. If you do not specify a time,
         * Amazon Data Lifecycle Manager selects a time within the next 24 hours.
         * </p>
         * 
         * @param times
         *        The time, in UTC, to start the operation. The supported format is hh:mm.</p>
         *        <p>
         *        The operation occurs within a one-hour window following the specified time. If you do not specify a
         *        time, Amazon Data Lifecycle Manager selects a time within the next 24 hours.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder times(String... times);

        /**
         * <p>
         * The schedule, as a Cron expression. The schedule interval must be between 1 hour and 1 year. For more
         * information, see the <a
         * href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-cron-expressions.html">Cron expressions
         * reference</a> in the <i>Amazon EventBridge User Guide</i>.
         * </p>
         * 
         * @param cronExpression
         *        The schedule, as a Cron expression. The schedule interval must be between 1 hour and 1 year. For more
         *        information, see the <a
         *        href="https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-cron-expressions.html">Cron
         *        expressions reference</a> in the <i>Amazon EventBridge User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cronExpression(String cronExpression);

        /**
         * <p>
         * <b>[Custom snapshot policies that target instances only]</b> Specifies pre and/or post scripts for a snapshot
         * lifecycle policy that targets instances. This is useful for creating application-consistent snapshots, or for
         * performing specific administrative tasks before or after Amazon Data Lifecycle Manager initiates snapshot
         * creation.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/automate-app-consistent-backups.html">Automating
         * application-consistent snapshots with pre and post scripts</a>.
         * </p>
         * 
         * @param scripts
         *        <b>[Custom snapshot policies that target instances only]</b> Specifies pre and/or post scripts for a
         *        snapshot lifecycle policy that targets instances. This is useful for creating application-consistent
         *        snapshots, or for performing specific administrative tasks before or after Amazon Data Lifecycle
         *        Manager initiates snapshot creation.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/automate-app-consistent-backups.html"
         *        >Automating application-consistent snapshots with pre and post scripts</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scripts(Collection<Script> scripts);

        /**
         * <p>
         * <b>[Custom snapshot policies that target instances only]</b> Specifies pre and/or post scripts for a snapshot
         * lifecycle policy that targets instances. This is useful for creating application-consistent snapshots, or for
         * performing specific administrative tasks before or after Amazon Data Lifecycle Manager initiates snapshot
         * creation.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/automate-app-consistent-backups.html">Automating
         * application-consistent snapshots with pre and post scripts</a>.
         * </p>
         * 
         * @param scripts
         *        <b>[Custom snapshot policies that target instances only]</b> Specifies pre and/or post scripts for a
         *        snapshot lifecycle policy that targets instances. This is useful for creating application-consistent
         *        snapshots, or for performing specific administrative tasks before or after Amazon Data Lifecycle
         *        Manager initiates snapshot creation.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/automate-app-consistent-backups.html"
         *        >Automating application-consistent snapshots with pre and post scripts</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scripts(Script... scripts);

        /**
         * <p>
         * <b>[Custom snapshot policies that target instances only]</b> Specifies pre and/or post scripts for a snapshot
         * lifecycle policy that targets instances. This is useful for creating application-consistent snapshots, or for
         * performing specific administrative tasks before or after Amazon Data Lifecycle Manager initiates snapshot
         * creation.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/automate-app-consistent-backups.html">Automating
         * application-consistent snapshots with pre and post scripts</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.dlm.model.Script.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.dlm.model.Script#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.dlm.model.Script.Builder#build()}
         * is called immediately and its result is passed to {@link #scripts(List<Script>)}.
         * 
         * @param scripts
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.dlm.model.Script.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scripts(java.util.Collection<Script>)
         */
        Builder scripts(Consumer<Script.Builder>... scripts);
    }

    static final class BuilderImpl implements Builder {
        private String location;

        private Integer interval;

        private String intervalUnit;

        private List<String> times = DefaultSdkAutoConstructList.getInstance();

        private String cronExpression;

        private List<Script> scripts = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateRule model) {
            location(model.location);
            interval(model.interval);
            intervalUnit(model.intervalUnit);
            times(model.times);
            cronExpression(model.cronExpression);
            scripts(model.scripts);
        }

        public final String getLocation() {
            return location;
        }

        public final void setLocation(String location) {
            this.location = location;
        }

        @Override
        public final Builder location(String location) {
            this.location = location;
            return this;
        }

        @Override
        public final Builder location(LocationValues location) {
            this.location(location == null ? null : location.toString());
            return this;
        }

        public final Integer getInterval() {
            return interval;
        }

        public final void setInterval(Integer interval) {
            this.interval = interval;
        }

        @Override
        public final Builder interval(Integer interval) {
            this.interval = interval;
            return this;
        }

        public final String getIntervalUnit() {
            return intervalUnit;
        }

        public final void setIntervalUnit(String intervalUnit) {
            this.intervalUnit = intervalUnit;
        }

        @Override
        public final Builder intervalUnit(String intervalUnit) {
            this.intervalUnit = intervalUnit;
            return this;
        }

        @Override
        public final Builder intervalUnit(IntervalUnitValues intervalUnit) {
            this.intervalUnit(intervalUnit == null ? null : intervalUnit.toString());
            return this;
        }

        public final Collection<String> getTimes() {
            if (times instanceof SdkAutoConstructList) {
                return null;
            }
            return times;
        }

        public final void setTimes(Collection<String> times) {
            this.times = TimesListCopier.copy(times);
        }

        @Override
        public final Builder times(Collection<String> times) {
            this.times = TimesListCopier.copy(times);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder times(String... times) {
            times(Arrays.asList(times));
            return this;
        }

        public final String getCronExpression() {
            return cronExpression;
        }

        public final void setCronExpression(String cronExpression) {
            this.cronExpression = cronExpression;
        }

        @Override
        public final Builder cronExpression(String cronExpression) {
            this.cronExpression = cronExpression;
            return this;
        }

        public final List<Script.Builder> getScripts() {
            List<Script.Builder> result = ScriptsListCopier.copyToBuilder(this.scripts);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setScripts(Collection<Script.BuilderImpl> scripts) {
            this.scripts = ScriptsListCopier.copyFromBuilder(scripts);
        }

        @Override
        public final Builder scripts(Collection<Script> scripts) {
            this.scripts = ScriptsListCopier.copy(scripts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder scripts(Script... scripts) {
            scripts(Arrays.asList(scripts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder scripts(Consumer<Script.Builder>... scripts) {
            scripts(Stream.of(scripts).map(c -> Script.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public CreateRule build() {
            return new CreateRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
