/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dlm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the configuration of a lifecycle policy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PolicyDetails implements SdkPojo, Serializable, ToCopyableBuilder<PolicyDetails.Builder, PolicyDetails> {
    private static final SdkField<String> POLICY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyType").getter(getter(PolicyDetails::policyTypeAsString)).setter(setter(Builder::policyType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyType").build()).build();

    private static final SdkField<List<String>> RESOURCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourceTypes")
            .getter(getter(PolicyDetails::resourceTypesAsStrings))
            .setter(setter(Builder::resourceTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> RESOURCE_LOCATIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourceLocations")
            .getter(getter(PolicyDetails::resourceLocationsAsStrings))
            .setter(setter(Builder::resourceLocationsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceLocations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TARGET_TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("TargetTags")
            .getter(getter(PolicyDetails::targetTags))
            .setter(setter(Builder::targetTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Schedule>> SCHEDULES_FIELD = SdkField
            .<List<Schedule>> builder(MarshallingType.LIST)
            .memberName("Schedules")
            .getter(getter(PolicyDetails::schedules))
            .setter(setter(Builder::schedules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Schedules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Schedule> builder(MarshallingType.SDK_POJO)
                                            .constructor(Schedule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Parameters> PARAMETERS_FIELD = SdkField.<Parameters> builder(MarshallingType.SDK_POJO)
            .memberName("Parameters").getter(getter(PolicyDetails::parameters)).setter(setter(Builder::parameters))
            .constructor(Parameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build()).build();

    private static final SdkField<EventSource> EVENT_SOURCE_FIELD = SdkField.<EventSource> builder(MarshallingType.SDK_POJO)
            .memberName("EventSource").getter(getter(PolicyDetails::eventSource)).setter(setter(Builder::eventSource))
            .constructor(EventSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventSource").build()).build();

    private static final SdkField<List<Action>> ACTIONS_FIELD = SdkField
            .<List<Action>> builder(MarshallingType.LIST)
            .memberName("Actions")
            .getter(getter(PolicyDetails::actions))
            .setter(setter(Builder::actions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Actions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Action> builder(MarshallingType.SDK_POJO)
                                            .constructor(Action::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> POLICY_LANGUAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyLanguage").getter(getter(PolicyDetails::policyLanguageAsString))
            .setter(setter(Builder::policyLanguage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyLanguage").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(PolicyDetails::resourceTypeAsString)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<Integer> CREATE_INTERVAL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("CreateInterval").getter(getter(PolicyDetails::createInterval)).setter(setter(Builder::createInterval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateInterval").build()).build();

    private static final SdkField<Integer> RETAIN_INTERVAL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("RetainInterval").getter(getter(PolicyDetails::retainInterval)).setter(setter(Builder::retainInterval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RetainInterval").build()).build();

    private static final SdkField<Boolean> COPY_TAGS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CopyTags").getter(getter(PolicyDetails::copyTags)).setter(setter(Builder::copyTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyTags").build()).build();

    private static final SdkField<List<CrossRegionCopyTarget>> CROSS_REGION_COPY_TARGETS_FIELD = SdkField
            .<List<CrossRegionCopyTarget>> builder(MarshallingType.LIST)
            .memberName("CrossRegionCopyTargets")
            .getter(getter(PolicyDetails::crossRegionCopyTargets))
            .setter(setter(Builder::crossRegionCopyTargets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CrossRegionCopyTargets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CrossRegionCopyTarget> builder(MarshallingType.SDK_POJO)
                                            .constructor(CrossRegionCopyTarget::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> EXTEND_DELETION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ExtendDeletion").getter(getter(PolicyDetails::extendDeletion)).setter(setter(Builder::extendDeletion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExtendDeletion").build()).build();

    private static final SdkField<Exclusions> EXCLUSIONS_FIELD = SdkField.<Exclusions> builder(MarshallingType.SDK_POJO)
            .memberName("Exclusions").getter(getter(PolicyDetails::exclusions)).setter(setter(Builder::exclusions))
            .constructor(Exclusions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Exclusions").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(POLICY_TYPE_FIELD,
            RESOURCE_TYPES_FIELD, RESOURCE_LOCATIONS_FIELD, TARGET_TAGS_FIELD, SCHEDULES_FIELD, PARAMETERS_FIELD,
            EVENT_SOURCE_FIELD, ACTIONS_FIELD, POLICY_LANGUAGE_FIELD, RESOURCE_TYPE_FIELD, CREATE_INTERVAL_FIELD,
            RETAIN_INTERVAL_FIELD, COPY_TAGS_FIELD, CROSS_REGION_COPY_TARGETS_FIELD, EXTEND_DELETION_FIELD, EXCLUSIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String policyType;

    private final List<String> resourceTypes;

    private final List<String> resourceLocations;

    private final List<Tag> targetTags;

    private final List<Schedule> schedules;

    private final Parameters parameters;

    private final EventSource eventSource;

    private final List<Action> actions;

    private final String policyLanguage;

    private final String resourceType;

    private final Integer createInterval;

    private final Integer retainInterval;

    private final Boolean copyTags;

    private final List<CrossRegionCopyTarget> crossRegionCopyTargets;

    private final Boolean extendDeletion;

    private final Exclusions exclusions;

    private PolicyDetails(BuilderImpl builder) {
        this.policyType = builder.policyType;
        this.resourceTypes = builder.resourceTypes;
        this.resourceLocations = builder.resourceLocations;
        this.targetTags = builder.targetTags;
        this.schedules = builder.schedules;
        this.parameters = builder.parameters;
        this.eventSource = builder.eventSource;
        this.actions = builder.actions;
        this.policyLanguage = builder.policyLanguage;
        this.resourceType = builder.resourceType;
        this.createInterval = builder.createInterval;
        this.retainInterval = builder.retainInterval;
        this.copyTags = builder.copyTags;
        this.crossRegionCopyTargets = builder.crossRegionCopyTargets;
        this.extendDeletion = builder.extendDeletion;
        this.exclusions = builder.exclusions;
    }

    /**
     * <p>
     * The type of policy. Specify <code>EBS_SNAPSHOT_MANAGEMENT</code> to create a lifecycle policy that manages the
     * lifecycle of Amazon EBS snapshots. Specify <code>IMAGE_MANAGEMENT</code> to create a lifecycle policy that
     * manages the lifecycle of EBS-backed AMIs. Specify <code>EVENT_BASED_POLICY </code> to create an event-based
     * policy that performs specific actions when a defined event occurs in your Amazon Web Services account.
     * </p>
     * <p>
     * The default is <code>EBS_SNAPSHOT_MANAGEMENT</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyType} will
     * return {@link PolicyTypeValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #policyTypeAsString}.
     * </p>
     * 
     * @return The type of policy. Specify <code>EBS_SNAPSHOT_MANAGEMENT</code> to create a lifecycle policy that
     *         manages the lifecycle of Amazon EBS snapshots. Specify <code>IMAGE_MANAGEMENT</code> to create a
     *         lifecycle policy that manages the lifecycle of EBS-backed AMIs. Specify <code>EVENT_BASED_POLICY </code>
     *         to create an event-based policy that performs specific actions when a defined event occurs in your Amazon
     *         Web Services account.</p>
     *         <p>
     *         The default is <code>EBS_SNAPSHOT_MANAGEMENT</code>.
     * @see PolicyTypeValues
     */
    public final PolicyTypeValues policyType() {
        return PolicyTypeValues.fromValue(policyType);
    }

    /**
     * <p>
     * The type of policy. Specify <code>EBS_SNAPSHOT_MANAGEMENT</code> to create a lifecycle policy that manages the
     * lifecycle of Amazon EBS snapshots. Specify <code>IMAGE_MANAGEMENT</code> to create a lifecycle policy that
     * manages the lifecycle of EBS-backed AMIs. Specify <code>EVENT_BASED_POLICY </code> to create an event-based
     * policy that performs specific actions when a defined event occurs in your Amazon Web Services account.
     * </p>
     * <p>
     * The default is <code>EBS_SNAPSHOT_MANAGEMENT</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyType} will
     * return {@link PolicyTypeValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #policyTypeAsString}.
     * </p>
     * 
     * @return The type of policy. Specify <code>EBS_SNAPSHOT_MANAGEMENT</code> to create a lifecycle policy that
     *         manages the lifecycle of Amazon EBS snapshots. Specify <code>IMAGE_MANAGEMENT</code> to create a
     *         lifecycle policy that manages the lifecycle of EBS-backed AMIs. Specify <code>EVENT_BASED_POLICY </code>
     *         to create an event-based policy that performs specific actions when a defined event occurs in your Amazon
     *         Web Services account.</p>
     *         <p>
     *         The default is <code>EBS_SNAPSHOT_MANAGEMENT</code>.
     * @see PolicyTypeValues
     */
    public final String policyTypeAsString() {
        return policyType;
    }

    /**
     * <p>
     * <b>[Custom snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
     * <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create multi-volume
     * snapshots from the volumes for an instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypes} method.
     * </p>
     * 
     * @return <b>[Custom snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies.
     *         Use <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create
     *         multi-volume snapshots from the volumes for an instance.
     */
    public final List<ResourceTypeValues> resourceTypes() {
        return ResourceTypeValuesListCopier.copyStringToEnum(resourceTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTypes() {
        return resourceTypes != null && !(resourceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <b>[Custom snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
     * <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create multi-volume
     * snapshots from the volumes for an instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypes} method.
     * </p>
     * 
     * @return <b>[Custom snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies.
     *         Use <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create
     *         multi-volume snapshots from the volumes for an instance.
     */
    public final List<String> resourceTypesAsStrings() {
        return resourceTypes;
    }

    /**
     * <p>
     * <b>[Custom snapshot and AMI policies only]</b> The location of the resources to backup.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the source resources are located in a Region, specify <code>CLOUD</code>. In this case, the policy targets all
     * resources of the specified type with matching target tags across all Availability Zones in the Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>[Custom snapshot policies only]</b> If the source resources are located in a Local Zone, specify
     * <code>LOCAL_ZONE</code>. In this case, the policy targets all resources of the specified type with matching
     * target tags across all Local Zones in the Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the source resources are located on an Outpost in your account, specify <code>OUTPOST</code>. In this case,
     * the policy targets all resources of the specified type with matching target tags across all of the Outposts in
     * your account.
     * </p>
     * </li>
     * </ul>
     * <p/>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceLocations} method.
     * </p>
     * 
     * @return <b>[Custom snapshot and AMI policies only]</b> The location of the resources to backup.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the source resources are located in a Region, specify <code>CLOUD</code>. In this case, the policy
     *         targets all resources of the specified type with matching target tags across all Availability Zones in
     *         the Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>[Custom snapshot policies only]</b> If the source resources are located in a Local Zone, specify
     *         <code>LOCAL_ZONE</code>. In this case, the policy targets all resources of the specified type with
     *         matching target tags across all Local Zones in the Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the source resources are located on an Outpost in your account, specify <code>OUTPOST</code>. In this
     *         case, the policy targets all resources of the specified type with matching target tags across all of the
     *         Outposts in your account.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final List<ResourceLocationValues> resourceLocations() {
        return ResourceLocationListCopier.copyStringToEnum(resourceLocations);
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceLocations property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceLocations() {
        return resourceLocations != null && !(resourceLocations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <b>[Custom snapshot and AMI policies only]</b> The location of the resources to backup.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the source resources are located in a Region, specify <code>CLOUD</code>. In this case, the policy targets all
     * resources of the specified type with matching target tags across all Availability Zones in the Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>[Custom snapshot policies only]</b> If the source resources are located in a Local Zone, specify
     * <code>LOCAL_ZONE</code>. In this case, the policy targets all resources of the specified type with matching
     * target tags across all Local Zones in the Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the source resources are located on an Outpost in your account, specify <code>OUTPOST</code>. In this case,
     * the policy targets all resources of the specified type with matching target tags across all of the Outposts in
     * your account.
     * </p>
     * </li>
     * </ul>
     * <p/>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceLocations} method.
     * </p>
     * 
     * @return <b>[Custom snapshot and AMI policies only]</b> The location of the resources to backup.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the source resources are located in a Region, specify <code>CLOUD</code>. In this case, the policy
     *         targets all resources of the specified type with matching target tags across all Availability Zones in
     *         the Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>[Custom snapshot policies only]</b> If the source resources are located in a Local Zone, specify
     *         <code>LOCAL_ZONE</code>. In this case, the policy targets all resources of the specified type with
     *         matching target tags across all Local Zones in the Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the source resources are located on an Outpost in your account, specify <code>OUTPOST</code>. In this
     *         case, the policy targets all resources of the specified type with matching target tags across all of the
     *         Outposts in your account.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final List<String> resourceLocationsAsStrings() {
        return resourceLocations;
    }

    /**
     * For responses, this returns true if the service returned a value for the TargetTags property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTargetTags() {
        return targetTags != null && !(targetTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <b>[Custom snapshot and AMI policies only]</b> The single tag that identifies targeted resources for this policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargetTags} method.
     * </p>
     * 
     * @return <b>[Custom snapshot and AMI policies only]</b> The single tag that identifies targeted resources for this
     *         policy.
     */
    public final List<Tag> targetTags() {
        return targetTags;
    }

    /**
     * For responses, this returns true if the service returned a value for the Schedules property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSchedules() {
        return schedules != null && !(schedules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <b>[Custom snapshot and AMI policies only]</b> The schedules of policy-defined actions for snapshot and AMI
     * lifecycle policies. A policy can have up to four schedules—one mandatory schedule and up to three optional
     * schedules.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSchedules} method.
     * </p>
     * 
     * @return <b>[Custom snapshot and AMI policies only]</b> The schedules of policy-defined actions for snapshot and
     *         AMI lifecycle policies. A policy can have up to four schedules—one mandatory schedule and up to three
     *         optional schedules.
     */
    public final List<Schedule> schedules() {
        return schedules;
    }

    /**
     * <p>
     * <b>[Custom snapshot and AMI policies only]</b> A set of optional parameters for snapshot and AMI lifecycle
     * policies.
     * </p>
     * <note>
     * <p>
     * If you are modifying a policy that was created or previously modified using the Amazon Data Lifecycle Manager
     * console, then you must include this parameter and specify either the default values or the new values that you
     * require. You can't omit this parameter or set its values to null.
     * </p>
     * </note>
     * 
     * @return <b>[Custom snapshot and AMI policies only]</b> A set of optional parameters for snapshot and AMI
     *         lifecycle policies. </p> <note>
     *         <p>
     *         If you are modifying a policy that was created or previously modified using the Amazon Data Lifecycle
     *         Manager console, then you must include this parameter and specify either the default values or the new
     *         values that you require. You can't omit this parameter or set its values to null.
     *         </p>
     */
    public final Parameters parameters() {
        return parameters;
    }

    /**
     * <p>
     * <b>[Event-based policies only]</b> The event that activates the event-based policy.
     * </p>
     * 
     * @return <b>[Event-based policies only]</b> The event that activates the event-based policy.
     */
    public final EventSource eventSource() {
        return eventSource;
    }

    /**
     * For responses, this returns true if the service returned a value for the Actions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasActions() {
        return actions != null && !(actions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <b>[Event-based policies only]</b> The actions to be performed when the event-based policy is activated. You can
     * specify only one action per policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasActions} method.
     * </p>
     * 
     * @return <b>[Event-based policies only]</b> The actions to be performed when the event-based policy is activated.
     *         You can specify only one action per policy.
     */
    public final List<Action> actions() {
        return actions;
    }

    /**
     * <p>
     * The type of policy to create. Specify one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SIMPLIFIED</code> To create a default policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STANDARD</code> To create a custom policy.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyLanguage}
     * will return {@link PolicyLanguageValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #policyLanguageAsString}.
     * </p>
     * 
     * @return The type of policy to create. Specify one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SIMPLIFIED</code> To create a default policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STANDARD</code> To create a custom policy.
     *         </p>
     *         </li>
     * @see PolicyLanguageValues
     */
    public final PolicyLanguageValues policyLanguage() {
        return PolicyLanguageValues.fromValue(policyLanguage);
    }

    /**
     * <p>
     * The type of policy to create. Specify one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SIMPLIFIED</code> To create a default policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STANDARD</code> To create a custom policy.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyLanguage}
     * will return {@link PolicyLanguageValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #policyLanguageAsString}.
     * </p>
     * 
     * @return The type of policy to create. Specify one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SIMPLIFIED</code> To create a default policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STANDARD</code> To create a custom policy.
     *         </p>
     *         </li>
     * @see PolicyLanguageValues
     */
    public final String policyLanguageAsString() {
        return policyLanguage;
    }

    /**
     * <p>
     * <b>[Default policies only]</b> Specify the type of default policy to create.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To create a default policy for EBS snapshots, that creates snapshots of all volumes in the Region that do not
     * have recent backups, specify <code>VOLUME</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To create a default policy for EBS-backed AMIs, that creates EBS-backed AMIs from all instances in the Region
     * that do not have recent backups, specify <code>INSTANCE</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceTypeValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return <b>[Default policies only]</b> Specify the type of default policy to create.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To create a default policy for EBS snapshots, that creates snapshots of all volumes in the Region that do
     *         not have recent backups, specify <code>VOLUME</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To create a default policy for EBS-backed AMIs, that creates EBS-backed AMIs from all instances in the
     *         Region that do not have recent backups, specify <code>INSTANCE</code>.
     *         </p>
     *         </li>
     * @see ResourceTypeValues
     */
    public final ResourceTypeValues resourceType() {
        return ResourceTypeValues.fromValue(resourceType);
    }

    /**
     * <p>
     * <b>[Default policies only]</b> Specify the type of default policy to create.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To create a default policy for EBS snapshots, that creates snapshots of all volumes in the Region that do not
     * have recent backups, specify <code>VOLUME</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To create a default policy for EBS-backed AMIs, that creates EBS-backed AMIs from all instances in the Region
     * that do not have recent backups, specify <code>INSTANCE</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceTypeValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return <b>[Default policies only]</b> Specify the type of default policy to create.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To create a default policy for EBS snapshots, that creates snapshots of all volumes in the Region that do
     *         not have recent backups, specify <code>VOLUME</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To create a default policy for EBS-backed AMIs, that creates EBS-backed AMIs from all instances in the
     *         Region that do not have recent backups, specify <code>INSTANCE</code>.
     *         </p>
     *         </li>
     * @see ResourceTypeValues
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * <p>
     * <b>[Default policies only]</b> Specifies how often the policy should run and create snapshots or AMIs. The
     * creation frequency can range from 1 to 7 days. If you do not specify a value, the default is 1.
     * </p>
     * <p>
     * Default: 1
     * </p>
     * 
     * @return <b>[Default policies only]</b> Specifies how often the policy should run and create snapshots or AMIs.
     *         The creation frequency can range from 1 to 7 days. If you do not specify a value, the default is 1.</p>
     *         <p>
     *         Default: 1
     */
    public final Integer createInterval() {
        return createInterval;
    }

    /**
     * <p>
     * <b>[Default policies only]</b> Specifies how long the policy should retain snapshots or AMIs before deleting
     * them. The retention period can range from 2 to 14 days, but it must be greater than the creation frequency to
     * ensure that the policy retains at least 1 snapshot or AMI at any given time. If you do not specify a value, the
     * default is 7.
     * </p>
     * <p>
     * Default: 7
     * </p>
     * 
     * @return <b>[Default policies only]</b> Specifies how long the policy should retain snapshots or AMIs before
     *         deleting them. The retention period can range from 2 to 14 days, but it must be greater than the creation
     *         frequency to ensure that the policy retains at least 1 snapshot or AMI at any given time. If you do not
     *         specify a value, the default is 7.</p>
     *         <p>
     *         Default: 7
     */
    public final Integer retainInterval() {
        return retainInterval;
    }

    /**
     * <p>
     * <b>[Default policies only]</b> Indicates whether the policy should copy tags from the source resource to the
     * snapshot or AMI. If you do not specify a value, the default is <code>false</code>.
     * </p>
     * <p>
     * Default: false
     * </p>
     * 
     * @return <b>[Default policies only]</b> Indicates whether the policy should copy tags from the source resource to
     *         the snapshot or AMI. If you do not specify a value, the default is <code>false</code>.</p>
     *         <p>
     *         Default: false
     */
    public final Boolean copyTags() {
        return copyTags;
    }

    /**
     * For responses, this returns true if the service returned a value for the CrossRegionCopyTargets property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCrossRegionCopyTargets() {
        return crossRegionCopyTargets != null && !(crossRegionCopyTargets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <b>[Default policies only]</b> Specifies destination Regions for snapshot or AMI copies. You can specify up to 3
     * destination Regions. If you do not want to create cross-Region copies, omit this parameter.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCrossRegionCopyTargets} method.
     * </p>
     * 
     * @return <b>[Default policies only]</b> Specifies destination Regions for snapshot or AMI copies. You can specify
     *         up to 3 destination Regions. If you do not want to create cross-Region copies, omit this parameter.
     */
    public final List<CrossRegionCopyTarget> crossRegionCopyTargets() {
        return crossRegionCopyTargets;
    }

    /**
     * <p>
     * <b>[Default policies only]</b> Defines the snapshot or AMI retention behavior for the policy if the source volume
     * or instance is deleted, or if the policy enters the error, disabled, or deleted state.
     * </p>
     * <p>
     * By default (<b>ExtendDeletion=false</b>):
     * </p>
     * <ul>
     * <li>
     * <p>
     * If a source resource is deleted, Amazon Data Lifecycle Manager will continue to delete previously created
     * snapshots or AMIs, up to but not including the last one, based on the specified retention period. If you want
     * Amazon Data Lifecycle Manager to delete all snapshots or AMIs, including the last one, specify <code>true</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a policy enters the error, disabled, or deleted state, Amazon Data Lifecycle Manager stops deleting snapshots
     * and AMIs. If you want Amazon Data Lifecycle Manager to continue deleting snapshots or AMIs, including the last
     * one, if the policy enters one of these states, specify <code>true</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you enable extended deletion (<b>ExtendDeletion=true</b>), you override both default behaviors simultaneously.
     * </p>
     * <p>
     * If you do not specify a value, the default is <code>false</code>.
     * </p>
     * <p>
     * Default: false
     * </p>
     * 
     * @return <b>[Default policies only]</b> Defines the snapshot or AMI retention behavior for the policy if the
     *         source volume or instance is deleted, or if the policy enters the error, disabled, or deleted state.</p>
     *         <p>
     *         By default (<b>ExtendDeletion=false</b>):
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If a source resource is deleted, Amazon Data Lifecycle Manager will continue to delete previously created
     *         snapshots or AMIs, up to but not including the last one, based on the specified retention period. If you
     *         want Amazon Data Lifecycle Manager to delete all snapshots or AMIs, including the last one, specify
     *         <code>true</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a policy enters the error, disabled, or deleted state, Amazon Data Lifecycle Manager stops deleting
     *         snapshots and AMIs. If you want Amazon Data Lifecycle Manager to continue deleting snapshots or AMIs,
     *         including the last one, if the policy enters one of these states, specify <code>true</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you enable extended deletion (<b>ExtendDeletion=true</b>), you override both default behaviors
     *         simultaneously.
     *         </p>
     *         <p>
     *         If you do not specify a value, the default is <code>false</code>.
     *         </p>
     *         <p>
     *         Default: false
     */
    public final Boolean extendDeletion() {
        return extendDeletion;
    }

    /**
     * <p>
     * <b>[Default policies only]</b> Specifies exclusion parameters for volumes or instances for which you do not want
     * to create snapshots or AMIs. The policy will not create snapshots or AMIs for target resources that match any of
     * the specified exclusion parameters.
     * </p>
     * 
     * @return <b>[Default policies only]</b> Specifies exclusion parameters for volumes or instances for which you do
     *         not want to create snapshots or AMIs. The policy will not create snapshots or AMIs for target resources
     *         that match any of the specified exclusion parameters.
     */
    public final Exclusions exclusions() {
        return exclusions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(policyTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTypes() ? resourceTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceLocations() ? resourceLocationsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTargetTags() ? targetTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSchedules() ? schedules() : null);
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(eventSource());
        hashCode = 31 * hashCode + Objects.hashCode(hasActions() ? actions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(policyLanguageAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createInterval());
        hashCode = 31 * hashCode + Objects.hashCode(retainInterval());
        hashCode = 31 * hashCode + Objects.hashCode(copyTags());
        hashCode = 31 * hashCode + Objects.hashCode(hasCrossRegionCopyTargets() ? crossRegionCopyTargets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(extendDeletion());
        hashCode = 31 * hashCode + Objects.hashCode(exclusions());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PolicyDetails)) {
            return false;
        }
        PolicyDetails other = (PolicyDetails) obj;
        return Objects.equals(policyTypeAsString(), other.policyTypeAsString()) && hasResourceTypes() == other.hasResourceTypes()
                && Objects.equals(resourceTypesAsStrings(), other.resourceTypesAsStrings())
                && hasResourceLocations() == other.hasResourceLocations()
                && Objects.equals(resourceLocationsAsStrings(), other.resourceLocationsAsStrings())
                && hasTargetTags() == other.hasTargetTags() && Objects.equals(targetTags(), other.targetTags())
                && hasSchedules() == other.hasSchedules() && Objects.equals(schedules(), other.schedules())
                && Objects.equals(parameters(), other.parameters()) && Objects.equals(eventSource(), other.eventSource())
                && hasActions() == other.hasActions() && Objects.equals(actions(), other.actions())
                && Objects.equals(policyLanguageAsString(), other.policyLanguageAsString())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && Objects.equals(createInterval(), other.createInterval())
                && Objects.equals(retainInterval(), other.retainInterval()) && Objects.equals(copyTags(), other.copyTags())
                && hasCrossRegionCopyTargets() == other.hasCrossRegionCopyTargets()
                && Objects.equals(crossRegionCopyTargets(), other.crossRegionCopyTargets())
                && Objects.equals(extendDeletion(), other.extendDeletion()) && Objects.equals(exclusions(), other.exclusions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PolicyDetails").add("PolicyType", policyTypeAsString())
                .add("ResourceTypes", hasResourceTypes() ? resourceTypesAsStrings() : null)
                .add("ResourceLocations", hasResourceLocations() ? resourceLocationsAsStrings() : null)
                .add("TargetTags", hasTargetTags() ? targetTags() : null).add("Schedules", hasSchedules() ? schedules() : null)
                .add("Parameters", parameters()).add("EventSource", eventSource())
                .add("Actions", hasActions() ? actions() : null).add("PolicyLanguage", policyLanguageAsString())
                .add("ResourceType", resourceTypeAsString()).add("CreateInterval", createInterval())
                .add("RetainInterval", retainInterval()).add("CopyTags", copyTags())
                .add("CrossRegionCopyTargets", hasCrossRegionCopyTargets() ? crossRegionCopyTargets() : null)
                .add("ExtendDeletion", extendDeletion()).add("Exclusions", exclusions()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PolicyType":
            return Optional.ofNullable(clazz.cast(policyTypeAsString()));
        case "ResourceTypes":
            return Optional.ofNullable(clazz.cast(resourceTypesAsStrings()));
        case "ResourceLocations":
            return Optional.ofNullable(clazz.cast(resourceLocationsAsStrings()));
        case "TargetTags":
            return Optional.ofNullable(clazz.cast(targetTags()));
        case "Schedules":
            return Optional.ofNullable(clazz.cast(schedules()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "EventSource":
            return Optional.ofNullable(clazz.cast(eventSource()));
        case "Actions":
            return Optional.ofNullable(clazz.cast(actions()));
        case "PolicyLanguage":
            return Optional.ofNullable(clazz.cast(policyLanguageAsString()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "CreateInterval":
            return Optional.ofNullable(clazz.cast(createInterval()));
        case "RetainInterval":
            return Optional.ofNullable(clazz.cast(retainInterval()));
        case "CopyTags":
            return Optional.ofNullable(clazz.cast(copyTags()));
        case "CrossRegionCopyTargets":
            return Optional.ofNullable(clazz.cast(crossRegionCopyTargets()));
        case "ExtendDeletion":
            return Optional.ofNullable(clazz.cast(extendDeletion()));
        case "Exclusions":
            return Optional.ofNullable(clazz.cast(exclusions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("PolicyType", POLICY_TYPE_FIELD);
        map.put("ResourceTypes", RESOURCE_TYPES_FIELD);
        map.put("ResourceLocations", RESOURCE_LOCATIONS_FIELD);
        map.put("TargetTags", TARGET_TAGS_FIELD);
        map.put("Schedules", SCHEDULES_FIELD);
        map.put("Parameters", PARAMETERS_FIELD);
        map.put("EventSource", EVENT_SOURCE_FIELD);
        map.put("Actions", ACTIONS_FIELD);
        map.put("PolicyLanguage", POLICY_LANGUAGE_FIELD);
        map.put("ResourceType", RESOURCE_TYPE_FIELD);
        map.put("CreateInterval", CREATE_INTERVAL_FIELD);
        map.put("RetainInterval", RETAIN_INTERVAL_FIELD);
        map.put("CopyTags", COPY_TAGS_FIELD);
        map.put("CrossRegionCopyTargets", CROSS_REGION_COPY_TARGETS_FIELD);
        map.put("ExtendDeletion", EXTEND_DELETION_FIELD);
        map.put("Exclusions", EXCLUSIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PolicyDetails, T> g) {
        return obj -> g.apply((PolicyDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PolicyDetails> {
        /**
         * <p>
         * The type of policy. Specify <code>EBS_SNAPSHOT_MANAGEMENT</code> to create a lifecycle policy that manages
         * the lifecycle of Amazon EBS snapshots. Specify <code>IMAGE_MANAGEMENT</code> to create a lifecycle policy
         * that manages the lifecycle of EBS-backed AMIs. Specify <code>EVENT_BASED_POLICY </code> to create an
         * event-based policy that performs specific actions when a defined event occurs in your Amazon Web Services
         * account.
         * </p>
         * <p>
         * The default is <code>EBS_SNAPSHOT_MANAGEMENT</code>.
         * </p>
         * 
         * @param policyType
         *        The type of policy. Specify <code>EBS_SNAPSHOT_MANAGEMENT</code> to create a lifecycle policy that
         *        manages the lifecycle of Amazon EBS snapshots. Specify <code>IMAGE_MANAGEMENT</code> to create a
         *        lifecycle policy that manages the lifecycle of EBS-backed AMIs. Specify
         *        <code>EVENT_BASED_POLICY </code> to create an event-based policy that performs specific actions when a
         *        defined event occurs in your Amazon Web Services account.</p>
         *        <p>
         *        The default is <code>EBS_SNAPSHOT_MANAGEMENT</code>.
         * @see PolicyTypeValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyTypeValues
         */
        Builder policyType(String policyType);

        /**
         * <p>
         * The type of policy. Specify <code>EBS_SNAPSHOT_MANAGEMENT</code> to create a lifecycle policy that manages
         * the lifecycle of Amazon EBS snapshots. Specify <code>IMAGE_MANAGEMENT</code> to create a lifecycle policy
         * that manages the lifecycle of EBS-backed AMIs. Specify <code>EVENT_BASED_POLICY </code> to create an
         * event-based policy that performs specific actions when a defined event occurs in your Amazon Web Services
         * account.
         * </p>
         * <p>
         * The default is <code>EBS_SNAPSHOT_MANAGEMENT</code>.
         * </p>
         * 
         * @param policyType
         *        The type of policy. Specify <code>EBS_SNAPSHOT_MANAGEMENT</code> to create a lifecycle policy that
         *        manages the lifecycle of Amazon EBS snapshots. Specify <code>IMAGE_MANAGEMENT</code> to create a
         *        lifecycle policy that manages the lifecycle of EBS-backed AMIs. Specify
         *        <code>EVENT_BASED_POLICY </code> to create an event-based policy that performs specific actions when a
         *        defined event occurs in your Amazon Web Services account.</p>
         *        <p>
         *        The default is <code>EBS_SNAPSHOT_MANAGEMENT</code>.
         * @see PolicyTypeValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyTypeValues
         */
        Builder policyType(PolicyTypeValues policyType);

        /**
         * <p>
         * <b>[Custom snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
         * <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create
         * multi-volume snapshots from the volumes for an instance.
         * </p>
         * 
         * @param resourceTypes
         *        <b>[Custom snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle
         *        policies. Use <code>VOLUME </code>to create snapshots of individual volumes or use
         *        <code>INSTANCE</code> to create multi-volume snapshots from the volumes for an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesWithStrings(Collection<String> resourceTypes);

        /**
         * <p>
         * <b>[Custom snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
         * <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create
         * multi-volume snapshots from the volumes for an instance.
         * </p>
         * 
         * @param resourceTypes
         *        <b>[Custom snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle
         *        policies. Use <code>VOLUME </code>to create snapshots of individual volumes or use
         *        <code>INSTANCE</code> to create multi-volume snapshots from the volumes for an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesWithStrings(String... resourceTypes);

        /**
         * <p>
         * <b>[Custom snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
         * <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create
         * multi-volume snapshots from the volumes for an instance.
         * </p>
         * 
         * @param resourceTypes
         *        <b>[Custom snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle
         *        policies. Use <code>VOLUME </code>to create snapshots of individual volumes or use
         *        <code>INSTANCE</code> to create multi-volume snapshots from the volumes for an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(Collection<ResourceTypeValues> resourceTypes);

        /**
         * <p>
         * <b>[Custom snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle policies. Use
         * <code>VOLUME </code>to create snapshots of individual volumes or use <code>INSTANCE</code> to create
         * multi-volume snapshots from the volumes for an instance.
         * </p>
         * 
         * @param resourceTypes
         *        <b>[Custom snapshot policies only]</b> The target resource type for snapshot and AMI lifecycle
         *        policies. Use <code>VOLUME </code>to create snapshots of individual volumes or use
         *        <code>INSTANCE</code> to create multi-volume snapshots from the volumes for an instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(ResourceTypeValues... resourceTypes);

        /**
         * <p>
         * <b>[Custom snapshot and AMI policies only]</b> The location of the resources to backup.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the source resources are located in a Region, specify <code>CLOUD</code>. In this case, the policy targets
         * all resources of the specified type with matching target tags across all Availability Zones in the Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>[Custom snapshot policies only]</b> If the source resources are located in a Local Zone, specify
         * <code>LOCAL_ZONE</code>. In this case, the policy targets all resources of the specified type with matching
         * target tags across all Local Zones in the Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the source resources are located on an Outpost in your account, specify <code>OUTPOST</code>. In this
         * case, the policy targets all resources of the specified type with matching target tags across all of the
         * Outposts in your account.
         * </p>
         * </li>
         * </ul>
         * <p/>
         * 
         * @param resourceLocations
         *        <b>[Custom snapshot and AMI policies only]</b> The location of the resources to backup.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the source resources are located in a Region, specify <code>CLOUD</code>. In this case, the policy
         *        targets all resources of the specified type with matching target tags across all Availability Zones in
         *        the Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>[Custom snapshot policies only]</b> If the source resources are located in a Local Zone, specify
         *        <code>LOCAL_ZONE</code>. In this case, the policy targets all resources of the specified type with
         *        matching target tags across all Local Zones in the Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the source resources are located on an Outpost in your account, specify <code>OUTPOST</code>. In
         *        this case, the policy targets all resources of the specified type with matching target tags across all
         *        of the Outposts in your account.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceLocationsWithStrings(Collection<String> resourceLocations);

        /**
         * <p>
         * <b>[Custom snapshot and AMI policies only]</b> The location of the resources to backup.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the source resources are located in a Region, specify <code>CLOUD</code>. In this case, the policy targets
         * all resources of the specified type with matching target tags across all Availability Zones in the Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>[Custom snapshot policies only]</b> If the source resources are located in a Local Zone, specify
         * <code>LOCAL_ZONE</code>. In this case, the policy targets all resources of the specified type with matching
         * target tags across all Local Zones in the Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the source resources are located on an Outpost in your account, specify <code>OUTPOST</code>. In this
         * case, the policy targets all resources of the specified type with matching target tags across all of the
         * Outposts in your account.
         * </p>
         * </li>
         * </ul>
         * <p/>
         * 
         * @param resourceLocations
         *        <b>[Custom snapshot and AMI policies only]</b> The location of the resources to backup.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the source resources are located in a Region, specify <code>CLOUD</code>. In this case, the policy
         *        targets all resources of the specified type with matching target tags across all Availability Zones in
         *        the Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>[Custom snapshot policies only]</b> If the source resources are located in a Local Zone, specify
         *        <code>LOCAL_ZONE</code>. In this case, the policy targets all resources of the specified type with
         *        matching target tags across all Local Zones in the Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the source resources are located on an Outpost in your account, specify <code>OUTPOST</code>. In
         *        this case, the policy targets all resources of the specified type with matching target tags across all
         *        of the Outposts in your account.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceLocationsWithStrings(String... resourceLocations);

        /**
         * <p>
         * <b>[Custom snapshot and AMI policies only]</b> The location of the resources to backup.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the source resources are located in a Region, specify <code>CLOUD</code>. In this case, the policy targets
         * all resources of the specified type with matching target tags across all Availability Zones in the Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>[Custom snapshot policies only]</b> If the source resources are located in a Local Zone, specify
         * <code>LOCAL_ZONE</code>. In this case, the policy targets all resources of the specified type with matching
         * target tags across all Local Zones in the Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the source resources are located on an Outpost in your account, specify <code>OUTPOST</code>. In this
         * case, the policy targets all resources of the specified type with matching target tags across all of the
         * Outposts in your account.
         * </p>
         * </li>
         * </ul>
         * <p/>
         * 
         * @param resourceLocations
         *        <b>[Custom snapshot and AMI policies only]</b> The location of the resources to backup.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the source resources are located in a Region, specify <code>CLOUD</code>. In this case, the policy
         *        targets all resources of the specified type with matching target tags across all Availability Zones in
         *        the Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>[Custom snapshot policies only]</b> If the source resources are located in a Local Zone, specify
         *        <code>LOCAL_ZONE</code>. In this case, the policy targets all resources of the specified type with
         *        matching target tags across all Local Zones in the Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the source resources are located on an Outpost in your account, specify <code>OUTPOST</code>. In
         *        this case, the policy targets all resources of the specified type with matching target tags across all
         *        of the Outposts in your account.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceLocations(Collection<ResourceLocationValues> resourceLocations);

        /**
         * <p>
         * <b>[Custom snapshot and AMI policies only]</b> The location of the resources to backup.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the source resources are located in a Region, specify <code>CLOUD</code>. In this case, the policy targets
         * all resources of the specified type with matching target tags across all Availability Zones in the Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>[Custom snapshot policies only]</b> If the source resources are located in a Local Zone, specify
         * <code>LOCAL_ZONE</code>. In this case, the policy targets all resources of the specified type with matching
         * target tags across all Local Zones in the Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the source resources are located on an Outpost in your account, specify <code>OUTPOST</code>. In this
         * case, the policy targets all resources of the specified type with matching target tags across all of the
         * Outposts in your account.
         * </p>
         * </li>
         * </ul>
         * <p/>
         * 
         * @param resourceLocations
         *        <b>[Custom snapshot and AMI policies only]</b> The location of the resources to backup.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the source resources are located in a Region, specify <code>CLOUD</code>. In this case, the policy
         *        targets all resources of the specified type with matching target tags across all Availability Zones in
         *        the Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>[Custom snapshot policies only]</b> If the source resources are located in a Local Zone, specify
         *        <code>LOCAL_ZONE</code>. In this case, the policy targets all resources of the specified type with
         *        matching target tags across all Local Zones in the Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the source resources are located on an Outpost in your account, specify <code>OUTPOST</code>. In
         *        this case, the policy targets all resources of the specified type with matching target tags across all
         *        of the Outposts in your account.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceLocations(ResourceLocationValues... resourceLocations);

        /**
         * <p>
         * <b>[Custom snapshot and AMI policies only]</b> The single tag that identifies targeted resources for this
         * policy.
         * </p>
         * 
         * @param targetTags
         *        <b>[Custom snapshot and AMI policies only]</b> The single tag that identifies targeted resources for
         *        this policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetTags(Collection<Tag> targetTags);

        /**
         * <p>
         * <b>[Custom snapshot and AMI policies only]</b> The single tag that identifies targeted resources for this
         * policy.
         * </p>
         * 
         * @param targetTags
         *        <b>[Custom snapshot and AMI policies only]</b> The single tag that identifies targeted resources for
         *        this policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetTags(Tag... targetTags);

        /**
         * <p>
         * <b>[Custom snapshot and AMI policies only]</b> The single tag that identifies targeted resources for this
         * policy.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.dlm.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.dlm.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.dlm.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #targetTags(List<Tag>)}.
         * 
         * @param targetTags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.dlm.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetTags(java.util.Collection<Tag>)
         */
        Builder targetTags(Consumer<Tag.Builder>... targetTags);

        /**
         * <p>
         * <b>[Custom snapshot and AMI policies only]</b> The schedules of policy-defined actions for snapshot and AMI
         * lifecycle policies. A policy can have up to four schedules—one mandatory schedule and up to three optional
         * schedules.
         * </p>
         * 
         * @param schedules
         *        <b>[Custom snapshot and AMI policies only]</b> The schedules of policy-defined actions for snapshot
         *        and AMI lifecycle policies. A policy can have up to four schedules—one mandatory schedule and up to
         *        three optional schedules.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schedules(Collection<Schedule> schedules);

        /**
         * <p>
         * <b>[Custom snapshot and AMI policies only]</b> The schedules of policy-defined actions for snapshot and AMI
         * lifecycle policies. A policy can have up to four schedules—one mandatory schedule and up to three optional
         * schedules.
         * </p>
         * 
         * @param schedules
         *        <b>[Custom snapshot and AMI policies only]</b> The schedules of policy-defined actions for snapshot
         *        and AMI lifecycle policies. A policy can have up to four schedules—one mandatory schedule and up to
         *        three optional schedules.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schedules(Schedule... schedules);

        /**
         * <p>
         * <b>[Custom snapshot and AMI policies only]</b> The schedules of policy-defined actions for snapshot and AMI
         * lifecycle policies. A policy can have up to four schedules—one mandatory schedule and up to three optional
         * schedules.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.dlm.model.Schedule.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.dlm.model.Schedule#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.dlm.model.Schedule.Builder#build()} is called immediately and its
         * result is passed to {@link #schedules(List<Schedule>)}.
         * 
         * @param schedules
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.dlm.model.Schedule.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #schedules(java.util.Collection<Schedule>)
         */
        Builder schedules(Consumer<Schedule.Builder>... schedules);

        /**
         * <p>
         * <b>[Custom snapshot and AMI policies only]</b> A set of optional parameters for snapshot and AMI lifecycle
         * policies.
         * </p>
         * <note>
         * <p>
         * If you are modifying a policy that was created or previously modified using the Amazon Data Lifecycle Manager
         * console, then you must include this parameter and specify either the default values or the new values that
         * you require. You can't omit this parameter or set its values to null.
         * </p>
         * </note>
         * 
         * @param parameters
         *        <b>[Custom snapshot and AMI policies only]</b> A set of optional parameters for snapshot and AMI
         *        lifecycle policies. </p> <note>
         *        <p>
         *        If you are modifying a policy that was created or previously modified using the Amazon Data Lifecycle
         *        Manager console, then you must include this parameter and specify either the default values or the new
         *        values that you require. You can't omit this parameter or set its values to null.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Parameters parameters);

        /**
         * <p>
         * <b>[Custom snapshot and AMI policies only]</b> A set of optional parameters for snapshot and AMI lifecycle
         * policies.
         * </p>
         * <note>
         * <p>
         * If you are modifying a policy that was created or previously modified using the Amazon Data Lifecycle Manager
         * console, then you must include this parameter and specify either the default values or the new values that
         * you require. You can't omit this parameter or set its values to null.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link Parameters.Builder} avoiding the
         * need to create one manually via {@link Parameters#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Parameters.Builder#build()} is called immediately and its result
         * is passed to {@link #parameters(Parameters)}.
         * 
         * @param parameters
         *        a consumer that will call methods on {@link Parameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(Parameters)
         */
        default Builder parameters(Consumer<Parameters.Builder> parameters) {
            return parameters(Parameters.builder().applyMutation(parameters).build());
        }

        /**
         * <p>
         * <b>[Event-based policies only]</b> The event that activates the event-based policy.
         * </p>
         * 
         * @param eventSource
         *        <b>[Event-based policies only]</b> The event that activates the event-based policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventSource(EventSource eventSource);

        /**
         * <p>
         * <b>[Event-based policies only]</b> The event that activates the event-based policy.
         * </p>
         * This is a convenience method that creates an instance of the {@link EventSource.Builder} avoiding the need to
         * create one manually via {@link EventSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EventSource.Builder#build()} is called immediately and its result
         * is passed to {@link #eventSource(EventSource)}.
         * 
         * @param eventSource
         *        a consumer that will call methods on {@link EventSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #eventSource(EventSource)
         */
        default Builder eventSource(Consumer<EventSource.Builder> eventSource) {
            return eventSource(EventSource.builder().applyMutation(eventSource).build());
        }

        /**
         * <p>
         * <b>[Event-based policies only]</b> The actions to be performed when the event-based policy is activated. You
         * can specify only one action per policy.
         * </p>
         * 
         * @param actions
         *        <b>[Event-based policies only]</b> The actions to be performed when the event-based policy is
         *        activated. You can specify only one action per policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(Collection<Action> actions);

        /**
         * <p>
         * <b>[Event-based policies only]</b> The actions to be performed when the event-based policy is activated. You
         * can specify only one action per policy.
         * </p>
         * 
         * @param actions
         *        <b>[Event-based policies only]</b> The actions to be performed when the event-based policy is
         *        activated. You can specify only one action per policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(Action... actions);

        /**
         * <p>
         * <b>[Event-based policies only]</b> The actions to be performed when the event-based policy is activated. You
         * can specify only one action per policy.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.dlm.model.Action.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.dlm.model.Action#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.dlm.model.Action.Builder#build()}
         * is called immediately and its result is passed to {@link #actions(List<Action>)}.
         * 
         * @param actions
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.dlm.model.Action.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #actions(java.util.Collection<Action>)
         */
        Builder actions(Consumer<Action.Builder>... actions);

        /**
         * <p>
         * The type of policy to create. Specify one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SIMPLIFIED</code> To create a default policy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STANDARD</code> To create a custom policy.
         * </p>
         * </li>
         * </ul>
         * 
         * @param policyLanguage
         *        The type of policy to create. Specify one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SIMPLIFIED</code> To create a default policy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STANDARD</code> To create a custom policy.
         *        </p>
         *        </li>
         * @see PolicyLanguageValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyLanguageValues
         */
        Builder policyLanguage(String policyLanguage);

        /**
         * <p>
         * The type of policy to create. Specify one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SIMPLIFIED</code> To create a default policy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STANDARD</code> To create a custom policy.
         * </p>
         * </li>
         * </ul>
         * 
         * @param policyLanguage
         *        The type of policy to create. Specify one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SIMPLIFIED</code> To create a default policy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STANDARD</code> To create a custom policy.
         *        </p>
         *        </li>
         * @see PolicyLanguageValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyLanguageValues
         */
        Builder policyLanguage(PolicyLanguageValues policyLanguage);

        /**
         * <p>
         * <b>[Default policies only]</b> Specify the type of default policy to create.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To create a default policy for EBS snapshots, that creates snapshots of all volumes in the Region that do not
         * have recent backups, specify <code>VOLUME</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To create a default policy for EBS-backed AMIs, that creates EBS-backed AMIs from all instances in the Region
         * that do not have recent backups, specify <code>INSTANCE</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resourceType
         *        <b>[Default policies only]</b> Specify the type of default policy to create.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To create a default policy for EBS snapshots, that creates snapshots of all volumes in the Region that
         *        do not have recent backups, specify <code>VOLUME</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To create a default policy for EBS-backed AMIs, that creates EBS-backed AMIs from all instances in the
         *        Region that do not have recent backups, specify <code>INSTANCE</code>.
         *        </p>
         *        </li>
         * @see ResourceTypeValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceTypeValues
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * <b>[Default policies only]</b> Specify the type of default policy to create.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To create a default policy for EBS snapshots, that creates snapshots of all volumes in the Region that do not
         * have recent backups, specify <code>VOLUME</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To create a default policy for EBS-backed AMIs, that creates EBS-backed AMIs from all instances in the Region
         * that do not have recent backups, specify <code>INSTANCE</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resourceType
         *        <b>[Default policies only]</b> Specify the type of default policy to create.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To create a default policy for EBS snapshots, that creates snapshots of all volumes in the Region that
         *        do not have recent backups, specify <code>VOLUME</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To create a default policy for EBS-backed AMIs, that creates EBS-backed AMIs from all instances in the
         *        Region that do not have recent backups, specify <code>INSTANCE</code>.
         *        </p>
         *        </li>
         * @see ResourceTypeValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceTypeValues
         */
        Builder resourceType(ResourceTypeValues resourceType);

        /**
         * <p>
         * <b>[Default policies only]</b> Specifies how often the policy should run and create snapshots or AMIs. The
         * creation frequency can range from 1 to 7 days. If you do not specify a value, the default is 1.
         * </p>
         * <p>
         * Default: 1
         * </p>
         * 
         * @param createInterval
         *        <b>[Default policies only]</b> Specifies how often the policy should run and create snapshots or AMIs.
         *        The creation frequency can range from 1 to 7 days. If you do not specify a value, the default is
         *        1.</p>
         *        <p>
         *        Default: 1
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createInterval(Integer createInterval);

        /**
         * <p>
         * <b>[Default policies only]</b> Specifies how long the policy should retain snapshots or AMIs before deleting
         * them. The retention period can range from 2 to 14 days, but it must be greater than the creation frequency to
         * ensure that the policy retains at least 1 snapshot or AMI at any given time. If you do not specify a value,
         * the default is 7.
         * </p>
         * <p>
         * Default: 7
         * </p>
         * 
         * @param retainInterval
         *        <b>[Default policies only]</b> Specifies how long the policy should retain snapshots or AMIs before
         *        deleting them. The retention period can range from 2 to 14 days, but it must be greater than the
         *        creation frequency to ensure that the policy retains at least 1 snapshot or AMI at any given time. If
         *        you do not specify a value, the default is 7.</p>
         *        <p>
         *        Default: 7
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retainInterval(Integer retainInterval);

        /**
         * <p>
         * <b>[Default policies only]</b> Indicates whether the policy should copy tags from the source resource to the
         * snapshot or AMI. If you do not specify a value, the default is <code>false</code>.
         * </p>
         * <p>
         * Default: false
         * </p>
         * 
         * @param copyTags
         *        <b>[Default policies only]</b> Indicates whether the policy should copy tags from the source resource
         *        to the snapshot or AMI. If you do not specify a value, the default is <code>false</code>.</p>
         *        <p>
         *        Default: false
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copyTags(Boolean copyTags);

        /**
         * <p>
         * <b>[Default policies only]</b> Specifies destination Regions for snapshot or AMI copies. You can specify up
         * to 3 destination Regions. If you do not want to create cross-Region copies, omit this parameter.
         * </p>
         * 
         * @param crossRegionCopyTargets
         *        <b>[Default policies only]</b> Specifies destination Regions for snapshot or AMI copies. You can
         *        specify up to 3 destination Regions. If you do not want to create cross-Region copies, omit this
         *        parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crossRegionCopyTargets(Collection<CrossRegionCopyTarget> crossRegionCopyTargets);

        /**
         * <p>
         * <b>[Default policies only]</b> Specifies destination Regions for snapshot or AMI copies. You can specify up
         * to 3 destination Regions. If you do not want to create cross-Region copies, omit this parameter.
         * </p>
         * 
         * @param crossRegionCopyTargets
         *        <b>[Default policies only]</b> Specifies destination Regions for snapshot or AMI copies. You can
         *        specify up to 3 destination Regions. If you do not want to create cross-Region copies, omit this
         *        parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crossRegionCopyTargets(CrossRegionCopyTarget... crossRegionCopyTargets);

        /**
         * <p>
         * <b>[Default policies only]</b> Specifies destination Regions for snapshot or AMI copies. You can specify up
         * to 3 destination Regions. If you do not want to create cross-Region copies, omit this parameter.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.dlm.model.CrossRegionCopyTarget.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.dlm.model.CrossRegionCopyTarget#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.dlm.model.CrossRegionCopyTarget.Builder#build()} is called immediately
         * and its result is passed to {@link #crossRegionCopyTargets(List<CrossRegionCopyTarget>)}.
         * 
         * @param crossRegionCopyTargets
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.dlm.model.CrossRegionCopyTarget.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #crossRegionCopyTargets(java.util.Collection<CrossRegionCopyTarget>)
         */
        Builder crossRegionCopyTargets(Consumer<CrossRegionCopyTarget.Builder>... crossRegionCopyTargets);

        /**
         * <p>
         * <b>[Default policies only]</b> Defines the snapshot or AMI retention behavior for the policy if the source
         * volume or instance is deleted, or if the policy enters the error, disabled, or deleted state.
         * </p>
         * <p>
         * By default (<b>ExtendDeletion=false</b>):
         * </p>
         * <ul>
         * <li>
         * <p>
         * If a source resource is deleted, Amazon Data Lifecycle Manager will continue to delete previously created
         * snapshots or AMIs, up to but not including the last one, based on the specified retention period. If you want
         * Amazon Data Lifecycle Manager to delete all snapshots or AMIs, including the last one, specify
         * <code>true</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * If a policy enters the error, disabled, or deleted state, Amazon Data Lifecycle Manager stops deleting
         * snapshots and AMIs. If you want Amazon Data Lifecycle Manager to continue deleting snapshots or AMIs,
         * including the last one, if the policy enters one of these states, specify <code>true</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you enable extended deletion (<b>ExtendDeletion=true</b>), you override both default behaviors
         * simultaneously.
         * </p>
         * <p>
         * If you do not specify a value, the default is <code>false</code>.
         * </p>
         * <p>
         * Default: false
         * </p>
         * 
         * @param extendDeletion
         *        <b>[Default policies only]</b> Defines the snapshot or AMI retention behavior for the policy if the
         *        source volume or instance is deleted, or if the policy enters the error, disabled, or deleted
         *        state.</p>
         *        <p>
         *        By default (<b>ExtendDeletion=false</b>):
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If a source resource is deleted, Amazon Data Lifecycle Manager will continue to delete previously
         *        created snapshots or AMIs, up to but not including the last one, based on the specified retention
         *        period. If you want Amazon Data Lifecycle Manager to delete all snapshots or AMIs, including the last
         *        one, specify <code>true</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a policy enters the error, disabled, or deleted state, Amazon Data Lifecycle Manager stops deleting
         *        snapshots and AMIs. If you want Amazon Data Lifecycle Manager to continue deleting snapshots or AMIs,
         *        including the last one, if the policy enters one of these states, specify <code>true</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you enable extended deletion (<b>ExtendDeletion=true</b>), you override both default behaviors
         *        simultaneously.
         *        </p>
         *        <p>
         *        If you do not specify a value, the default is <code>false</code>.
         *        </p>
         *        <p>
         *        Default: false
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder extendDeletion(Boolean extendDeletion);

        /**
         * <p>
         * <b>[Default policies only]</b> Specifies exclusion parameters for volumes or instances for which you do not
         * want to create snapshots or AMIs. The policy will not create snapshots or AMIs for target resources that
         * match any of the specified exclusion parameters.
         * </p>
         * 
         * @param exclusions
         *        <b>[Default policies only]</b> Specifies exclusion parameters for volumes or instances for which you
         *        do not want to create snapshots or AMIs. The policy will not create snapshots or AMIs for target
         *        resources that match any of the specified exclusion parameters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusions(Exclusions exclusions);

        /**
         * <p>
         * <b>[Default policies only]</b> Specifies exclusion parameters for volumes or instances for which you do not
         * want to create snapshots or AMIs. The policy will not create snapshots or AMIs for target resources that
         * match any of the specified exclusion parameters.
         * </p>
         * This is a convenience method that creates an instance of the {@link Exclusions.Builder} avoiding the need to
         * create one manually via {@link Exclusions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Exclusions.Builder#build()} is called immediately and its result
         * is passed to {@link #exclusions(Exclusions)}.
         * 
         * @param exclusions
         *        a consumer that will call methods on {@link Exclusions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #exclusions(Exclusions)
         */
        default Builder exclusions(Consumer<Exclusions.Builder> exclusions) {
            return exclusions(Exclusions.builder().applyMutation(exclusions).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String policyType;

        private List<String> resourceTypes = DefaultSdkAutoConstructList.getInstance();

        private List<String> resourceLocations = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> targetTags = DefaultSdkAutoConstructList.getInstance();

        private List<Schedule> schedules = DefaultSdkAutoConstructList.getInstance();

        private Parameters parameters;

        private EventSource eventSource;

        private List<Action> actions = DefaultSdkAutoConstructList.getInstance();

        private String policyLanguage;

        private String resourceType;

        private Integer createInterval;

        private Integer retainInterval;

        private Boolean copyTags;

        private List<CrossRegionCopyTarget> crossRegionCopyTargets = DefaultSdkAutoConstructList.getInstance();

        private Boolean extendDeletion;

        private Exclusions exclusions;

        private BuilderImpl() {
        }

        private BuilderImpl(PolicyDetails model) {
            policyType(model.policyType);
            resourceTypesWithStrings(model.resourceTypes);
            resourceLocationsWithStrings(model.resourceLocations);
            targetTags(model.targetTags);
            schedules(model.schedules);
            parameters(model.parameters);
            eventSource(model.eventSource);
            actions(model.actions);
            policyLanguage(model.policyLanguage);
            resourceType(model.resourceType);
            createInterval(model.createInterval);
            retainInterval(model.retainInterval);
            copyTags(model.copyTags);
            crossRegionCopyTargets(model.crossRegionCopyTargets);
            extendDeletion(model.extendDeletion);
            exclusions(model.exclusions);
        }

        public final String getPolicyType() {
            return policyType;
        }

        public final void setPolicyType(String policyType) {
            this.policyType = policyType;
        }

        @Override
        public final Builder policyType(String policyType) {
            this.policyType = policyType;
            return this;
        }

        @Override
        public final Builder policyType(PolicyTypeValues policyType) {
            this.policyType(policyType == null ? null : policyType.toString());
            return this;
        }

        public final Collection<String> getResourceTypes() {
            if (resourceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceTypes;
        }

        public final void setResourceTypes(Collection<String> resourceTypes) {
            this.resourceTypes = ResourceTypeValuesListCopier.copy(resourceTypes);
        }

        @Override
        public final Builder resourceTypesWithStrings(Collection<String> resourceTypes) {
            this.resourceTypes = ResourceTypeValuesListCopier.copy(resourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypesWithStrings(String... resourceTypes) {
            resourceTypesWithStrings(Arrays.asList(resourceTypes));
            return this;
        }

        @Override
        public final Builder resourceTypes(Collection<ResourceTypeValues> resourceTypes) {
            this.resourceTypes = ResourceTypeValuesListCopier.copyEnumToString(resourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypes(ResourceTypeValues... resourceTypes) {
            resourceTypes(Arrays.asList(resourceTypes));
            return this;
        }

        public final Collection<String> getResourceLocations() {
            if (resourceLocations instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceLocations;
        }

        public final void setResourceLocations(Collection<String> resourceLocations) {
            this.resourceLocations = ResourceLocationListCopier.copy(resourceLocations);
        }

        @Override
        public final Builder resourceLocationsWithStrings(Collection<String> resourceLocations) {
            this.resourceLocations = ResourceLocationListCopier.copy(resourceLocations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceLocationsWithStrings(String... resourceLocations) {
            resourceLocationsWithStrings(Arrays.asList(resourceLocations));
            return this;
        }

        @Override
        public final Builder resourceLocations(Collection<ResourceLocationValues> resourceLocations) {
            this.resourceLocations = ResourceLocationListCopier.copyEnumToString(resourceLocations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceLocations(ResourceLocationValues... resourceLocations) {
            resourceLocations(Arrays.asList(resourceLocations));
            return this;
        }

        public final List<Tag.Builder> getTargetTags() {
            List<Tag.Builder> result = TargetTagListCopier.copyToBuilder(this.targetTags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTargetTags(Collection<Tag.BuilderImpl> targetTags) {
            this.targetTags = TargetTagListCopier.copyFromBuilder(targetTags);
        }

        @Override
        public final Builder targetTags(Collection<Tag> targetTags) {
            this.targetTags = TargetTagListCopier.copy(targetTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetTags(Tag... targetTags) {
            targetTags(Arrays.asList(targetTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetTags(Consumer<Tag.Builder>... targetTags) {
            targetTags(Stream.of(targetTags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<Schedule.Builder> getSchedules() {
            List<Schedule.Builder> result = ScheduleListCopier.copyToBuilder(this.schedules);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSchedules(Collection<Schedule.BuilderImpl> schedules) {
            this.schedules = ScheduleListCopier.copyFromBuilder(schedules);
        }

        @Override
        public final Builder schedules(Collection<Schedule> schedules) {
            this.schedules = ScheduleListCopier.copy(schedules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder schedules(Schedule... schedules) {
            schedules(Arrays.asList(schedules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder schedules(Consumer<Schedule.Builder>... schedules) {
            schedules(Stream.of(schedules).map(c -> Schedule.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Parameters.Builder getParameters() {
            return parameters != null ? parameters.toBuilder() : null;
        }

        public final void setParameters(Parameters.BuilderImpl parameters) {
            this.parameters = parameters != null ? parameters.build() : null;
        }

        @Override
        public final Builder parameters(Parameters parameters) {
            this.parameters = parameters;
            return this;
        }

        public final EventSource.Builder getEventSource() {
            return eventSource != null ? eventSource.toBuilder() : null;
        }

        public final void setEventSource(EventSource.BuilderImpl eventSource) {
            this.eventSource = eventSource != null ? eventSource.build() : null;
        }

        @Override
        public final Builder eventSource(EventSource eventSource) {
            this.eventSource = eventSource;
            return this;
        }

        public final List<Action.Builder> getActions() {
            List<Action.Builder> result = ActionListCopier.copyToBuilder(this.actions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setActions(Collection<Action.BuilderImpl> actions) {
            this.actions = ActionListCopier.copyFromBuilder(actions);
        }

        @Override
        public final Builder actions(Collection<Action> actions) {
            this.actions = ActionListCopier.copy(actions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(Action... actions) {
            actions(Arrays.asList(actions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(Consumer<Action.Builder>... actions) {
            actions(Stream.of(actions).map(c -> Action.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getPolicyLanguage() {
            return policyLanguage;
        }

        public final void setPolicyLanguage(String policyLanguage) {
            this.policyLanguage = policyLanguage;
        }

        @Override
        public final Builder policyLanguage(String policyLanguage) {
            this.policyLanguage = policyLanguage;
            return this;
        }

        @Override
        public final Builder policyLanguage(PolicyLanguageValues policyLanguage) {
            this.policyLanguage(policyLanguage == null ? null : policyLanguage.toString());
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceTypeValues resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final Integer getCreateInterval() {
            return createInterval;
        }

        public final void setCreateInterval(Integer createInterval) {
            this.createInterval = createInterval;
        }

        @Override
        public final Builder createInterval(Integer createInterval) {
            this.createInterval = createInterval;
            return this;
        }

        public final Integer getRetainInterval() {
            return retainInterval;
        }

        public final void setRetainInterval(Integer retainInterval) {
            this.retainInterval = retainInterval;
        }

        @Override
        public final Builder retainInterval(Integer retainInterval) {
            this.retainInterval = retainInterval;
            return this;
        }

        public final Boolean getCopyTags() {
            return copyTags;
        }

        public final void setCopyTags(Boolean copyTags) {
            this.copyTags = copyTags;
        }

        @Override
        public final Builder copyTags(Boolean copyTags) {
            this.copyTags = copyTags;
            return this;
        }

        public final List<CrossRegionCopyTarget.Builder> getCrossRegionCopyTargets() {
            List<CrossRegionCopyTarget.Builder> result = CrossRegionCopyTargetListCopier
                    .copyToBuilder(this.crossRegionCopyTargets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCrossRegionCopyTargets(Collection<CrossRegionCopyTarget.BuilderImpl> crossRegionCopyTargets) {
            this.crossRegionCopyTargets = CrossRegionCopyTargetListCopier.copyFromBuilder(crossRegionCopyTargets);
        }

        @Override
        public final Builder crossRegionCopyTargets(Collection<CrossRegionCopyTarget> crossRegionCopyTargets) {
            this.crossRegionCopyTargets = CrossRegionCopyTargetListCopier.copy(crossRegionCopyTargets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder crossRegionCopyTargets(CrossRegionCopyTarget... crossRegionCopyTargets) {
            crossRegionCopyTargets(Arrays.asList(crossRegionCopyTargets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder crossRegionCopyTargets(Consumer<CrossRegionCopyTarget.Builder>... crossRegionCopyTargets) {
            crossRegionCopyTargets(Stream.of(crossRegionCopyTargets)
                    .map(c -> CrossRegionCopyTarget.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Boolean getExtendDeletion() {
            return extendDeletion;
        }

        public final void setExtendDeletion(Boolean extendDeletion) {
            this.extendDeletion = extendDeletion;
        }

        @Override
        public final Builder extendDeletion(Boolean extendDeletion) {
            this.extendDeletion = extendDeletion;
            return this;
        }

        public final Exclusions.Builder getExclusions() {
            return exclusions != null ? exclusions.toBuilder() : null;
        }

        public final void setExclusions(Exclusions.BuilderImpl exclusions) {
            this.exclusions = exclusions != null ? exclusions.build() : null;
        }

        @Override
        public final Builder exclusions(Exclusions exclusions) {
            this.exclusions = exclusions;
            return this;
        }

        @Override
        public PolicyDetails build() {
            return new PolicyDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
