/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.docdb.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the input to <a>CopyDBClusterSnapshot</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CopyDbClusterSnapshotRequest extends DocDbRequest implements
        ToCopyableBuilder<CopyDbClusterSnapshotRequest.Builder, CopyDbClusterSnapshotRequest> {
    private static final SdkField<String> SOURCE_DB_CLUSTER_SNAPSHOT_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SourceDBClusterSnapshotIdentifier")
            .getter(getter(CopyDbClusterSnapshotRequest::sourceDBClusterSnapshotIdentifier))
            .setter(setter(Builder::sourceDBClusterSnapshotIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceDBClusterSnapshotIdentifier")
                    .build()).build();

    private static final SdkField<String> TARGET_DB_CLUSTER_SNAPSHOT_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TargetDBClusterSnapshotIdentifier")
            .getter(getter(CopyDbClusterSnapshotRequest::targetDBClusterSnapshotIdentifier))
            .setter(setter(Builder::targetDBClusterSnapshotIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetDBClusterSnapshotIdentifier")
                    .build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(CopyDbClusterSnapshotRequest::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<String> PRE_SIGNED_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreSignedUrl").getter(getter(CopyDbClusterSnapshotRequest::preSignedUrl))
            .setter(setter(Builder::preSignedUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreSignedUrl").build()).build();

    private static final SdkField<Boolean> COPY_TAGS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CopyTags").getter(getter(CopyDbClusterSnapshotRequest::copyTags)).setter(setter(Builder::copyTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyTags").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CopyDbClusterSnapshotRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Tag")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Tag").build()).build()).build()).build();

    private static final SdkField<String> SOURCE_REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceRegion").getter(getter(CopyDbClusterSnapshotRequest::sourceRegion))
            .setter(setter(Builder::sourceRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceRegion").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            SOURCE_DB_CLUSTER_SNAPSHOT_IDENTIFIER_FIELD, TARGET_DB_CLUSTER_SNAPSHOT_IDENTIFIER_FIELD, KMS_KEY_ID_FIELD,
            PRE_SIGNED_URL_FIELD, COPY_TAGS_FIELD, TAGS_FIELD, SOURCE_REGION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String sourceDBClusterSnapshotIdentifier;

    private final String targetDBClusterSnapshotIdentifier;

    private final String kmsKeyId;

    private final String preSignedUrl;

    private final Boolean copyTags;

    private final List<Tag> tags;

    private final String sourceRegion;

    private CopyDbClusterSnapshotRequest(BuilderImpl builder) {
        super(builder);
        this.sourceDBClusterSnapshotIdentifier = builder.sourceDBClusterSnapshotIdentifier;
        this.targetDBClusterSnapshotIdentifier = builder.targetDBClusterSnapshotIdentifier;
        this.kmsKeyId = builder.kmsKeyId;
        this.preSignedUrl = builder.preSignedUrl;
        this.copyTags = builder.copyTags;
        this.tags = builder.tags;
        this.sourceRegion = builder.sourceRegion;
    }

    /**
     * <p>
     * The identifier of the cluster snapshot to copy. This parameter is not case sensitive.
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must specify a valid system snapshot in the <i>available</i> state.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the source snapshot is in the same Amazon Web Services Region as the copy, specify a valid snapshot
     * identifier.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the source snapshot is in a different Amazon Web Services Region than the copy, specify a valid cluster
     * snapshot ARN.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Example: <code>my-cluster-snapshot1</code>
     * </p>
     * 
     * @return The identifier of the cluster snapshot to copy. This parameter is not case sensitive.</p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Must specify a valid system snapshot in the <i>available</i> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the source snapshot is in the same Amazon Web Services Region as the copy, specify a valid snapshot
     *         identifier.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If the source snapshot is in a different Amazon Web Services Region than the copy, specify a valid
     *         cluster snapshot ARN.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Example: <code>my-cluster-snapshot1</code>
     */
    public final String sourceDBClusterSnapshotIdentifier() {
        return sourceDBClusterSnapshotIdentifier;
    }

    /**
     * <p>
     * The identifier of the new cluster snapshot to create from the source cluster snapshot. This parameter is not case
     * sensitive.
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must contain from 1 to 63 letters, numbers, or hyphens.
     * </p>
     * </li>
     * <li>
     * <p>
     * The first character must be a letter.
     * </p>
     * </li>
     * <li>
     * <p>
     * Cannot end with a hyphen or contain two consecutive hyphens.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Example: <code>my-cluster-snapshot2</code>
     * </p>
     * 
     * @return The identifier of the new cluster snapshot to create from the source cluster snapshot. This parameter is
     *         not case sensitive.</p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Must contain from 1 to 63 letters, numbers, or hyphens.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The first character must be a letter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Cannot end with a hyphen or contain two consecutive hyphens.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Example: <code>my-cluster-snapshot2</code>
     */
    public final String targetDBClusterSnapshotIdentifier() {
        return targetDBClusterSnapshotIdentifier;
    }

    /**
     * <p>
     * The KMS key ID for an encrypted cluster snapshot. The KMS key ID is the Amazon Resource Name (ARN), KMS key
     * identifier, or the KMS key alias for the KMS encryption key.
     * </p>
     * <p>
     * If you copy an encrypted cluster snapshot from your Amazon Web Services account, you can specify a value for
     * <code>KmsKeyId</code> to encrypt the copy with a new KMS encryption key. If you don't specify a value for
     * <code>KmsKeyId</code>, then the copy of the cluster snapshot is encrypted with the same KMS key as the source
     * cluster snapshot.
     * </p>
     * <p>
     * If you copy an encrypted cluster snapshot that is shared from another Amazon Web Services account, then you must
     * specify a value for <code>KmsKeyId</code>.
     * </p>
     * <p>
     * To copy an encrypted cluster snapshot to another Amazon Web Services Region, set <code>KmsKeyId</code> to the KMS
     * key ID that you want to use to encrypt the copy of the cluster snapshot in the destination Region. KMS encryption
     * keys are specific to the Amazon Web Services Region that they are created in, and you can't use encryption keys
     * from one Amazon Web Services Region in another Amazon Web Services Region.
     * </p>
     * <p>
     * If you copy an unencrypted cluster snapshot and specify a value for the <code>KmsKeyId</code> parameter, an error
     * is returned.
     * </p>
     * 
     * @return The KMS key ID for an encrypted cluster snapshot. The KMS key ID is the Amazon Resource Name (ARN), KMS
     *         key identifier, or the KMS key alias for the KMS encryption key. </p>
     *         <p>
     *         If you copy an encrypted cluster snapshot from your Amazon Web Services account, you can specify a value
     *         for <code>KmsKeyId</code> to encrypt the copy with a new KMS encryption key. If you don't specify a value
     *         for <code>KmsKeyId</code>, then the copy of the cluster snapshot is encrypted with the same KMS key as
     *         the source cluster snapshot.
     *         </p>
     *         <p>
     *         If you copy an encrypted cluster snapshot that is shared from another Amazon Web Services account, then
     *         you must specify a value for <code>KmsKeyId</code>.
     *         </p>
     *         <p>
     *         To copy an encrypted cluster snapshot to another Amazon Web Services Region, set <code>KmsKeyId</code> to
     *         the KMS key ID that you want to use to encrypt the copy of the cluster snapshot in the destination
     *         Region. KMS encryption keys are specific to the Amazon Web Services Region that they are created in, and
     *         you can't use encryption keys from one Amazon Web Services Region in another Amazon Web Services Region.
     *         </p>
     *         <p>
     *         If you copy an unencrypted cluster snapshot and specify a value for the <code>KmsKeyId</code> parameter,
     *         an error is returned.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The URL that contains a Signature Version 4 signed request for the<code>CopyDBClusterSnapshot</code> API action
     * in the Amazon Web Services Region that contains the source cluster snapshot to copy. You must use the
     * <code>PreSignedUrl</code> parameter when copying a cluster snapshot from another Amazon Web Services Region.
     * </p>
     * <p>
     * If you are using an Amazon Web Services SDK tool or the CLI, you can specify <code>SourceRegion</code> (or
     * <code>--source-region</code> for the CLI) instead of specifying <code>PreSignedUrl</code> manually. Specifying
     * <code>SourceRegion</code> autogenerates a pre-signed URL that is a valid request for the operation that can be
     * executed in the source Amazon Web Services Region.
     * </p>
     * <p>
     * The presigned URL must be a valid request for the <code>CopyDBClusterSnapshot</code> API action that can be
     * executed in the source Amazon Web Services Region that contains the cluster snapshot to be copied. The presigned
     * URL request must contain the following parameter values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SourceRegion</code> - The ID of the region that contains the snapshot to be copied.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SourceDBClusterSnapshotIdentifier</code> - The identifier for the the encrypted cluster snapshot to be
     * copied. This identifier must be in the Amazon Resource Name (ARN) format for the source Amazon Web Services
     * Region. For example, if you are copying an encrypted cluster snapshot from the us-east-1 Amazon Web Services
     * Region, then your <code>SourceDBClusterSnapshotIdentifier</code> looks something like the following:
     * <code>arn:aws:rds:us-east-1:12345678012:sample-cluster:sample-cluster-snapshot</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TargetDBClusterSnapshotIdentifier</code> - The identifier for the new cluster snapshot to be created. This
     * parameter isn't case sensitive.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The URL that contains a Signature Version 4 signed request for the<code>CopyDBClusterSnapshot</code> API
     *         action in the Amazon Web Services Region that contains the source cluster snapshot to copy. You must use
     *         the <code>PreSignedUrl</code> parameter when copying a cluster snapshot from another Amazon Web Services
     *         Region.</p>
     *         <p>
     *         If you are using an Amazon Web Services SDK tool or the CLI, you can specify <code>SourceRegion</code>
     *         (or <code>--source-region</code> for the CLI) instead of specifying <code>PreSignedUrl</code> manually.
     *         Specifying <code>SourceRegion</code> autogenerates a pre-signed URL that is a valid request for the
     *         operation that can be executed in the source Amazon Web Services Region.
     *         </p>
     *         <p>
     *         The presigned URL must be a valid request for the <code>CopyDBClusterSnapshot</code> API action that can
     *         be executed in the source Amazon Web Services Region that contains the cluster snapshot to be copied. The
     *         presigned URL request must contain the following parameter values:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SourceRegion</code> - The ID of the region that contains the snapshot to be copied.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SourceDBClusterSnapshotIdentifier</code> - The identifier for the the encrypted cluster snapshot to
     *         be copied. This identifier must be in the Amazon Resource Name (ARN) format for the source Amazon Web
     *         Services Region. For example, if you are copying an encrypted cluster snapshot from the us-east-1 Amazon
     *         Web Services Region, then your <code>SourceDBClusterSnapshotIdentifier</code> looks something like the
     *         following: <code>arn:aws:rds:us-east-1:12345678012:sample-cluster:sample-cluster-snapshot</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TargetDBClusterSnapshotIdentifier</code> - The identifier for the new cluster snapshot to be
     *         created. This parameter isn't case sensitive.
     *         </p>
     *         </li>
     */
    public final String preSignedUrl() {
        return preSignedUrl;
    }

    /**
     * <p>
     * Set to <code>true</code> to copy all tags from the source cluster snapshot to the target cluster snapshot, and
     * otherwise <code>false</code>. The default is <code>false</code>.
     * </p>
     * 
     * @return Set to <code>true</code> to copy all tags from the source cluster snapshot to the target cluster
     *         snapshot, and otherwise <code>false</code>. The default is <code>false</code>.
     */
    public final Boolean copyTags() {
        return copyTags;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags to be assigned to the cluster snapshot.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags to be assigned to the cluster snapshot.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * If PreSignedUrl is not specified, this is the region where the source snapshot is located. A PreSignedUrl will be
     * generated automatically by the SDK.
     * 
     * @return If PreSignedUrl is not specified, this is the region where the source snapshot is located. A PreSignedUrl
     *         will be generated automatically by the SDK.
     */
    public final String sourceRegion() {
        return sourceRegion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(sourceDBClusterSnapshotIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(targetDBClusterSnapshotIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(preSignedUrl());
        hashCode = 31 * hashCode + Objects.hashCode(copyTags());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sourceRegion());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CopyDbClusterSnapshotRequest)) {
            return false;
        }
        CopyDbClusterSnapshotRequest other = (CopyDbClusterSnapshotRequest) obj;
        return Objects.equals(sourceDBClusterSnapshotIdentifier(), other.sourceDBClusterSnapshotIdentifier())
                && Objects.equals(targetDBClusterSnapshotIdentifier(), other.targetDBClusterSnapshotIdentifier())
                && Objects.equals(kmsKeyId(), other.kmsKeyId()) && Objects.equals(preSignedUrl(), other.preSignedUrl())
                && Objects.equals(copyTags(), other.copyTags()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(sourceRegion(), other.sourceRegion());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CopyDbClusterSnapshotRequest")
                .add("SourceDBClusterSnapshotIdentifier", sourceDBClusterSnapshotIdentifier())
                .add("TargetDBClusterSnapshotIdentifier", targetDBClusterSnapshotIdentifier()).add("KmsKeyId", kmsKeyId())
                .add("PreSignedUrl", preSignedUrl()).add("CopyTags", copyTags()).add("Tags", hasTags() ? tags() : null)
                .add("SourceRegion", sourceRegion()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SourceDBClusterSnapshotIdentifier":
            return Optional.ofNullable(clazz.cast(sourceDBClusterSnapshotIdentifier()));
        case "TargetDBClusterSnapshotIdentifier":
            return Optional.ofNullable(clazz.cast(targetDBClusterSnapshotIdentifier()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "PreSignedUrl":
            return Optional.ofNullable(clazz.cast(preSignedUrl()));
        case "CopyTags":
            return Optional.ofNullable(clazz.cast(copyTags()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "SourceRegion":
            return Optional.ofNullable(clazz.cast(sourceRegion()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SourceDBClusterSnapshotIdentifier", SOURCE_DB_CLUSTER_SNAPSHOT_IDENTIFIER_FIELD);
        map.put("TargetDBClusterSnapshotIdentifier", TARGET_DB_CLUSTER_SNAPSHOT_IDENTIFIER_FIELD);
        map.put("KmsKeyId", KMS_KEY_ID_FIELD);
        map.put("PreSignedUrl", PRE_SIGNED_URL_FIELD);
        map.put("CopyTags", COPY_TAGS_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("SourceRegion", SOURCE_REGION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CopyDbClusterSnapshotRequest, T> g) {
        return obj -> g.apply((CopyDbClusterSnapshotRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DocDbRequest.Builder, SdkPojo, CopyableBuilder<Builder, CopyDbClusterSnapshotRequest> {
        /**
         * <p>
         * The identifier of the cluster snapshot to copy. This parameter is not case sensitive.
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must specify a valid system snapshot in the <i>available</i> state.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the source snapshot is in the same Amazon Web Services Region as the copy, specify a valid snapshot
         * identifier.
         * </p>
         * </li>
         * <li>
         * <p>
         * If the source snapshot is in a different Amazon Web Services Region than the copy, specify a valid cluster
         * snapshot ARN.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Example: <code>my-cluster-snapshot1</code>
         * </p>
         * 
         * @param sourceDBClusterSnapshotIdentifier
         *        The identifier of the cluster snapshot to copy. This parameter is not case sensitive.</p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must specify a valid system snapshot in the <i>available</i> state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the source snapshot is in the same Amazon Web Services Region as the copy, specify a valid snapshot
         *        identifier.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If the source snapshot is in a different Amazon Web Services Region than the copy, specify a valid
         *        cluster snapshot ARN.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Example: <code>my-cluster-snapshot1</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceDBClusterSnapshotIdentifier(String sourceDBClusterSnapshotIdentifier);

        /**
         * <p>
         * The identifier of the new cluster snapshot to create from the source cluster snapshot. This parameter is not
         * case sensitive.
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must contain from 1 to 63 letters, numbers, or hyphens.
         * </p>
         * </li>
         * <li>
         * <p>
         * The first character must be a letter.
         * </p>
         * </li>
         * <li>
         * <p>
         * Cannot end with a hyphen or contain two consecutive hyphens.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Example: <code>my-cluster-snapshot2</code>
         * </p>
         * 
         * @param targetDBClusterSnapshotIdentifier
         *        The identifier of the new cluster snapshot to create from the source cluster snapshot. This parameter
         *        is not case sensitive.</p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must contain from 1 to 63 letters, numbers, or hyphens.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The first character must be a letter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cannot end with a hyphen or contain two consecutive hyphens.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Example: <code>my-cluster-snapshot2</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetDBClusterSnapshotIdentifier(String targetDBClusterSnapshotIdentifier);

        /**
         * <p>
         * The KMS key ID for an encrypted cluster snapshot. The KMS key ID is the Amazon Resource Name (ARN), KMS key
         * identifier, or the KMS key alias for the KMS encryption key.
         * </p>
         * <p>
         * If you copy an encrypted cluster snapshot from your Amazon Web Services account, you can specify a value for
         * <code>KmsKeyId</code> to encrypt the copy with a new KMS encryption key. If you don't specify a value for
         * <code>KmsKeyId</code>, then the copy of the cluster snapshot is encrypted with the same KMS key as the source
         * cluster snapshot.
         * </p>
         * <p>
         * If you copy an encrypted cluster snapshot that is shared from another Amazon Web Services account, then you
         * must specify a value for <code>KmsKeyId</code>.
         * </p>
         * <p>
         * To copy an encrypted cluster snapshot to another Amazon Web Services Region, set <code>KmsKeyId</code> to the
         * KMS key ID that you want to use to encrypt the copy of the cluster snapshot in the destination Region. KMS
         * encryption keys are specific to the Amazon Web Services Region that they are created in, and you can't use
         * encryption keys from one Amazon Web Services Region in another Amazon Web Services Region.
         * </p>
         * <p>
         * If you copy an unencrypted cluster snapshot and specify a value for the <code>KmsKeyId</code> parameter, an
         * error is returned.
         * </p>
         * 
         * @param kmsKeyId
         *        The KMS key ID for an encrypted cluster snapshot. The KMS key ID is the Amazon Resource Name (ARN),
         *        KMS key identifier, or the KMS key alias for the KMS encryption key. </p>
         *        <p>
         *        If you copy an encrypted cluster snapshot from your Amazon Web Services account, you can specify a
         *        value for <code>KmsKeyId</code> to encrypt the copy with a new KMS encryption key. If you don't
         *        specify a value for <code>KmsKeyId</code>, then the copy of the cluster snapshot is encrypted with the
         *        same KMS key as the source cluster snapshot.
         *        </p>
         *        <p>
         *        If you copy an encrypted cluster snapshot that is shared from another Amazon Web Services account,
         *        then you must specify a value for <code>KmsKeyId</code>.
         *        </p>
         *        <p>
         *        To copy an encrypted cluster snapshot to another Amazon Web Services Region, set <code>KmsKeyId</code>
         *        to the KMS key ID that you want to use to encrypt the copy of the cluster snapshot in the destination
         *        Region. KMS encryption keys are specific to the Amazon Web Services Region that they are created in,
         *        and you can't use encryption keys from one Amazon Web Services Region in another Amazon Web Services
         *        Region.
         *        </p>
         *        <p>
         *        If you copy an unencrypted cluster snapshot and specify a value for the <code>KmsKeyId</code>
         *        parameter, an error is returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The URL that contains a Signature Version 4 signed request for the<code>CopyDBClusterSnapshot</code> API
         * action in the Amazon Web Services Region that contains the source cluster snapshot to copy. You must use the
         * <code>PreSignedUrl</code> parameter when copying a cluster snapshot from another Amazon Web Services Region.
         * </p>
         * <p>
         * If you are using an Amazon Web Services SDK tool or the CLI, you can specify <code>SourceRegion</code> (or
         * <code>--source-region</code> for the CLI) instead of specifying <code>PreSignedUrl</code> manually.
         * Specifying <code>SourceRegion</code> autogenerates a pre-signed URL that is a valid request for the operation
         * that can be executed in the source Amazon Web Services Region.
         * </p>
         * <p>
         * The presigned URL must be a valid request for the <code>CopyDBClusterSnapshot</code> API action that can be
         * executed in the source Amazon Web Services Region that contains the cluster snapshot to be copied. The
         * presigned URL request must contain the following parameter values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SourceRegion</code> - The ID of the region that contains the snapshot to be copied.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SourceDBClusterSnapshotIdentifier</code> - The identifier for the the encrypted cluster snapshot to be
         * copied. This identifier must be in the Amazon Resource Name (ARN) format for the source Amazon Web Services
         * Region. For example, if you are copying an encrypted cluster snapshot from the us-east-1 Amazon Web Services
         * Region, then your <code>SourceDBClusterSnapshotIdentifier</code> looks something like the following:
         * <code>arn:aws:rds:us-east-1:12345678012:sample-cluster:sample-cluster-snapshot</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TargetDBClusterSnapshotIdentifier</code> - The identifier for the new cluster snapshot to be created.
         * This parameter isn't case sensitive.
         * </p>
         * </li>
         * </ul>
         * 
         * @param preSignedUrl
         *        The URL that contains a Signature Version 4 signed request for the<code>CopyDBClusterSnapshot</code>
         *        API action in the Amazon Web Services Region that contains the source cluster snapshot to copy. You
         *        must use the <code>PreSignedUrl</code> parameter when copying a cluster snapshot from another Amazon
         *        Web Services Region.</p>
         *        <p>
         *        If you are using an Amazon Web Services SDK tool or the CLI, you can specify <code>SourceRegion</code>
         *        (or <code>--source-region</code> for the CLI) instead of specifying <code>PreSignedUrl</code>
         *        manually. Specifying <code>SourceRegion</code> autogenerates a pre-signed URL that is a valid request
         *        for the operation that can be executed in the source Amazon Web Services Region.
         *        </p>
         *        <p>
         *        The presigned URL must be a valid request for the <code>CopyDBClusterSnapshot</code> API action that
         *        can be executed in the source Amazon Web Services Region that contains the cluster snapshot to be
         *        copied. The presigned URL request must contain the following parameter values:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SourceRegion</code> - The ID of the region that contains the snapshot to be copied.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SourceDBClusterSnapshotIdentifier</code> - The identifier for the the encrypted cluster snapshot
         *        to be copied. This identifier must be in the Amazon Resource Name (ARN) format for the source Amazon
         *        Web Services Region. For example, if you are copying an encrypted cluster snapshot from the us-east-1
         *        Amazon Web Services Region, then your <code>SourceDBClusterSnapshotIdentifier</code> looks something
         *        like the following:
         *        <code>arn:aws:rds:us-east-1:12345678012:sample-cluster:sample-cluster-snapshot</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TargetDBClusterSnapshotIdentifier</code> - The identifier for the new cluster snapshot to be
         *        created. This parameter isn't case sensitive.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preSignedUrl(String preSignedUrl);

        /**
         * <p>
         * Set to <code>true</code> to copy all tags from the source cluster snapshot to the target cluster snapshot,
         * and otherwise <code>false</code>. The default is <code>false</code>.
         * </p>
         * 
         * @param copyTags
         *        Set to <code>true</code> to copy all tags from the source cluster snapshot to the target cluster
         *        snapshot, and otherwise <code>false</code>. The default is <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copyTags(Boolean copyTags);

        /**
         * <p>
         * The tags to be assigned to the cluster snapshot.
         * </p>
         * 
         * @param tags
         *        The tags to be assigned to the cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags to be assigned to the cluster snapshot.
         * </p>
         * 
         * @param tags
         *        The tags to be assigned to the cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags to be assigned to the cluster snapshot.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.docdb.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.docdb.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.docdb.model.Tag.Builder#build()}
         * is called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.docdb.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * If PreSignedUrl is not specified, this is the region where the source snapshot is located. A PreSignedUrl
         * will be generated automatically by the SDK.
         * 
         * @param sourceRegion
         *        If PreSignedUrl is not specified, this is the region where the source snapshot is located. A
         *        PreSignedUrl will be generated automatically by the SDK.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceRegion(String sourceRegion);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DocDbRequest.BuilderImpl implements Builder {
        private String sourceDBClusterSnapshotIdentifier;

        private String targetDBClusterSnapshotIdentifier;

        private String kmsKeyId;

        private String preSignedUrl;

        private Boolean copyTags;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String sourceRegion;

        private BuilderImpl() {
        }

        private BuilderImpl(CopyDbClusterSnapshotRequest model) {
            super(model);
            sourceDBClusterSnapshotIdentifier(model.sourceDBClusterSnapshotIdentifier);
            targetDBClusterSnapshotIdentifier(model.targetDBClusterSnapshotIdentifier);
            kmsKeyId(model.kmsKeyId);
            preSignedUrl(model.preSignedUrl);
            copyTags(model.copyTags);
            tags(model.tags);
            sourceRegion(model.sourceRegion);
        }

        public final String getSourceDBClusterSnapshotIdentifier() {
            return sourceDBClusterSnapshotIdentifier;
        }

        public final void setSourceDBClusterSnapshotIdentifier(String sourceDBClusterSnapshotIdentifier) {
            this.sourceDBClusterSnapshotIdentifier = sourceDBClusterSnapshotIdentifier;
        }

        @Override
        public final Builder sourceDBClusterSnapshotIdentifier(String sourceDBClusterSnapshotIdentifier) {
            this.sourceDBClusterSnapshotIdentifier = sourceDBClusterSnapshotIdentifier;
            return this;
        }

        public final String getTargetDBClusterSnapshotIdentifier() {
            return targetDBClusterSnapshotIdentifier;
        }

        public final void setTargetDBClusterSnapshotIdentifier(String targetDBClusterSnapshotIdentifier) {
            this.targetDBClusterSnapshotIdentifier = targetDBClusterSnapshotIdentifier;
        }

        @Override
        public final Builder targetDBClusterSnapshotIdentifier(String targetDBClusterSnapshotIdentifier) {
            this.targetDBClusterSnapshotIdentifier = targetDBClusterSnapshotIdentifier;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final String getPreSignedUrl() {
            return preSignedUrl;
        }

        public final void setPreSignedUrl(String preSignedUrl) {
            this.preSignedUrl = preSignedUrl;
        }

        @Override
        public final Builder preSignedUrl(String preSignedUrl) {
            this.preSignedUrl = preSignedUrl;
            return this;
        }

        public final Boolean getCopyTags() {
            return copyTags;
        }

        public final void setCopyTags(Boolean copyTags) {
            this.copyTags = copyTags;
        }

        @Override
        public final Builder copyTags(Boolean copyTags) {
            this.copyTags = copyTags;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getSourceRegion() {
            return sourceRegion;
        }

        public final void setSourceRegion(String sourceRegion) {
            this.sourceRegion = sourceRegion;
        }

        @Override
        public final Builder sourceRegion(String sourceRegion) {
            this.sourceRegion = sourceRegion;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CopyDbClusterSnapshotRequest build() {
            return new CopyDbClusterSnapshotRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
