/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dsql.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.dsql.DsqlClient;
import software.amazon.awssdk.services.dsql.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.dsql.model.DsqlRequest;
import software.amazon.awssdk.services.dsql.model.GetClusterRequest;
import software.amazon.awssdk.services.dsql.model.GetClusterResponse;
import software.amazon.awssdk.services.dsql.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultDsqlWaiter implements DsqlWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final DsqlClient client;

    private final AttributeMap managedResources;

    private final Waiter<GetClusterResponse> clusterActiveWaiter;

    private final Waiter<GetClusterResponse> clusterNotExistsWaiter;

    private DefaultDsqlWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = DsqlClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.clusterActiveWaiter = Waiter.builder(GetClusterResponse.class).acceptors(clusterActiveWaiterAcceptors())
                .overrideConfiguration(clusterActiveWaiterConfig(builder.overrideConfiguration)).build();
        this.clusterNotExistsWaiter = Waiter.builder(GetClusterResponse.class).acceptors(clusterNotExistsWaiterAcceptors())
                .overrideConfiguration(clusterNotExistsWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<GetClusterResponse> waitUntilClusterActive(GetClusterRequest getClusterRequest) {
        return clusterActiveWaiter.run(() -> client.getCluster(applyWaitersUserAgent(getClusterRequest)));
    }

    @Override
    public WaiterResponse<GetClusterResponse> waitUntilClusterActive(GetClusterRequest getClusterRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return clusterActiveWaiter.run(() -> client.getCluster(applyWaitersUserAgent(getClusterRequest)),
                clusterActiveWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<GetClusterResponse> waitUntilClusterNotExists(GetClusterRequest getClusterRequest) {
        return clusterNotExistsWaiter.run(() -> client.getCluster(applyWaitersUserAgent(getClusterRequest)));
    }

    @Override
    public WaiterResponse<GetClusterResponse> waitUntilClusterNotExists(GetClusterRequest getClusterRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return clusterNotExistsWaiter.run(() -> client.getCluster(applyWaitersUserAgent(getClusterRequest)),
                clusterNotExistsWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetClusterResponse>> clusterActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetClusterResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("status").value(), "ACTIVE");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetClusterResponse>> clusterNotExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super GetClusterResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration clusterActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration clusterNotExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static DsqlWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends DsqlRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements DsqlWaiter.Builder {
        private DsqlClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public DsqlWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public DsqlWaiter.Builder client(DsqlClient client) {
            this.client = client;
            return this;
        }

        public DsqlWaiter build() {
            return new DefaultDsqlWaiter(this);
        }
    }
}
