/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dsql;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.dsql.model.CreateClusterRequest;
import software.amazon.awssdk.services.dsql.model.CreateClusterResponse;
import software.amazon.awssdk.services.dsql.model.CreateMultiRegionClustersRequest;
import software.amazon.awssdk.services.dsql.model.CreateMultiRegionClustersResponse;
import software.amazon.awssdk.services.dsql.model.DeleteClusterRequest;
import software.amazon.awssdk.services.dsql.model.DeleteClusterResponse;
import software.amazon.awssdk.services.dsql.model.DeleteMultiRegionClustersRequest;
import software.amazon.awssdk.services.dsql.model.DeleteMultiRegionClustersResponse;
import software.amazon.awssdk.services.dsql.model.GetClusterRequest;
import software.amazon.awssdk.services.dsql.model.GetClusterResponse;
import software.amazon.awssdk.services.dsql.model.ListClustersRequest;
import software.amazon.awssdk.services.dsql.model.ListClustersResponse;
import software.amazon.awssdk.services.dsql.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.dsql.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.dsql.model.TagResourceRequest;
import software.amazon.awssdk.services.dsql.model.TagResourceResponse;
import software.amazon.awssdk.services.dsql.model.UntagResourceRequest;
import software.amazon.awssdk.services.dsql.model.UntagResourceResponse;
import software.amazon.awssdk.services.dsql.model.UpdateClusterRequest;
import software.amazon.awssdk.services.dsql.model.UpdateClusterResponse;
import software.amazon.awssdk.services.dsql.paginators.ListClustersPublisher;
import software.amazon.awssdk.services.dsql.waiters.DsqlAsyncWaiter;

/**
 * Service client for accessing Amazon Aurora DSQL asynchronously. This can be created using the static
 * {@link #builder()} method.The asynchronous client performs non-blocking I/O when configured with any
 * {@code SdkAsyncHttpClient} supported in the SDK. However, full non-blocking is not guaranteed as the async client may
 * perform blocking calls in some cases such as credentials retrieval and endpoint discovery as part of the async API
 * call.
 *
 * <p>
 * This is an interface reference for Amazon Aurora DSQL. It contains documentation for one of the programming or
 * command line interfaces you can use to manage Amazon Aurora DSQL.
 * </p>
 * <p>
 * Amazon Aurora DSQL is a serverless, distributed SQL database suitable for workloads of any size. Aurora DSQL is
 * available in both single-Region and multi-Region configurations, so your clusters and databases are always available
 * even if an Availability Zone or an Amazon Web Services Region are unavailable. Aurora DSQL lets you focus on using
 * your data to acquire new insights for your business and customers.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface DsqlAsyncClient extends AwsClient {
    String SERVICE_NAME = "dsql";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "dsql";

    /**
     * Creates an instance of {@link DsqlUtilities} object with the configuration set on this client.
     */
    default DsqlUtilities utilities() {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a cluster in Amazon Aurora DSQL.
     * </p>
     *
     * @param createClusterRequest
     * @return A Java Future containing the result of the CreateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The service limit was exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ConflictException The submitted action has conflicts.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateClusterResponse> createCluster(CreateClusterRequest createClusterRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a cluster in Amazon Aurora DSQL.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateClusterRequest.Builder} avoiding the need to
     * create one manually via {@link CreateClusterRequest#builder()}
     * </p>
     *
     * @param createClusterRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.CreateClusterRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The service limit was exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ConflictException The submitted action has conflicts.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateClusterResponse> createCluster(Consumer<CreateClusterRequest.Builder> createClusterRequest) {
        return createCluster(CreateClusterRequest.builder().applyMutation(createClusterRequest).build());
    }

    /**
     * <p>
     * Creates multi-Region clusters in Amazon Aurora DSQL. Multi-Region clusters require a linked Region list, which is
     * an array of the Regions in which you want to create linked clusters. Multi-Region clusters require a witness
     * Region, which participates in quorum in failure scenarios.
     * </p>
     *
     * @param createMultiRegionClustersRequest
     * @return A Java Future containing the result of the CreateMultiRegionClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The service limit was exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ConflictException The submitted action has conflicts.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.CreateMultiRegionClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/CreateMultiRegionClusters"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateMultiRegionClustersResponse> createMultiRegionClusters(
            CreateMultiRegionClustersRequest createMultiRegionClustersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates multi-Region clusters in Amazon Aurora DSQL. Multi-Region clusters require a linked Region list, which is
     * an array of the Regions in which you want to create linked clusters. Multi-Region clusters require a witness
     * Region, which participates in quorum in failure scenarios.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateMultiRegionClustersRequest.Builder} avoiding
     * the need to create one manually via {@link CreateMultiRegionClustersRequest#builder()}
     * </p>
     *
     * @param createMultiRegionClustersRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.CreateMultiRegionClustersRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the CreateMultiRegionClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The service limit was exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ConflictException The submitted action has conflicts.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.CreateMultiRegionClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/CreateMultiRegionClusters"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateMultiRegionClustersResponse> createMultiRegionClusters(
            Consumer<CreateMultiRegionClustersRequest.Builder> createMultiRegionClustersRequest) {
        return createMultiRegionClusters(CreateMultiRegionClustersRequest.builder()
                .applyMutation(createMultiRegionClustersRequest).build());
    }

    /**
     * <p>
     * Deletes a cluster in Amazon Aurora DSQL.
     * </p>
     *
     * @param deleteClusterRequest
     * @return A Java Future containing the result of the DeleteCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ConflictException The submitted action has conflicts.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.DeleteCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteClusterResponse> deleteCluster(DeleteClusterRequest deleteClusterRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a cluster in Amazon Aurora DSQL.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteClusterRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteClusterRequest#builder()}
     * </p>
     *
     * @param deleteClusterRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.DeleteClusterRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ConflictException The submitted action has conflicts.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.DeleteCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteClusterResponse> deleteCluster(Consumer<DeleteClusterRequest.Builder> deleteClusterRequest) {
        return deleteCluster(DeleteClusterRequest.builder().applyMutation(deleteClusterRequest).build());
    }

    /**
     * <p>
     * Deletes a multi-Region cluster in Amazon Aurora DSQL.
     * </p>
     *
     * @param deleteMultiRegionClustersRequest
     * @return A Java Future containing the result of the DeleteMultiRegionClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ConflictException The submitted action has conflicts.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.DeleteMultiRegionClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/DeleteMultiRegionClusters"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteMultiRegionClustersResponse> deleteMultiRegionClusters(
            DeleteMultiRegionClustersRequest deleteMultiRegionClustersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a multi-Region cluster in Amazon Aurora DSQL.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteMultiRegionClustersRequest.Builder} avoiding
     * the need to create one manually via {@link DeleteMultiRegionClustersRequest#builder()}
     * </p>
     *
     * @param deleteMultiRegionClustersRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.DeleteMultiRegionClustersRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DeleteMultiRegionClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ConflictException The submitted action has conflicts.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.DeleteMultiRegionClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/DeleteMultiRegionClusters"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteMultiRegionClustersResponse> deleteMultiRegionClusters(
            Consumer<DeleteMultiRegionClustersRequest.Builder> deleteMultiRegionClustersRequest) {
        return deleteMultiRegionClusters(DeleteMultiRegionClustersRequest.builder()
                .applyMutation(deleteMultiRegionClustersRequest).build());
    }

    /**
     * <p>
     * Retrieves information about a cluster.
     * </p>
     *
     * @param getClusterRequest
     * @return A Java Future containing the result of the GetCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.GetCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/GetCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetClusterResponse> getCluster(GetClusterRequest getClusterRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves information about a cluster.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetClusterRequest.Builder} avoiding the need to
     * create one manually via {@link GetClusterRequest#builder()}
     * </p>
     *
     * @param getClusterRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.GetClusterRequest.Builder} to create a request.
     * @return A Java Future containing the result of the GetCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.GetCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/GetCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetClusterResponse> getCluster(Consumer<GetClusterRequest.Builder> getClusterRequest) {
        return getCluster(GetClusterRequest.builder().applyMutation(getClusterRequest).build());
    }

    /**
     * <p>
     * Retrieves information about a list of clusters.
     * </p>
     *
     * @param listClustersRequest
     * @return A Java Future containing the result of the ListClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListClustersResponse> listClusters(ListClustersRequest listClustersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves information about a list of clusters.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListClustersRequest.Builder} avoiding the need to
     * create one manually via {@link ListClustersRequest#builder()}
     * </p>
     *
     * @param listClustersRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.ListClustersRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListClustersResponse> listClusters(Consumer<ListClustersRequest.Builder> listClustersRequest) {
        return listClusters(ListClustersRequest.builder().applyMutation(listClustersRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #listClusters(software.amazon.awssdk.services.dsql.model.ListClustersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dsql.paginators.ListClustersPublisher publisher = client.listClustersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dsql.paginators.ListClustersPublisher publisher = client.listClustersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.dsql.model.ListClustersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.dsql.model.ListClustersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listClusters(software.amazon.awssdk.services.dsql.model.ListClustersRequest)} operation.</b>
     * </p>
     *
     * @param listClustersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    default ListClustersPublisher listClustersPaginator(ListClustersRequest listClustersRequest) {
        return new ListClustersPublisher(this, listClustersRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #listClusters(software.amazon.awssdk.services.dsql.model.ListClustersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dsql.paginators.ListClustersPublisher publisher = client.listClustersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dsql.paginators.ListClustersPublisher publisher = client.listClustersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.dsql.model.ListClustersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.dsql.model.ListClustersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listClusters(software.amazon.awssdk.services.dsql.model.ListClustersRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListClustersRequest.Builder} avoiding the need to
     * create one manually via {@link ListClustersRequest#builder()}
     * </p>
     *
     * @param listClustersRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.ListClustersRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    default ListClustersPublisher listClustersPaginator(Consumer<ListClustersRequest.Builder> listClustersRequest) {
        return listClustersPaginator(ListClustersRequest.builder().applyMutation(listClustersRequest).build());
    }

    /**
     * <p>
     * Lists all of the tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all of the tags for a resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.ListTagsForResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Tags a resource with a map of key and value pairs.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The service limit was exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Tags a resource with a map of key and value pairs.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.TagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException The service limit was exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.UntagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest) {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates a cluster.
     * </p>
     *
     * @param updateClusterRequest
     * @return A Java Future containing the result of the UpdateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ConflictException The submitted action has conflicts.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.UpdateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/UpdateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateClusterResponse> updateCluster(UpdateClusterRequest updateClusterRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a cluster.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateClusterRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateClusterRequest#builder()}
     * </p>
     *
     * @param updateClusterRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.UpdateClusterRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UpdateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input failed to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ConflictException The submitted action has conflicts.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DsqlException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DsqlAsyncClient.UpdateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/UpdateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateClusterResponse> updateCluster(Consumer<UpdateClusterRequest.Builder> updateClusterRequest) {
        return updateCluster(UpdateClusterRequest.builder().applyMutation(updateClusterRequest).build());
    }

    /**
     * Create an instance of {@link DsqlAsyncWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link DsqlAsyncWaiter}
     */
    default DsqlAsyncWaiter waiter() {
        throw new UnsupportedOperationException();
    }

    @Override
    default DsqlServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link DsqlAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static DsqlAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link DsqlAsyncClient}.
     */
    static DsqlAsyncClientBuilder builder() {
        return new DefaultDsqlAsyncClientBuilder();
    }
}
