/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dsql;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.dsql.model.AccessDeniedException;
import software.amazon.awssdk.services.dsql.model.ConflictException;
import software.amazon.awssdk.services.dsql.model.CreateClusterRequest;
import software.amazon.awssdk.services.dsql.model.CreateClusterResponse;
import software.amazon.awssdk.services.dsql.model.DeleteClusterPolicyRequest;
import software.amazon.awssdk.services.dsql.model.DeleteClusterPolicyResponse;
import software.amazon.awssdk.services.dsql.model.DeleteClusterRequest;
import software.amazon.awssdk.services.dsql.model.DeleteClusterResponse;
import software.amazon.awssdk.services.dsql.model.DsqlException;
import software.amazon.awssdk.services.dsql.model.GetClusterPolicyRequest;
import software.amazon.awssdk.services.dsql.model.GetClusterPolicyResponse;
import software.amazon.awssdk.services.dsql.model.GetClusterRequest;
import software.amazon.awssdk.services.dsql.model.GetClusterResponse;
import software.amazon.awssdk.services.dsql.model.GetVpcEndpointServiceNameRequest;
import software.amazon.awssdk.services.dsql.model.GetVpcEndpointServiceNameResponse;
import software.amazon.awssdk.services.dsql.model.InternalServerException;
import software.amazon.awssdk.services.dsql.model.ListClustersRequest;
import software.amazon.awssdk.services.dsql.model.ListClustersResponse;
import software.amazon.awssdk.services.dsql.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.dsql.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.dsql.model.PutClusterPolicyRequest;
import software.amazon.awssdk.services.dsql.model.PutClusterPolicyResponse;
import software.amazon.awssdk.services.dsql.model.ResourceNotFoundException;
import software.amazon.awssdk.services.dsql.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.dsql.model.TagResourceRequest;
import software.amazon.awssdk.services.dsql.model.TagResourceResponse;
import software.amazon.awssdk.services.dsql.model.ThrottlingException;
import software.amazon.awssdk.services.dsql.model.UntagResourceRequest;
import software.amazon.awssdk.services.dsql.model.UntagResourceResponse;
import software.amazon.awssdk.services.dsql.model.UpdateClusterRequest;
import software.amazon.awssdk.services.dsql.model.UpdateClusterResponse;
import software.amazon.awssdk.services.dsql.model.ValidationException;
import software.amazon.awssdk.services.dsql.paginators.ListClustersIterable;
import software.amazon.awssdk.services.dsql.waiters.DsqlWaiter;

/**
 * Service client for accessing Amazon Aurora DSQL. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * This is an interface reference for Amazon Aurora DSQL. It contains documentation for one of the programming or
 * command line interfaces you can use to manage Amazon Aurora DSQL.
 * </p>
 * <p>
 * Amazon Aurora DSQL is a serverless, distributed SQL database suitable for workloads of any size. is available in both
 * single-Region and multi-Region configurations, so your clusters and databases are always available even if an
 * Availability Zone or an Amazon Web Services Region are unavailable. lets you focus on using your data to acquire new
 * insights for your business and customers.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface DsqlClient extends AwsClient {
    String SERVICE_NAME = "dsql";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "dsql";

    /**
     * <p>
     * The CreateCluster API allows you to create both single-Region clusters and multi-Region clusters. With the
     * addition of the <i>multiRegionProperties</i> parameter, you can create a cluster with witness Region support and
     * establish peer relationships with clusters in other Regions during creation.
     * </p>
     * <note>
     * <p>
     * Creating multi-Region clusters requires additional IAM permissions beyond those needed for single-Region
     * clusters, as detailed in the <b>Required permissions</b> section below.
     * </p>
     * </note>
     * <p>
     * <b>Required permissions</b>
     * </p>
     * <dl>
     * <dt>dsql:CreateCluster</dt>
     * <dd>
     * <p>
     * Required to create a cluster.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:region:account-id:cluster/*</code>
     * </p>
     * </dd>
     * <dt>dsql:TagResource</dt>
     * <dd>
     * <p>
     * Permission to add tags to a resource.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:region:account-id:cluster/*</code>
     * </p>
     * </dd>
     * <dt>dsql:PutMultiRegionProperties</dt>
     * <dd>
     * <p>
     * Permission to configure multi-Region properties for a cluster.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:region:account-id:cluster/*</code>
     * </p>
     * </dd>
     * <dt>dsql:AddPeerCluster</dt>
     * <dd>
     * <p>
     * When specifying <code>multiRegionProperties.clusters</code>, permission to add peer clusters.
     * </p>
     * <p>
     * Resources:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Local cluster: <code>arn:aws:dsql:region:account-id:cluster/*</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Each peer cluster: exact ARN of each specified peer cluster
     * </p>
     * </li>
     * </ul>
     * </dd>
     * <dt>dsql:PutWitnessRegion</dt>
     * <dd>
     * <p>
     * When specifying <code>multiRegionProperties.witnessRegion</code>, permission to set a witness Region. This
     * permission is checked both in the cluster Region and in the witness Region.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:region:account-id:cluster/*</code>
     * </p>
     * <p>
     * Condition Keys: <code>dsql:WitnessRegion</code> (matching the specified witness region)
     * </p>
     * </dd>
     * </dl>
     * <important>
     * <ul>
     * <li>
     * <p>
     * The witness Region specified in <code>multiRegionProperties.witnessRegion</code> cannot be the same as the
     * cluster's Region.
     * </p>
     * </li>
     * </ul>
     * </important>
     *
     * @param createClusterRequest
     * @return Result of the CreateCluster operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The service limit was exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateClusterResponse createCluster(CreateClusterRequest createClusterRequest) throws ServiceQuotaExceededException,
            ThrottlingException, ValidationException, AccessDeniedException, InternalServerException, ConflictException,
            AwsServiceException, SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * The CreateCluster API allows you to create both single-Region clusters and multi-Region clusters. With the
     * addition of the <i>multiRegionProperties</i> parameter, you can create a cluster with witness Region support and
     * establish peer relationships with clusters in other Regions during creation.
     * </p>
     * <note>
     * <p>
     * Creating multi-Region clusters requires additional IAM permissions beyond those needed for single-Region
     * clusters, as detailed in the <b>Required permissions</b> section below.
     * </p>
     * </note>
     * <p>
     * <b>Required permissions</b>
     * </p>
     * <dl>
     * <dt>dsql:CreateCluster</dt>
     * <dd>
     * <p>
     * Required to create a cluster.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:region:account-id:cluster/*</code>
     * </p>
     * </dd>
     * <dt>dsql:TagResource</dt>
     * <dd>
     * <p>
     * Permission to add tags to a resource.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:region:account-id:cluster/*</code>
     * </p>
     * </dd>
     * <dt>dsql:PutMultiRegionProperties</dt>
     * <dd>
     * <p>
     * Permission to configure multi-Region properties for a cluster.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:region:account-id:cluster/*</code>
     * </p>
     * </dd>
     * <dt>dsql:AddPeerCluster</dt>
     * <dd>
     * <p>
     * When specifying <code>multiRegionProperties.clusters</code>, permission to add peer clusters.
     * </p>
     * <p>
     * Resources:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Local cluster: <code>arn:aws:dsql:region:account-id:cluster/*</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Each peer cluster: exact ARN of each specified peer cluster
     * </p>
     * </li>
     * </ul>
     * </dd>
     * <dt>dsql:PutWitnessRegion</dt>
     * <dd>
     * <p>
     * When specifying <code>multiRegionProperties.witnessRegion</code>, permission to set a witness Region. This
     * permission is checked both in the cluster Region and in the witness Region.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:region:account-id:cluster/*</code>
     * </p>
     * <p>
     * Condition Keys: <code>dsql:WitnessRegion</code> (matching the specified witness region)
     * </p>
     * </dd>
     * </dl>
     * <important>
     * <ul>
     * <li>
     * <p>
     * The witness Region specified in <code>multiRegionProperties.witnessRegion</code> cannot be the same as the
     * cluster's Region.
     * </p>
     * </li>
     * </ul>
     * </important><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateClusterRequest.Builder} avoiding the need to
     * create one manually via {@link CreateClusterRequest#builder()}
     * </p>
     *
     * @param createClusterRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.CreateClusterRequest.Builder} to create a request.
     * @return Result of the CreateCluster operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The service limit was exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateClusterResponse createCluster(Consumer<CreateClusterRequest.Builder> createClusterRequest)
            throws ServiceQuotaExceededException, ThrottlingException, ValidationException, AccessDeniedException,
            InternalServerException, ConflictException, AwsServiceException, SdkClientException, DsqlException {
        return createCluster(CreateClusterRequest.builder().applyMutation(createClusterRequest).build());
    }

    /**
     * <p>
     * Deletes a cluster in Amazon Aurora DSQL.
     * </p>
     *
     * @param deleteClusterRequest
     * @return Result of the DeleteCluster operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.DeleteCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteClusterResponse deleteCluster(DeleteClusterRequest deleteClusterRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, ConflictException,
            AwsServiceException, SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a cluster in Amazon Aurora DSQL.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteClusterRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteClusterRequest#builder()}
     * </p>
     *
     * @param deleteClusterRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.DeleteClusterRequest.Builder} to create a request.
     * @return Result of the DeleteCluster operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.DeleteCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteClusterResponse deleteCluster(Consumer<DeleteClusterRequest.Builder> deleteClusterRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, ConflictException, AwsServiceException, SdkClientException, DsqlException {
        return deleteCluster(DeleteClusterRequest.builder().applyMutation(deleteClusterRequest).build());
    }

    /**
     * <p>
     * Deletes the resource-based policy attached to a cluster. This removes all access permissions defined by the
     * policy, reverting to default access controls.
     * </p>
     *
     * @param deleteClusterPolicyRequest
     * @return Result of the DeleteClusterPolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.DeleteClusterPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/DeleteClusterPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteClusterPolicyResponse deleteClusterPolicy(DeleteClusterPolicyRequest deleteClusterPolicyRequest)
            throws ThrottlingException, ValidationException, AccessDeniedException, InternalServerException,
            ResourceNotFoundException, ConflictException, AwsServiceException, SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the resource-based policy attached to a cluster. This removes all access permissions defined by the
     * policy, reverting to default access controls.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteClusterPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteClusterPolicyRequest#builder()}
     * </p>
     *
     * @param deleteClusterPolicyRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.DeleteClusterPolicyRequest.Builder} to create a request.
     * @return Result of the DeleteClusterPolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.DeleteClusterPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/DeleteClusterPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteClusterPolicyResponse deleteClusterPolicy(
            Consumer<DeleteClusterPolicyRequest.Builder> deleteClusterPolicyRequest) throws ThrottlingException,
            ValidationException, AccessDeniedException, InternalServerException, ResourceNotFoundException, ConflictException,
            AwsServiceException, SdkClientException, DsqlException {
        return deleteClusterPolicy(DeleteClusterPolicyRequest.builder().applyMutation(deleteClusterPolicyRequest).build());
    }

    /**
     * <p>
     * Retrieves information about a cluster.
     * </p>
     *
     * @param getClusterRequest
     * @return Result of the GetCluster operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.GetCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/GetCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default GetClusterResponse getCluster(GetClusterRequest getClusterRequest) throws ThrottlingException, AccessDeniedException,
            ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves information about a cluster.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetClusterRequest.Builder} avoiding the need to
     * create one manually via {@link GetClusterRequest#builder()}
     * </p>
     *
     * @param getClusterRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.GetClusterRequest.Builder} to create a request.
     * @return Result of the GetCluster operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.GetCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/GetCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default GetClusterResponse getCluster(Consumer<GetClusterRequest.Builder> getClusterRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, DsqlException {
        return getCluster(GetClusterRequest.builder().applyMutation(getClusterRequest).build());
    }

    /**
     * <p>
     * Retrieves the resource-based policy document attached to a cluster. This policy defines the access permissions
     * and conditions for the cluster.
     * </p>
     *
     * @param getClusterPolicyRequest
     * @return Result of the GetClusterPolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.GetClusterPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/GetClusterPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    default GetClusterPolicyResponse getClusterPolicy(GetClusterPolicyRequest getClusterPolicyRequest)
            throws ThrottlingException, ValidationException, AccessDeniedException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the resource-based policy document attached to a cluster. This policy defines the access permissions
     * and conditions for the cluster.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetClusterPolicyRequest.Builder} avoiding the need
     * to create one manually via {@link GetClusterPolicyRequest#builder()}
     * </p>
     *
     * @param getClusterPolicyRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.GetClusterPolicyRequest.Builder} to create a request.
     * @return Result of the GetClusterPolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.GetClusterPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/GetClusterPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    default GetClusterPolicyResponse getClusterPolicy(Consumer<GetClusterPolicyRequest.Builder> getClusterPolicyRequest)
            throws ThrottlingException, ValidationException, AccessDeniedException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, DsqlException {
        return getClusterPolicy(GetClusterPolicyRequest.builder().applyMutation(getClusterPolicyRequest).build());
    }

    /**
     * <p>
     * Retrieves the VPC endpoint service name.
     * </p>
     *
     * @param getVpcEndpointServiceNameRequest
     * @return Result of the GetVpcEndpointServiceName operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.GetVpcEndpointServiceName
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/GetVpcEndpointServiceName"
     *      target="_top">AWS API Documentation</a>
     */
    default GetVpcEndpointServiceNameResponse getVpcEndpointServiceName(
            GetVpcEndpointServiceNameRequest getVpcEndpointServiceNameRequest) throws ThrottlingException, ValidationException,
            AccessDeniedException, InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the VPC endpoint service name.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetVpcEndpointServiceNameRequest.Builder} avoiding
     * the need to create one manually via {@link GetVpcEndpointServiceNameRequest#builder()}
     * </p>
     *
     * @param getVpcEndpointServiceNameRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.GetVpcEndpointServiceNameRequest.Builder} to create a
     *        request.
     * @return Result of the GetVpcEndpointServiceName operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.GetVpcEndpointServiceName
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/GetVpcEndpointServiceName"
     *      target="_top">AWS API Documentation</a>
     */
    default GetVpcEndpointServiceNameResponse getVpcEndpointServiceName(
            Consumer<GetVpcEndpointServiceNameRequest.Builder> getVpcEndpointServiceNameRequest) throws ThrottlingException,
            ValidationException, AccessDeniedException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, DsqlException {
        return getVpcEndpointServiceName(GetVpcEndpointServiceNameRequest.builder()
                .applyMutation(getVpcEndpointServiceNameRequest).build());
    }

    /**
     * <p>
     * Retrieves information about a list of clusters.
     * </p>
     *
     * @param listClustersRequest
     * @return Result of the ListClusters operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    default ListClustersResponse listClusters(ListClustersRequest listClustersRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves information about a list of clusters.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListClustersRequest.Builder} avoiding the need to
     * create one manually via {@link ListClustersRequest#builder()}
     * </p>
     *
     * @param listClustersRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.ListClustersRequest.Builder} to create a request.
     * @return Result of the ListClusters operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    default ListClustersResponse listClusters(Consumer<ListClustersRequest.Builder> listClustersRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, DsqlException {
        return listClusters(ListClustersRequest.builder().applyMutation(listClustersRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #listClusters(software.amazon.awssdk.services.dsql.model.ListClustersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dsql.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.dsql.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     *     for (software.amazon.awssdk.services.dsql.model.ListClustersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dsql.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listClusters(software.amazon.awssdk.services.dsql.model.ListClustersRequest)} operation.</b>
     * </p>
     *
     * @param listClustersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    default ListClustersIterable listClustersPaginator(ListClustersRequest listClustersRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, DsqlException {
        return new ListClustersIterable(this, listClustersRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #listClusters(software.amazon.awssdk.services.dsql.model.ListClustersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dsql.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.dsql.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     *     for (software.amazon.awssdk.services.dsql.model.ListClustersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dsql.paginators.ListClustersIterable responses = client.listClustersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listClusters(software.amazon.awssdk.services.dsql.model.ListClustersRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListClustersRequest.Builder} avoiding the need to
     * create one manually via {@link ListClustersRequest#builder()}
     * </p>
     *
     * @param listClustersRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.ListClustersRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.ListClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListClusters" target="_top">AWS API
     *      Documentation</a>
     */
    default ListClustersIterable listClustersPaginator(Consumer<ListClustersRequest.Builder> listClustersRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, DsqlException {
        return listClustersPaginator(ListClustersRequest.builder().applyMutation(listClustersRequest).build());
    }

    /**
     * <p>
     * Lists all of the tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all of the tags for a resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.ListTagsForResourceRequest.Builder} to create a request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, DsqlException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Attaches a resource-based policy to a cluster. This policy defines access permissions and conditions for the
     * cluster, allowing you to control which principals can perform actions on the cluster.
     * </p>
     *
     * @param putClusterPolicyRequest
     * @return Result of the PutClusterPolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.PutClusterPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/PutClusterPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    default PutClusterPolicyResponse putClusterPolicy(PutClusterPolicyRequest putClusterPolicyRequest)
            throws ThrottlingException, ValidationException, AccessDeniedException, InternalServerException,
            ResourceNotFoundException, ConflictException, AwsServiceException, SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Attaches a resource-based policy to a cluster. This policy defines access permissions and conditions for the
     * cluster, allowing you to control which principals can perform actions on the cluster.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutClusterPolicyRequest.Builder} avoiding the need
     * to create one manually via {@link PutClusterPolicyRequest#builder()}
     * </p>
     *
     * @param putClusterPolicyRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.PutClusterPolicyRequest.Builder} to create a request.
     * @return Result of the PutClusterPolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.PutClusterPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/PutClusterPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    default PutClusterPolicyResponse putClusterPolicy(Consumer<PutClusterPolicyRequest.Builder> putClusterPolicyRequest)
            throws ThrottlingException, ValidationException, AccessDeniedException, InternalServerException,
            ResourceNotFoundException, ConflictException, AwsServiceException, SdkClientException, DsqlException {
        return putClusterPolicy(PutClusterPolicyRequest.builder().applyMutation(putClusterPolicyRequest).build());
    }

    /**
     * <p>
     * Tags a resource with a map of key and value pairs.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The service limit was exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ServiceQuotaExceededException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Tags a resource with a map of key and value pairs.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.TagResourceRequest.Builder} to create a request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         The service limit was exceeded.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest)
            throws ServiceQuotaExceededException, ThrottlingException, AccessDeniedException, ValidationException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, DsqlException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.UntagResourceRequest.Builder} to create a request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, DsqlException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * The <i>UpdateCluster</i> API allows you to modify both single-Region and multi-Region cluster configurations.
     * With the <i>multiRegionProperties</i> parameter, you can add or modify witness Region support and manage peer
     * relationships with clusters in other Regions.
     * </p>
     * <note>
     * <p>
     * Note that updating multi-Region clusters requires additional IAM permissions beyond those needed for standard
     * cluster updates, as detailed in the Permissions section.
     * </p>
     * </note>
     * <p>
     * <b>Required permissions</b>
     * </p>
     * <dl>
     * <dt>dsql:UpdateCluster</dt>
     * <dd>
     * <p>
     * Permission to update a DSQL cluster.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:<i>region</i>:<i>account-id</i>:cluster/<i>cluster-id</i> </code>
     * </p>
     * </dd>
     * </dl>
     * <dl>
     * <dt>dsql:PutMultiRegionProperties</dt>
     * <dd>
     * <p>
     * Permission to configure multi-Region properties for a cluster.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:<i>region</i>:<i>account-id</i>:cluster/<i>cluster-id</i> </code>
     * </p>
     * </dd>
     * </dl>
     * <dl>
     * <dt>dsql:GetCluster</dt>
     * <dd>
     * <p>
     * Permission to retrieve cluster information.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:<i>region</i>:<i>account-id</i>:cluster/<i>cluster-id</i> </code>
     * </p>
     * </dd>
     * <dt>dsql:AddPeerCluster</dt>
     * <dd>
     * <p>
     * Permission to add peer clusters.
     * </p>
     * <p>
     * Resources:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Local cluster: <code>arn:aws:dsql:<i>region</i>:<i>account-id</i>:cluster/<i>cluster-id</i> </code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Each peer cluster: exact ARN of each specified peer cluster
     * </p>
     * </li>
     * </ul>
     * </dd>
     * <dt>dsql:RemovePeerCluster</dt>
     * <dd>
     * <p>
     * Permission to remove peer clusters. The <i>dsql:RemovePeerCluster</i> permission uses a wildcard ARN pattern to
     * simplify permission management during updates.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:*:<i>account-id</i>:cluster/*</code>
     * </p>
     * </dd>
     * </dl>
     * <dl>
     * <dt>dsql:PutWitnessRegion</dt>
     * <dd>
     * <p>
     * Permission to set a witness Region.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:<i>region</i>:<i>account-id</i>:cluster/<i>cluster-id</i> </code>
     * </p>
     * <p>
     * Condition Keys: dsql:WitnessRegion (matching the specified witness Region)
     * </p>
     * <p>
     * <b>This permission is checked both in the cluster Region and in the witness Region.</b>
     * </p>
     * </dd>
     * </dl>
     * <important>
     * <ul>
     * <li>
     * <p>
     * The witness region specified in <code>multiRegionProperties.witnessRegion</code> cannot be the same as the
     * cluster's Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * When updating clusters with peer relationships, permissions are checked for both adding and removing peers.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>dsql:RemovePeerCluster</code> permission uses a wildcard ARN pattern to simplify permission management
     * during updates.
     * </p>
     * </li>
     * </ul>
     * </important>
     *
     * @param updateClusterRequest
     * @return Result of the UpdateCluster operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.UpdateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/UpdateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateClusterResponse updateCluster(UpdateClusterRequest updateClusterRequest) throws ThrottlingException,
            ValidationException, AccessDeniedException, InternalServerException, ResourceNotFoundException, ConflictException,
            AwsServiceException, SdkClientException, DsqlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * The <i>UpdateCluster</i> API allows you to modify both single-Region and multi-Region cluster configurations.
     * With the <i>multiRegionProperties</i> parameter, you can add or modify witness Region support and manage peer
     * relationships with clusters in other Regions.
     * </p>
     * <note>
     * <p>
     * Note that updating multi-Region clusters requires additional IAM permissions beyond those needed for standard
     * cluster updates, as detailed in the Permissions section.
     * </p>
     * </note>
     * <p>
     * <b>Required permissions</b>
     * </p>
     * <dl>
     * <dt>dsql:UpdateCluster</dt>
     * <dd>
     * <p>
     * Permission to update a DSQL cluster.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:<i>region</i>:<i>account-id</i>:cluster/<i>cluster-id</i> </code>
     * </p>
     * </dd>
     * </dl>
     * <dl>
     * <dt>dsql:PutMultiRegionProperties</dt>
     * <dd>
     * <p>
     * Permission to configure multi-Region properties for a cluster.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:<i>region</i>:<i>account-id</i>:cluster/<i>cluster-id</i> </code>
     * </p>
     * </dd>
     * </dl>
     * <dl>
     * <dt>dsql:GetCluster</dt>
     * <dd>
     * <p>
     * Permission to retrieve cluster information.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:<i>region</i>:<i>account-id</i>:cluster/<i>cluster-id</i> </code>
     * </p>
     * </dd>
     * <dt>dsql:AddPeerCluster</dt>
     * <dd>
     * <p>
     * Permission to add peer clusters.
     * </p>
     * <p>
     * Resources:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Local cluster: <code>arn:aws:dsql:<i>region</i>:<i>account-id</i>:cluster/<i>cluster-id</i> </code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Each peer cluster: exact ARN of each specified peer cluster
     * </p>
     * </li>
     * </ul>
     * </dd>
     * <dt>dsql:RemovePeerCluster</dt>
     * <dd>
     * <p>
     * Permission to remove peer clusters. The <i>dsql:RemovePeerCluster</i> permission uses a wildcard ARN pattern to
     * simplify permission management during updates.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:*:<i>account-id</i>:cluster/*</code>
     * </p>
     * </dd>
     * </dl>
     * <dl>
     * <dt>dsql:PutWitnessRegion</dt>
     * <dd>
     * <p>
     * Permission to set a witness Region.
     * </p>
     * <p>
     * Resources: <code>arn:aws:dsql:<i>region</i>:<i>account-id</i>:cluster/<i>cluster-id</i> </code>
     * </p>
     * <p>
     * Condition Keys: dsql:WitnessRegion (matching the specified witness Region)
     * </p>
     * <p>
     * <b>This permission is checked both in the cluster Region and in the witness Region.</b>
     * </p>
     * </dd>
     * </dl>
     * <important>
     * <ul>
     * <li>
     * <p>
     * The witness region specified in <code>multiRegionProperties.witnessRegion</code> cannot be the same as the
     * cluster's Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * When updating clusters with peer relationships, permissions are checked for both adding and removing peers.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>dsql:RemovePeerCluster</code> permission uses a wildcard ARN pattern to simplify permission management
     * during updates.
     * </p>
     * </li>
     * </ul>
     * </important><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateClusterRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateClusterRequest#builder()}
     * </p>
     *
     * @param updateClusterRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.dsql.model.UpdateClusterRequest.Builder} to create a request.
     * @return Result of the UpdateCluster operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input failed to satisfy the constraints specified by an Amazon Web Services service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ConflictException
     *         The submitted action has conflicts.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DsqlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DsqlClient.UpdateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dsql-2018-05-10/UpdateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateClusterResponse updateCluster(Consumer<UpdateClusterRequest.Builder> updateClusterRequest)
            throws ThrottlingException, ValidationException, AccessDeniedException, InternalServerException,
            ResourceNotFoundException, ConflictException, AwsServiceException, SdkClientException, DsqlException {
        return updateCluster(UpdateClusterRequest.builder().applyMutation(updateClusterRequest).build());
    }

    /**
     * Creates an instance of {@link DsqlUtilities} object with the configuration set on this client.
     */
    default DsqlUtilities utilities() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create an instance of {@link DsqlWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link DsqlWaiter}
     */
    default DsqlWaiter waiter() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link DsqlClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static DsqlClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link DsqlClient}.
     */
    static DsqlClientBuilder builder() {
        return new DefaultDsqlClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default DsqlServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
