/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dsql.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines the structure for multi-Region cluster configurations, containing the witness region and linked cluster
 * settings.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MultiRegionProperties implements SdkPojo, Serializable,
        ToCopyableBuilder<MultiRegionProperties.Builder, MultiRegionProperties> {
    private static final SdkField<String> WITNESS_REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("witnessRegion").getter(getter(MultiRegionProperties::witnessRegion))
            .setter(setter(Builder::witnessRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("witnessRegion").build()).build();

    private static final SdkField<List<String>> CLUSTERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("clusters")
            .getter(getter(MultiRegionProperties::clusters))
            .setter(setter(Builder::clusters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clusters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(WITNESS_REGION_FIELD,
            CLUSTERS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String witnessRegion;

    private final List<String> clusters;

    private MultiRegionProperties(BuilderImpl builder) {
        this.witnessRegion = builder.witnessRegion;
        this.clusters = builder.clusters;
    }

    /**
     * <p>
     * The Region that serves as the witness region for a multi-Region cluster. The witness Region helps maintain
     * cluster consistency and quorum.
     * </p>
     * 
     * @return The Region that serves as the witness region for a multi-Region cluster. The witness Region helps
     *         maintain cluster consistency and quorum.
     */
    public final String witnessRegion() {
        return witnessRegion;
    }

    /**
     * For responses, this returns true if the service returned a value for the Clusters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasClusters() {
        return clusters != null && !(clusters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The set of peered clusters that form the multi-Region cluster configuration. Each peered cluster represents a
     * database instance in a different Region.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasClusters} method.
     * </p>
     * 
     * @return The set of peered clusters that form the multi-Region cluster configuration. Each peered cluster
     *         represents a database instance in a different Region.
     */
    public final List<String> clusters() {
        return clusters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(witnessRegion());
        hashCode = 31 * hashCode + Objects.hashCode(hasClusters() ? clusters() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MultiRegionProperties)) {
            return false;
        }
        MultiRegionProperties other = (MultiRegionProperties) obj;
        return Objects.equals(witnessRegion(), other.witnessRegion()) && hasClusters() == other.hasClusters()
                && Objects.equals(clusters(), other.clusters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MultiRegionProperties").add("WitnessRegion", witnessRegion())
                .add("Clusters", hasClusters() ? clusters() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "witnessRegion":
            return Optional.ofNullable(clazz.cast(witnessRegion()));
        case "clusters":
            return Optional.ofNullable(clazz.cast(clusters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("witnessRegion", WITNESS_REGION_FIELD);
        map.put("clusters", CLUSTERS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MultiRegionProperties, T> g) {
        return obj -> g.apply((MultiRegionProperties) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MultiRegionProperties> {
        /**
         * <p>
         * The Region that serves as the witness region for a multi-Region cluster. The witness Region helps maintain
         * cluster consistency and quorum.
         * </p>
         * 
         * @param witnessRegion
         *        The Region that serves as the witness region for a multi-Region cluster. The witness Region helps
         *        maintain cluster consistency and quorum.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder witnessRegion(String witnessRegion);

        /**
         * <p>
         * The set of peered clusters that form the multi-Region cluster configuration. Each peered cluster represents a
         * database instance in a different Region.
         * </p>
         * 
         * @param clusters
         *        The set of peered clusters that form the multi-Region cluster configuration. Each peered cluster
         *        represents a database instance in a different Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusters(Collection<String> clusters);

        /**
         * <p>
         * The set of peered clusters that form the multi-Region cluster configuration. Each peered cluster represents a
         * database instance in a different Region.
         * </p>
         * 
         * @param clusters
         *        The set of peered clusters that form the multi-Region cluster configuration. Each peered cluster
         *        represents a database instance in a different Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusters(String... clusters);
    }

    static final class BuilderImpl implements Builder {
        private String witnessRegion;

        private List<String> clusters = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MultiRegionProperties model) {
            witnessRegion(model.witnessRegion);
            clusters(model.clusters);
        }

        public final String getWitnessRegion() {
            return witnessRegion;
        }

        public final void setWitnessRegion(String witnessRegion) {
            this.witnessRegion = witnessRegion;
        }

        @Override
        public final Builder witnessRegion(String witnessRegion) {
            this.witnessRegion = witnessRegion;
            return this;
        }

        public final Collection<String> getClusters() {
            if (clusters instanceof SdkAutoConstructList) {
                return null;
            }
            return clusters;
        }

        public final void setClusters(Collection<String> clusters) {
            this.clusters = ClusterArnListCopier.copy(clusters);
        }

        @Override
        public final Builder clusters(Collection<String> clusters) {
            this.clusters = ClusterArnListCopier.copy(clusters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder clusters(String... clusters) {
            clusters(Arrays.asList(clusters));
            return this;
        }

        @Override
        public MultiRegionProperties build() {
            return new MultiRegionProperties(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
