/*
 * Copyright 2010-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package software.amazon.awssdk.extensions.dynamodb.mappingclient.operations;

import static software.amazon.awssdk.extensions.dynamodb.mappingclient.core.Utils.readAndTransformPaginatedItems;

import java.util.Map;
import java.util.function.Function;

import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.extensions.dynamodb.mappingclient.Expression;
import software.amazon.awssdk.extensions.dynamodb.mappingclient.IndexOperation;
import software.amazon.awssdk.extensions.dynamodb.mappingclient.MapperExtension;
import software.amazon.awssdk.extensions.dynamodb.mappingclient.OperationContext;
import software.amazon.awssdk.extensions.dynamodb.mappingclient.Page;
import software.amazon.awssdk.extensions.dynamodb.mappingclient.TableMetadata;
import software.amazon.awssdk.extensions.dynamodb.mappingclient.TableOperation;
import software.amazon.awssdk.extensions.dynamodb.mappingclient.TableSchema;
import software.amazon.awssdk.extensions.dynamodb.mappingclient.core.TransformIterable;
import software.amazon.awssdk.services.dynamodb.DynamoDbClient;
import software.amazon.awssdk.services.dynamodb.model.AttributeValue;
import software.amazon.awssdk.services.dynamodb.model.ScanRequest;
import software.amazon.awssdk.services.dynamodb.model.ScanResponse;
import software.amazon.awssdk.services.dynamodb.paginators.ScanIterable;

@SdkPublicApi
public class Scan<T> implements TableOperation<T, ScanRequest, ScanIterable, Iterable<Page<T>>>,
                                IndexOperation<T, ScanRequest, ScanIterable, Iterable<Page<T>>> {
    private final Map<String, AttributeValue> exclusiveStartKey;
    private final Integer limit;
    private final Boolean consistentRead;
    private final Expression filterExpression;

    private Scan(Map<String, AttributeValue> exclusiveStartKey,
                 Integer limit, Boolean consistentRead,
                 Expression filterExpression) {
        this.exclusiveStartKey = exclusiveStartKey;
        this.limit = limit;
        this.consistentRead = consistentRead;
        this.filterExpression = filterExpression;
    }

    public static <T> Scan<T> create() {
        return new Scan<>(null, null, null, null);
    }

    public static Builder builder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder().exclusiveStartKey(exclusiveStartKey)
                            .limit(limit)
                            .consistentRead(consistentRead)
                            .filterExpression(filterExpression);
    }

    @Override
    public ScanRequest generateRequest(TableSchema<T> tableSchema,
                                       OperationContext operationContext,
                                       MapperExtension mapperExtension) {
        ScanRequest.Builder scanRequest = ScanRequest.builder()
            .tableName(operationContext.getTableName())
            .limit(limit)
            .exclusiveStartKey(exclusiveStartKey)
            .consistentRead(consistentRead);

        if (!TableMetadata.getPrimaryIndexName().equals(operationContext.getIndexName())) {
            scanRequest = scanRequest.indexName(operationContext.getIndexName());
        }

        if (filterExpression != null) {
            scanRequest = scanRequest.filterExpression(filterExpression.getExpression())
                                     .expressionAttributeValues(filterExpression.getExpressionValues())
                                     .expressionAttributeNames(filterExpression.getExpressionNames());
        }

        return scanRequest.build();
    }

    @Override
    public Iterable<Page<T>> transformResponse(ScanIterable response,
                                               TableSchema<T> tableSchema,
                                               OperationContext operationContext,
                                               MapperExtension mapperExtension) {
        return TransformIterable.of(response, getScanResponseIterator(tableSchema, operationContext, mapperExtension));
    }

    @Override
    public Function<ScanRequest, ScanIterable> getServiceCall(DynamoDbClient dynamoDbClient) {
        return dynamoDbClient::scanPaginator;
    }

    public Map<String, AttributeValue> getExclusiveStartKey() {
        return exclusiveStartKey;
    }

    public Integer getLimit() {
        return limit;
    }

    public Boolean getConsistentRead() {
        return consistentRead;
    }

    public Expression getFilterExpression() {
        return filterExpression;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        Scan<?> scan = (Scan<?>) o;

        if (exclusiveStartKey != null ? ! exclusiveStartKey.equals(scan.exclusiveStartKey) :
            scan.exclusiveStartKey != null) {
            return false;
        }
        if (limit != null ? ! limit.equals(scan.limit) : scan.limit != null) {
            return false;
        }
        if (consistentRead != null ? ! consistentRead.equals(scan.consistentRead) : scan.consistentRead != null) {
            return false;
        }
        return filterExpression != null ? filterExpression.equals(scan.filterExpression) : scan.filterExpression == null;
    }

    @Override
    public int hashCode() {
        int result = exclusiveStartKey != null ? exclusiveStartKey.hashCode() : 0;
        result = 31 * result + (limit != null ? limit.hashCode() : 0);
        result = 31 * result + (consistentRead != null ? consistentRead.hashCode() : 0);
        result = 31 * result + (filterExpression != null ? filterExpression.hashCode() : 0);
        return result;
    }

    private Function<ScanResponse, Page<T>> getScanResponseIterator(TableSchema<T> tableSchema,
                                                                    OperationContext operationContext,
                                                                    MapperExtension mapperExtension) {
        return readAndTransformPaginatedItems(tableSchema,
                                              operationContext,
                                              mapperExtension,
                                              ScanResponse::items,
                                              ScanResponse::lastEvaluatedKey);
    }

    public static final class Builder {
        private Map<String, AttributeValue> exclusiveStartKey;
        private Integer limit;
        private Boolean consistentRead;
        private Expression filterExpression;

        private Builder() {
        }

        public <T> Scan<T> build() {
            return new Scan<>(exclusiveStartKey, limit, consistentRead, filterExpression);
        }

        public Builder exclusiveStartKey(Map<String, AttributeValue> exclusiveStartKey) {
            this.exclusiveStartKey = exclusiveStartKey;
            return this;
        }

        public Builder limit(Integer limit) {
            this.limit = limit;
            return this;
        }

        public Builder consistentRead(Boolean consistentRead) {
            this.consistentRead = consistentRead;
            return this;
        }

        public Builder filterExpression(Expression filterExpression) {
            this.filterExpression = filterExpression;
            return this;
        }
    }
}
