/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.dynamodb.transform.TableDescriptionMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the properties of a table.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class TableDescription implements StructuredPojo, ToCopyableBuilder<TableDescription.Builder, TableDescription> {
    private final List<AttributeDefinition> attributeDefinitions;

    private final String tableName;

    private final List<KeySchemaElement> keySchema;

    private final String tableStatus;

    private final Instant creationDateTime;

    private final ProvisionedThroughputDescription provisionedThroughput;

    private final Long tableSizeBytes;

    private final Long itemCount;

    private final String tableArn;

    private final List<LocalSecondaryIndexDescription> localSecondaryIndexes;

    private final List<GlobalSecondaryIndexDescription> globalSecondaryIndexes;

    private final StreamSpecification streamSpecification;

    private final String latestStreamLabel;

    private final String latestStreamArn;

    private TableDescription(BuilderImpl builder) {
        this.attributeDefinitions = builder.attributeDefinitions;
        this.tableName = builder.tableName;
        this.keySchema = builder.keySchema;
        this.tableStatus = builder.tableStatus;
        this.creationDateTime = builder.creationDateTime;
        this.provisionedThroughput = builder.provisionedThroughput;
        this.tableSizeBytes = builder.tableSizeBytes;
        this.itemCount = builder.itemCount;
        this.tableArn = builder.tableArn;
        this.localSecondaryIndexes = builder.localSecondaryIndexes;
        this.globalSecondaryIndexes = builder.globalSecondaryIndexes;
        this.streamSpecification = builder.streamSpecification;
        this.latestStreamLabel = builder.latestStreamLabel;
        this.latestStreamArn = builder.latestStreamArn;
    }

    /**
     * <p>
     * An array of <code>AttributeDefinition</code> objects. Each of these objects describes one attribute in the table
     * and index key schema.
     * </p>
     * <p>
     * Each <code>AttributeDefinition</code> object in this array is composed of:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AttributeName</code> - The name of the attribute.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AttributeType</code> - The data type for the attribute.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of <code>AttributeDefinition</code> objects. Each of these objects describes one attribute in
     *         the table and index key schema.</p>
     *         <p>
     *         Each <code>AttributeDefinition</code> object in this array is composed of:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AttributeName</code> - The name of the attribute.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AttributeType</code> - The data type for the attribute.
     *         </p>
     *         </li>
     */
    public List<AttributeDefinition> attributeDefinitions() {
        return attributeDefinitions;
    }

    /**
     * <p>
     * The name of the table.
     * </p>
     * 
     * @return The name of the table.
     */
    public String tableName() {
        return tableName;
    }

    /**
     * <p>
     * The primary key structure for the table. Each <code>KeySchemaElement</code> consists of:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AttributeName</code> - The name of the attribute.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyType</code> - The role of the attribute:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>HASH</code> - partition key
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RANGE</code> - sort key
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives from
     * DynamoDB' usage of an internal hash function to evenly distribute data items across partitions, based on their
     * partition key values.
     * </p>
     * <p>
     * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from the
     * way DynamoDB stores items with the same partition key physically close together, in sorted order by the sort key
     * value.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * For more information about primary keys, see <a
     * href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataModel.html#DataModelPrimaryKey">Primary
     * Key</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The primary key structure for the table. Each <code>KeySchemaElement</code> consists of:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AttributeName</code> - The name of the attribute.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KeyType</code> - The role of the attribute:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>HASH</code> - partition key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RANGE</code> - sort key
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
     *         derives from DynamoDB' usage of an internal hash function to evenly distribute data items across
     *         partitions, based on their partition key values.
     *         </p>
     *         <p>
     *         The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives
     *         from the way DynamoDB stores items with the same partition key physically close together, in sorted order
     *         by the sort key value.
     *         </p>
     *         </note></li>
     *         </ul>
     *         <p>
     *         For more information about primary keys, see <a href=
     *         "http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataModel.html#DataModelPrimaryKey"
     *         >Primary Key</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     */
    public List<KeySchemaElement> keySchema() {
        return keySchema;
    }

    /**
     * <p>
     * The current state of the table:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The table is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The table is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The table is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The table is ready for use.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tableStatus} will
     * return {@link TableStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tableStatusAsString}.
     * </p>
     * 
     * @return The current state of the table:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The table is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The table is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The table is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The table is ready for use.
     *         </p>
     *         </li>
     * @see TableStatus
     */
    public TableStatus tableStatus() {
        return TableStatus.fromValue(tableStatus);
    }

    /**
     * <p>
     * The current state of the table:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The table is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The table is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The table is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The table is ready for use.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tableStatus} will
     * return {@link TableStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tableStatusAsString}.
     * </p>
     * 
     * @return The current state of the table:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The table is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The table is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The table is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The table is ready for use.
     *         </p>
     *         </li>
     * @see TableStatus
     */
    public String tableStatusAsString() {
        return tableStatus;
    }

    /**
     * <p>
     * The date and time when the table was created, in <a href="http://www.epochconverter.com/">UNIX epoch time</a>
     * format.
     * </p>
     * 
     * @return The date and time when the table was created, in <a href="http://www.epochconverter.com/">UNIX epoch
     *         time</a> format.
     */
    public Instant creationDateTime() {
        return creationDateTime;
    }

    /**
     * <p>
     * The provisioned throughput settings for the table, consisting of read and write capacity units, along with data
     * about increases and decreases.
     * </p>
     * 
     * @return The provisioned throughput settings for the table, consisting of read and write capacity units, along
     *         with data about increases and decreases.
     */
    public ProvisionedThroughputDescription provisionedThroughput() {
        return provisionedThroughput;
    }

    /**
     * <p>
     * The total size of the specified table, in bytes. DynamoDB updates this value approximately every six hours.
     * Recent changes might not be reflected in this value.
     * </p>
     * 
     * @return The total size of the specified table, in bytes. DynamoDB updates this value approximately every six
     *         hours. Recent changes might not be reflected in this value.
     */
    public Long tableSizeBytes() {
        return tableSizeBytes;
    }

    /**
     * <p>
     * The number of items in the specified table. DynamoDB updates this value approximately every six hours. Recent
     * changes might not be reflected in this value.
     * </p>
     * 
     * @return The number of items in the specified table. DynamoDB updates this value approximately every six hours.
     *         Recent changes might not be reflected in this value.
     */
    public Long itemCount() {
        return itemCount;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that uniquely identifies the table.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that uniquely identifies the table.
     */
    public String tableArn() {
        return tableArn;
    }

    /**
     * <p>
     * Represents one or more local secondary indexes on the table. Each index is scoped to a given partition key value.
     * Tables with one or more local secondary indexes are subject to an item collection size limit, where the amount of
     * data within a given item collection cannot exceed 10 GB. Each element is composed of:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>IndexName</code> - The name of the local secondary index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema must be
     * between 1 and 255 characters (inclusive). The key schema must begin with the same partition key as the table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index. These
     * are in addition to the primary key attributes and index key attributes, which are automatically projected. Each
     * attribute specification is composed of:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ProjectionType</code> - One of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of projected
     * attributes are in <code>NonKeyAttributes</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ALL</code> - All of the table attributes are projected into the index.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
     * secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across all of
     * the secondary indexes, must not exceed 20. If you project the same attribute into two different indexes, this
     * counts as two distinct attributes when determining the total.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>IndexSizeBytes</code> - Represents the total size of the index, in bytes. DynamoDB updates this value
     * approximately every six hours. Recent changes might not be reflected in this value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ItemCount</code> - Represents the number of items in the index. DynamoDB updates this value approximately
     * every six hours. Recent changes might not be reflected in this value.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Represents one or more local secondary indexes on the table. Each index is scoped to a given partition
     *         key value. Tables with one or more local secondary indexes are subject to an item collection size limit,
     *         where the amount of data within a given item collection cannot exceed 10 GB. Each element is composed
     *         of:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>IndexName</code> - The name of the local secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema
     *         must be between 1 and 255 characters (inclusive). The key schema must begin with the same partition key
     *         as the table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index.
     *         These are in addition to the primary key attributes and index key attributes, which are automatically
     *         projected. Each attribute specification is composed of:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ProjectionType</code> - One of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
     *         projected attributes are in <code>NonKeyAttributes</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ALL</code> - All of the table attributes are projected into the index.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
     *         secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across
     *         all of the secondary indexes, must not exceed 20. If you project the same attribute into two different
     *         indexes, this counts as two distinct attributes when determining the total.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IndexSizeBytes</code> - Represents the total size of the index, in bytes. DynamoDB updates this
     *         value approximately every six hours. Recent changes might not be reflected in this value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ItemCount</code> - Represents the number of items in the index. DynamoDB updates this value
     *         approximately every six hours. Recent changes might not be reflected in this value.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
     */
    public List<LocalSecondaryIndexDescription> localSecondaryIndexes() {
        return localSecondaryIndexes;
    }

    /**
     * <p>
     * The global secondary indexes, if any, on the table. Each index is scoped to a given partition key value. Each
     * element is composed of:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Backfilling</code> - If true, then the index is currently in the backfilling phase. Backfilling occurs only
     * when a new global secondary index is added to the table; it is the process by which DynamoDB populates the new
     * index with data from the table. (This attribute does not appear for indexes that were created during a
     * <code>CreateTable</code> operation.)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IndexName</code> - The name of the global secondary index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IndexSizeBytes</code> - The total size of the global secondary index, in bytes. DynamoDB updates this value
     * approximately every six hours. Recent changes might not be reflected in this value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IndexStatus</code> - The current status of the global secondary index:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The index is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The index is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The index is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The index is ready for use.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>ItemCount</code> - The number of items in the global secondary index. DynamoDB updates this value
     * approximately every six hours. Recent changes might not be reflected in this value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema must be
     * between 1 and 255 characters (inclusive). The key schema must begin with the same partition key as the table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index. These
     * are in addition to the primary key attributes and index key attributes, which are automatically projected. Each
     * attribute specification is composed of:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ProjectionType</code> - One of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of projected
     * attributes are in <code>NonKeyAttributes</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ALL</code> - All of the table attributes are projected into the index.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
     * secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across all of
     * the secondary indexes, must not exceed 20. If you project the same attribute into two different indexes, this
     * counts as two distinct attributes when determining the total.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>ProvisionedThroughput</code> - The provisioned throughput settings for the global secondary index,
     * consisting of read and write capacity units, along with data about increases and decreases.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The global secondary indexes, if any, on the table. Each index is scoped to a given partition key value.
     *         Each element is composed of:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Backfilling</code> - If true, then the index is currently in the backfilling phase. Backfilling
     *         occurs only when a new global secondary index is added to the table; it is the process by which DynamoDB
     *         populates the new index with data from the table. (This attribute does not appear for indexes that were
     *         created during a <code>CreateTable</code> operation.)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IndexName</code> - The name of the global secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IndexSizeBytes</code> - The total size of the global secondary index, in bytes. DynamoDB updates
     *         this value approximately every six hours. Recent changes might not be reflected in this value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IndexStatus</code> - The current status of the global secondary index:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The index is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The index is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The index is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The index is ready for use.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ItemCount</code> - The number of items in the global secondary index. DynamoDB updates this value
     *         approximately every six hours. Recent changes might not be reflected in this value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema
     *         must be between 1 and 255 characters (inclusive). The key schema must begin with the same partition key
     *         as the table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index.
     *         These are in addition to the primary key attributes and index key attributes, which are automatically
     *         projected. Each attribute specification is composed of:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ProjectionType</code> - One of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
     *         projected attributes are in <code>NonKeyAttributes</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ALL</code> - All of the table attributes are projected into the index.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
     *         secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across
     *         all of the secondary indexes, must not exceed 20. If you project the same attribute into two different
     *         indexes, this counts as two distinct attributes when determining the total.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ProvisionedThroughput</code> - The provisioned throughput settings for the global secondary index,
     *         consisting of read and write capacity units, along with data about increases and decreases.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
     */
    public List<GlobalSecondaryIndexDescription> globalSecondaryIndexes() {
        return globalSecondaryIndexes;
    }

    /**
     * <p>
     * The current DynamoDB Streams configuration for the table.
     * </p>
     * 
     * @return The current DynamoDB Streams configuration for the table.
     */
    public StreamSpecification streamSpecification() {
        return streamSpecification;
    }

    /**
     * <p>
     * A timestamp, in ISO 8601 format, for this stream.
     * </p>
     * <p>
     * Note that <code>LatestStreamLabel</code> is not a unique identifier for the stream, because it is possible that a
     * stream from another table might have the same timestamp. However, the combination of the following three elements
     * is guaranteed to be unique:
     * </p>
     * <ul>
     * <li>
     * <p>
     * the AWS customer ID.
     * </p>
     * </li>
     * <li>
     * <p>
     * the table name.
     * </p>
     * </li>
     * <li>
     * <p>
     * the <code>StreamLabel</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return A timestamp, in ISO 8601 format, for this stream.</p>
     *         <p>
     *         Note that <code>LatestStreamLabel</code> is not a unique identifier for the stream, because it is
     *         possible that a stream from another table might have the same timestamp. However, the combination of the
     *         following three elements is guaranteed to be unique:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         the AWS customer ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         the table name.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         the <code>StreamLabel</code>.
     *         </p>
     *         </li>
     */
    public String latestStreamLabel() {
        return latestStreamLabel;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that uniquely identifies the latest stream for this table.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that uniquely identifies the latest stream for this table.
     */
    public String latestStreamArn() {
        return latestStreamArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(attributeDefinitions());
        hashCode = 31 * hashCode + Objects.hashCode(tableName());
        hashCode = 31 * hashCode + Objects.hashCode(keySchema());
        hashCode = 31 * hashCode + Objects.hashCode(tableStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(provisionedThroughput());
        hashCode = 31 * hashCode + Objects.hashCode(tableSizeBytes());
        hashCode = 31 * hashCode + Objects.hashCode(itemCount());
        hashCode = 31 * hashCode + Objects.hashCode(tableArn());
        hashCode = 31 * hashCode + Objects.hashCode(localSecondaryIndexes());
        hashCode = 31 * hashCode + Objects.hashCode(globalSecondaryIndexes());
        hashCode = 31 * hashCode + Objects.hashCode(streamSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(latestStreamLabel());
        hashCode = 31 * hashCode + Objects.hashCode(latestStreamArn());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TableDescription)) {
            return false;
        }
        TableDescription other = (TableDescription) obj;
        return Objects.equals(attributeDefinitions(), other.attributeDefinitions())
                && Objects.equals(tableName(), other.tableName()) && Objects.equals(keySchema(), other.keySchema())
                && Objects.equals(tableStatusAsString(), other.tableStatusAsString())
                && Objects.equals(creationDateTime(), other.creationDateTime())
                && Objects.equals(provisionedThroughput(), other.provisionedThroughput())
                && Objects.equals(tableSizeBytes(), other.tableSizeBytes()) && Objects.equals(itemCount(), other.itemCount())
                && Objects.equals(tableArn(), other.tableArn())
                && Objects.equals(localSecondaryIndexes(), other.localSecondaryIndexes())
                && Objects.equals(globalSecondaryIndexes(), other.globalSecondaryIndexes())
                && Objects.equals(streamSpecification(), other.streamSpecification())
                && Objects.equals(latestStreamLabel(), other.latestStreamLabel())
                && Objects.equals(latestStreamArn(), other.latestStreamArn());
    }

    @Override
    public String toString() {
        return ToString.builder("TableDescription").add("AttributeDefinitions", attributeDefinitions())
                .add("TableName", tableName()).add("KeySchema", keySchema()).add("TableStatus", tableStatusAsString())
                .add("CreationDateTime", creationDateTime()).add("ProvisionedThroughput", provisionedThroughput())
                .add("TableSizeBytes", tableSizeBytes()).add("ItemCount", itemCount()).add("TableArn", tableArn())
                .add("LocalSecondaryIndexes", localSecondaryIndexes()).add("GlobalSecondaryIndexes", globalSecondaryIndexes())
                .add("StreamSpecification", streamSpecification()).add("LatestStreamLabel", latestStreamLabel())
                .add("LatestStreamArn", latestStreamArn()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AttributeDefinitions":
            return Optional.of(clazz.cast(attributeDefinitions()));
        case "TableName":
            return Optional.of(clazz.cast(tableName()));
        case "KeySchema":
            return Optional.of(clazz.cast(keySchema()));
        case "TableStatus":
            return Optional.of(clazz.cast(tableStatusAsString()));
        case "CreationDateTime":
            return Optional.of(clazz.cast(creationDateTime()));
        case "ProvisionedThroughput":
            return Optional.of(clazz.cast(provisionedThroughput()));
        case "TableSizeBytes":
            return Optional.of(clazz.cast(tableSizeBytes()));
        case "ItemCount":
            return Optional.of(clazz.cast(itemCount()));
        case "TableArn":
            return Optional.of(clazz.cast(tableArn()));
        case "LocalSecondaryIndexes":
            return Optional.of(clazz.cast(localSecondaryIndexes()));
        case "GlobalSecondaryIndexes":
            return Optional.of(clazz.cast(globalSecondaryIndexes()));
        case "StreamSpecification":
            return Optional.of(clazz.cast(streamSpecification()));
        case "LatestStreamLabel":
            return Optional.of(clazz.cast(latestStreamLabel()));
        case "LatestStreamArn":
            return Optional.of(clazz.cast(latestStreamArn()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        TableDescriptionMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, TableDescription> {
        /**
         * <p>
         * An array of <code>AttributeDefinition</code> objects. Each of these objects describes one attribute in the
         * table and index key schema.
         * </p>
         * <p>
         * Each <code>AttributeDefinition</code> object in this array is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AttributeName</code> - The name of the attribute.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AttributeType</code> - The data type for the attribute.
         * </p>
         * </li>
         * </ul>
         * 
         * @param attributeDefinitions
         *        An array of <code>AttributeDefinition</code> objects. Each of these objects describes one attribute in
         *        the table and index key schema.</p>
         *        <p>
         *        Each <code>AttributeDefinition</code> object in this array is composed of:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AttributeName</code> - The name of the attribute.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AttributeType</code> - The data type for the attribute.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeDefinitions(Collection<AttributeDefinition> attributeDefinitions);

        /**
         * <p>
         * An array of <code>AttributeDefinition</code> objects. Each of these objects describes one attribute in the
         * table and index key schema.
         * </p>
         * <p>
         * Each <code>AttributeDefinition</code> object in this array is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AttributeName</code> - The name of the attribute.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AttributeType</code> - The data type for the attribute.
         * </p>
         * </li>
         * </ul>
         * 
         * @param attributeDefinitions
         *        An array of <code>AttributeDefinition</code> objects. Each of these objects describes one attribute in
         *        the table and index key schema.</p>
         *        <p>
         *        Each <code>AttributeDefinition</code> object in this array is composed of:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AttributeName</code> - The name of the attribute.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AttributeType</code> - The data type for the attribute.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeDefinitions(AttributeDefinition... attributeDefinitions);

        /**
         * <p>
         * An array of <code>AttributeDefinition</code> objects. Each of these objects describes one attribute in the
         * table and index key schema.
         * </p>
         * <p>
         * Each <code>AttributeDefinition</code> object in this array is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AttributeName</code> - The name of the attribute.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AttributeType</code> - The data type for the attribute.
         * </p>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link List<AttributeDefinition>.Builder} avoiding the
         * need to create one manually via {@link List<AttributeDefinition>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AttributeDefinition>.Builder#build()} is called immediately
         * and its result is passed to {@link #attributeDefinitions(List<AttributeDefinition>)}.
         * 
         * @param attributeDefinitions
         *        a consumer that will call methods on {@link List<AttributeDefinition>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attributeDefinitions(List<AttributeDefinition>)
         */
        Builder attributeDefinitions(Consumer<AttributeDefinition.Builder>... attributeDefinitions);

        /**
         * <p>
         * The name of the table.
         * </p>
         * 
         * @param tableName
         *        The name of the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableName(String tableName);

        /**
         * <p>
         * The primary key structure for the table. Each <code>KeySchemaElement</code> consists of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AttributeName</code> - The name of the attribute.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KeyType</code> - The role of the attribute:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB' usage of an internal hash function to evenly distribute data items across partitions, based on
         * their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note></li>
         * </ul>
         * <p>
         * For more information about primary keys, see <a
         * href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataModel.html#DataModelPrimaryKey"
         * >Primary Key</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param keySchema
         *        The primary key structure for the table. Each <code>KeySchemaElement</code> consists of:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AttributeName</code> - The name of the attribute.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KeyType</code> - The role of the attribute:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HASH</code> - partition key
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RANGE</code> - sort key
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
         *        derives from DynamoDB' usage of an internal hash function to evenly distribute data items across
         *        partitions, based on their partition key values.
         *        </p>
         *        <p>
         *        The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute"
         *        derives from the way DynamoDB stores items with the same partition key physically close together, in
         *        sorted order by the sort key value.
         *        </p>
         *        </note></li>
         *        </ul>
         *        <p>
         *        For more information about primary keys, see <a href=
         *        "http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataModel.html#DataModelPrimaryKey"
         *        >Primary Key</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keySchema(Collection<KeySchemaElement> keySchema);

        /**
         * <p>
         * The primary key structure for the table. Each <code>KeySchemaElement</code> consists of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AttributeName</code> - The name of the attribute.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KeyType</code> - The role of the attribute:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB' usage of an internal hash function to evenly distribute data items across partitions, based on
         * their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note></li>
         * </ul>
         * <p>
         * For more information about primary keys, see <a
         * href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataModel.html#DataModelPrimaryKey"
         * >Primary Key</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param keySchema
         *        The primary key structure for the table. Each <code>KeySchemaElement</code> consists of:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AttributeName</code> - The name of the attribute.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KeyType</code> - The role of the attribute:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HASH</code> - partition key
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RANGE</code> - sort key
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
         *        derives from DynamoDB' usage of an internal hash function to evenly distribute data items across
         *        partitions, based on their partition key values.
         *        </p>
         *        <p>
         *        The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute"
         *        derives from the way DynamoDB stores items with the same partition key physically close together, in
         *        sorted order by the sort key value.
         *        </p>
         *        </note></li>
         *        </ul>
         *        <p>
         *        For more information about primary keys, see <a href=
         *        "http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataModel.html#DataModelPrimaryKey"
         *        >Primary Key</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keySchema(KeySchemaElement... keySchema);

        /**
         * <p>
         * The primary key structure for the table. Each <code>KeySchemaElement</code> consists of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AttributeName</code> - The name of the attribute.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KeyType</code> - The role of the attribute:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB' usage of an internal hash function to evenly distribute data items across partitions, based on
         * their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note></li>
         * </ul>
         * <p>
         * For more information about primary keys, see <a
         * href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataModel.html#DataModelPrimaryKey"
         * >Primary Key</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<KeySchemaElement>.Builder} avoiding the
         * need to create one manually via {@link List<KeySchemaElement>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeySchemaElement>.Builder#build()} is called immediately and
         * its result is passed to {@link #keySchema(List<KeySchemaElement>)}.
         * 
         * @param keySchema
         *        a consumer that will call methods on {@link List<KeySchemaElement>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #keySchema(List<KeySchemaElement>)
         */
        Builder keySchema(Consumer<KeySchemaElement.Builder>... keySchema);

        /**
         * <p>
         * The current state of the table:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The table is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The table is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The table is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The table is ready for use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tableStatus
         *        The current state of the table:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The table is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The table is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The table is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The table is ready for use.
         *        </p>
         *        </li>
         * @see TableStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TableStatus
         */
        Builder tableStatus(String tableStatus);

        /**
         * <p>
         * The current state of the table:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The table is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The table is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The table is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The table is ready for use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tableStatus
         *        The current state of the table:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The table is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The table is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The table is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The table is ready for use.
         *        </p>
         *        </li>
         * @see TableStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TableStatus
         */
        Builder tableStatus(TableStatus tableStatus);

        /**
         * <p>
         * The date and time when the table was created, in <a href="http://www.epochconverter.com/">UNIX epoch time</a>
         * format.
         * </p>
         * 
         * @param creationDateTime
         *        The date and time when the table was created, in <a href="http://www.epochconverter.com/">UNIX epoch
         *        time</a> format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDateTime(Instant creationDateTime);

        /**
         * <p>
         * The provisioned throughput settings for the table, consisting of read and write capacity units, along with
         * data about increases and decreases.
         * </p>
         * 
         * @param provisionedThroughput
         *        The provisioned throughput settings for the table, consisting of read and write capacity units, along
         *        with data about increases and decreases.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder provisionedThroughput(ProvisionedThroughputDescription provisionedThroughput);

        /**
         * <p>
         * The provisioned throughput settings for the table, consisting of read and write capacity units, along with
         * data about increases and decreases.
         * </p>
         * This is a convenience that creates an instance of the {@link ProvisionedThroughputDescription.Builder}
         * avoiding the need to create one manually via {@link ProvisionedThroughputDescription#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProvisionedThroughputDescription.Builder#build()} is called
         * immediately and its result is passed to {@link #provisionedThroughput(ProvisionedThroughputDescription)}.
         * 
         * @param provisionedThroughput
         *        a consumer that will call methods on {@link ProvisionedThroughputDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #provisionedThroughput(ProvisionedThroughputDescription)
         */
        default Builder provisionedThroughput(Consumer<ProvisionedThroughputDescription.Builder> provisionedThroughput) {
            return provisionedThroughput(ProvisionedThroughputDescription.builder().apply(provisionedThroughput).build());
        }

        /**
         * <p>
         * The total size of the specified table, in bytes. DynamoDB updates this value approximately every six hours.
         * Recent changes might not be reflected in this value.
         * </p>
         * 
         * @param tableSizeBytes
         *        The total size of the specified table, in bytes. DynamoDB updates this value approximately every six
         *        hours. Recent changes might not be reflected in this value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableSizeBytes(Long tableSizeBytes);

        /**
         * <p>
         * The number of items in the specified table. DynamoDB updates this value approximately every six hours. Recent
         * changes might not be reflected in this value.
         * </p>
         * 
         * @param itemCount
         *        The number of items in the specified table. DynamoDB updates this value approximately every six hours.
         *        Recent changes might not be reflected in this value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder itemCount(Long itemCount);

        /**
         * <p>
         * The Amazon Resource Name (ARN) that uniquely identifies the table.
         * </p>
         * 
         * @param tableArn
         *        The Amazon Resource Name (ARN) that uniquely identifies the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableArn(String tableArn);

        /**
         * <p>
         * Represents one or more local secondary indexes on the table. Each index is scoped to a given partition key
         * value. Tables with one or more local secondary indexes are subject to an item collection size limit, where
         * the amount of data within a given item collection cannot exceed 10 GB. Each element is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>IndexName</code> - The name of the local secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema must
         * be between 1 and 255 characters (inclusive). The key schema must begin with the same partition key as the
         * table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index.
         * These are in addition to the primary key attributes and index key attributes, which are automatically
         * projected. Each attribute specification is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ProjectionType</code> - One of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         * projected attributes are in <code>NonKeyAttributes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code> - All of the table attributes are projected into the index.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
         * secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across all
         * of the secondary indexes, must not exceed 20. If you project the same attribute into two different indexes,
         * this counts as two distinct attributes when determining the total.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>IndexSizeBytes</code> - Represents the total size of the index, in bytes. DynamoDB updates this value
         * approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ItemCount</code> - Represents the number of items in the index. DynamoDB updates this value
         * approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * </p>
         * 
         * @param localSecondaryIndexes
         *        Represents one or more local secondary indexes on the table. Each index is scoped to a given partition
         *        key value. Tables with one or more local secondary indexes are subject to an item collection size
         *        limit, where the amount of data within a given item collection cannot exceed 10 GB. Each element is
         *        composed of:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>IndexName</code> - The name of the local secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key
         *        schema must be between 1 and 255 characters (inclusive). The key schema must begin with the same
         *        partition key as the table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the
         *        index. These are in addition to the primary key attributes and index key attributes, which are
         *        automatically projected. Each attribute specification is composed of:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ProjectionType</code> - One of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         *        projected attributes are in <code>NonKeyAttributes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code> - All of the table attributes are projected into the index.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into
         *        the secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed
         *        across all of the secondary indexes, must not exceed 20. If you project the same attribute into two
         *        different indexes, this counts as two distinct attributes when determining the total.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IndexSizeBytes</code> - Represents the total size of the index, in bytes. DynamoDB updates this
         *        value approximately every six hours. Recent changes might not be reflected in this value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ItemCount</code> - Represents the number of items in the index. DynamoDB updates this value
         *        approximately every six hours. Recent changes might not be reflected in this value.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localSecondaryIndexes(Collection<LocalSecondaryIndexDescription> localSecondaryIndexes);

        /**
         * <p>
         * Represents one or more local secondary indexes on the table. Each index is scoped to a given partition key
         * value. Tables with one or more local secondary indexes are subject to an item collection size limit, where
         * the amount of data within a given item collection cannot exceed 10 GB. Each element is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>IndexName</code> - The name of the local secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema must
         * be between 1 and 255 characters (inclusive). The key schema must begin with the same partition key as the
         * table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index.
         * These are in addition to the primary key attributes and index key attributes, which are automatically
         * projected. Each attribute specification is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ProjectionType</code> - One of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         * projected attributes are in <code>NonKeyAttributes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code> - All of the table attributes are projected into the index.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
         * secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across all
         * of the secondary indexes, must not exceed 20. If you project the same attribute into two different indexes,
         * this counts as two distinct attributes when determining the total.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>IndexSizeBytes</code> - Represents the total size of the index, in bytes. DynamoDB updates this value
         * approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ItemCount</code> - Represents the number of items in the index. DynamoDB updates this value
         * approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * </p>
         * 
         * @param localSecondaryIndexes
         *        Represents one or more local secondary indexes on the table. Each index is scoped to a given partition
         *        key value. Tables with one or more local secondary indexes are subject to an item collection size
         *        limit, where the amount of data within a given item collection cannot exceed 10 GB. Each element is
         *        composed of:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>IndexName</code> - The name of the local secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key
         *        schema must be between 1 and 255 characters (inclusive). The key schema must begin with the same
         *        partition key as the table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the
         *        index. These are in addition to the primary key attributes and index key attributes, which are
         *        automatically projected. Each attribute specification is composed of:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ProjectionType</code> - One of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         *        projected attributes are in <code>NonKeyAttributes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code> - All of the table attributes are projected into the index.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into
         *        the secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed
         *        across all of the secondary indexes, must not exceed 20. If you project the same attribute into two
         *        different indexes, this counts as two distinct attributes when determining the total.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IndexSizeBytes</code> - Represents the total size of the index, in bytes. DynamoDB updates this
         *        value approximately every six hours. Recent changes might not be reflected in this value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ItemCount</code> - Represents the number of items in the index. DynamoDB updates this value
         *        approximately every six hours. Recent changes might not be reflected in this value.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localSecondaryIndexes(LocalSecondaryIndexDescription... localSecondaryIndexes);

        /**
         * <p>
         * Represents one or more local secondary indexes on the table. Each index is scoped to a given partition key
         * value. Tables with one or more local secondary indexes are subject to an item collection size limit, where
         * the amount of data within a given item collection cannot exceed 10 GB. Each element is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>IndexName</code> - The name of the local secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema must
         * be between 1 and 255 characters (inclusive). The key schema must begin with the same partition key as the
         * table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index.
         * These are in addition to the primary key attributes and index key attributes, which are automatically
         * projected. Each attribute specification is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ProjectionType</code> - One of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         * projected attributes are in <code>NonKeyAttributes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code> - All of the table attributes are projected into the index.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
         * secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across all
         * of the secondary indexes, must not exceed 20. If you project the same attribute into two different indexes,
         * this counts as two distinct attributes when determining the total.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>IndexSizeBytes</code> - Represents the total size of the index, in bytes. DynamoDB updates this value
         * approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ItemCount</code> - Represents the number of items in the index. DynamoDB updates this value
         * approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * </p>
         * This is a convenience that creates an instance of the {@link List<LocalSecondaryIndexDescription>.Builder}
         * avoiding the need to create one manually via {@link List<LocalSecondaryIndexDescription>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<LocalSecondaryIndexDescription>.Builder#build()} is called
         * immediately and its result is passed to {@link #localSecondaryIndexes(List<LocalSecondaryIndexDescription>)}.
         * 
         * @param localSecondaryIndexes
         *        a consumer that will call methods on {@link List<LocalSecondaryIndexDescription>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #localSecondaryIndexes(List<LocalSecondaryIndexDescription>)
         */
        Builder localSecondaryIndexes(Consumer<LocalSecondaryIndexDescription.Builder>... localSecondaryIndexes);

        /**
         * <p>
         * The global secondary indexes, if any, on the table. Each index is scoped to a given partition key value. Each
         * element is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Backfilling</code> - If true, then the index is currently in the backfilling phase. Backfilling occurs
         * only when a new global secondary index is added to the table; it is the process by which DynamoDB populates
         * the new index with data from the table. (This attribute does not appear for indexes that were created during
         * a <code>CreateTable</code> operation.)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IndexName</code> - The name of the global secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IndexSizeBytes</code> - The total size of the global secondary index, in bytes. DynamoDB updates this
         * value approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IndexStatus</code> - The current status of the global secondary index:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The index is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The index is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The index is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The index is ready for use.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>ItemCount</code> - The number of items in the global secondary index. DynamoDB updates this value
         * approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema must
         * be between 1 and 255 characters (inclusive). The key schema must begin with the same partition key as the
         * table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index.
         * These are in addition to the primary key attributes and index key attributes, which are automatically
         * projected. Each attribute specification is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ProjectionType</code> - One of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         * projected attributes are in <code>NonKeyAttributes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code> - All of the table attributes are projected into the index.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
         * secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across all
         * of the secondary indexes, must not exceed 20. If you project the same attribute into two different indexes,
         * this counts as two distinct attributes when determining the total.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>ProvisionedThroughput</code> - The provisioned throughput settings for the global secondary index,
         * consisting of read and write capacity units, along with data about increases and decreases.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * </p>
         * 
         * @param globalSecondaryIndexes
         *        The global secondary indexes, if any, on the table. Each index is scoped to a given partition key
         *        value. Each element is composed of:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Backfilling</code> - If true, then the index is currently in the backfilling phase. Backfilling
         *        occurs only when a new global secondary index is added to the table; it is the process by which
         *        DynamoDB populates the new index with data from the table. (This attribute does not appear for indexes
         *        that were created during a <code>CreateTable</code> operation.)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IndexName</code> - The name of the global secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IndexSizeBytes</code> - The total size of the global secondary index, in bytes. DynamoDB updates
         *        this value approximately every six hours. Recent changes might not be reflected in this value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IndexStatus</code> - The current status of the global secondary index:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The index is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The index is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The index is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The index is ready for use.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ItemCount</code> - The number of items in the global secondary index. DynamoDB updates this
         *        value approximately every six hours. Recent changes might not be reflected in this value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key
         *        schema must be between 1 and 255 characters (inclusive). The key schema must begin with the same
         *        partition key as the table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the
         *        index. These are in addition to the primary key attributes and index key attributes, which are
         *        automatically projected. Each attribute specification is composed of:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ProjectionType</code> - One of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         *        projected attributes are in <code>NonKeyAttributes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code> - All of the table attributes are projected into the index.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into
         *        the secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed
         *        across all of the secondary indexes, must not exceed 20. If you project the same attribute into two
         *        different indexes, this counts as two distinct attributes when determining the total.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ProvisionedThroughput</code> - The provisioned throughput settings for the global secondary
         *        index, consisting of read and write capacity units, along with data about increases and decreases.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder globalSecondaryIndexes(Collection<GlobalSecondaryIndexDescription> globalSecondaryIndexes);

        /**
         * <p>
         * The global secondary indexes, if any, on the table. Each index is scoped to a given partition key value. Each
         * element is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Backfilling</code> - If true, then the index is currently in the backfilling phase. Backfilling occurs
         * only when a new global secondary index is added to the table; it is the process by which DynamoDB populates
         * the new index with data from the table. (This attribute does not appear for indexes that were created during
         * a <code>CreateTable</code> operation.)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IndexName</code> - The name of the global secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IndexSizeBytes</code> - The total size of the global secondary index, in bytes. DynamoDB updates this
         * value approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IndexStatus</code> - The current status of the global secondary index:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The index is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The index is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The index is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The index is ready for use.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>ItemCount</code> - The number of items in the global secondary index. DynamoDB updates this value
         * approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema must
         * be between 1 and 255 characters (inclusive). The key schema must begin with the same partition key as the
         * table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index.
         * These are in addition to the primary key attributes and index key attributes, which are automatically
         * projected. Each attribute specification is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ProjectionType</code> - One of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         * projected attributes are in <code>NonKeyAttributes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code> - All of the table attributes are projected into the index.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
         * secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across all
         * of the secondary indexes, must not exceed 20. If you project the same attribute into two different indexes,
         * this counts as two distinct attributes when determining the total.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>ProvisionedThroughput</code> - The provisioned throughput settings for the global secondary index,
         * consisting of read and write capacity units, along with data about increases and decreases.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * </p>
         * 
         * @param globalSecondaryIndexes
         *        The global secondary indexes, if any, on the table. Each index is scoped to a given partition key
         *        value. Each element is composed of:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Backfilling</code> - If true, then the index is currently in the backfilling phase. Backfilling
         *        occurs only when a new global secondary index is added to the table; it is the process by which
         *        DynamoDB populates the new index with data from the table. (This attribute does not appear for indexes
         *        that were created during a <code>CreateTable</code> operation.)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IndexName</code> - The name of the global secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IndexSizeBytes</code> - The total size of the global secondary index, in bytes. DynamoDB updates
         *        this value approximately every six hours. Recent changes might not be reflected in this value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IndexStatus</code> - The current status of the global secondary index:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The index is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The index is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The index is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The index is ready for use.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ItemCount</code> - The number of items in the global secondary index. DynamoDB updates this
         *        value approximately every six hours. Recent changes might not be reflected in this value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key
         *        schema must be between 1 and 255 characters (inclusive). The key schema must begin with the same
         *        partition key as the table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the
         *        index. These are in addition to the primary key attributes and index key attributes, which are
         *        automatically projected. Each attribute specification is composed of:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ProjectionType</code> - One of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         *        projected attributes are in <code>NonKeyAttributes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code> - All of the table attributes are projected into the index.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into
         *        the secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed
         *        across all of the secondary indexes, must not exceed 20. If you project the same attribute into two
         *        different indexes, this counts as two distinct attributes when determining the total.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ProvisionedThroughput</code> - The provisioned throughput settings for the global secondary
         *        index, consisting of read and write capacity units, along with data about increases and decreases.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder globalSecondaryIndexes(GlobalSecondaryIndexDescription... globalSecondaryIndexes);

        /**
         * <p>
         * The global secondary indexes, if any, on the table. Each index is scoped to a given partition key value. Each
         * element is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Backfilling</code> - If true, then the index is currently in the backfilling phase. Backfilling occurs
         * only when a new global secondary index is added to the table; it is the process by which DynamoDB populates
         * the new index with data from the table. (This attribute does not appear for indexes that were created during
         * a <code>CreateTable</code> operation.)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IndexName</code> - The name of the global secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IndexSizeBytes</code> - The total size of the global secondary index, in bytes. DynamoDB updates this
         * value approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IndexStatus</code> - The current status of the global secondary index:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The index is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The index is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The index is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The index is ready for use.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>ItemCount</code> - The number of items in the global secondary index. DynamoDB updates this value
         * approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema must
         * be between 1 and 255 characters (inclusive). The key schema must begin with the same partition key as the
         * table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index.
         * These are in addition to the primary key attributes and index key attributes, which are automatically
         * projected. Each attribute specification is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ProjectionType</code> - One of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         * projected attributes are in <code>NonKeyAttributes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code> - All of the table attributes are projected into the index.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
         * secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across all
         * of the secondary indexes, must not exceed 20. If you project the same attribute into two different indexes,
         * this counts as two distinct attributes when determining the total.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>ProvisionedThroughput</code> - The provisioned throughput settings for the global secondary index,
         * consisting of read and write capacity units, along with data about increases and decreases.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * </p>
         * This is a convenience that creates an instance of the {@link List<GlobalSecondaryIndexDescription>.Builder}
         * avoiding the need to create one manually via {@link List<GlobalSecondaryIndexDescription>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<GlobalSecondaryIndexDescription>.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #globalSecondaryIndexes(List<GlobalSecondaryIndexDescription>)}.
         * 
         * @param globalSecondaryIndexes
         *        a consumer that will call methods on {@link List<GlobalSecondaryIndexDescription>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #globalSecondaryIndexes(List<GlobalSecondaryIndexDescription>)
         */
        Builder globalSecondaryIndexes(Consumer<GlobalSecondaryIndexDescription.Builder>... globalSecondaryIndexes);

        /**
         * <p>
         * The current DynamoDB Streams configuration for the table.
         * </p>
         * 
         * @param streamSpecification
         *        The current DynamoDB Streams configuration for the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamSpecification(StreamSpecification streamSpecification);

        /**
         * <p>
         * The current DynamoDB Streams configuration for the table.
         * </p>
         * This is a convenience that creates an instance of the {@link StreamSpecification.Builder} avoiding the need
         * to create one manually via {@link StreamSpecification#builder()}.
         *
         * When the {@link Consumer} completes, {@link StreamSpecification.Builder#build()} is called immediately and
         * its result is passed to {@link #streamSpecification(StreamSpecification)}.
         * 
         * @param streamSpecification
         *        a consumer that will call methods on {@link StreamSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #streamSpecification(StreamSpecification)
         */
        default Builder streamSpecification(Consumer<StreamSpecification.Builder> streamSpecification) {
            return streamSpecification(StreamSpecification.builder().apply(streamSpecification).build());
        }

        /**
         * <p>
         * A timestamp, in ISO 8601 format, for this stream.
         * </p>
         * <p>
         * Note that <code>LatestStreamLabel</code> is not a unique identifier for the stream, because it is possible
         * that a stream from another table might have the same timestamp. However, the combination of the following
         * three elements is guaranteed to be unique:
         * </p>
         * <ul>
         * <li>
         * <p>
         * the AWS customer ID.
         * </p>
         * </li>
         * <li>
         * <p>
         * the table name.
         * </p>
         * </li>
         * <li>
         * <p>
         * the <code>StreamLabel</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param latestStreamLabel
         *        A timestamp, in ISO 8601 format, for this stream.</p>
         *        <p>
         *        Note that <code>LatestStreamLabel</code> is not a unique identifier for the stream, because it is
         *        possible that a stream from another table might have the same timestamp. However, the combination of
         *        the following three elements is guaranteed to be unique:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        the AWS customer ID.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        the table name.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        the <code>StreamLabel</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder latestStreamLabel(String latestStreamLabel);

        /**
         * <p>
         * The Amazon Resource Name (ARN) that uniquely identifies the latest stream for this table.
         * </p>
         * 
         * @param latestStreamArn
         *        The Amazon Resource Name (ARN) that uniquely identifies the latest stream for this table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder latestStreamArn(String latestStreamArn);
    }

    static final class BuilderImpl implements Builder {
        private List<AttributeDefinition> attributeDefinitions;

        private String tableName;

        private List<KeySchemaElement> keySchema;

        private String tableStatus;

        private Instant creationDateTime;

        private ProvisionedThroughputDescription provisionedThroughput;

        private Long tableSizeBytes;

        private Long itemCount;

        private String tableArn;

        private List<LocalSecondaryIndexDescription> localSecondaryIndexes;

        private List<GlobalSecondaryIndexDescription> globalSecondaryIndexes;

        private StreamSpecification streamSpecification;

        private String latestStreamLabel;

        private String latestStreamArn;

        private BuilderImpl() {
        }

        private BuilderImpl(TableDescription model) {
            attributeDefinitions(model.attributeDefinitions);
            tableName(model.tableName);
            keySchema(model.keySchema);
            tableStatus(model.tableStatus);
            creationDateTime(model.creationDateTime);
            provisionedThroughput(model.provisionedThroughput);
            tableSizeBytes(model.tableSizeBytes);
            itemCount(model.itemCount);
            tableArn(model.tableArn);
            localSecondaryIndexes(model.localSecondaryIndexes);
            globalSecondaryIndexes(model.globalSecondaryIndexes);
            streamSpecification(model.streamSpecification);
            latestStreamLabel(model.latestStreamLabel);
            latestStreamArn(model.latestStreamArn);
        }

        public final Collection<AttributeDefinition.Builder> getAttributeDefinitions() {
            return attributeDefinitions != null ? attributeDefinitions.stream().map(AttributeDefinition::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder attributeDefinitions(Collection<AttributeDefinition> attributeDefinitions) {
            this.attributeDefinitions = AttributeDefinitionsCopier.copy(attributeDefinitions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributeDefinitions(AttributeDefinition... attributeDefinitions) {
            attributeDefinitions(Arrays.asList(attributeDefinitions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributeDefinitions(Consumer<AttributeDefinition.Builder>... attributeDefinitions) {
            attributeDefinitions(Stream.of(attributeDefinitions).map(c -> AttributeDefinition.builder().apply(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setAttributeDefinitions(Collection<AttributeDefinition.BuilderImpl> attributeDefinitions) {
            this.attributeDefinitions = AttributeDefinitionsCopier.copyFromBuilder(attributeDefinitions);
        }

        public final String getTableName() {
            return tableName;
        }

        @Override
        public final Builder tableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        public final void setTableName(String tableName) {
            this.tableName = tableName;
        }

        public final Collection<KeySchemaElement.Builder> getKeySchema() {
            return keySchema != null ? keySchema.stream().map(KeySchemaElement::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder keySchema(Collection<KeySchemaElement> keySchema) {
            this.keySchema = KeySchemaCopier.copy(keySchema);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder keySchema(KeySchemaElement... keySchema) {
            keySchema(Arrays.asList(keySchema));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder keySchema(Consumer<KeySchemaElement.Builder>... keySchema) {
            keySchema(Stream.of(keySchema).map(c -> KeySchemaElement.builder().apply(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setKeySchema(Collection<KeySchemaElement.BuilderImpl> keySchema) {
            this.keySchema = KeySchemaCopier.copyFromBuilder(keySchema);
        }

        public final String getTableStatus() {
            return tableStatus;
        }

        @Override
        public final Builder tableStatus(String tableStatus) {
            this.tableStatus = tableStatus;
            return this;
        }

        @Override
        public final Builder tableStatus(TableStatus tableStatus) {
            this.tableStatus(tableStatus.toString());
            return this;
        }

        public final void setTableStatus(String tableStatus) {
            this.tableStatus = tableStatus;
        }

        public final Instant getCreationDateTime() {
            return creationDateTime;
        }

        @Override
        public final Builder creationDateTime(Instant creationDateTime) {
            this.creationDateTime = creationDateTime;
            return this;
        }

        public final void setCreationDateTime(Instant creationDateTime) {
            this.creationDateTime = creationDateTime;
        }

        public final ProvisionedThroughputDescription.Builder getProvisionedThroughput() {
            return provisionedThroughput != null ? provisionedThroughput.toBuilder() : null;
        }

        @Override
        public final Builder provisionedThroughput(ProvisionedThroughputDescription provisionedThroughput) {
            this.provisionedThroughput = provisionedThroughput;
            return this;
        }

        public final void setProvisionedThroughput(ProvisionedThroughputDescription.BuilderImpl provisionedThroughput) {
            this.provisionedThroughput = provisionedThroughput != null ? provisionedThroughput.build() : null;
        }

        public final Long getTableSizeBytes() {
            return tableSizeBytes;
        }

        @Override
        public final Builder tableSizeBytes(Long tableSizeBytes) {
            this.tableSizeBytes = tableSizeBytes;
            return this;
        }

        public final void setTableSizeBytes(Long tableSizeBytes) {
            this.tableSizeBytes = tableSizeBytes;
        }

        public final Long getItemCount() {
            return itemCount;
        }

        @Override
        public final Builder itemCount(Long itemCount) {
            this.itemCount = itemCount;
            return this;
        }

        public final void setItemCount(Long itemCount) {
            this.itemCount = itemCount;
        }

        public final String getTableArn() {
            return tableArn;
        }

        @Override
        public final Builder tableArn(String tableArn) {
            this.tableArn = tableArn;
            return this;
        }

        public final void setTableArn(String tableArn) {
            this.tableArn = tableArn;
        }

        public final Collection<LocalSecondaryIndexDescription.Builder> getLocalSecondaryIndexes() {
            return localSecondaryIndexes != null ? localSecondaryIndexes.stream().map(LocalSecondaryIndexDescription::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder localSecondaryIndexes(Collection<LocalSecondaryIndexDescription> localSecondaryIndexes) {
            this.localSecondaryIndexes = LocalSecondaryIndexDescriptionListCopier.copy(localSecondaryIndexes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder localSecondaryIndexes(LocalSecondaryIndexDescription... localSecondaryIndexes) {
            localSecondaryIndexes(Arrays.asList(localSecondaryIndexes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder localSecondaryIndexes(Consumer<LocalSecondaryIndexDescription.Builder>... localSecondaryIndexes) {
            localSecondaryIndexes(Stream.of(localSecondaryIndexes)
                    .map(c -> LocalSecondaryIndexDescription.builder().apply(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setLocalSecondaryIndexes(Collection<LocalSecondaryIndexDescription.BuilderImpl> localSecondaryIndexes) {
            this.localSecondaryIndexes = LocalSecondaryIndexDescriptionListCopier.copyFromBuilder(localSecondaryIndexes);
        }

        public final Collection<GlobalSecondaryIndexDescription.Builder> getGlobalSecondaryIndexes() {
            return globalSecondaryIndexes != null ? globalSecondaryIndexes.stream()
                    .map(GlobalSecondaryIndexDescription::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder globalSecondaryIndexes(Collection<GlobalSecondaryIndexDescription> globalSecondaryIndexes) {
            this.globalSecondaryIndexes = GlobalSecondaryIndexDescriptionListCopier.copy(globalSecondaryIndexes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder globalSecondaryIndexes(GlobalSecondaryIndexDescription... globalSecondaryIndexes) {
            globalSecondaryIndexes(Arrays.asList(globalSecondaryIndexes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder globalSecondaryIndexes(Consumer<GlobalSecondaryIndexDescription.Builder>... globalSecondaryIndexes) {
            globalSecondaryIndexes(Stream.of(globalSecondaryIndexes)
                    .map(c -> GlobalSecondaryIndexDescription.builder().apply(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setGlobalSecondaryIndexes(Collection<GlobalSecondaryIndexDescription.BuilderImpl> globalSecondaryIndexes) {
            this.globalSecondaryIndexes = GlobalSecondaryIndexDescriptionListCopier.copyFromBuilder(globalSecondaryIndexes);
        }

        public final StreamSpecification.Builder getStreamSpecification() {
            return streamSpecification != null ? streamSpecification.toBuilder() : null;
        }

        @Override
        public final Builder streamSpecification(StreamSpecification streamSpecification) {
            this.streamSpecification = streamSpecification;
            return this;
        }

        public final void setStreamSpecification(StreamSpecification.BuilderImpl streamSpecification) {
            this.streamSpecification = streamSpecification != null ? streamSpecification.build() : null;
        }

        public final String getLatestStreamLabel() {
            return latestStreamLabel;
        }

        @Override
        public final Builder latestStreamLabel(String latestStreamLabel) {
            this.latestStreamLabel = latestStreamLabel;
            return this;
        }

        public final void setLatestStreamLabel(String latestStreamLabel) {
            this.latestStreamLabel = latestStreamLabel;
        }

        public final String getLatestStreamArn() {
            return latestStreamArn;
        }

        @Override
        public final Builder latestStreamArn(String latestStreamArn) {
            this.latestStreamArn = latestStreamArn;
            return this;
        }

        public final void setLatestStreamArn(String latestStreamArn) {
            this.latestStreamArn = latestStreamArn;
        }

        @Override
        public TableDescription build() {
            return new TableDescription(this);
        }
    }
}
