/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.streams.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedItemsIterable;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.services.dynamodb.model.ListStreamsRequest;
import software.amazon.awssdk.services.dynamodb.model.ListStreamsResponse;
import software.amazon.awssdk.services.dynamodb.model.Stream;
import software.amazon.awssdk.services.dynamodb.streams.DynamoDbStreamsClient;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.dynamodb.streams.DynamoDbStreamsClient#listStreamsPaginator(software.amazon.awssdk.services.dynamodb.model.ListStreamsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.dynamodb.model.ListStreamsResponse} that can be used to iterate through all
 * the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.dynamodb.streams.paginators.ListStreamsIterable responses = client.listStreamsPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.dynamodb.streams.paginators.ListStreamsIterable responses = client
 *             .listStreamsPaginator(request);
 *     for (software.amazon.awssdk.services.dynamodb.model.ListStreamsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.dynamodb.streams.paginators.ListStreamsIterable responses = client.listStreamsPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listStreams(software.amazon.awssdk.services.dynamodb.model.ListStreamsRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListStreamsIterable implements SdkIterable<ListStreamsResponse> {
    private final DynamoDbStreamsClient client;

    private final ListStreamsRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListStreamsIterable(DynamoDbStreamsClient client, ListStreamsRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new ListStreamsResponseFetcher();
    }

    @Override
    public Iterator<ListStreamsResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    /**
     * Returns an iterable to iterate through the paginated {@link ListStreamsResponse#streams()} member. The returned
     * iterable is used to iterate through the results across all response pages and not a single page.
     *
     * This method is useful if you are interested in iterating over the paginated member in the response pages instead
     * of the top level pages. Similar to iteration over pages, this method internally makes service calls to get the
     * next list of results until the iteration stops or there are no more results.
     */
    public final SdkIterable<Stream> streams() {
        Function<ListStreamsResponse, Iterator<Stream>> getIterator = response -> {
            if (response != null && response.streams() != null) {
                return response.streams().iterator();
            }
            return Collections.emptyIterator();
        };
        return PaginatedItemsIterable.builder().pagesIterable(this).itemIteratorFunction(getIterator).build();
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link ListStreamsIterable} that can be used to retrieve the consecutive
     * pages that follows the input page.
     * </p>
     */
    private final ListStreamsIterable resume(ListStreamsResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new ListStreamsIterable(client, firstRequest.toBuilder()
                    .exclusiveStartStreamArn(lastSuccessfulPage.lastEvaluatedStreamArn()).build());
        }
        return new ListStreamsIterable(client, firstRequest) {
            @Override
            public Iterator<ListStreamsResponse> iterator() {
                return Collections.emptyIterator();
            }
        };
    }

    private class ListStreamsResponseFetcher implements SyncPageFetcher<ListStreamsResponse> {
        @Override
        public boolean hasNextPage(ListStreamsResponse previousPage) {
            return previousPage.lastEvaluatedStreamArn() != null
                    && !SdkAutoConstructList.class.isInstance(previousPage.lastEvaluatedStreamArn())
                    && !SdkAutoConstructMap.class.isInstance(previousPage.lastEvaluatedStreamArn());
        }

        @Override
        public ListStreamsResponse nextPage(ListStreamsResponse previousPage) {
            if (previousPage == null) {
                return client.listStreams(firstRequest);
            }
            return client.listStreams(firstRequest.toBuilder().exclusiveStartStreamArn(previousPage.lastEvaluatedStreamArn())
                    .build());
        }
    }
}
