/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.dynamodb.transform.AutoScalingSettingsDescriptionMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the autoscaling settings for a global table or global secondary index.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutoScalingSettingsDescription implements StructuredPojo,
        ToCopyableBuilder<AutoScalingSettingsDescription.Builder, AutoScalingSettingsDescription> {
    private final Long minimumUnits;

    private final Long maximumUnits;

    private final Boolean autoScalingDisabled;

    private final String autoScalingRoleArn;

    private final List<AutoScalingPolicyDescription> scalingPolicies;

    private AutoScalingSettingsDescription(BuilderImpl builder) {
        this.minimumUnits = builder.minimumUnits;
        this.maximumUnits = builder.maximumUnits;
        this.autoScalingDisabled = builder.autoScalingDisabled;
        this.autoScalingRoleArn = builder.autoScalingRoleArn;
        this.scalingPolicies = builder.scalingPolicies;
    }

    /**
     * <p>
     * The minimum capacity units that a global table or global secondary index should be scaled down to.
     * </p>
     * 
     * @return The minimum capacity units that a global table or global secondary index should be scaled down to.
     */
    public Long minimumUnits() {
        return minimumUnits;
    }

    /**
     * <p>
     * The maximum capacity units that a global table or global secondary index should be scaled up to.
     * </p>
     * 
     * @return The maximum capacity units that a global table or global secondary index should be scaled up to.
     */
    public Long maximumUnits() {
        return maximumUnits;
    }

    /**
     * <p>
     * Disabled autoscaling for this global table or global secondary index.
     * </p>
     * 
     * @return Disabled autoscaling for this global table or global secondary index.
     */
    public Boolean autoScalingDisabled() {
        return autoScalingDisabled;
    }

    /**
     * <p>
     * Role ARN used for configuring autoScaling policy.
     * </p>
     * 
     * @return Role ARN used for configuring autoScaling policy.
     */
    public String autoScalingRoleArn() {
        return autoScalingRoleArn;
    }

    /**
     * <p>
     * Information about the scaling policies.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Information about the scaling policies.
     */
    public List<AutoScalingPolicyDescription> scalingPolicies() {
        return scalingPolicies;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(minimumUnits());
        hashCode = 31 * hashCode + Objects.hashCode(maximumUnits());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingDisabled());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(scalingPolicies());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoScalingSettingsDescription)) {
            return false;
        }
        AutoScalingSettingsDescription other = (AutoScalingSettingsDescription) obj;
        return Objects.equals(minimumUnits(), other.minimumUnits()) && Objects.equals(maximumUnits(), other.maximumUnits())
                && Objects.equals(autoScalingDisabled(), other.autoScalingDisabled())
                && Objects.equals(autoScalingRoleArn(), other.autoScalingRoleArn())
                && Objects.equals(scalingPolicies(), other.scalingPolicies());
    }

    @Override
    public String toString() {
        return ToString.builder("AutoScalingSettingsDescription").add("MinimumUnits", minimumUnits())
                .add("MaximumUnits", maximumUnits()).add("AutoScalingDisabled", autoScalingDisabled())
                .add("AutoScalingRoleArn", autoScalingRoleArn()).add("ScalingPolicies", scalingPolicies()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MinimumUnits":
            return Optional.ofNullable(clazz.cast(minimumUnits()));
        case "MaximumUnits":
            return Optional.ofNullable(clazz.cast(maximumUnits()));
        case "AutoScalingDisabled":
            return Optional.ofNullable(clazz.cast(autoScalingDisabled()));
        case "AutoScalingRoleArn":
            return Optional.ofNullable(clazz.cast(autoScalingRoleArn()));
        case "ScalingPolicies":
            return Optional.ofNullable(clazz.cast(scalingPolicies()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        AutoScalingSettingsDescriptionMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, AutoScalingSettingsDescription> {
        /**
         * <p>
         * The minimum capacity units that a global table or global secondary index should be scaled down to.
         * </p>
         * 
         * @param minimumUnits
         *        The minimum capacity units that a global table or global secondary index should be scaled down to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minimumUnits(Long minimumUnits);

        /**
         * <p>
         * The maximum capacity units that a global table or global secondary index should be scaled up to.
         * </p>
         * 
         * @param maximumUnits
         *        The maximum capacity units that a global table or global secondary index should be scaled up to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumUnits(Long maximumUnits);

        /**
         * <p>
         * Disabled autoscaling for this global table or global secondary index.
         * </p>
         * 
         * @param autoScalingDisabled
         *        Disabled autoscaling for this global table or global secondary index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingDisabled(Boolean autoScalingDisabled);

        /**
         * <p>
         * Role ARN used for configuring autoScaling policy.
         * </p>
         * 
         * @param autoScalingRoleArn
         *        Role ARN used for configuring autoScaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingRoleArn(String autoScalingRoleArn);

        /**
         * <p>
         * Information about the scaling policies.
         * </p>
         * 
         * @param scalingPolicies
         *        Information about the scaling policies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scalingPolicies(Collection<AutoScalingPolicyDescription> scalingPolicies);

        /**
         * <p>
         * Information about the scaling policies.
         * </p>
         * 
         * @param scalingPolicies
         *        Information about the scaling policies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scalingPolicies(AutoScalingPolicyDescription... scalingPolicies);

        /**
         * <p>
         * Information about the scaling policies.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AutoScalingPolicyDescription>.Builder}
         * avoiding the need to create one manually via {@link List<AutoScalingPolicyDescription>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AutoScalingPolicyDescription>.Builder#build()} is called
         * immediately and its result is passed to {@link #scalingPolicies(List<AutoScalingPolicyDescription>)}.
         * 
         * @param scalingPolicies
         *        a consumer that will call methods on {@link List<AutoScalingPolicyDescription>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scalingPolicies(List<AutoScalingPolicyDescription>)
         */
        Builder scalingPolicies(Consumer<AutoScalingPolicyDescription.Builder>... scalingPolicies);
    }

    static final class BuilderImpl implements Builder {
        private Long minimumUnits;

        private Long maximumUnits;

        private Boolean autoScalingDisabled;

        private String autoScalingRoleArn;

        private List<AutoScalingPolicyDescription> scalingPolicies = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AutoScalingSettingsDescription model) {
            minimumUnits(model.minimumUnits);
            maximumUnits(model.maximumUnits);
            autoScalingDisabled(model.autoScalingDisabled);
            autoScalingRoleArn(model.autoScalingRoleArn);
            scalingPolicies(model.scalingPolicies);
        }

        public final Long getMinimumUnits() {
            return minimumUnits;
        }

        @Override
        public final Builder minimumUnits(Long minimumUnits) {
            this.minimumUnits = minimumUnits;
            return this;
        }

        public final void setMinimumUnits(Long minimumUnits) {
            this.minimumUnits = minimumUnits;
        }

        public final Long getMaximumUnits() {
            return maximumUnits;
        }

        @Override
        public final Builder maximumUnits(Long maximumUnits) {
            this.maximumUnits = maximumUnits;
            return this;
        }

        public final void setMaximumUnits(Long maximumUnits) {
            this.maximumUnits = maximumUnits;
        }

        public final Boolean getAutoScalingDisabled() {
            return autoScalingDisabled;
        }

        @Override
        public final Builder autoScalingDisabled(Boolean autoScalingDisabled) {
            this.autoScalingDisabled = autoScalingDisabled;
            return this;
        }

        public final void setAutoScalingDisabled(Boolean autoScalingDisabled) {
            this.autoScalingDisabled = autoScalingDisabled;
        }

        public final String getAutoScalingRoleArn() {
            return autoScalingRoleArn;
        }

        @Override
        public final Builder autoScalingRoleArn(String autoScalingRoleArn) {
            this.autoScalingRoleArn = autoScalingRoleArn;
            return this;
        }

        public final void setAutoScalingRoleArn(String autoScalingRoleArn) {
            this.autoScalingRoleArn = autoScalingRoleArn;
        }

        public final Collection<AutoScalingPolicyDescription.Builder> getScalingPolicies() {
            return scalingPolicies != null ? scalingPolicies.stream().map(AutoScalingPolicyDescription::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder scalingPolicies(Collection<AutoScalingPolicyDescription> scalingPolicies) {
            this.scalingPolicies = AutoScalingPolicyDescriptionListCopier.copy(scalingPolicies);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder scalingPolicies(AutoScalingPolicyDescription... scalingPolicies) {
            scalingPolicies(Arrays.asList(scalingPolicies));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder scalingPolicies(Consumer<AutoScalingPolicyDescription.Builder>... scalingPolicies) {
            scalingPolicies(Stream.of(scalingPolicies).map(c -> AutoScalingPolicyDescription.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setScalingPolicies(Collection<AutoScalingPolicyDescription.BuilderImpl> scalingPolicies) {
            this.scalingPolicies = AutoScalingPolicyDescriptionListCopier.copyFromBuilder(scalingPolicies);
        }

        @Override
        public AutoScalingSettingsDescription build() {
            return new AutoScalingSettingsDescription(this);
        }
    }
}
