/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.dynamodb.transform.GlobalTableDescriptionMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about the global table.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GlobalTableDescription implements StructuredPojo,
        ToCopyableBuilder<GlobalTableDescription.Builder, GlobalTableDescription> {
    private final List<ReplicaDescription> replicationGroup;

    private final String globalTableArn;

    private final Instant creationDateTime;

    private final String globalTableStatus;

    private final String globalTableName;

    private GlobalTableDescription(BuilderImpl builder) {
        this.replicationGroup = builder.replicationGroup;
        this.globalTableArn = builder.globalTableArn;
        this.creationDateTime = builder.creationDateTime;
        this.globalTableStatus = builder.globalTableStatus;
        this.globalTableName = builder.globalTableName;
    }

    /**
     * <p>
     * The regions where the global table has replicas.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The regions where the global table has replicas.
     */
    public List<ReplicaDescription> replicationGroup() {
        return replicationGroup;
    }

    /**
     * <p>
     * The unique identifier of the global table.
     * </p>
     * 
     * @return The unique identifier of the global table.
     */
    public String globalTableArn() {
        return globalTableArn;
    }

    /**
     * <p>
     * The creation time of the global table.
     * </p>
     * 
     * @return The creation time of the global table.
     */
    public Instant creationDateTime() {
        return creationDateTime;
    }

    /**
     * <p>
     * The current state of the global table:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The global table is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The global table is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The global table is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The global table is ready for use.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #globalTableStatus}
     * will return {@link GlobalTableStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #globalTableStatusAsString}.
     * </p>
     * 
     * @return The current state of the global table:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The global table is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The global table is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The global table is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The global table is ready for use.
     *         </p>
     *         </li>
     * @see GlobalTableStatus
     */
    public GlobalTableStatus globalTableStatus() {
        return GlobalTableStatus.fromValue(globalTableStatus);
    }

    /**
     * <p>
     * The current state of the global table:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The global table is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The global table is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The global table is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The global table is ready for use.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #globalTableStatus}
     * will return {@link GlobalTableStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #globalTableStatusAsString}.
     * </p>
     * 
     * @return The current state of the global table:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The global table is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The global table is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The global table is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The global table is ready for use.
     *         </p>
     *         </li>
     * @see GlobalTableStatus
     */
    public String globalTableStatusAsString() {
        return globalTableStatus;
    }

    /**
     * <p>
     * The global table name.
     * </p>
     * 
     * @return The global table name.
     */
    public String globalTableName() {
        return globalTableName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(replicationGroup());
        hashCode = 31 * hashCode + Objects.hashCode(globalTableArn());
        hashCode = 31 * hashCode + Objects.hashCode(creationDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(globalTableStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(globalTableName());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GlobalTableDescription)) {
            return false;
        }
        GlobalTableDescription other = (GlobalTableDescription) obj;
        return Objects.equals(replicationGroup(), other.replicationGroup())
                && Objects.equals(globalTableArn(), other.globalTableArn())
                && Objects.equals(creationDateTime(), other.creationDateTime())
                && Objects.equals(globalTableStatusAsString(), other.globalTableStatusAsString())
                && Objects.equals(globalTableName(), other.globalTableName());
    }

    @Override
    public String toString() {
        return ToString.builder("GlobalTableDescription").add("ReplicationGroup", replicationGroup())
                .add("GlobalTableArn", globalTableArn()).add("CreationDateTime", creationDateTime())
                .add("GlobalTableStatus", globalTableStatusAsString()).add("GlobalTableName", globalTableName()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ReplicationGroup":
            return Optional.ofNullable(clazz.cast(replicationGroup()));
        case "GlobalTableArn":
            return Optional.ofNullable(clazz.cast(globalTableArn()));
        case "CreationDateTime":
            return Optional.ofNullable(clazz.cast(creationDateTime()));
        case "GlobalTableStatus":
            return Optional.ofNullable(clazz.cast(globalTableStatusAsString()));
        case "GlobalTableName":
            return Optional.ofNullable(clazz.cast(globalTableName()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        GlobalTableDescriptionMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, GlobalTableDescription> {
        /**
         * <p>
         * The regions where the global table has replicas.
         * </p>
         * 
         * @param replicationGroup
         *        The regions where the global table has replicas.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationGroup(Collection<ReplicaDescription> replicationGroup);

        /**
         * <p>
         * The regions where the global table has replicas.
         * </p>
         * 
         * @param replicationGroup
         *        The regions where the global table has replicas.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationGroup(ReplicaDescription... replicationGroup);

        /**
         * <p>
         * The regions where the global table has replicas.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ReplicaDescription>.Builder} avoiding the
         * need to create one manually via {@link List<ReplicaDescription>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ReplicaDescription>.Builder#build()} is called immediately
         * and its result is passed to {@link #replicationGroup(List<ReplicaDescription>)}.
         * 
         * @param replicationGroup
         *        a consumer that will call methods on {@link List<ReplicaDescription>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #replicationGroup(List<ReplicaDescription>)
         */
        Builder replicationGroup(Consumer<ReplicaDescription.Builder>... replicationGroup);

        /**
         * <p>
         * The unique identifier of the global table.
         * </p>
         * 
         * @param globalTableArn
         *        The unique identifier of the global table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder globalTableArn(String globalTableArn);

        /**
         * <p>
         * The creation time of the global table.
         * </p>
         * 
         * @param creationDateTime
         *        The creation time of the global table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDateTime(Instant creationDateTime);

        /**
         * <p>
         * The current state of the global table:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The global table is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The global table is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The global table is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The global table is ready for use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param globalTableStatus
         *        The current state of the global table:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The global table is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The global table is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The global table is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The global table is ready for use.
         *        </p>
         *        </li>
         * @see GlobalTableStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GlobalTableStatus
         */
        Builder globalTableStatus(String globalTableStatus);

        /**
         * <p>
         * The current state of the global table:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The global table is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The global table is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The global table is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The global table is ready for use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param globalTableStatus
         *        The current state of the global table:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The global table is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The global table is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The global table is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The global table is ready for use.
         *        </p>
         *        </li>
         * @see GlobalTableStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GlobalTableStatus
         */
        Builder globalTableStatus(GlobalTableStatus globalTableStatus);

        /**
         * <p>
         * The global table name.
         * </p>
         * 
         * @param globalTableName
         *        The global table name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder globalTableName(String globalTableName);
    }

    static final class BuilderImpl implements Builder {
        private List<ReplicaDescription> replicationGroup = DefaultSdkAutoConstructList.getInstance();

        private String globalTableArn;

        private Instant creationDateTime;

        private String globalTableStatus;

        private String globalTableName;

        private BuilderImpl() {
        }

        private BuilderImpl(GlobalTableDescription model) {
            replicationGroup(model.replicationGroup);
            globalTableArn(model.globalTableArn);
            creationDateTime(model.creationDateTime);
            globalTableStatus(model.globalTableStatus);
            globalTableName(model.globalTableName);
        }

        public final Collection<ReplicaDescription.Builder> getReplicationGroup() {
            return replicationGroup != null ? replicationGroup.stream().map(ReplicaDescription::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder replicationGroup(Collection<ReplicaDescription> replicationGroup) {
            this.replicationGroup = ReplicaDescriptionListCopier.copy(replicationGroup);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder replicationGroup(ReplicaDescription... replicationGroup) {
            replicationGroup(Arrays.asList(replicationGroup));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder replicationGroup(Consumer<ReplicaDescription.Builder>... replicationGroup) {
            replicationGroup(Stream.of(replicationGroup).map(c -> ReplicaDescription.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setReplicationGroup(Collection<ReplicaDescription.BuilderImpl> replicationGroup) {
            this.replicationGroup = ReplicaDescriptionListCopier.copyFromBuilder(replicationGroup);
        }

        public final String getGlobalTableArn() {
            return globalTableArn;
        }

        @Override
        public final Builder globalTableArn(String globalTableArn) {
            this.globalTableArn = globalTableArn;
            return this;
        }

        public final void setGlobalTableArn(String globalTableArn) {
            this.globalTableArn = globalTableArn;
        }

        public final Instant getCreationDateTime() {
            return creationDateTime;
        }

        @Override
        public final Builder creationDateTime(Instant creationDateTime) {
            this.creationDateTime = creationDateTime;
            return this;
        }

        public final void setCreationDateTime(Instant creationDateTime) {
            this.creationDateTime = creationDateTime;
        }

        public final String getGlobalTableStatus() {
            return globalTableStatus;
        }

        @Override
        public final Builder globalTableStatus(String globalTableStatus) {
            this.globalTableStatus = globalTableStatus;
            return this;
        }

        @Override
        public final Builder globalTableStatus(GlobalTableStatus globalTableStatus) {
            this.globalTableStatus(globalTableStatus.toString());
            return this;
        }

        public final void setGlobalTableStatus(String globalTableStatus) {
            this.globalTableStatus = globalTableStatus;
        }

        public final String getGlobalTableName() {
            return globalTableName;
        }

        @Override
        public final Builder globalTableName(String globalTableName) {
            this.globalTableName = globalTableName;
            return this;
        }

        public final void setGlobalTableName(String globalTableName) {
            this.globalTableName = globalTableName;
        }

        @Override
        public GlobalTableDescription build() {
            return new GlobalTableDescription(this);
        }
    }
}
