/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.dynamodb.transform.SSESpecificationMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the settings used to enable server-side encryption.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SSESpecification implements StructuredPojo, ToCopyableBuilder<SSESpecification.Builder, SSESpecification> {
    private final Boolean enabled;

    private final String sseType;

    private final String kmsMasterKeyId;

    private SSESpecification(BuilderImpl builder) {
        this.enabled = builder.enabled;
        this.sseType = builder.sseType;
        this.kmsMasterKeyId = builder.kmsMasterKeyId;
    }

    /**
     * <p>
     * Indicates whether server-side encryption is enabled (true) or disabled (false) on the table.
     * </p>
     * 
     * @return Indicates whether server-side encryption is enabled (true) or disabled (false) on the table.
     */
    public Boolean enabled() {
        return enabled;
    }

    /**
     * <p>
     * Server-side encryption type:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AES256</code> - Server-side encryption which uses the AES256 algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KMS</code> - Server-side encryption which uses AWS Key Management Service. (default)
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sseType} will
     * return {@link SSEType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sseTypeAsString}.
     * </p>
     * 
     * @return Server-side encryption type:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AES256</code> - Server-side encryption which uses the AES256 algorithm.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KMS</code> - Server-side encryption which uses AWS Key Management Service. (default)
     *         </p>
     *         </li>
     * @see SSEType
     */
    public SSEType sseType() {
        return SSEType.fromValue(sseType);
    }

    /**
     * <p>
     * Server-side encryption type:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AES256</code> - Server-side encryption which uses the AES256 algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KMS</code> - Server-side encryption which uses AWS Key Management Service. (default)
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sseType} will
     * return {@link SSEType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sseTypeAsString}.
     * </p>
     * 
     * @return Server-side encryption type:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AES256</code> - Server-side encryption which uses the AES256 algorithm.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KMS</code> - Server-side encryption which uses AWS Key Management Service. (default)
     *         </p>
     *         </li>
     * @see SSEType
     */
    public String sseTypeAsString() {
        return sseType;
    }

    /**
     * <p>
     * The KMS Master Key (CMK) which should be used for the KMS encryption. To specify a CMK, use its key ID, Amazon
     * Resource Name (ARN), alias name, or alias ARN. Note that you should only provide this parameter if the key is
     * different from the default DynamoDB KMS Master Key alias/aws/dynamodb.
     * </p>
     * 
     * @return The KMS Master Key (CMK) which should be used for the KMS encryption. To specify a CMK, use its key ID,
     *         Amazon Resource Name (ARN), alias name, or alias ARN. Note that you should only provide this parameter if
     *         the key is different from the default DynamoDB KMS Master Key alias/aws/dynamodb.
     */
    public String kmsMasterKeyId() {
        return kmsMasterKeyId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(sseTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(kmsMasterKeyId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SSESpecification)) {
            return false;
        }
        SSESpecification other = (SSESpecification) obj;
        return Objects.equals(enabled(), other.enabled()) && Objects.equals(sseTypeAsString(), other.sseTypeAsString())
                && Objects.equals(kmsMasterKeyId(), other.kmsMasterKeyId());
    }

    @Override
    public String toString() {
        return ToString.builder("SSESpecification").add("Enabled", enabled()).add("SSEType", sseTypeAsString())
                .add("KMSMasterKeyId", kmsMasterKeyId()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "SSEType":
            return Optional.ofNullable(clazz.cast(sseTypeAsString()));
        case "KMSMasterKeyId":
            return Optional.ofNullable(clazz.cast(kmsMasterKeyId()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        SSESpecificationMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, SSESpecification> {
        /**
         * <p>
         * Indicates whether server-side encryption is enabled (true) or disabled (false) on the table.
         * </p>
         * 
         * @param enabled
         *        Indicates whether server-side encryption is enabled (true) or disabled (false) on the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * Server-side encryption type:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AES256</code> - Server-side encryption which uses the AES256 algorithm.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KMS</code> - Server-side encryption which uses AWS Key Management Service. (default)
         * </p>
         * </li>
         * </ul>
         * 
         * @param sseType
         *        Server-side encryption type:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AES256</code> - Server-side encryption which uses the AES256 algorithm.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KMS</code> - Server-side encryption which uses AWS Key Management Service. (default)
         *        </p>
         *        </li>
         * @see SSEType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SSEType
         */
        Builder sseType(String sseType);

        /**
         * <p>
         * Server-side encryption type:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AES256</code> - Server-side encryption which uses the AES256 algorithm.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KMS</code> - Server-side encryption which uses AWS Key Management Service. (default)
         * </p>
         * </li>
         * </ul>
         * 
         * @param sseType
         *        Server-side encryption type:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AES256</code> - Server-side encryption which uses the AES256 algorithm.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KMS</code> - Server-side encryption which uses AWS Key Management Service. (default)
         *        </p>
         *        </li>
         * @see SSEType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SSEType
         */
        Builder sseType(SSEType sseType);

        /**
         * <p>
         * The KMS Master Key (CMK) which should be used for the KMS encryption. To specify a CMK, use its key ID,
         * Amazon Resource Name (ARN), alias name, or alias ARN. Note that you should only provide this parameter if the
         * key is different from the default DynamoDB KMS Master Key alias/aws/dynamodb.
         * </p>
         * 
         * @param kmsMasterKeyId
         *        The KMS Master Key (CMK) which should be used for the KMS encryption. To specify a CMK, use its key
         *        ID, Amazon Resource Name (ARN), alias name, or alias ARN. Note that you should only provide this
         *        parameter if the key is different from the default DynamoDB KMS Master Key alias/aws/dynamodb.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsMasterKeyId(String kmsMasterKeyId);
    }

    static final class BuilderImpl implements Builder {
        private Boolean enabled;

        private String sseType;

        private String kmsMasterKeyId;

        private BuilderImpl() {
        }

        private BuilderImpl(SSESpecification model) {
            enabled(model.enabled);
            sseType(model.sseType);
            kmsMasterKeyId(model.kmsMasterKeyId);
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        public final String getSSEType() {
            return sseType;
        }

        @Override
        public final Builder sseType(String sseType) {
            this.sseType = sseType;
            return this;
        }

        @Override
        public final Builder sseType(SSEType sseType) {
            this.sseType(sseType.toString());
            return this;
        }

        public final void setSSEType(String sseType) {
            this.sseType = sseType;
        }

        public final String getKMSMasterKeyId() {
            return kmsMasterKeyId;
        }

        @Override
        public final Builder kmsMasterKeyId(String kmsMasterKeyId) {
            this.kmsMasterKeyId = kmsMasterKeyId;
            return this;
        }

        public final void setKMSMasterKeyId(String kmsMasterKeyId) {
            this.kmsMasterKeyId = kmsMasterKeyId;
        }

        @Override
        public SSESpecification build() {
            return new SSESpecification(this);
        }
    }
}
