/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import javax.annotation.Generated;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.protocol.ProtocolMarshaller;
import software.amazon.awssdk.protocol.StructuredPojo;
import software.amazon.awssdk.services.dynamodb.transform.TableDescriptionMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the properties of a table.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class TableDescription implements StructuredPojo, ToCopyableBuilder<TableDescription.Builder, TableDescription> {
    private final List<AttributeDefinition> attributeDefinitions;

    private final String tableName;

    private final List<KeySchemaElement> keySchema;

    private final String tableStatus;

    private final Instant creationDateTime;

    private final ProvisionedThroughputDescription provisionedThroughput;

    private final Long tableSizeBytes;

    private final Long itemCount;

    private final String tableArn;

    private final List<LocalSecondaryIndexDescription> localSecondaryIndexes;

    private final List<GlobalSecondaryIndexDescription> globalSecondaryIndexes;

    private final StreamSpecification streamSpecification;

    private final String latestStreamLabel;

    private final String latestStreamArn;

    private TableDescription(BuilderImpl builder) {
        this.attributeDefinitions = builder.attributeDefinitions;
        this.tableName = builder.tableName;
        this.keySchema = builder.keySchema;
        this.tableStatus = builder.tableStatus;
        this.creationDateTime = builder.creationDateTime;
        this.provisionedThroughput = builder.provisionedThroughput;
        this.tableSizeBytes = builder.tableSizeBytes;
        this.itemCount = builder.itemCount;
        this.tableArn = builder.tableArn;
        this.localSecondaryIndexes = builder.localSecondaryIndexes;
        this.globalSecondaryIndexes = builder.globalSecondaryIndexes;
        this.streamSpecification = builder.streamSpecification;
        this.latestStreamLabel = builder.latestStreamLabel;
        this.latestStreamArn = builder.latestStreamArn;
    }

    /**
     * <p>
     * An array of <code>AttributeDefinition</code> objects. Each of these objects describes one attribute in the table
     * and index key schema.
     * </p>
     * <p>
     * Each <code>AttributeDefinition</code> object in this array is composed of:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AttributeName</code> - The name of the attribute.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AttributeType</code> - The data type for the attribute.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of <code>AttributeDefinition</code> objects. Each of these objects describes one attribute in
     *         the table and index key schema.</p>
     *         <p>
     *         Each <code>AttributeDefinition</code> object in this array is composed of:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AttributeName</code> - The name of the attribute.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AttributeType</code> - The data type for the attribute.
     *         </p>
     *         </li>
     */
    public List<AttributeDefinition> attributeDefinitions() {
        return attributeDefinitions;
    }

    /**
     * <p>
     * The name of the table.
     * </p>
     * 
     * @return The name of the table.
     */
    public String tableName() {
        return tableName;
    }

    /**
     * <p>
     * The primary key structure for the table. Each <code>KeySchemaElement</code> consists of:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AttributeName</code> - The name of the attribute.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeyType</code> - The role of the attribute:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>HASH</code> - partition key
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RANGE</code> - sort key
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives from
     * DynamoDB' usage of an internal hash function to evenly distribute data items across partitions, based on their
     * partition key values.
     * </p>
     * <p>
     * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from the
     * way DynamoDB stores items with the same partition key physically close together, in sorted order by the sort key
     * value.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * For more information about primary keys, see <a
     * href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataModel.html#DataModelPrimaryKey">Primary
     * Key</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The primary key structure for the table. Each <code>KeySchemaElement</code> consists of:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AttributeName</code> - The name of the attribute.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KeyType</code> - The role of the attribute:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>HASH</code> - partition key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RANGE</code> - sort key
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
     *         derives from DynamoDB' usage of an internal hash function to evenly distribute data items across
     *         partitions, based on their partition key values.
     *         </p>
     *         <p>
     *         The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives
     *         from the way DynamoDB stores items with the same partition key physically close together, in sorted order
     *         by the sort key value.
     *         </p>
     *         </note></li>
     *         </ul>
     *         <p>
     *         For more information about primary keys, see <a href=
     *         "http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataModel.html#DataModelPrimaryKey"
     *         >Primary Key</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     */
    public List<KeySchemaElement> keySchema() {
        return keySchema;
    }

    /**
     * <p>
     * The current state of the table:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The table is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The table is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The table is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The table is ready for use.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The current state of the table:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The table is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The table is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The table is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The table is ready for use.
     *         </p>
     *         </li>
     * @see TableStatus
     */
    public String tableStatus() {
        return tableStatus;
    }

    /**
     * <p>
     * The date and time when the table was created, in <a href="http://www.epochconverter.com/">UNIX epoch time</a>
     * format.
     * </p>
     * 
     * @return The date and time when the table was created, in <a href="http://www.epochconverter.com/">UNIX epoch
     *         time</a> format.
     */
    public Instant creationDateTime() {
        return creationDateTime;
    }

    /**
     * <p>
     * The provisioned throughput settings for the table, consisting of read and write capacity units, along with data
     * about increases and decreases.
     * </p>
     * 
     * @return The provisioned throughput settings for the table, consisting of read and write capacity units, along
     *         with data about increases and decreases.
     */
    public ProvisionedThroughputDescription provisionedThroughput() {
        return provisionedThroughput;
    }

    /**
     * <p>
     * The total size of the specified table, in bytes. DynamoDB updates this value approximately every six hours.
     * Recent changes might not be reflected in this value.
     * </p>
     * 
     * @return The total size of the specified table, in bytes. DynamoDB updates this value approximately every six
     *         hours. Recent changes might not be reflected in this value.
     */
    public Long tableSizeBytes() {
        return tableSizeBytes;
    }

    /**
     * <p>
     * The number of items in the specified table. DynamoDB updates this value approximately every six hours. Recent
     * changes might not be reflected in this value.
     * </p>
     * 
     * @return The number of items in the specified table. DynamoDB updates this value approximately every six hours.
     *         Recent changes might not be reflected in this value.
     */
    public Long itemCount() {
        return itemCount;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that uniquely identifies the table.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that uniquely identifies the table.
     */
    public String tableArn() {
        return tableArn;
    }

    /**
     * <p>
     * Represents one or more local secondary indexes on the table. Each index is scoped to a given partition key value.
     * Tables with one or more local secondary indexes are subject to an item collection size limit, where the amount of
     * data within a given item collection cannot exceed 10 GB. Each element is composed of:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>IndexName</code> - The name of the local secondary index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema must be
     * between 1 and 255 characters (inclusive). The key schema must begin with the same partition key as the table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index. These
     * are in addition to the primary key attributes and index key attributes, which are automatically projected. Each
     * attribute specification is composed of:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ProjectionType</code> - One of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of projected
     * attributes are in <code>NonKeyAttributes</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ALL</code> - All of the table attributes are projected into the index.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
     * secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across all of
     * the secondary indexes, must not exceed 20. If you project the same attribute into two different indexes, this
     * counts as two distinct attributes when determining the total.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>IndexSizeBytes</code> - Represents the total size of the index, in bytes. DynamoDB updates this value
     * approximately every six hours. Recent changes might not be reflected in this value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ItemCount</code> - Represents the number of items in the index. DynamoDB updates this value approximately
     * every six hours. Recent changes might not be reflected in this value.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Represents one or more local secondary indexes on the table. Each index is scoped to a given partition
     *         key value. Tables with one or more local secondary indexes are subject to an item collection size limit,
     *         where the amount of data within a given item collection cannot exceed 10 GB. Each element is composed
     *         of:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>IndexName</code> - The name of the local secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema
     *         must be between 1 and 255 characters (inclusive). The key schema must begin with the same partition key
     *         as the table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index.
     *         These are in addition to the primary key attributes and index key attributes, which are automatically
     *         projected. Each attribute specification is composed of:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ProjectionType</code> - One of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
     *         projected attributes are in <code>NonKeyAttributes</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ALL</code> - All of the table attributes are projected into the index.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
     *         secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across
     *         all of the secondary indexes, must not exceed 20. If you project the same attribute into two different
     *         indexes, this counts as two distinct attributes when determining the total.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IndexSizeBytes</code> - Represents the total size of the index, in bytes. DynamoDB updates this
     *         value approximately every six hours. Recent changes might not be reflected in this value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ItemCount</code> - Represents the number of items in the index. DynamoDB updates this value
     *         approximately every six hours. Recent changes might not be reflected in this value.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
     */
    public List<LocalSecondaryIndexDescription> localSecondaryIndexes() {
        return localSecondaryIndexes;
    }

    /**
     * <p>
     * The global secondary indexes, if any, on the table. Each index is scoped to a given partition key value. Each
     * element is composed of:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Backfilling</code> - If true, then the index is currently in the backfilling phase. Backfilling occurs only
     * when a new global secondary index is added to the table; it is the process by which DynamoDB populates the new
     * index with data from the table. (This attribute does not appear for indexes that were created during a
     * <code>CreateTable</code> operation.)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IndexName</code> - The name of the global secondary index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IndexSizeBytes</code> - The total size of the global secondary index, in bytes. DynamoDB updates this value
     * approximately every six hours. Recent changes might not be reflected in this value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IndexStatus</code> - The current status of the global secondary index:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The index is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The index is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The index is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The index is ready for use.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>ItemCount</code> - The number of items in the global secondary index. DynamoDB updates this value
     * approximately every six hours. Recent changes might not be reflected in this value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema must be
     * between 1 and 255 characters (inclusive). The key schema must begin with the same partition key as the table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index. These
     * are in addition to the primary key attributes and index key attributes, which are automatically projected. Each
     * attribute specification is composed of:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ProjectionType</code> - One of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of projected
     * attributes are in <code>NonKeyAttributes</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ALL</code> - All of the table attributes are projected into the index.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
     * secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across all of
     * the secondary indexes, must not exceed 20. If you project the same attribute into two different indexes, this
     * counts as two distinct attributes when determining the total.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>ProvisionedThroughput</code> - The provisioned throughput settings for the global secondary index,
     * consisting of read and write capacity units, along with data about increases and decreases.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The global secondary indexes, if any, on the table. Each index is scoped to a given partition key value.
     *         Each element is composed of:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Backfilling</code> - If true, then the index is currently in the backfilling phase. Backfilling
     *         occurs only when a new global secondary index is added to the table; it is the process by which DynamoDB
     *         populates the new index with data from the table. (This attribute does not appear for indexes that were
     *         created during a <code>CreateTable</code> operation.)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IndexName</code> - The name of the global secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IndexSizeBytes</code> - The total size of the global secondary index, in bytes. DynamoDB updates
     *         this value approximately every six hours. Recent changes might not be reflected in this value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IndexStatus</code> - The current status of the global secondary index:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The index is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The index is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The index is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The index is ready for use.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ItemCount</code> - The number of items in the global secondary index. DynamoDB updates this value
     *         approximately every six hours. Recent changes might not be reflected in this value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema
     *         must be between 1 and 255 characters (inclusive). The key schema must begin with the same partition key
     *         as the table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index.
     *         These are in addition to the primary key attributes and index key attributes, which are automatically
     *         projected. Each attribute specification is composed of:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ProjectionType</code> - One of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
     *         projected attributes are in <code>NonKeyAttributes</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ALL</code> - All of the table attributes are projected into the index.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
     *         secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across
     *         all of the secondary indexes, must not exceed 20. If you project the same attribute into two different
     *         indexes, this counts as two distinct attributes when determining the total.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ProvisionedThroughput</code> - The provisioned throughput settings for the global secondary index,
     *         consisting of read and write capacity units, along with data about increases and decreases.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
     */
    public List<GlobalSecondaryIndexDescription> globalSecondaryIndexes() {
        return globalSecondaryIndexes;
    }

    /**
     * <p>
     * The current DynamoDB Streams configuration for the table.
     * </p>
     * 
     * @return The current DynamoDB Streams configuration for the table.
     */
    public StreamSpecification streamSpecification() {
        return streamSpecification;
    }

    /**
     * <p>
     * A timestamp, in ISO 8601 format, for this stream.
     * </p>
     * <p>
     * Note that <code>LatestStreamLabel</code> is not a unique identifier for the stream, because it is possible that a
     * stream from another table might have the same timestamp. However, the combination of the following three elements
     * is guaranteed to be unique:
     * </p>
     * <ul>
     * <li>
     * <p>
     * the AWS customer ID.
     * </p>
     * </li>
     * <li>
     * <p>
     * the table name.
     * </p>
     * </li>
     * <li>
     * <p>
     * the <code>StreamLabel</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return A timestamp, in ISO 8601 format, for this stream.</p>
     *         <p>
     *         Note that <code>LatestStreamLabel</code> is not a unique identifier for the stream, because it is
     *         possible that a stream from another table might have the same timestamp. However, the combination of the
     *         following three elements is guaranteed to be unique:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         the AWS customer ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         the table name.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         the <code>StreamLabel</code>.
     *         </p>
     *         </li>
     */
    public String latestStreamLabel() {
        return latestStreamLabel;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that uniquely identifies the latest stream for this table.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that uniquely identifies the latest stream for this table.
     */
    public String latestStreamArn() {
        return latestStreamArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((attributeDefinitions() == null) ? 0 : attributeDefinitions().hashCode());
        hashCode = 31 * hashCode + ((tableName() == null) ? 0 : tableName().hashCode());
        hashCode = 31 * hashCode + ((keySchema() == null) ? 0 : keySchema().hashCode());
        hashCode = 31 * hashCode + ((tableStatus() == null) ? 0 : tableStatus().hashCode());
        hashCode = 31 * hashCode + ((creationDateTime() == null) ? 0 : creationDateTime().hashCode());
        hashCode = 31 * hashCode + ((provisionedThroughput() == null) ? 0 : provisionedThroughput().hashCode());
        hashCode = 31 * hashCode + ((tableSizeBytes() == null) ? 0 : tableSizeBytes().hashCode());
        hashCode = 31 * hashCode + ((itemCount() == null) ? 0 : itemCount().hashCode());
        hashCode = 31 * hashCode + ((tableArn() == null) ? 0 : tableArn().hashCode());
        hashCode = 31 * hashCode + ((localSecondaryIndexes() == null) ? 0 : localSecondaryIndexes().hashCode());
        hashCode = 31 * hashCode + ((globalSecondaryIndexes() == null) ? 0 : globalSecondaryIndexes().hashCode());
        hashCode = 31 * hashCode + ((streamSpecification() == null) ? 0 : streamSpecification().hashCode());
        hashCode = 31 * hashCode + ((latestStreamLabel() == null) ? 0 : latestStreamLabel().hashCode());
        hashCode = 31 * hashCode + ((latestStreamArn() == null) ? 0 : latestStreamArn().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TableDescription)) {
            return false;
        }
        TableDescription other = (TableDescription) obj;
        if (other.attributeDefinitions() == null ^ this.attributeDefinitions() == null) {
            return false;
        }
        if (other.attributeDefinitions() != null && !other.attributeDefinitions().equals(this.attributeDefinitions())) {
            return false;
        }
        if (other.tableName() == null ^ this.tableName() == null) {
            return false;
        }
        if (other.tableName() != null && !other.tableName().equals(this.tableName())) {
            return false;
        }
        if (other.keySchema() == null ^ this.keySchema() == null) {
            return false;
        }
        if (other.keySchema() != null && !other.keySchema().equals(this.keySchema())) {
            return false;
        }
        if (other.tableStatus() == null ^ this.tableStatus() == null) {
            return false;
        }
        if (other.tableStatus() != null && !other.tableStatus().equals(this.tableStatus())) {
            return false;
        }
        if (other.creationDateTime() == null ^ this.creationDateTime() == null) {
            return false;
        }
        if (other.creationDateTime() != null && !other.creationDateTime().equals(this.creationDateTime())) {
            return false;
        }
        if (other.provisionedThroughput() == null ^ this.provisionedThroughput() == null) {
            return false;
        }
        if (other.provisionedThroughput() != null && !other.provisionedThroughput().equals(this.provisionedThroughput())) {
            return false;
        }
        if (other.tableSizeBytes() == null ^ this.tableSizeBytes() == null) {
            return false;
        }
        if (other.tableSizeBytes() != null && !other.tableSizeBytes().equals(this.tableSizeBytes())) {
            return false;
        }
        if (other.itemCount() == null ^ this.itemCount() == null) {
            return false;
        }
        if (other.itemCount() != null && !other.itemCount().equals(this.itemCount())) {
            return false;
        }
        if (other.tableArn() == null ^ this.tableArn() == null) {
            return false;
        }
        if (other.tableArn() != null && !other.tableArn().equals(this.tableArn())) {
            return false;
        }
        if (other.localSecondaryIndexes() == null ^ this.localSecondaryIndexes() == null) {
            return false;
        }
        if (other.localSecondaryIndexes() != null && !other.localSecondaryIndexes().equals(this.localSecondaryIndexes())) {
            return false;
        }
        if (other.globalSecondaryIndexes() == null ^ this.globalSecondaryIndexes() == null) {
            return false;
        }
        if (other.globalSecondaryIndexes() != null && !other.globalSecondaryIndexes().equals(this.globalSecondaryIndexes())) {
            return false;
        }
        if (other.streamSpecification() == null ^ this.streamSpecification() == null) {
            return false;
        }
        if (other.streamSpecification() != null && !other.streamSpecification().equals(this.streamSpecification())) {
            return false;
        }
        if (other.latestStreamLabel() == null ^ this.latestStreamLabel() == null) {
            return false;
        }
        if (other.latestStreamLabel() != null && !other.latestStreamLabel().equals(this.latestStreamLabel())) {
            return false;
        }
        if (other.latestStreamArn() == null ^ this.latestStreamArn() == null) {
            return false;
        }
        if (other.latestStreamArn() != null && !other.latestStreamArn().equals(this.latestStreamArn())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (attributeDefinitions() != null) {
            sb.append("AttributeDefinitions: ").append(attributeDefinitions()).append(",");
        }
        if (tableName() != null) {
            sb.append("TableName: ").append(tableName()).append(",");
        }
        if (keySchema() != null) {
            sb.append("KeySchema: ").append(keySchema()).append(",");
        }
        if (tableStatus() != null) {
            sb.append("TableStatus: ").append(tableStatus()).append(",");
        }
        if (creationDateTime() != null) {
            sb.append("CreationDateTime: ").append(creationDateTime()).append(",");
        }
        if (provisionedThroughput() != null) {
            sb.append("ProvisionedThroughput: ").append(provisionedThroughput()).append(",");
        }
        if (tableSizeBytes() != null) {
            sb.append("TableSizeBytes: ").append(tableSizeBytes()).append(",");
        }
        if (itemCount() != null) {
            sb.append("ItemCount: ").append(itemCount()).append(",");
        }
        if (tableArn() != null) {
            sb.append("TableArn: ").append(tableArn()).append(",");
        }
        if (localSecondaryIndexes() != null) {
            sb.append("LocalSecondaryIndexes: ").append(localSecondaryIndexes()).append(",");
        }
        if (globalSecondaryIndexes() != null) {
            sb.append("GlobalSecondaryIndexes: ").append(globalSecondaryIndexes()).append(",");
        }
        if (streamSpecification() != null) {
            sb.append("StreamSpecification: ").append(streamSpecification()).append(",");
        }
        if (latestStreamLabel() != null) {
            sb.append("LatestStreamLabel: ").append(latestStreamLabel()).append(",");
        }
        if (latestStreamArn() != null) {
            sb.append("LatestStreamArn: ").append(latestStreamArn()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        TableDescriptionMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, TableDescription> {
        /**
         * <p>
         * An array of <code>AttributeDefinition</code> objects. Each of these objects describes one attribute in the
         * table and index key schema.
         * </p>
         * <p>
         * Each <code>AttributeDefinition</code> object in this array is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AttributeName</code> - The name of the attribute.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AttributeType</code> - The data type for the attribute.
         * </p>
         * </li>
         * </ul>
         * 
         * @param attributeDefinitions
         *        An array of <code>AttributeDefinition</code> objects. Each of these objects describes one attribute in
         *        the table and index key schema.</p>
         *        <p>
         *        Each <code>AttributeDefinition</code> object in this array is composed of:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AttributeName</code> - The name of the attribute.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AttributeType</code> - The data type for the attribute.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeDefinitions(Collection<AttributeDefinition> attributeDefinitions);

        /**
         * <p>
         * An array of <code>AttributeDefinition</code> objects. Each of these objects describes one attribute in the
         * table and index key schema.
         * </p>
         * <p>
         * Each <code>AttributeDefinition</code> object in this array is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AttributeName</code> - The name of the attribute.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AttributeType</code> - The data type for the attribute.
         * </p>
         * </li>
         * </ul>
         * 
         * @param attributeDefinitions
         *        An array of <code>AttributeDefinition</code> objects. Each of these objects describes one attribute in
         *        the table and index key schema.</p>
         *        <p>
         *        Each <code>AttributeDefinition</code> object in this array is composed of:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AttributeName</code> - The name of the attribute.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AttributeType</code> - The data type for the attribute.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeDefinitions(AttributeDefinition... attributeDefinitions);

        /**
         * <p>
         * The name of the table.
         * </p>
         * 
         * @param tableName
         *        The name of the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableName(String tableName);

        /**
         * <p>
         * The primary key structure for the table. Each <code>KeySchemaElement</code> consists of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AttributeName</code> - The name of the attribute.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KeyType</code> - The role of the attribute:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB' usage of an internal hash function to evenly distribute data items across partitions, based on
         * their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note></li>
         * </ul>
         * <p>
         * For more information about primary keys, see <a
         * href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataModel.html#DataModelPrimaryKey"
         * >Primary Key</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param keySchema
         *        The primary key structure for the table. Each <code>KeySchemaElement</code> consists of:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AttributeName</code> - The name of the attribute.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KeyType</code> - The role of the attribute:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HASH</code> - partition key
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RANGE</code> - sort key
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
         *        derives from DynamoDB' usage of an internal hash function to evenly distribute data items across
         *        partitions, based on their partition key values.
         *        </p>
         *        <p>
         *        The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute"
         *        derives from the way DynamoDB stores items with the same partition key physically close together, in
         *        sorted order by the sort key value.
         *        </p>
         *        </note></li>
         *        </ul>
         *        <p>
         *        For more information about primary keys, see <a href=
         *        "http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataModel.html#DataModelPrimaryKey"
         *        >Primary Key</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keySchema(Collection<KeySchemaElement> keySchema);

        /**
         * <p>
         * The primary key structure for the table. Each <code>KeySchemaElement</code> consists of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AttributeName</code> - The name of the attribute.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KeyType</code> - The role of the attribute:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB' usage of an internal hash function to evenly distribute data items across partitions, based on
         * their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note></li>
         * </ul>
         * <p>
         * For more information about primary keys, see <a
         * href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataModel.html#DataModelPrimaryKey"
         * >Primary Key</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param keySchema
         *        The primary key structure for the table. Each <code>KeySchemaElement</code> consists of:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AttributeName</code> - The name of the attribute.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KeyType</code> - The role of the attribute:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HASH</code> - partition key
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RANGE</code> - sort key
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
         *        derives from DynamoDB' usage of an internal hash function to evenly distribute data items across
         *        partitions, based on their partition key values.
         *        </p>
         *        <p>
         *        The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute"
         *        derives from the way DynamoDB stores items with the same partition key physically close together, in
         *        sorted order by the sort key value.
         *        </p>
         *        </note></li>
         *        </ul>
         *        <p>
         *        For more information about primary keys, see <a href=
         *        "http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/DataModel.html#DataModelPrimaryKey"
         *        >Primary Key</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keySchema(KeySchemaElement... keySchema);

        /**
         * <p>
         * The current state of the table:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The table is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The table is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The table is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The table is ready for use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tableStatus
         *        The current state of the table:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The table is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The table is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The table is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The table is ready for use.
         *        </p>
         *        </li>
         * @see TableStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TableStatus
         */
        Builder tableStatus(String tableStatus);

        /**
         * <p>
         * The current state of the table:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The table is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The table is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The table is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The table is ready for use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tableStatus
         *        The current state of the table:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The table is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The table is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The table is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The table is ready for use.
         *        </p>
         *        </li>
         * @see TableStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TableStatus
         */
        Builder tableStatus(TableStatus tableStatus);

        /**
         * <p>
         * The date and time when the table was created, in <a href="http://www.epochconverter.com/">UNIX epoch time</a>
         * format.
         * </p>
         * 
         * @param creationDateTime
         *        The date and time when the table was created, in <a href="http://www.epochconverter.com/">UNIX epoch
         *        time</a> format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDateTime(Instant creationDateTime);

        /**
         * <p>
         * The provisioned throughput settings for the table, consisting of read and write capacity units, along with
         * data about increases and decreases.
         * </p>
         * 
         * @param provisionedThroughput
         *        The provisioned throughput settings for the table, consisting of read and write capacity units, along
         *        with data about increases and decreases.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder provisionedThroughput(ProvisionedThroughputDescription provisionedThroughput);

        /**
         * <p>
         * The total size of the specified table, in bytes. DynamoDB updates this value approximately every six hours.
         * Recent changes might not be reflected in this value.
         * </p>
         * 
         * @param tableSizeBytes
         *        The total size of the specified table, in bytes. DynamoDB updates this value approximately every six
         *        hours. Recent changes might not be reflected in this value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableSizeBytes(Long tableSizeBytes);

        /**
         * <p>
         * The number of items in the specified table. DynamoDB updates this value approximately every six hours. Recent
         * changes might not be reflected in this value.
         * </p>
         * 
         * @param itemCount
         *        The number of items in the specified table. DynamoDB updates this value approximately every six hours.
         *        Recent changes might not be reflected in this value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder itemCount(Long itemCount);

        /**
         * <p>
         * The Amazon Resource Name (ARN) that uniquely identifies the table.
         * </p>
         * 
         * @param tableArn
         *        The Amazon Resource Name (ARN) that uniquely identifies the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableArn(String tableArn);

        /**
         * <p>
         * Represents one or more local secondary indexes on the table. Each index is scoped to a given partition key
         * value. Tables with one or more local secondary indexes are subject to an item collection size limit, where
         * the amount of data within a given item collection cannot exceed 10 GB. Each element is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>IndexName</code> - The name of the local secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema must
         * be between 1 and 255 characters (inclusive). The key schema must begin with the same partition key as the
         * table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index.
         * These are in addition to the primary key attributes and index key attributes, which are automatically
         * projected. Each attribute specification is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ProjectionType</code> - One of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         * projected attributes are in <code>NonKeyAttributes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code> - All of the table attributes are projected into the index.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
         * secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across all
         * of the secondary indexes, must not exceed 20. If you project the same attribute into two different indexes,
         * this counts as two distinct attributes when determining the total.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>IndexSizeBytes</code> - Represents the total size of the index, in bytes. DynamoDB updates this value
         * approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ItemCount</code> - Represents the number of items in the index. DynamoDB updates this value
         * approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * </p>
         * 
         * @param localSecondaryIndexes
         *        Represents one or more local secondary indexes on the table. Each index is scoped to a given partition
         *        key value. Tables with one or more local secondary indexes are subject to an item collection size
         *        limit, where the amount of data within a given item collection cannot exceed 10 GB. Each element is
         *        composed of:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>IndexName</code> - The name of the local secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key
         *        schema must be between 1 and 255 characters (inclusive). The key schema must begin with the same
         *        partition key as the table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the
         *        index. These are in addition to the primary key attributes and index key attributes, which are
         *        automatically projected. Each attribute specification is composed of:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ProjectionType</code> - One of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         *        projected attributes are in <code>NonKeyAttributes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code> - All of the table attributes are projected into the index.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into
         *        the secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed
         *        across all of the secondary indexes, must not exceed 20. If you project the same attribute into two
         *        different indexes, this counts as two distinct attributes when determining the total.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IndexSizeBytes</code> - Represents the total size of the index, in bytes. DynamoDB updates this
         *        value approximately every six hours. Recent changes might not be reflected in this value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ItemCount</code> - Represents the number of items in the index. DynamoDB updates this value
         *        approximately every six hours. Recent changes might not be reflected in this value.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localSecondaryIndexes(Collection<LocalSecondaryIndexDescription> localSecondaryIndexes);

        /**
         * <p>
         * Represents one or more local secondary indexes on the table. Each index is scoped to a given partition key
         * value. Tables with one or more local secondary indexes are subject to an item collection size limit, where
         * the amount of data within a given item collection cannot exceed 10 GB. Each element is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>IndexName</code> - The name of the local secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema must
         * be between 1 and 255 characters (inclusive). The key schema must begin with the same partition key as the
         * table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index.
         * These are in addition to the primary key attributes and index key attributes, which are automatically
         * projected. Each attribute specification is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ProjectionType</code> - One of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         * projected attributes are in <code>NonKeyAttributes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code> - All of the table attributes are projected into the index.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
         * secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across all
         * of the secondary indexes, must not exceed 20. If you project the same attribute into two different indexes,
         * this counts as two distinct attributes when determining the total.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>IndexSizeBytes</code> - Represents the total size of the index, in bytes. DynamoDB updates this value
         * approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ItemCount</code> - Represents the number of items in the index. DynamoDB updates this value
         * approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * </p>
         * 
         * @param localSecondaryIndexes
         *        Represents one or more local secondary indexes on the table. Each index is scoped to a given partition
         *        key value. Tables with one or more local secondary indexes are subject to an item collection size
         *        limit, where the amount of data within a given item collection cannot exceed 10 GB. Each element is
         *        composed of:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>IndexName</code> - The name of the local secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key
         *        schema must be between 1 and 255 characters (inclusive). The key schema must begin with the same
         *        partition key as the table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the
         *        index. These are in addition to the primary key attributes and index key attributes, which are
         *        automatically projected. Each attribute specification is composed of:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ProjectionType</code> - One of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         *        projected attributes are in <code>NonKeyAttributes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code> - All of the table attributes are projected into the index.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into
         *        the secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed
         *        across all of the secondary indexes, must not exceed 20. If you project the same attribute into two
         *        different indexes, this counts as two distinct attributes when determining the total.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IndexSizeBytes</code> - Represents the total size of the index, in bytes. DynamoDB updates this
         *        value approximately every six hours. Recent changes might not be reflected in this value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ItemCount</code> - Represents the number of items in the index. DynamoDB updates this value
         *        approximately every six hours. Recent changes might not be reflected in this value.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localSecondaryIndexes(LocalSecondaryIndexDescription... localSecondaryIndexes);

        /**
         * <p>
         * The global secondary indexes, if any, on the table. Each index is scoped to a given partition key value. Each
         * element is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Backfilling</code> - If true, then the index is currently in the backfilling phase. Backfilling occurs
         * only when a new global secondary index is added to the table; it is the process by which DynamoDB populates
         * the new index with data from the table. (This attribute does not appear for indexes that were created during
         * a <code>CreateTable</code> operation.)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IndexName</code> - The name of the global secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IndexSizeBytes</code> - The total size of the global secondary index, in bytes. DynamoDB updates this
         * value approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IndexStatus</code> - The current status of the global secondary index:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The index is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The index is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The index is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The index is ready for use.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>ItemCount</code> - The number of items in the global secondary index. DynamoDB updates this value
         * approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema must
         * be between 1 and 255 characters (inclusive). The key schema must begin with the same partition key as the
         * table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index.
         * These are in addition to the primary key attributes and index key attributes, which are automatically
         * projected. Each attribute specification is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ProjectionType</code> - One of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         * projected attributes are in <code>NonKeyAttributes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code> - All of the table attributes are projected into the index.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
         * secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across all
         * of the secondary indexes, must not exceed 20. If you project the same attribute into two different indexes,
         * this counts as two distinct attributes when determining the total.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>ProvisionedThroughput</code> - The provisioned throughput settings for the global secondary index,
         * consisting of read and write capacity units, along with data about increases and decreases.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * </p>
         * 
         * @param globalSecondaryIndexes
         *        The global secondary indexes, if any, on the table. Each index is scoped to a given partition key
         *        value. Each element is composed of:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Backfilling</code> - If true, then the index is currently in the backfilling phase. Backfilling
         *        occurs only when a new global secondary index is added to the table; it is the process by which
         *        DynamoDB populates the new index with data from the table. (This attribute does not appear for indexes
         *        that were created during a <code>CreateTable</code> operation.)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IndexName</code> - The name of the global secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IndexSizeBytes</code> - The total size of the global secondary index, in bytes. DynamoDB updates
         *        this value approximately every six hours. Recent changes might not be reflected in this value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IndexStatus</code> - The current status of the global secondary index:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The index is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The index is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The index is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The index is ready for use.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ItemCount</code> - The number of items in the global secondary index. DynamoDB updates this
         *        value approximately every six hours. Recent changes might not be reflected in this value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key
         *        schema must be between 1 and 255 characters (inclusive). The key schema must begin with the same
         *        partition key as the table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the
         *        index. These are in addition to the primary key attributes and index key attributes, which are
         *        automatically projected. Each attribute specification is composed of:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ProjectionType</code> - One of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         *        projected attributes are in <code>NonKeyAttributes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code> - All of the table attributes are projected into the index.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into
         *        the secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed
         *        across all of the secondary indexes, must not exceed 20. If you project the same attribute into two
         *        different indexes, this counts as two distinct attributes when determining the total.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ProvisionedThroughput</code> - The provisioned throughput settings for the global secondary
         *        index, consisting of read and write capacity units, along with data about increases and decreases.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder globalSecondaryIndexes(Collection<GlobalSecondaryIndexDescription> globalSecondaryIndexes);

        /**
         * <p>
         * The global secondary indexes, if any, on the table. Each index is scoped to a given partition key value. Each
         * element is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Backfilling</code> - If true, then the index is currently in the backfilling phase. Backfilling occurs
         * only when a new global secondary index is added to the table; it is the process by which DynamoDB populates
         * the new index with data from the table. (This attribute does not appear for indexes that were created during
         * a <code>CreateTable</code> operation.)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IndexName</code> - The name of the global secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IndexSizeBytes</code> - The total size of the global secondary index, in bytes. DynamoDB updates this
         * value approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IndexStatus</code> - The current status of the global secondary index:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The index is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The index is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The index is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The index is ready for use.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>ItemCount</code> - The number of items in the global secondary index. DynamoDB updates this value
         * approximately every six hours. Recent changes might not be reflected in this value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key schema must
         * be between 1 and 255 characters (inclusive). The key schema must begin with the same partition key as the
         * table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the index.
         * These are in addition to the primary key attributes and index key attributes, which are automatically
         * projected. Each attribute specification is composed of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ProjectionType</code> - One of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         * projected attributes are in <code>NonKeyAttributes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code> - All of the table attributes are projected into the index.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into the
         * secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed across all
         * of the secondary indexes, must not exceed 20. If you project the same attribute into two different indexes,
         * this counts as two distinct attributes when determining the total.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>ProvisionedThroughput</code> - The provisioned throughput settings for the global secondary index,
         * consisting of read and write capacity units, along with data about increases and decreases.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * </p>
         * 
         * @param globalSecondaryIndexes
         *        The global secondary indexes, if any, on the table. Each index is scoped to a given partition key
         *        value. Each element is composed of:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Backfilling</code> - If true, then the index is currently in the backfilling phase. Backfilling
         *        occurs only when a new global secondary index is added to the table; it is the process by which
         *        DynamoDB populates the new index with data from the table. (This attribute does not appear for indexes
         *        that were created during a <code>CreateTable</code> operation.)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IndexName</code> - The name of the global secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IndexSizeBytes</code> - The total size of the global secondary index, in bytes. DynamoDB updates
         *        this value approximately every six hours. Recent changes might not be reflected in this value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IndexStatus</code> - The current status of the global secondary index:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The index is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The index is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The index is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The index is ready for use.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ItemCount</code> - The number of items in the global secondary index. DynamoDB updates this
         *        value approximately every six hours. Recent changes might not be reflected in this value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KeySchema</code> - Specifies the complete index key schema. The attribute names in the key
         *        schema must be between 1 and 255 characters (inclusive). The key schema must begin with the same
         *        partition key as the table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Projection</code> - Specifies attributes that are copied (projected) from the table into the
         *        index. These are in addition to the primary key attributes and index key attributes, which are
         *        automatically projected. Each attribute specification is composed of:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ProjectionType</code> - One of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         *        projected attributes are in <code>NonKeyAttributes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code> - All of the table attributes are projected into the index.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NonKeyAttributes</code> - A list of one or more non-key attribute names that are projected into
         *        the secondary index. The total count of attributes provided in <code>NonKeyAttributes</code>, summed
         *        across all of the secondary indexes, must not exceed 20. If you project the same attribute into two
         *        different indexes, this counts as two distinct attributes when determining the total.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ProvisionedThroughput</code> - The provisioned throughput settings for the global secondary
         *        index, consisting of read and write capacity units, along with data about increases and decreases.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If the table is in the <code>DELETING</code> state, no information about indexes will be returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder globalSecondaryIndexes(GlobalSecondaryIndexDescription... globalSecondaryIndexes);

        /**
         * <p>
         * The current DynamoDB Streams configuration for the table.
         * </p>
         * 
         * @param streamSpecification
         *        The current DynamoDB Streams configuration for the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamSpecification(StreamSpecification streamSpecification);

        /**
         * <p>
         * A timestamp, in ISO 8601 format, for this stream.
         * </p>
         * <p>
         * Note that <code>LatestStreamLabel</code> is not a unique identifier for the stream, because it is possible
         * that a stream from another table might have the same timestamp. However, the combination of the following
         * three elements is guaranteed to be unique:
         * </p>
         * <ul>
         * <li>
         * <p>
         * the AWS customer ID.
         * </p>
         * </li>
         * <li>
         * <p>
         * the table name.
         * </p>
         * </li>
         * <li>
         * <p>
         * the <code>StreamLabel</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param latestStreamLabel
         *        A timestamp, in ISO 8601 format, for this stream.</p>
         *        <p>
         *        Note that <code>LatestStreamLabel</code> is not a unique identifier for the stream, because it is
         *        possible that a stream from another table might have the same timestamp. However, the combination of
         *        the following three elements is guaranteed to be unique:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        the AWS customer ID.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        the table name.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        the <code>StreamLabel</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder latestStreamLabel(String latestStreamLabel);

        /**
         * <p>
         * The Amazon Resource Name (ARN) that uniquely identifies the latest stream for this table.
         * </p>
         * 
         * @param latestStreamArn
         *        The Amazon Resource Name (ARN) that uniquely identifies the latest stream for this table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder latestStreamArn(String latestStreamArn);
    }

    private static final class BuilderImpl implements Builder {
        private List<AttributeDefinition> attributeDefinitions;

        private String tableName;

        private List<KeySchemaElement> keySchema;

        private String tableStatus;

        private Instant creationDateTime;

        private ProvisionedThroughputDescription provisionedThroughput;

        private Long tableSizeBytes;

        private Long itemCount;

        private String tableArn;

        private List<LocalSecondaryIndexDescription> localSecondaryIndexes;

        private List<GlobalSecondaryIndexDescription> globalSecondaryIndexes;

        private StreamSpecification streamSpecification;

        private String latestStreamLabel;

        private String latestStreamArn;

        private BuilderImpl() {
        }

        private BuilderImpl(TableDescription model) {
            setAttributeDefinitions(model.attributeDefinitions);
            setTableName(model.tableName);
            setKeySchema(model.keySchema);
            setTableStatus(model.tableStatus);
            setCreationDateTime(model.creationDateTime);
            setProvisionedThroughput(model.provisionedThroughput);
            setTableSizeBytes(model.tableSizeBytes);
            setItemCount(model.itemCount);
            setTableArn(model.tableArn);
            setLocalSecondaryIndexes(model.localSecondaryIndexes);
            setGlobalSecondaryIndexes(model.globalSecondaryIndexes);
            setStreamSpecification(model.streamSpecification);
            setLatestStreamLabel(model.latestStreamLabel);
            setLatestStreamArn(model.latestStreamArn);
        }

        public final Collection<AttributeDefinition> getAttributeDefinitions() {
            return attributeDefinitions;
        }

        @Override
        public final Builder attributeDefinitions(Collection<AttributeDefinition> attributeDefinitions) {
            this.attributeDefinitions = AttributeDefinitionsCopier.copy(attributeDefinitions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributeDefinitions(AttributeDefinition... attributeDefinitions) {
            attributeDefinitions(Arrays.asList(attributeDefinitions));
            return this;
        }

        public final void setAttributeDefinitions(Collection<AttributeDefinition> attributeDefinitions) {
            this.attributeDefinitions = AttributeDefinitionsCopier.copy(attributeDefinitions);
        }

        @SafeVarargs
        public final void setAttributeDefinitions(AttributeDefinition... attributeDefinitions) {
            attributeDefinitions(Arrays.asList(attributeDefinitions));
        }

        public final String getTableName() {
            return tableName;
        }

        @Override
        public final Builder tableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        public final void setTableName(String tableName) {
            this.tableName = tableName;
        }

        public final Collection<KeySchemaElement> getKeySchema() {
            return keySchema;
        }

        @Override
        public final Builder keySchema(Collection<KeySchemaElement> keySchema) {
            this.keySchema = KeySchemaCopier.copy(keySchema);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder keySchema(KeySchemaElement... keySchema) {
            keySchema(Arrays.asList(keySchema));
            return this;
        }

        public final void setKeySchema(Collection<KeySchemaElement> keySchema) {
            this.keySchema = KeySchemaCopier.copy(keySchema);
        }

        @SafeVarargs
        public final void setKeySchema(KeySchemaElement... keySchema) {
            keySchema(Arrays.asList(keySchema));
        }

        public final String getTableStatus() {
            return tableStatus;
        }

        @Override
        public final Builder tableStatus(String tableStatus) {
            this.tableStatus = tableStatus;
            return this;
        }

        @Override
        public final Builder tableStatus(TableStatus tableStatus) {
            this.tableStatus(tableStatus.toString());
            return this;
        }

        public final void setTableStatus(String tableStatus) {
            this.tableStatus = tableStatus;
        }

        public final void setTableStatus(TableStatus tableStatus) {
            this.tableStatus(tableStatus.toString());
        }

        public final Instant getCreationDateTime() {
            return creationDateTime;
        }

        @Override
        public final Builder creationDateTime(Instant creationDateTime) {
            this.creationDateTime = creationDateTime;
            return this;
        }

        public final void setCreationDateTime(Instant creationDateTime) {
            this.creationDateTime = creationDateTime;
        }

        public final ProvisionedThroughputDescription getProvisionedThroughput() {
            return provisionedThroughput;
        }

        @Override
        public final Builder provisionedThroughput(ProvisionedThroughputDescription provisionedThroughput) {
            this.provisionedThroughput = provisionedThroughput;
            return this;
        }

        public final void setProvisionedThroughput(ProvisionedThroughputDescription provisionedThroughput) {
            this.provisionedThroughput = provisionedThroughput;
        }

        public final Long getTableSizeBytes() {
            return tableSizeBytes;
        }

        @Override
        public final Builder tableSizeBytes(Long tableSizeBytes) {
            this.tableSizeBytes = tableSizeBytes;
            return this;
        }

        public final void setTableSizeBytes(Long tableSizeBytes) {
            this.tableSizeBytes = tableSizeBytes;
        }

        public final Long getItemCount() {
            return itemCount;
        }

        @Override
        public final Builder itemCount(Long itemCount) {
            this.itemCount = itemCount;
            return this;
        }

        public final void setItemCount(Long itemCount) {
            this.itemCount = itemCount;
        }

        public final String getTableArn() {
            return tableArn;
        }

        @Override
        public final Builder tableArn(String tableArn) {
            this.tableArn = tableArn;
            return this;
        }

        public final void setTableArn(String tableArn) {
            this.tableArn = tableArn;
        }

        public final Collection<LocalSecondaryIndexDescription> getLocalSecondaryIndexes() {
            return localSecondaryIndexes;
        }

        @Override
        public final Builder localSecondaryIndexes(Collection<LocalSecondaryIndexDescription> localSecondaryIndexes) {
            this.localSecondaryIndexes = LocalSecondaryIndexDescriptionListCopier.copy(localSecondaryIndexes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder localSecondaryIndexes(LocalSecondaryIndexDescription... localSecondaryIndexes) {
            localSecondaryIndexes(Arrays.asList(localSecondaryIndexes));
            return this;
        }

        public final void setLocalSecondaryIndexes(Collection<LocalSecondaryIndexDescription> localSecondaryIndexes) {
            this.localSecondaryIndexes = LocalSecondaryIndexDescriptionListCopier.copy(localSecondaryIndexes);
        }

        @SafeVarargs
        public final void setLocalSecondaryIndexes(LocalSecondaryIndexDescription... localSecondaryIndexes) {
            localSecondaryIndexes(Arrays.asList(localSecondaryIndexes));
        }

        public final Collection<GlobalSecondaryIndexDescription> getGlobalSecondaryIndexes() {
            return globalSecondaryIndexes;
        }

        @Override
        public final Builder globalSecondaryIndexes(Collection<GlobalSecondaryIndexDescription> globalSecondaryIndexes) {
            this.globalSecondaryIndexes = GlobalSecondaryIndexDescriptionListCopier.copy(globalSecondaryIndexes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder globalSecondaryIndexes(GlobalSecondaryIndexDescription... globalSecondaryIndexes) {
            globalSecondaryIndexes(Arrays.asList(globalSecondaryIndexes));
            return this;
        }

        public final void setGlobalSecondaryIndexes(Collection<GlobalSecondaryIndexDescription> globalSecondaryIndexes) {
            this.globalSecondaryIndexes = GlobalSecondaryIndexDescriptionListCopier.copy(globalSecondaryIndexes);
        }

        @SafeVarargs
        public final void setGlobalSecondaryIndexes(GlobalSecondaryIndexDescription... globalSecondaryIndexes) {
            globalSecondaryIndexes(Arrays.asList(globalSecondaryIndexes));
        }

        public final StreamSpecification getStreamSpecification() {
            return streamSpecification;
        }

        @Override
        public final Builder streamSpecification(StreamSpecification streamSpecification) {
            this.streamSpecification = streamSpecification;
            return this;
        }

        public final void setStreamSpecification(StreamSpecification streamSpecification) {
            this.streamSpecification = streamSpecification;
        }

        public final String getLatestStreamLabel() {
            return latestStreamLabel;
        }

        @Override
        public final Builder latestStreamLabel(String latestStreamLabel) {
            this.latestStreamLabel = latestStreamLabel;
            return this;
        }

        public final void setLatestStreamLabel(String latestStreamLabel) {
            this.latestStreamLabel = latestStreamLabel;
        }

        public final String getLatestStreamArn() {
            return latestStreamArn;
        }

        @Override
        public final Builder latestStreamArn(String latestStreamArn) {
            this.latestStreamArn = latestStreamArn;
            return this;
        }

        public final void setLatestStreamArn(String latestStreamArn) {
            this.latestStreamArn = latestStreamArn;
        }

        @Override
        public TableDescription build() {
            return new TableDescription(this);
        }
    }
}
