/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.util.Map;
import javax.annotation.Generated;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.protocol.ProtocolMarshaller;
import software.amazon.awssdk.protocol.StructuredPojo;
import software.amazon.awssdk.services.dynamodb.transform.ConsumedCapacityMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The capacity units consumed by an operation. The data returned includes the total provisioned throughput consumed,
 * along with statistics for the table and any indexes involved in the operation. <code>ConsumedCapacity</code> is only
 * returned if the request asked for it. For more information, see <a
 * href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ProvisionedThroughputIntro.html">Provisioned
 * Throughput</a> in the <i>Amazon DynamoDB Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ConsumedCapacity implements StructuredPojo, ToCopyableBuilder<ConsumedCapacity.Builder, ConsumedCapacity> {
    private final String tableName;

    private final Double capacityUnits;

    private final Capacity table;

    private final Map<String, Capacity> localSecondaryIndexes;

    private final Map<String, Capacity> globalSecondaryIndexes;

    private ConsumedCapacity(BuilderImpl builder) {
        this.tableName = builder.tableName;
        this.capacityUnits = builder.capacityUnits;
        this.table = builder.table;
        this.localSecondaryIndexes = builder.localSecondaryIndexes;
        this.globalSecondaryIndexes = builder.globalSecondaryIndexes;
    }

    /**
     * <p>
     * The name of the table that was affected by the operation.
     * </p>
     * 
     * @return The name of the table that was affected by the operation.
     */
    public String tableName() {
        return tableName;
    }

    /**
     * <p>
     * The total number of capacity units consumed by the operation.
     * </p>
     * 
     * @return The total number of capacity units consumed by the operation.
     */
    public Double capacityUnits() {
        return capacityUnits;
    }

    /**
     * <p>
     * The amount of throughput consumed on the table affected by the operation.
     * </p>
     * 
     * @return The amount of throughput consumed on the table affected by the operation.
     */
    public Capacity table() {
        return table;
    }

    /**
     * <p>
     * The amount of throughput consumed on each local index affected by the operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The amount of throughput consumed on each local index affected by the operation.
     */
    public Map<String, Capacity> localSecondaryIndexes() {
        return localSecondaryIndexes;
    }

    /**
     * <p>
     * The amount of throughput consumed on each global index affected by the operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The amount of throughput consumed on each global index affected by the operation.
     */
    public Map<String, Capacity> globalSecondaryIndexes() {
        return globalSecondaryIndexes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((tableName() == null) ? 0 : tableName().hashCode());
        hashCode = 31 * hashCode + ((capacityUnits() == null) ? 0 : capacityUnits().hashCode());
        hashCode = 31 * hashCode + ((table() == null) ? 0 : table().hashCode());
        hashCode = 31 * hashCode + ((localSecondaryIndexes() == null) ? 0 : localSecondaryIndexes().hashCode());
        hashCode = 31 * hashCode + ((globalSecondaryIndexes() == null) ? 0 : globalSecondaryIndexes().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConsumedCapacity)) {
            return false;
        }
        ConsumedCapacity other = (ConsumedCapacity) obj;
        if (other.tableName() == null ^ this.tableName() == null) {
            return false;
        }
        if (other.tableName() != null && !other.tableName().equals(this.tableName())) {
            return false;
        }
        if (other.capacityUnits() == null ^ this.capacityUnits() == null) {
            return false;
        }
        if (other.capacityUnits() != null && !other.capacityUnits().equals(this.capacityUnits())) {
            return false;
        }
        if (other.table() == null ^ this.table() == null) {
            return false;
        }
        if (other.table() != null && !other.table().equals(this.table())) {
            return false;
        }
        if (other.localSecondaryIndexes() == null ^ this.localSecondaryIndexes() == null) {
            return false;
        }
        if (other.localSecondaryIndexes() != null && !other.localSecondaryIndexes().equals(this.localSecondaryIndexes())) {
            return false;
        }
        if (other.globalSecondaryIndexes() == null ^ this.globalSecondaryIndexes() == null) {
            return false;
        }
        if (other.globalSecondaryIndexes() != null && !other.globalSecondaryIndexes().equals(this.globalSecondaryIndexes())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (tableName() != null) {
            sb.append("TableName: ").append(tableName()).append(",");
        }
        if (capacityUnits() != null) {
            sb.append("CapacityUnits: ").append(capacityUnits()).append(",");
        }
        if (table() != null) {
            sb.append("Table: ").append(table()).append(",");
        }
        if (localSecondaryIndexes() != null) {
            sb.append("LocalSecondaryIndexes: ").append(localSecondaryIndexes()).append(",");
        }
        if (globalSecondaryIndexes() != null) {
            sb.append("GlobalSecondaryIndexes: ").append(globalSecondaryIndexes()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ConsumedCapacityMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ConsumedCapacity> {
        /**
         * <p>
         * The name of the table that was affected by the operation.
         * </p>
         * 
         * @param tableName
         *        The name of the table that was affected by the operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableName(String tableName);

        /**
         * <p>
         * The total number of capacity units consumed by the operation.
         * </p>
         * 
         * @param capacityUnits
         *        The total number of capacity units consumed by the operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityUnits(Double capacityUnits);

        /**
         * <p>
         * The amount of throughput consumed on the table affected by the operation.
         * </p>
         * 
         * @param table
         *        The amount of throughput consumed on the table affected by the operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder table(Capacity table);

        /**
         * <p>
         * The amount of throughput consumed on each local index affected by the operation.
         * </p>
         * 
         * @param localSecondaryIndexes
         *        The amount of throughput consumed on each local index affected by the operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localSecondaryIndexes(Map<String, Capacity> localSecondaryIndexes);

        /**
         * <p>
         * The amount of throughput consumed on each global index affected by the operation.
         * </p>
         * 
         * @param globalSecondaryIndexes
         *        The amount of throughput consumed on each global index affected by the operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder globalSecondaryIndexes(Map<String, Capacity> globalSecondaryIndexes);
    }

    private static final class BuilderImpl implements Builder {
        private String tableName;

        private Double capacityUnits;

        private Capacity table;

        private Map<String, Capacity> localSecondaryIndexes;

        private Map<String, Capacity> globalSecondaryIndexes;

        private BuilderImpl() {
        }

        private BuilderImpl(ConsumedCapacity model) {
            setTableName(model.tableName);
            setCapacityUnits(model.capacityUnits);
            setTable(model.table);
            setLocalSecondaryIndexes(model.localSecondaryIndexes);
            setGlobalSecondaryIndexes(model.globalSecondaryIndexes);
        }

        public final String getTableName() {
            return tableName;
        }

        @Override
        public final Builder tableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        public final void setTableName(String tableName) {
            this.tableName = tableName;
        }

        public final Double getCapacityUnits() {
            return capacityUnits;
        }

        @Override
        public final Builder capacityUnits(Double capacityUnits) {
            this.capacityUnits = capacityUnits;
            return this;
        }

        public final void setCapacityUnits(Double capacityUnits) {
            this.capacityUnits = capacityUnits;
        }

        public final Capacity getTable() {
            return table;
        }

        @Override
        public final Builder table(Capacity table) {
            this.table = table;
            return this;
        }

        public final void setTable(Capacity table) {
            this.table = table;
        }

        public final Map<String, Capacity> getLocalSecondaryIndexes() {
            return localSecondaryIndexes;
        }

        @Override
        public final Builder localSecondaryIndexes(Map<String, Capacity> localSecondaryIndexes) {
            this.localSecondaryIndexes = SecondaryIndexesCapacityMapCopier.copy(localSecondaryIndexes);
            return this;
        }

        public final void setLocalSecondaryIndexes(Map<String, Capacity> localSecondaryIndexes) {
            this.localSecondaryIndexes = SecondaryIndexesCapacityMapCopier.copy(localSecondaryIndexes);
        }

        public final Map<String, Capacity> getGlobalSecondaryIndexes() {
            return globalSecondaryIndexes;
        }

        @Override
        public final Builder globalSecondaryIndexes(Map<String, Capacity> globalSecondaryIndexes) {
            this.globalSecondaryIndexes = SecondaryIndexesCapacityMapCopier.copy(globalSecondaryIndexes);
            return this;
        }

        public final void setGlobalSecondaryIndexes(Map<String, Capacity> globalSecondaryIndexes) {
            this.globalSecondaryIndexes = SecondaryIndexesCapacityMapCopier.copy(globalSecondaryIndexes);
        }

        @Override
        public ConsumedCapacity build() {
            return new ConsumedCapacity(this);
        }
    }
}
