/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import javax.annotation.Generated;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.protocol.ProtocolMarshaller;
import software.amazon.awssdk.protocol.StructuredPojo;
import software.amazon.awssdk.services.dynamodb.transform.ProjectionMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents attributes that are copied (projected) from the table into an index. These are in addition to the primary
 * key attributes and index key attributes, which are automatically projected.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Projection implements StructuredPojo, ToCopyableBuilder<Projection.Builder, Projection> {
    private final String projectionType;

    private final List<String> nonKeyAttributes;

    private Projection(BuilderImpl builder) {
        this.projectionType = builder.projectionType;
        this.nonKeyAttributes = builder.nonKeyAttributes;
    }

    /**
     * <p>
     * The set of attributes that are projected into the index:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of projected
     * attributes are in <code>NonKeyAttributes</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ALL</code> - All of the table attributes are projected into the index.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The set of attributes that are projected into the index:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
     *         projected attributes are in <code>NonKeyAttributes</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ALL</code> - All of the table attributes are projected into the index.
     *         </p>
     *         </li>
     * @see ProjectionType
     */
    public String projectionType() {
        return projectionType;
    }

    /**
     * <p>
     * Represents the non-key attribute names which will be projected into the index.
     * </p>
     * <p>
     * For local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across all of the local
     * secondary indexes, must not exceed 20. If you project the same attribute into two different indexes, this counts
     * as two distinct attributes when determining the total.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Represents the non-key attribute names which will be projected into the index.</p>
     *         <p>
     *         For local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across all of the
     *         local secondary indexes, must not exceed 20. If you project the same attribute into two different
     *         indexes, this counts as two distinct attributes when determining the total.
     */
    public List<String> nonKeyAttributes() {
        return nonKeyAttributes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((projectionType() == null) ? 0 : projectionType().hashCode());
        hashCode = 31 * hashCode + ((nonKeyAttributes() == null) ? 0 : nonKeyAttributes().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Projection)) {
            return false;
        }
        Projection other = (Projection) obj;
        if (other.projectionType() == null ^ this.projectionType() == null) {
            return false;
        }
        if (other.projectionType() != null && !other.projectionType().equals(this.projectionType())) {
            return false;
        }
        if (other.nonKeyAttributes() == null ^ this.nonKeyAttributes() == null) {
            return false;
        }
        if (other.nonKeyAttributes() != null && !other.nonKeyAttributes().equals(this.nonKeyAttributes())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (projectionType() != null) {
            sb.append("ProjectionType: ").append(projectionType()).append(",");
        }
        if (nonKeyAttributes() != null) {
            sb.append("NonKeyAttributes: ").append(nonKeyAttributes()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ProjectionMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Projection> {
        /**
         * <p>
         * The set of attributes that are projected into the index:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         * projected attributes are in <code>NonKeyAttributes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code> - All of the table attributes are projected into the index.
         * </p>
         * </li>
         * </ul>
         * 
         * @param projectionType
         *        The set of attributes that are projected into the index:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         *        projected attributes are in <code>NonKeyAttributes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code> - All of the table attributes are projected into the index.
         *        </p>
         *        </li>
         * @see ProjectionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProjectionType
         */
        Builder projectionType(String projectionType);

        /**
         * <p>
         * The set of attributes that are projected into the index:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         * projected attributes are in <code>NonKeyAttributes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code> - All of the table attributes are projected into the index.
         * </p>
         * </li>
         * </ul>
         * 
         * @param projectionType
         *        The set of attributes that are projected into the index:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         *        projected attributes are in <code>NonKeyAttributes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code> - All of the table attributes are projected into the index.
         *        </p>
         *        </li>
         * @see ProjectionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProjectionType
         */
        Builder projectionType(ProjectionType projectionType);

        /**
         * <p>
         * Represents the non-key attribute names which will be projected into the index.
         * </p>
         * <p>
         * For local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across all of the local
         * secondary indexes, must not exceed 20. If you project the same attribute into two different indexes, this
         * counts as two distinct attributes when determining the total.
         * </p>
         * 
         * @param nonKeyAttributes
         *        Represents the non-key attribute names which will be projected into the index.</p>
         *        <p>
         *        For local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across all of the
         *        local secondary indexes, must not exceed 20. If you project the same attribute into two different
         *        indexes, this counts as two distinct attributes when determining the total.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nonKeyAttributes(Collection<String> nonKeyAttributes);

        /**
         * <p>
         * Represents the non-key attribute names which will be projected into the index.
         * </p>
         * <p>
         * For local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across all of the local
         * secondary indexes, must not exceed 20. If you project the same attribute into two different indexes, this
         * counts as two distinct attributes when determining the total.
         * </p>
         * 
         * @param nonKeyAttributes
         *        Represents the non-key attribute names which will be projected into the index.</p>
         *        <p>
         *        For local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across all of the
         *        local secondary indexes, must not exceed 20. If you project the same attribute into two different
         *        indexes, this counts as two distinct attributes when determining the total.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nonKeyAttributes(String... nonKeyAttributes);
    }

    private static final class BuilderImpl implements Builder {
        private String projectionType;

        private List<String> nonKeyAttributes;

        private BuilderImpl() {
        }

        private BuilderImpl(Projection model) {
            setProjectionType(model.projectionType);
            setNonKeyAttributes(model.nonKeyAttributes);
        }

        public final String getProjectionType() {
            return projectionType;
        }

        @Override
        public final Builder projectionType(String projectionType) {
            this.projectionType = projectionType;
            return this;
        }

        @Override
        public final Builder projectionType(ProjectionType projectionType) {
            this.projectionType(projectionType.toString());
            return this;
        }

        public final void setProjectionType(String projectionType) {
            this.projectionType = projectionType;
        }

        public final Collection<String> getNonKeyAttributes() {
            return nonKeyAttributes;
        }

        @Override
        public final Builder nonKeyAttributes(Collection<String> nonKeyAttributes) {
            this.nonKeyAttributes = NonKeyAttributeNameListCopier.copy(nonKeyAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nonKeyAttributes(String... nonKeyAttributes) {
            nonKeyAttributes(Arrays.asList(nonKeyAttributes));
            return this;
        }

        public final void setNonKeyAttributes(Collection<String> nonKeyAttributes) {
            this.nonKeyAttributes = NonKeyAttributeNameListCopier.copy(nonKeyAttributes);
        }

        @Override
        public Projection build() {
            return new Projection(this);
        }
    }
}
