/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import javax.annotation.Generated;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.protocol.ProtocolMarshaller;
import software.amazon.awssdk.protocol.StructuredPojo;
import software.amazon.awssdk.services.dynamodb.transform.ConditionMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the selection criteria for a <code>Query</code> or <code>Scan</code> operation:
 * </p>
 * <ul>
 * <li>
 * <p>
 * For a <code>Query</code> operation, <code>Condition</code> is used for specifying the <code>KeyConditions</code> to
 * use when querying a table or an index. For <code>KeyConditions</code>, only the following comparison operators are
 * supported:
 * </p>
 * <p>
 * <code>EQ | LE | LT | GE | GT | BEGINS_WITH | BETWEEN</code>
 * </p>
 * <p>
 * <code>Condition</code> is also used in a <code>QueryFilter</code>, which evaluates the query results and returns only
 * the desired values.
 * </p>
 * </li>
 * <li>
 * <p>
 * For a <code>Scan</code> operation, <code>Condition</code> is used in a <code>ScanFilter</code>, which evaluates the
 * scan results and returns only the desired values.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public class Condition implements StructuredPojo, ToCopyableBuilder<Condition.Builder, Condition> {
    private final List<AttributeValue> attributeValueList;

    private final String comparisonOperator;

    private Condition(BuilderImpl builder) {
        this.attributeValueList = builder.attributeValueList;
        this.comparisonOperator = builder.comparisonOperator;
    }

    /**
     * <p>
     * One or more values to evaluate against the supplied attribute. The number of values in the list depends on the
     * <code>ComparisonOperator</code> being used.
     * </p>
     * <p>
     * For type Number, value comparisons are numeric.
     * </p>
     * <p>
     * String value comparisons for greater than, equals, or less than are based on ASCII character code values. For
     * example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than <code>B</code>. For a
     * list of code values, see <a
     * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">http://en.wikipedia
     * .org/wiki/ASCII#ASCII_printable_characters</a>.
     * </p>
     * <p>
     * For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return One or more values to evaluate against the supplied attribute. The number of values in the list depends
     *         on the <code>ComparisonOperator</code> being used.</p>
     *         <p>
     *         For type Number, value comparisons are numeric.
     *         </p>
     *         <p>
     *         String value comparisons for greater than, equals, or less than are based on ASCII character code values.
     *         For example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than
     *         <code>B</code>. For a list of code values, see <a
     *         href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
     *         >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
     *         </p>
     *         <p>
     *         For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
     */
    public List<AttributeValue> attributeValueList() {
        return attributeValueList;
    }

    /**
     * <p>
     * A comparator for evaluating attributes. For example, equals, greater than, less than, etc.
     * </p>
     * <p>
     * The following comparison operators are available:
     * </p>
     * <p>
     * <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
     * </p>
     * <p>
     * The following are descriptions of each comparison operator.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> element of a
     * different type than the one provided in the request, the value does not match. For example,
     * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
     * <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and maps.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number, Binary,
     * String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> of a different type
     * than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not
     * equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>LE</code> : Less than or equal.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
     * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>LT</code> : Less than.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number, or
     * Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than the
     * one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>GE</code> : Greater than or equal.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
     * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>GT</code> : Greater than.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
     * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types, including
     * lists and maps.
     * </p>
     * <note>
     * <p>
     * This operator tests for the existence of an attribute, not its data type. If the data type of attribute "
     * <code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a Boolean
     * <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type is not relevant to
     * the <code>NOT_NULL</code> comparison operator.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types, including
     * lists and maps.
     * </p>
     * <note>
     * <p>
     * This operator tests for the nonexistence of an attribute, not its data type. If the data type of attribute "
     * <code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a Boolean <code>false</code>.
     * This is because the attribute "<code>a</code>" exists; its data type is not relevant to the <code>NULL</code>
     * comparison operator.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If the target attribute of the comparison is of type String, then the operator checks
     * for a substring match. If the target attribute of the comparison is of type Binary, then the operator looks for a
     * subsequence of the target that matches the input. If the target attribute of the comparison is a set ("
     * <code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator evaluates to true if it finds an
     * exact match with any member of the set.
     * </p>
     * <p>
     * CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
     * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If the target attribute of the comparison is a String, then the operator checks for
     * the absence of a substring match. If the target attribute of the comparison is Binary, then the operator checks
     * for the absence of a subsequence of the target that matches the input. If the target attribute of the comparison
     * is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator evaluates to true if it
     * <i>does not</i> find an exact match with any member of the set.
     * </p>
     * <p>
     * NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
     * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BEGINS_WITH</code> : Checks for a prefix.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or Binary (not a
     * Number or a set type). The target attribute of the comparison must be of type String or Binary (not a Number or a
     * set type).
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>IN</code> : Checks for matching elements in a list.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type String,
     * Number, or Binary. These attributes are compared against an existing attribute of an item. If any elements of the
     * input are equal to the item attribute, the expression evaluates to true.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second value.
     * </p>
     * <p>
     * <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same type, either
     * String, Number, or Binary (not a set type). A target attribute matches if the target value is greater than, or
     * equal to, the first element and less than, or equal to, the second element. If an item contains an
     * <code>AttributeValue</code> element of a different type than the one provided in the request, the value does not
     * match. For example, <code>{"S":"6"}</code> does not compare to <code>{"N":"6"}</code>. Also,
     * <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For usage examples of <code>AttributeValueList</code> and <code>ComparisonOperator</code>, see <a
     * href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.html">Legacy
     * Conditional Parameters</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * 
     * @return A comparator for evaluating attributes. For example, equals, greater than, less than, etc.</p>
     *         <p>
     *         The following comparison operators are available:
     *         </p>
     *         <p>
     *         <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
     *         </p>
     *         <p>
     *         The following are descriptions of each comparison operator.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code>
     *         element of a different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
     *         <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and maps.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number,
     *         Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
     *         <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>LE</code> : Less than or equal.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
     *         compare to <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>LT</code> : Less than.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number,
     *         or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
     *         type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code>
     *         does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
     *         <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>GE</code> : Greater than or equal.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
     *         compare to <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>GT</code> : Greater than.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
     *         compare to <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
     *         including lists and maps.
     *         </p>
     *         <note>
     *         <p>
     *         This operator tests for the existence of an attribute, not its data type. If the data type of attribute "
     *         <code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a Boolean
     *         <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type is not
     *         relevant to the <code>NOT_NULL</code> comparison operator.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
     *         including lists and maps.
     *         </p>
     *         <note>
     *         <p>
     *         This operator tests for the nonexistence of an attribute, not its data type. If the data type of
     *         attribute "<code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a Boolean
     *         <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is not relevant
     *         to the <code>NULL</code> comparison operator.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If the target attribute of the comparison is of type String, then the
     *         operator checks for a substring match. If the target attribute of the comparison is of type Binary, then
     *         the operator looks for a subsequence of the target that matches the input. If the target attribute of the
     *         comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator
     *         evaluates to true if it finds an exact match with any member of the set.
     *         </p>
     *         <p>
     *         CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
     *         " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If the target attribute of the comparison is a String, then the
     *         operator checks for the absence of a substring match. If the target attribute of the comparison is
     *         Binary, then the operator checks for the absence of a subsequence of the target that matches the input.
     *         If the target attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "
     *         <code>BS</code>"), then the operator evaluates to true if it <i>does not</i> find an exact match with any
     *         member of the set.
     *         </p>
     *         <p>
     *         NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
     *         " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BEGINS_WITH</code> : Checks for a prefix.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or Binary
     *         (not a Number or a set type). The target attribute of the comparison must be of type String or Binary
     *         (not a Number or a set type).
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>IN</code> : Checks for matching elements in a list.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type
     *         String, Number, or Binary. These attributes are compared against an existing attribute of an item. If any
     *         elements of the input are equal to the item attribute, the expression evaluates to true.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second
     *         value.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same type,
     *         either String, Number, or Binary (not a set type). A target attribute matches if the target value is
     *         greater than, or equal to, the first element and less than, or equal to, the second element. If an item
     *         contains an <code>AttributeValue</code> element of a different type than the one provided in the request,
     *         the value does not match. For example, <code>{"S":"6"}</code> does not compare to <code>{"N":"6"}</code>.
     *         Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For usage examples of <code>AttributeValueList</code> and <code>ComparisonOperator</code>, see <a
     *         href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.html"
     *         >Legacy Conditional Parameters</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * @see ComparisonOperator
     */
    public String comparisonOperator() {
        return comparisonOperator;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((attributeValueList() == null) ? 0 : attributeValueList().hashCode());
        hashCode = 31 * hashCode + ((comparisonOperator() == null) ? 0 : comparisonOperator().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Condition)) {
            return false;
        }
        Condition other = (Condition) obj;
        if (other.attributeValueList() == null ^ this.attributeValueList() == null) {
            return false;
        }
        if (other.attributeValueList() != null && !other.attributeValueList().equals(this.attributeValueList())) {
            return false;
        }
        if (other.comparisonOperator() == null ^ this.comparisonOperator() == null) {
            return false;
        }
        if (other.comparisonOperator() != null && !other.comparisonOperator().equals(this.comparisonOperator())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (attributeValueList() != null) {
            sb.append("AttributeValueList: ").append(attributeValueList()).append(",");
        }
        if (comparisonOperator() != null) {
            sb.append("ComparisonOperator: ").append(comparisonOperator()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ConditionMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Condition> {
        /**
         * <p>
         * One or more values to evaluate against the supplied attribute. The number of values in the list depends on
         * the <code>ComparisonOperator</code> being used.
         * </p>
         * <p>
         * For type Number, value comparisons are numeric.
         * </p>
         * <p>
         * String value comparisons for greater than, equals, or less than are based on ASCII character code values. For
         * example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than <code>B</code>.
         * For a list of code values, see <a
         * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">http://
         * en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         * </p>
         * <p>
         * For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
         * </p>
         * 
         * @param attributeValueList
         *        One or more values to evaluate against the supplied attribute. The number of values in the list
         *        depends on the <code>ComparisonOperator</code> being used.</p>
         *        <p>
         *        For type Number, value comparisons are numeric.
         *        </p>
         *        <p>
         *        String value comparisons for greater than, equals, or less than are based on ASCII character code
         *        values. For example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than
         *        <code>B</code>. For a list of code values, see <a
         *        href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
         *        >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         *        </p>
         *        <p>
         *        For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeValueList(Collection<AttributeValue> attributeValueList);

        /**
         * <p>
         * One or more values to evaluate against the supplied attribute. The number of values in the list depends on
         * the <code>ComparisonOperator</code> being used.
         * </p>
         * <p>
         * For type Number, value comparisons are numeric.
         * </p>
         * <p>
         * String value comparisons for greater than, equals, or less than are based on ASCII character code values. For
         * example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than <code>B</code>.
         * For a list of code values, see <a
         * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">http://
         * en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         * </p>
         * <p>
         * For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
         * </p>
         * 
         * @param attributeValueList
         *        One or more values to evaluate against the supplied attribute. The number of values in the list
         *        depends on the <code>ComparisonOperator</code> being used.</p>
         *        <p>
         *        For type Number, value comparisons are numeric.
         *        </p>
         *        <p>
         *        String value comparisons for greater than, equals, or less than are based on ASCII character code
         *        values. For example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than
         *        <code>B</code>. For a list of code values, see <a
         *        href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
         *        >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         *        </p>
         *        <p>
         *        For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeValueList(AttributeValue... attributeValueList);

        /**
         * <p>
         * A comparator for evaluating attributes. For example, equals, greater than, less than, etc.
         * </p>
         * <p>
         * The following comparison operators are available:
         * </p>
         * <p>
         * <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
         * </p>
         * <p>
         * The following are descriptions of each comparison operator.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code>
         * element of a different type than the one provided in the request, the value does not match. For example,
         * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and maps.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number,
         * Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> of a
         * different type than the one provided in the request, the value does not match. For example,
         * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>LE</code> : Less than or equal.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>LT</code> : Less than.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number, or
         * Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
         * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
         * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>GE</code> : Greater than or equal.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>GT</code> : Greater than.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
         * including lists and maps.
         * </p>
         * <note>
         * <p>
         * This operator tests for the existence of an attribute, not its data type. If the data type of attribute "
         * <code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a Boolean
         * <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type is not
         * relevant to the <code>NOT_NULL</code> comparison operator.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
         * including lists and maps.
         * </p>
         * <note>
         * <p>
         * This operator tests for the nonexistence of an attribute, not its data type. If the data type of attribute "
         * <code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a Boolean
         * <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is not relevant to
         * the <code>NULL</code> comparison operator.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If the target attribute of the comparison is of type String, then the
         * operator checks for a substring match. If the target attribute of the comparison is of type Binary, then the
         * operator looks for a subsequence of the target that matches the input. If the target attribute of the
         * comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator evaluates
         * to true if it finds an exact match with any member of the set.
         * </p>
         * <p>
         * CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
         * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If the target attribute of the comparison is a String, then the operator
         * checks for the absence of a substring match. If the target attribute of the comparison is Binary, then the
         * operator checks for the absence of a subsequence of the target that matches the input. If the target
         * attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the
         * operator evaluates to true if it <i>does not</i> find an exact match with any member of the set.
         * </p>
         * <p>
         * NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
         * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BEGINS_WITH</code> : Checks for a prefix.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or Binary
         * (not a Number or a set type). The target attribute of the comparison must be of type String or Binary (not a
         * Number or a set type).
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>IN</code> : Checks for matching elements in a list.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type String,
         * Number, or Binary. These attributes are compared against an existing attribute of an item. If any elements of
         * the input are equal to the item attribute, the expression evaluates to true.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second value.
         * </p>
         * <p>
         * <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same type,
         * either String, Number, or Binary (not a set type). A target attribute matches if the target value is greater
         * than, or equal to, the first element and less than, or equal to, the second element. If an item contains an
         * <code>AttributeValue</code> element of a different type than the one provided in the request, the value does
         * not match. For example, <code>{"S":"6"}</code> does not compare to <code>{"N":"6"}</code>. Also,
         * <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For usage examples of <code>AttributeValueList</code> and <code>ComparisonOperator</code>, see <a
         * href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.html"
         * >Legacy Conditional Parameters</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param comparisonOperator
         *        A comparator for evaluating attributes. For example, equals, greater than, less than, etc.</p>
         *        <p>
         *        The following comparison operators are available:
         *        </p>
         *        <p>
         *        <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
         *        </p>
         *        <p>
         *        The following are descriptions of each comparison operator.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, Binary, String Set, Number Set, or Binary Set. If an item contains an
         *        <code>AttributeValue</code> element of a different type than the one provided in the request, the
         *        value does not match. For example, <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also,
         *        <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and
         *        maps.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String,
         *        Number, Binary, String Set, Number Set, or Binary Set. If an item contains an
         *        <code>AttributeValue</code> of a different type than the one provided in the request, the value does
         *        not match. For example, <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also,
         *        <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>LE</code> : Less than or equal.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>LT</code> : Less than.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String,
         *        Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
         *        different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>GE</code> : Greater than or equal.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>GT</code> : Greater than.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
         *        including lists and maps.
         *        </p>
         *        <note>
         *        <p>
         *        This operator tests for the existence of an attribute, not its data type. If the data type of
         *        attribute "<code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a
         *        Boolean <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type
         *        is not relevant to the <code>NOT_NULL</code> comparison operator.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
         *        including lists and maps.
         *        </p>
         *        <note>
         *        <p>
         *        This operator tests for the nonexistence of an attribute, not its data type. If the data type of
         *        attribute "<code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a
         *        Boolean <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is
         *        not relevant to the <code>NULL</code> comparison operator.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If the target attribute of the comparison is of type
         *        String, then the operator checks for a substring match. If the target attribute of the comparison is
         *        of type Binary, then the operator looks for a subsequence of the target that matches the input. If the
         *        target attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>
         *        "), then the operator evaluates to true if it finds an exact match with any member of the set.
         *        </p>
         *        <p>
         *        CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
         *        " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If the target attribute of the comparison is a String,
         *        then the operator checks for the absence of a substring match. If the target attribute of the
         *        comparison is Binary, then the operator checks for the absence of a subsequence of the target that
         *        matches the input. If the target attribute of the comparison is a set ("<code>SS</code>", "
         *        <code>NS</code>", or "<code>BS</code>"), then the operator evaluates to true if it <i>does not</i>
         *        find an exact match with any member of the set.
         *        </p>
         *        <p>
         *        NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
         *        " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BEGINS_WITH</code> : Checks for a prefix.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or
         *        Binary (not a Number or a set type). The target attribute of the comparison must be of type String or
         *        Binary (not a Number or a set type).
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>IN</code> : Checks for matching elements in a list.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type
         *        String, Number, or Binary. These attributes are compared against an existing attribute of an item. If
         *        any elements of the input are equal to the item attribute, the expression evaluates to true.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second
         *        value.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same
         *        type, either String, Number, or Binary (not a set type). A target attribute matches if the target
         *        value is greater than, or equal to, the first element and less than, or equal to, the second element.
         *        If an item contains an <code>AttributeValue</code> element of a different type than the one provided
         *        in the request, the value does not match. For example, <code>{"S":"6"}</code> does not compare to
         *        <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         *        <code>{"NS":["6", "2", "1"]}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For usage examples of <code>AttributeValueList</code> and <code>ComparisonOperator</code>, see <a
         *        href=
         *        "http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.html"
         *        >Legacy Conditional Parameters</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(String comparisonOperator);

        /**
         * <p>
         * A comparator for evaluating attributes. For example, equals, greater than, less than, etc.
         * </p>
         * <p>
         * The following comparison operators are available:
         * </p>
         * <p>
         * <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
         * </p>
         * <p>
         * The following are descriptions of each comparison operator.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code>
         * element of a different type than the one provided in the request, the value does not match. For example,
         * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and maps.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number,
         * Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> of a
         * different type than the one provided in the request, the value does not match. For example,
         * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>LE</code> : Less than or equal.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>LT</code> : Less than.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number, or
         * Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
         * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
         * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>GE</code> : Greater than or equal.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>GT</code> : Greater than.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
         * including lists and maps.
         * </p>
         * <note>
         * <p>
         * This operator tests for the existence of an attribute, not its data type. If the data type of attribute "
         * <code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a Boolean
         * <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type is not
         * relevant to the <code>NOT_NULL</code> comparison operator.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
         * including lists and maps.
         * </p>
         * <note>
         * <p>
         * This operator tests for the nonexistence of an attribute, not its data type. If the data type of attribute "
         * <code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a Boolean
         * <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is not relevant to
         * the <code>NULL</code> comparison operator.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If the target attribute of the comparison is of type String, then the
         * operator checks for a substring match. If the target attribute of the comparison is of type Binary, then the
         * operator looks for a subsequence of the target that matches the input. If the target attribute of the
         * comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator evaluates
         * to true if it finds an exact match with any member of the set.
         * </p>
         * <p>
         * CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
         * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If the target attribute of the comparison is a String, then the operator
         * checks for the absence of a substring match. If the target attribute of the comparison is Binary, then the
         * operator checks for the absence of a subsequence of the target that matches the input. If the target
         * attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the
         * operator evaluates to true if it <i>does not</i> find an exact match with any member of the set.
         * </p>
         * <p>
         * NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
         * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BEGINS_WITH</code> : Checks for a prefix.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or Binary
         * (not a Number or a set type). The target attribute of the comparison must be of type String or Binary (not a
         * Number or a set type).
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>IN</code> : Checks for matching elements in a list.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type String,
         * Number, or Binary. These attributes are compared against an existing attribute of an item. If any elements of
         * the input are equal to the item attribute, the expression evaluates to true.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second value.
         * </p>
         * <p>
         * <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same type,
         * either String, Number, or Binary (not a set type). A target attribute matches if the target value is greater
         * than, or equal to, the first element and less than, or equal to, the second element. If an item contains an
         * <code>AttributeValue</code> element of a different type than the one provided in the request, the value does
         * not match. For example, <code>{"S":"6"}</code> does not compare to <code>{"N":"6"}</code>. Also,
         * <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For usage examples of <code>AttributeValueList</code> and <code>ComparisonOperator</code>, see <a
         * href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.html"
         * >Legacy Conditional Parameters</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param comparisonOperator
         *        A comparator for evaluating attributes. For example, equals, greater than, less than, etc.</p>
         *        <p>
         *        The following comparison operators are available:
         *        </p>
         *        <p>
         *        <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
         *        </p>
         *        <p>
         *        The following are descriptions of each comparison operator.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, Binary, String Set, Number Set, or Binary Set. If an item contains an
         *        <code>AttributeValue</code> element of a different type than the one provided in the request, the
         *        value does not match. For example, <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also,
         *        <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and
         *        maps.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String,
         *        Number, Binary, String Set, Number Set, or Binary Set. If an item contains an
         *        <code>AttributeValue</code> of a different type than the one provided in the request, the value does
         *        not match. For example, <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also,
         *        <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>LE</code> : Less than or equal.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>LT</code> : Less than.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String,
         *        Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
         *        different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>GE</code> : Greater than or equal.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>GT</code> : Greater than.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
         *        including lists and maps.
         *        </p>
         *        <note>
         *        <p>
         *        This operator tests for the existence of an attribute, not its data type. If the data type of
         *        attribute "<code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a
         *        Boolean <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type
         *        is not relevant to the <code>NOT_NULL</code> comparison operator.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
         *        including lists and maps.
         *        </p>
         *        <note>
         *        <p>
         *        This operator tests for the nonexistence of an attribute, not its data type. If the data type of
         *        attribute "<code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a
         *        Boolean <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is
         *        not relevant to the <code>NULL</code> comparison operator.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If the target attribute of the comparison is of type
         *        String, then the operator checks for a substring match. If the target attribute of the comparison is
         *        of type Binary, then the operator looks for a subsequence of the target that matches the input. If the
         *        target attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>
         *        "), then the operator evaluates to true if it finds an exact match with any member of the set.
         *        </p>
         *        <p>
         *        CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
         *        " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If the target attribute of the comparison is a String,
         *        then the operator checks for the absence of a substring match. If the target attribute of the
         *        comparison is Binary, then the operator checks for the absence of a subsequence of the target that
         *        matches the input. If the target attribute of the comparison is a set ("<code>SS</code>", "
         *        <code>NS</code>", or "<code>BS</code>"), then the operator evaluates to true if it <i>does not</i>
         *        find an exact match with any member of the set.
         *        </p>
         *        <p>
         *        NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
         *        " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BEGINS_WITH</code> : Checks for a prefix.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or
         *        Binary (not a Number or a set type). The target attribute of the comparison must be of type String or
         *        Binary (not a Number or a set type).
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>IN</code> : Checks for matching elements in a list.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type
         *        String, Number, or Binary. These attributes are compared against an existing attribute of an item. If
         *        any elements of the input are equal to the item attribute, the expression evaluates to true.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second
         *        value.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same
         *        type, either String, Number, or Binary (not a set type). A target attribute matches if the target
         *        value is greater than, or equal to, the first element and less than, or equal to, the second element.
         *        If an item contains an <code>AttributeValue</code> element of a different type than the one provided
         *        in the request, the value does not match. For example, <code>{"S":"6"}</code> does not compare to
         *        <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         *        <code>{"NS":["6", "2", "1"]}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For usage examples of <code>AttributeValueList</code> and <code>ComparisonOperator</code>, see <a
         *        href=
         *        "http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.html"
         *        >Legacy Conditional Parameters</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(ComparisonOperator comparisonOperator);
    }

    private static final class BuilderImpl implements Builder {
        private List<AttributeValue> attributeValueList;

        private String comparisonOperator;

        private BuilderImpl() {
        }

        private BuilderImpl(Condition model) {
            setAttributeValueList(model.attributeValueList);
            setComparisonOperator(model.comparisonOperator);
        }

        public final Collection<AttributeValue> getAttributeValueList() {
            return attributeValueList;
        }

        @Override
        public final Builder attributeValueList(Collection<AttributeValue> attributeValueList) {
            this.attributeValueList = AttributeValueListCopier.copy(attributeValueList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributeValueList(AttributeValue... attributeValueList) {
            attributeValueList(Arrays.asList(attributeValueList));
            return this;
        }

        public final void setAttributeValueList(Collection<AttributeValue> attributeValueList) {
            this.attributeValueList = AttributeValueListCopier.copy(attributeValueList);
        }

        public final String getComparisonOperator() {
            return comparisonOperator;
        }

        @Override
        public final Builder comparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
            return this;
        }

        @Override
        public final Builder comparisonOperator(ComparisonOperator comparisonOperator) {
            this.comparisonOperator(comparisonOperator.toString());
            return this;
        }

        public final void setComparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
        }

        @Override
        public Condition build() {
            return new Condition(this);
        }
    }
}
