/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.dynamodb.transform.KeySchemaElementMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents <i>a single element</i> of a key schema. A key schema specifies the attributes that make up the primary
 * key of a table, or the key attributes of an index.
 * </p>
 * <p>
 * A <code>KeySchemaElement</code> represents exactly one attribute of the primary key. For example, a simple primary
 * key would be represented by one <code>KeySchemaElement</code> (for the partition key). A composite primary key would
 * require one <code>KeySchemaElement</code> for the partition key, and another <code>KeySchemaElement</code> for the
 * sort key.
 * </p>
 * <p>
 * A <code>KeySchemaElement</code> must be a scalar, top-level attribute (not a nested attribute). The data type must be
 * one of String, Number, or Binary. The attribute cannot be nested within a List or a Map.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class KeySchemaElement implements StructuredPojo, ToCopyableBuilder<KeySchemaElement.Builder, KeySchemaElement> {
    private final String attributeName;

    private final String keyType;

    private KeySchemaElement(BuilderImpl builder) {
        this.attributeName = builder.attributeName;
        this.keyType = builder.keyType;
    }

    /**
     * <p>
     * The name of a key attribute.
     * </p>
     * 
     * @return The name of a key attribute.
     */
    public String attributeName() {
        return attributeName;
    }

    /**
     * <p>
     * The role that this key attribute will assume:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>HASH</code> - partition key
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RANGE</code> - sort key
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives from
     * DynamoDB' usage of an internal hash function to evenly distribute data items across partitions, based on their
     * partition key values.
     * </p>
     * <p>
     * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from the
     * way DynamoDB stores items with the same partition key physically close together, in sorted order by the sort key
     * value.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyType} will
     * return {@link KeyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyTypeString}.
     * </p>
     * 
     * @return The role that this key attribute will assume:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>HASH</code> - partition key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RANGE</code> - sort key
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
     *         derives from DynamoDB' usage of an internal hash function to evenly distribute data items across
     *         partitions, based on their partition key values.
     *         </p>
     *         <p>
     *         The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives
     *         from the way DynamoDB stores items with the same partition key physically close together, in sorted order
     *         by the sort key value.
     *         </p>
     * @see KeyType
     */
    public KeyType keyType() {
        return KeyType.fromValue(keyType);
    }

    /**
     * <p>
     * The role that this key attribute will assume:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>HASH</code> - partition key
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RANGE</code> - sort key
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives from
     * DynamoDB' usage of an internal hash function to evenly distribute data items across partitions, based on their
     * partition key values.
     * </p>
     * <p>
     * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from the
     * way DynamoDB stores items with the same partition key physically close together, in sorted order by the sort key
     * value.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyType} will
     * return {@link KeyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyTypeString}.
     * </p>
     * 
     * @return The role that this key attribute will assume:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>HASH</code> - partition key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RANGE</code> - sort key
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
     *         derives from DynamoDB' usage of an internal hash function to evenly distribute data items across
     *         partitions, based on their partition key values.
     *         </p>
     *         <p>
     *         The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives
     *         from the way DynamoDB stores items with the same partition key physically close together, in sorted order
     *         by the sort key value.
     *         </p>
     * @see KeyType
     */
    public String keyTypeString() {
        return keyType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(attributeName());
        hashCode = 31 * hashCode + Objects.hashCode(keyTypeString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KeySchemaElement)) {
            return false;
        }
        KeySchemaElement other = (KeySchemaElement) obj;
        return Objects.equals(attributeName(), other.attributeName()) && Objects.equals(keyTypeString(), other.keyTypeString());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (attributeName() != null) {
            sb.append("AttributeName: ").append(attributeName()).append(",");
        }
        if (keyTypeString() != null) {
            sb.append("KeyType: ").append(keyTypeString()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AttributeName":
            return Optional.of(clazz.cast(attributeName()));
        case "KeyType":
            return Optional.of(clazz.cast(keyTypeString()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        KeySchemaElementMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, KeySchemaElement> {
        /**
         * <p>
         * The name of a key attribute.
         * </p>
         * 
         * @param attributeName
         *        The name of a key attribute.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeName(String attributeName);

        /**
         * <p>
         * The role that this key attribute will assume:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB' usage of an internal hash function to evenly distribute data items across partitions, based on
         * their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note>
         * 
         * @param keyType
         *        The role that this key attribute will assume:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HASH</code> - partition key
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RANGE</code> - sort key
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
         *        derives from DynamoDB' usage of an internal hash function to evenly distribute data items across
         *        partitions, based on their partition key values.
         *        </p>
         *        <p>
         *        The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute"
         *        derives from the way DynamoDB stores items with the same partition key physically close together, in
         *        sorted order by the sort key value.
         *        </p>
         * @see KeyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyType
         */
        Builder keyType(String keyType);

        /**
         * <p>
         * The role that this key attribute will assume:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB' usage of an internal hash function to evenly distribute data items across partitions, based on
         * their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note>
         * 
         * @param keyType
         *        The role that this key attribute will assume:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HASH</code> - partition key
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RANGE</code> - sort key
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
         *        derives from DynamoDB' usage of an internal hash function to evenly distribute data items across
         *        partitions, based on their partition key values.
         *        </p>
         *        <p>
         *        The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute"
         *        derives from the way DynamoDB stores items with the same partition key physically close together, in
         *        sorted order by the sort key value.
         *        </p>
         * @see KeyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyType
         */
        Builder keyType(KeyType keyType);
    }

    static final class BuilderImpl implements Builder {
        private String attributeName;

        private String keyType;

        private BuilderImpl() {
        }

        private BuilderImpl(KeySchemaElement model) {
            attributeName(model.attributeName);
            keyType(model.keyType);
        }

        public final String getAttributeName() {
            return attributeName;
        }

        @Override
        public final Builder attributeName(String attributeName) {
            this.attributeName = attributeName;
            return this;
        }

        public final void setAttributeName(String attributeName) {
            this.attributeName = attributeName;
        }

        public final String getKeyType() {
            return keyType;
        }

        @Override
        public final Builder keyType(String keyType) {
            this.keyType = keyType;
            return this;
        }

        @Override
        public final Builder keyType(KeyType keyType) {
            this.keyType(keyType.toString());
            return this;
        }

        public final void setKeyType(String keyType) {
            this.keyType = keyType;
        }

        @Override
        public KeySchemaElement build() {
            return new KeySchemaElement(this);
        }
    }
}
