/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodbstreams;

import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.AmazonServiceException;
import software.amazon.awssdk.core.SdkBaseException;
import software.amazon.awssdk.core.SdkClientException;
import software.amazon.awssdk.core.client.ClientExecutionParams;
import software.amazon.awssdk.core.client.ClientHandler;
import software.amazon.awssdk.core.client.SdkClientHandler;
import software.amazon.awssdk.core.config.ClientConfiguration;
import software.amazon.awssdk.core.config.SyncClientConfiguration;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.protocol.json.JsonClientMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorResponseMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorShapeMetadata;
import software.amazon.awssdk.core.protocol.json.JsonOperationMetadata;
import software.amazon.awssdk.core.protocol.json.SdkJsonProtocolFactory;
import software.amazon.awssdk.services.dynamodb.model.DescribeStreamRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeStreamResponse;
import software.amazon.awssdk.services.dynamodb.model.DynamoDBException;
import software.amazon.awssdk.services.dynamodb.model.ExpiredIteratorException;
import software.amazon.awssdk.services.dynamodb.model.GetRecordsRequest;
import software.amazon.awssdk.services.dynamodb.model.GetRecordsResponse;
import software.amazon.awssdk.services.dynamodb.model.GetShardIteratorRequest;
import software.amazon.awssdk.services.dynamodb.model.GetShardIteratorResponse;
import software.amazon.awssdk.services.dynamodb.model.InternalServerErrorException;
import software.amazon.awssdk.services.dynamodb.model.LimitExceededException;
import software.amazon.awssdk.services.dynamodb.model.ListStreamsRequest;
import software.amazon.awssdk.services.dynamodb.model.ListStreamsResponse;
import software.amazon.awssdk.services.dynamodb.model.ResourceNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.TrimmedDataAccessException;
import software.amazon.awssdk.services.dynamodb.transform.DescribeStreamResponseUnmarshaller;
import software.amazon.awssdk.services.dynamodb.transform.GetRecordsResponseUnmarshaller;
import software.amazon.awssdk.services.dynamodb.transform.GetShardIteratorResponseUnmarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ListStreamsResponseUnmarshaller;
import software.amazon.awssdk.services.dynamodbstreams.paginators.DescribeStreamPaginator;
import software.amazon.awssdk.services.dynamodbstreams.paginators.ListStreamsPaginator;
import software.amazon.awssdk.services.dynamodbstreams.transform.DescribeStreamRequestMarshaller;
import software.amazon.awssdk.services.dynamodbstreams.transform.GetRecordsRequestMarshaller;
import software.amazon.awssdk.services.dynamodbstreams.transform.GetShardIteratorRequestMarshaller;
import software.amazon.awssdk.services.dynamodbstreams.transform.ListStreamsRequestMarshaller;

/**
 * Internal implementation of {@link DynamoDBStreamsClient}.
 *
 * @see DynamoDBStreamsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDynamoDBStreamsClient implements DynamoDBStreamsClient {
    private final ClientHandler clientHandler;

    private final SdkJsonProtocolFactory protocolFactory;

    private final ClientConfiguration clientConfiguration;

    protected DefaultDynamoDBStreamsClient(SyncClientConfiguration clientConfiguration) {
        this.clientHandler = new SdkClientHandler(clientConfiguration, null);
        this.protocolFactory = init();
        this.clientConfiguration = clientConfiguration;
    }

    /**
     * <p>
     * Returns information about a stream, including the current status of the stream, its Amazon Resource Name (ARN),
     * the composition of its shards, and its corresponding DynamoDB table.
     * </p>
     * <note>
     * <p>
     * You can call <code>DescribeStream</code> at a maximum rate of 10 times per second.
     * </p>
     * </note>
     * <p>
     * Each shard in the stream has a <code>SequenceNumberRange</code> associated with it. If the
     * <code>SequenceNumberRange</code> has a <code>StartingSequenceNumber</code> but no
     * <code>EndingSequenceNumber</code>, then the shard is still open (able to receive more stream records). If both
     * <code>StartingSequenceNumber</code> and <code>EndingSequenceNumber</code> are present, then that shard is closed
     * and can no longer receive more data.
     * </p>
     *
     * @param describeStreamRequest
     *        Represents the input of a <code>DescribeStream</code> operation.
     * @return Result of the DescribeStream operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent stream.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDBStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDBStreamsClient.DescribeStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/streams-dynamodb-2012-08-10/DescribeStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeStreamResponse describeStream(DescribeStreamRequest describeStreamRequest) throws ResourceNotFoundException,
            InternalServerErrorException, SdkBaseException, SdkClientException, DynamoDBException {

        HttpResponseHandler<DescribeStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new DescribeStreamResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeStreamRequest, DescribeStreamResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(describeStreamRequest).withMarshaller(new DescribeStreamRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns information about a stream, including the current status of the stream, its Amazon Resource Name (ARN),
     * the composition of its shards, and its corresponding DynamoDB table.
     * </p>
     * <note>
     * <p>
     * You can call <code>DescribeStream</code> at a maximum rate of 10 times per second.
     * </p>
     * </note>
     * <p>
     * Each shard in the stream has a <code>SequenceNumberRange</code> associated with it. If the
     * <code>SequenceNumberRange</code> has a <code>StartingSequenceNumber</code> but no
     * <code>EndingSequenceNumber</code>, then the shard is still open (able to receive more stream records). If both
     * <code>StartingSequenceNumber</code> and <code>EndingSequenceNumber</code> are present, then that shard is closed
     * and can no longer receive more data.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeStream(software.amazon.awssdk.services.dynamodb.model.DescribeStreamRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dynamodbstreams.paginators.DescribeStreamPaginator responses = client.describeStreamIterable(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.dynamodbstreams.paginators.DescribeStreamPaginator responses = client
     *             .describeStreamIterable(request);
     *     for (software.amazon.awssdk.services.dynamodb.model.DescribeStreamResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dynamodbstreams.paginators.DescribeStreamPaginator responses = client.describeStreamIterable(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeStream(software.amazon.awssdk.services.dynamodb.model.DescribeStreamRequest)} operation.</b>
     * </p>
     *
     * @param describeStreamRequest
     *        Represents the input of a <code>DescribeStream</code> operation.
     * @return Result of the DescribeStream operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent stream.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDBStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDBStreamsClient.DescribeStream
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/streams-dynamodb-2012-08-10/DescribeStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeStreamPaginator describeStreamIterable(DescribeStreamRequest describeStreamRequest)
            throws ResourceNotFoundException, InternalServerErrorException, SdkBaseException, SdkClientException,
            DynamoDBException {
        return new DescribeStreamPaginator(this, describeStreamRequest);
    }

    /**
     * <p>
     * Retrieves the stream records from a given shard.
     * </p>
     * <p>
     * Specify a shard iterator using the <code>ShardIterator</code> parameter. The shard iterator specifies the
     * position in the shard from which you want to start reading stream records sequentially. If there are no stream
     * records available in the portion of the shard that the iterator points to, <code>GetRecords</code> returns an
     * empty list. Note that it might take multiple calls to get to a portion of the shard that contains stream records.
     * </p>
     * <note>
     * <p>
     * <code>GetRecords</code> can retrieve a maximum of 1 MB of data or 1000 stream records, whichever comes first.
     * </p>
     * </note>
     *
     * @param getRecordsRequest
     *        Represents the input of a <code>GetRecords</code> operation.
     * @return Result of the GetRecords operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent stream.
     * @throws LimitExceededException
     *         Your request rate is too high. The AWS SDKs for DynamoDB automatically retry requests that receive this
     *         exception. Your request is eventually successful, unless your retry queue is too large to finish. Reduce
     *         the frequency of requests and use exponential backoff. For more information, go to <a
     *         href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ErrorHandling.html#APIRetries"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws ExpiredIteratorException
     *         The shard iterator has expired and can no longer be used to retrieve stream records. A shard iterator
     *         expires 15 minutes after it is retrieved using the <code>GetShardIterator</code> action.
     * @throws TrimmedDataAccessException
     *         The operation attempted to read past the oldest stream record in a shard.</p>
     *         <p>
     *         In DynamoDB Streams, there is a 24 hour limit on data retention. Stream records whose age exceeds this
     *         limit are subject to removal (trimming) from the stream. You might receive a TrimmedDataAccessException
     *         if:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You request a shard iterator with a sequence number older than the trim point (24 hours).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You obtain a shard iterator, but before you use the iterator in a <code>GetRecords</code> request, a
     *         stream record in the shard exceeds the 24 hour period and is trimmed. This causes the iterator to access
     *         a record that no longer exists.
     *         </p>
     *         </li>
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDBStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDBStreamsClient.GetRecords
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/streams-dynamodb-2012-08-10/GetRecords" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetRecordsResponse getRecords(GetRecordsRequest getRecordsRequest) throws ResourceNotFoundException,
            LimitExceededException, InternalServerErrorException, ExpiredIteratorException, TrimmedDataAccessException,
            SdkBaseException, SdkClientException, DynamoDBException {

        HttpResponseHandler<GetRecordsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetRecordsResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetRecordsRequest, GetRecordsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(getRecordsRequest)
                .withMarshaller(new GetRecordsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a shard iterator. A shard iterator provides information about how to retrieve the stream records from
     * within a shard. Use the shard iterator in a subsequent <code>GetRecords</code> request to read the stream records
     * from the shard.
     * </p>
     * <note>
     * <p>
     * A shard iterator expires 15 minutes after it is returned to the requester.
     * </p>
     * </note>
     *
     * @param getShardIteratorRequest
     *        Represents the input of a <code>GetShardIterator</code> operation.
     * @return Result of the GetShardIterator operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent stream.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws TrimmedDataAccessException
     *         The operation attempted to read past the oldest stream record in a shard.</p>
     *         <p>
     *         In DynamoDB Streams, there is a 24 hour limit on data retention. Stream records whose age exceeds this
     *         limit are subject to removal (trimming) from the stream. You might receive a TrimmedDataAccessException
     *         if:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You request a shard iterator with a sequence number older than the trim point (24 hours).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You obtain a shard iterator, but before you use the iterator in a <code>GetRecords</code> request, a
     *         stream record in the shard exceeds the 24 hour period and is trimmed. This causes the iterator to access
     *         a record that no longer exists.
     *         </p>
     *         </li>
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDBStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDBStreamsClient.GetShardIterator
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/streams-dynamodb-2012-08-10/GetShardIterator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetShardIteratorResponse getShardIterator(GetShardIteratorRequest getShardIteratorRequest)
            throws ResourceNotFoundException, InternalServerErrorException, TrimmedDataAccessException, SdkBaseException,
            SdkClientException, DynamoDBException {

        HttpResponseHandler<GetShardIteratorResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new GetShardIteratorResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetShardIteratorRequest, GetShardIteratorResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(getShardIteratorRequest).withMarshaller(new GetShardIteratorRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns an array of stream ARNs associated with the current account and endpoint. If the <code>TableName</code>
     * parameter is present, then <code>ListStreams</code> will return only the streams ARNs for that table.
     * </p>
     * <note>
     * <p>
     * You can call <code>ListStreams</code> at a maximum rate of 5 times per second.
     * </p>
     * </note>
     *
     * @param listStreamsRequest
     *        Represents the input of a <code>ListStreams</code> operation.
     * @return Result of the ListStreams operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent stream.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDBStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDBStreamsClient.ListStreams
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/streams-dynamodb-2012-08-10/ListStreams" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListStreamsResponse listStreams(ListStreamsRequest listStreamsRequest) throws ResourceNotFoundException,
            InternalServerErrorException, SdkBaseException, SdkClientException, DynamoDBException {

        HttpResponseHandler<ListStreamsResponse> responseHandler = protocolFactory.createResponseHandler(
                new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                new ListStreamsResponseUnmarshaller());

        HttpResponseHandler<AmazonServiceException> errorResponseHandler = createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListStreamsRequest, ListStreamsResponse>()
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                .withInput(listStreamsRequest).withMarshaller(new ListStreamsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns an array of stream ARNs associated with the current account and endpoint. If the <code>TableName</code>
     * parameter is present, then <code>ListStreams</code> will return only the streams ARNs for that table.
     * </p>
     * <note>
     * <p>
     * You can call <code>ListStreams</code> at a maximum rate of 5 times per second.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of {@link #listStreams(software.amazon.awssdk.services.dynamodb.model.ListStreamsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dynamodbstreams.paginators.ListStreamsPaginator responses = client.listStreamsIterable(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.dynamodbstreams.paginators.ListStreamsPaginator responses = client
     *             .listStreamsIterable(request);
     *     for (software.amazon.awssdk.services.dynamodb.model.ListStreamsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dynamodbstreams.paginators.ListStreamsPaginator responses = client.listStreamsIterable(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStreams(software.amazon.awssdk.services.dynamodb.model.ListStreamsRequest)} operation.</b>
     * </p>
     *
     * @param listStreamsRequest
     *        Represents the input of a <code>ListStreams</code> operation.
     * @return Result of the ListStreams operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent stream.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkBaseException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDBStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDBStreamsClient.ListStreams
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/streams-dynamodb-2012-08-10/ListStreams" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListStreamsPaginator listStreamsIterable(ListStreamsRequest listStreamsRequest) throws ResourceNotFoundException,
            InternalServerErrorException, SdkBaseException, SdkClientException, DynamoDBException {
        return new ListStreamsPaginator(this, listStreamsRequest);
    }

    private HttpResponseHandler<AmazonServiceException> createErrorResponseHandler() {
        return protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());
    }

    private software.amazon.awssdk.core.protocol.json.SdkJsonProtocolFactory init() {
        return new SdkJsonProtocolFactory(new JsonClientMetadata()
                .withProtocolVersion("1.0")
                .withSupportsCbor(false)
                .withSupportsIon(false)
                .withBaseServiceExceptionClass(software.amazon.awssdk.services.dynamodb.model.DynamoDBException.class)
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("TrimmedDataAccessException").withModeledClass(
                                TrimmedDataAccessException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ResourceNotFoundException").withModeledClass(
                                ResourceNotFoundException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("ExpiredIteratorException").withModeledClass(
                                ExpiredIteratorException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("InternalServerError").withModeledClass(
                                InternalServerErrorException.class))
                .addErrorMetadata(
                        new JsonErrorShapeMetadata().withErrorCode("LimitExceededException").withModeledClass(
                                LimitExceededException.class)));
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
